/***************************************************************

	FV_IBMPC.C  --  Fast Video routines
			for Simple Software.

			Implementation for IBM PC or Compatible
			utilizing direct RAM video manipulation.

			This implementation contains code specific
			to the Microsoft Quick C (tm) compiler.

			The use of Simple Software is governed by the
			"Simple Software User Agreement" (see file
			SS.DOC).

			Copyright (c) 1988, Ted A. Campbell

			Bywater Software
			Box 4023
			Duke Station,
			Durham, NC  27706

***************************************************************/

#include "fv.h"
#include "dos.h"
#include "malloc.h"
#include "bios.h"

/*   DEFINITIONS OF IBM MONOCHROME ATTRIBUTE BITS              */

#define   UNDERLINE  1
#define   BRIGHT     8
#define   REVERSE    16
#define   BLINK      128

/*   DEFINITIONS OF IBM COLOR BITS

     Definitions preceded by F indicate Foreground colors, definitions
     preceded by B indicate background colors.

*/

#define   FBLUE      1
#define   FGREEN     2
#define   FRED       4
#define   FWHITE     7

#define   BBLUE      16
#define   BGREEN     32
#define   BRED       64
#define   BWHITE     112

union REGS ibm_registers;      /* structure for passing and receiving
				  register values defined in dos.h     */
char far *ibm_videoram;        /* stores the address of the VIDEO RAM */
char adapter;                  /* stores the video adapter in use */

#define  BDOS       0x21       /*  BDOS interrupt         */
#define  VIDEOINT   0x10       /*  Video system interrupt */
#define  ERROR      0xff       /*  return error */

#define BIT3        0x08

char iscolor;                  /*  flag denoting color video */

/**************************************************************

	FUNCTION:       fv_init()

	DESCRIPTION:    This function provides any initialization
			necessary for the video system.

	INPUT:          none.

	RETURNS:        none.

	NOTES:          The principal task that fv_init() has
			to accomplish in the PC-compatible
			family is to determine what video
			system is in use, since different systems
			utilize different RAM addresses for the
			beginning of the video RAM.

**************************************************************/

fv_init()
	{

	/* First call ibm_get_ga() to get the graphics adapter */

	adapter = ibm_get_ga();

	/* Now set certain values based on this information */

	switch ( adapter )
		{
		case 0:                   /* IBM Monochrome Video */
			iscolor = 0;
			ibm_videoram = (char far *) 0xb0000000;
			break;
		case 1:                   /* Hercules Graphics Card */
			iscolor = 0;
			ibm_videoram = (char far *) 0xb0000000;
			break;
		case 2:                   /* CGA graphics            */
			iscolor = 1;
			ibm_videoram = (char far *) 0xb8000000;
			break;
		case 3:                  /* EGA: set to CGA mode     */
			iscolor = 1;
			ibm_videoram = (char far *) 0xb8000000;
			break;
		}
	}

/**************************************************************

	FUNCTION:       ibm_get_ga()

	DESCRIPTION:    This function determines the graphics
			adapte currently in use.

	INPUT:          none.

	RETURNS:        This function returns a number indicating
			the graphics adapter detected.  Conventions
			are as follows:

			0 = IBM Monochrome Adapter or Compatible
			1 = Hercules Graphics Card or Compatible
			2 = IBM Color Graphics Adapter or Compatible
			3 = IBM Enhanced Graphics Adapter or Compatible

**************************************************************/

ibm_get_ga()
	{
	if ( ibm_is_ega() == 1 )
		{
		return 3;
		}
	switch ( ibm_video_device() )
		{
		case 0:
			return 2;
			break;
		case 1:
			return 1;
			break;
		case 3:
			return 0;
			break;
		default:
			return 0xff;
			break;
		}
	}

/**************************************************************

	FUNCTION:       ibm_is_ega()

	DESCRIPTION:    This function is called by ibm_get_ga()
			and determines if an EGA card is installed.

	INPUT:          none.

	RETURNS:        The function returns 1 if an EGA card
			is detected, else 0.

**************************************************************/

ibm_is_ega()
	{
	char far *p;
	p = (char far *) 0x00400087;
	if ( *p == 0 )
		{
		return 0;
		}
	else
		{
		return 1;
		}
	}

/**************************************************************

	FUNCTION:       ibm_video_device()

	DESCRIPTION:    This function determines the video
			card for non-EGA PCs.

	INPUT:          none.

	RETURNS:        The function returns one of the following
			values:

			0       CGA compatible graphics
			1       Hercules (tm) compatible graphics
			3       No graphics

**************************************************************/

ibm_video_device()
	{
	int86( 0x11, &ibm_registers, &ibm_registers );
	return ( ( ibm_registers.x.ax & 0x18 ) >> 4 );
	}

/**************************************************************

	FUNCTION:       ibm_get_mode()

	DESCRIPTION:    This function determines the currently
			set video by making a call to the BIOS
			video service.

	INPUT:          none.

	RETURNS:        The function returns an integer repre-
			senting the currently set mode (see a
			DOS technical manual for specific codes).

**************************************************************/

ibm_get_mode()
	{
	ibm_registers.x.ax = 15*256;
	int86( VIDEOINT, &ibm_registers, &ibm_registers );
	return (ibm_registers.x.ax % 256 );
	}

/**************************************************************

	FUNCTION:       ibm_set_mode()

	DESCRIPTION:    This function calls the video BIOS to set
			a requested video mode.

	INPUT:          <mode_to_set> is the mode requested (see
			a DOS technical manual for specific
			information).

	RETURNS:        none.

**************************************************************/

ibm_set_mode( mode_to_set )
	int mode_to_set;
	{
	ibm_registers.x.ax = mode_to_set;
	int86( VIDEOINT, &ibm_registers, &ibm_registers );
	}

/**************************************************************

	FUNCTION:       ibm_getatt()

	DESCRIPTION:    This function converts a Simple Software
			Fast Video (FV) attribute convention (see
			FV.H) into a PC-compatible attribute.

	INPUT:          <fv_att> is the Simple Software Fast Video
			attribute convention input to the
			function (see FV.H).

	RETURNS:        The function returns an attribute which
			can be poked into an attribute position
			in the PC's video RAM.

**************************************************************/

ibm_getatt( fv_att )
	int fv_att;
	{
	if ( iscolor == 1 )
		{
		switch ( fv_att )
			{
			case FV_SPEC1:
				return FWHITE | BBLUE;
			case FV_SPEC2:
				return FBLUE;
			default:
				return FWHITE;
			}
		}
	else
		{
		switch ( fv_att )
			{
			case FV_SPEC1:
				return BWHITE;
			case FV_SPEC2:
				return BWHITE;
			default:
				return FWHITE;
			}
		}
	}

/**************************************************************

	FUNCTION:       fv_deinit()

	DESCRIPTION:    This function should provide any
			deinitialization necessary for the
			video system, i.e., it should return
			the video system to its default state.

	INPUT:          none.

	RETURNS:        none.

**************************************************************/

fv_deinit()
	{
	}

/**************************************************************

	FUNCTION:       fv_cls()

	DESCRIPTION:    This function clears the screen.

	INPUT:          none.

	RETURNS:        none.

**************************************************************/

fv_cls()
	{
	ibm_registers.x.ax = 6*256;
	ibm_registers.x.bx = 0;
	ibm_registers.x.cx = 0;
	ibm_registers.x.dx = 24*256 + 79;
	int86( VIDEOINT, &ibm_registers, &ibm_registers );
	fv_adr( 0, 0 );
	}

/**************************************************************

	FUNCTION:       fv_adr()

	DESCRIPTION:    This function addresses the cursor to a
			specific line and column on the video
			screen.

	INPUT:          <line> and <column> denote the position
			to  which the cursor is to be addressed.

	RETURNS:        none.

**************************************************************/

fv_adr( line, column )
	int line, column;
	{
	ibm_registers.x.ax = 2*256;
	ibm_registers.x.dx = column + (line * 256);
	ibm_registers.x.bx = 0;
	int86( VIDEOINT, &ibm_registers, &ibm_registers );
	}

/**************************************************************

	FUNCTION:       fv_cachr()

	DESCRIPTION:    This function prints a specified character
			on the screen at a specified location with
			a specified video attribute.

	INPUT:          <line> and <column> denote the position
			at which the character is to be printed;
			<chr> is the ASCII code for the character,
			and <attribute> gives the simple software
			FV convention for the video attribute
			to be set (see FV.H).

	RETURNS:        none.

**************************************************************/

fv_cachr( line, column, chr, attribute )
	 int line, chr, column, attribute;
	{
	char far *p;
	if( adapter == 2 )
		{
		while( (inp( 0x03DA ) & BIT3) == 0 ) { ; }
		}
	p = ibm_videoram + (line*160)+(column*2);
	*p = chr;
	++p;
	*p = ibm_getatt( attribute );
	}

/**************************************************************

	FUNCTION:       fv_castr()

	DESCRIPTION:    This function places a specified string
			on the video screen at a specified posi-
			tion with a specified attribute.

	INPUT:          <line> and <column> denote the starting
			position for the string; <string> is a
			pointer to a buffer holding the string,
			and <attribute> gives the Simple Software
			FV convention for the video attribute
			for the string (see FV.H).

	RETURNS:        none.

**************************************************************/

fv_castr( line, column, string, attribute )
	int line, column, attribute;
	char *string;
	{
	char *pointer;
	int c;
	c = column;
	pointer = string;
	while( *pointer != 0 )
		{
		fv_cachr( line, c, *pointer, attribute );
		++c;
		++pointer;
		}
	}

/**************************************************************

	FUNCTION:       fv_cfill()

	DESCRIPTION:    This function fills a specified rectangular
			area of the screen with a specified character
			with a specified attribute.

	INPUT:          <top>, <left>, <bottom>, and <right>
			denote the coordinates for the rectangular
			area to be filled.  <character> is the character
			with which the area is to be filled, and
			<attribute> is the video attribute to be
			attached to the character.

	RETURNS:        none.

**************************************************************/

fv_cfill( top, left, bottom, right, character, attribute )
	int top, left, bottom, right, character, attribute;
	{
	int l, c;
	l = top;
	c = left;
	while( l <= bottom )
		{
		while( c <= right )
			{
			fv_cachr( l, c, character, attribute );
			++c;
			}
		c = left;
		++l;
		}
	}

