/**********************************************************************

	SE3.C           Simple EDIT

			A text editor
			based on Simple Software
			principles and conventions.

			(Part 3)

			Copyright (c) 1988, Ted A. Campbell

			Bywater Software
			Box 4023
			Duke Station,
			Durham, NC  27706

			The use of Simple EDIT is governed by the
			"Simple Software User Agreement" (see file
			SS.DOC).

**********************************************************************/

#include "se.h"

#ifdef  FAST_VIDEO
#include "stdio.h"
#include "fv.h"
#else
#include "curses.h"
#endif

#include "kb.h"

#ifdef  UINCLUDE
#include "ctype.h"
#include "malloc.h"
#endif

extern FILE *se_fp;

/**********************************************************************

	FUNCTION:  se_dlch()

	DESCRIPTION:  This function deletes a character at the 
	current cursor position.

	INPUT:  None.  

	RETURNS:  The function returns SS_ERROR if it is unable to 
	delete the character (i.e., the last LF in a file);
	otherwise returns SS_NOMINAL.  

**********************************************************************/

se_dlch()
	{
	register unsigned int bpointer, cpointer;
	struct se_line *nextline;

	/***    Is it the last character of a line?  If so, this
		line and the next will have to be joined together  ***/

	if ( se_current->characters[ se_bpos ] == LF )
		{

		/***    Is it the last character of the very last line?
			If so, don't even <think> about deleting it */

		if ( se_current->next == se_bbot )
			{
			return SS_ERROR;
			}

		nextline = se_current->next;
		bpointer = se_bpos;
		for ( cpointer = 0; nextline->characters[ cpointer ] != LF;
			++cpointer )
			{
			se_current->characters[ bpointer ] =
				nextline->characters[ cpointer ];
			++bpointer;
			}
		se_current->characters[ bpointer ] = LF;
		se_current->next = nextline->next;
		nextline = nextline->next;
		nextline-> prev = se_current;
		se_screen( se_current, se_lpos );
		se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
		return SS_NOMINAL;
		}

	/***    Now we are ready to adjust the line, in the
		buffer removing the deleted character and 
		adjusting all other characters on the line 
		up one position.  ***/

	for ( bpointer = 0; se_current->characters [ se_bpos + bpointer ]
		!= LF; ++bpointer )
		{
		se_current->characters [ se_bpos + bpointer ] =
			se_current->characters [ se_bpos + bpointer + 1 ];
		}

	/***	Now the adjusted line is to be displayed on 
		the screen.  ***/

	se_displ( se_current, se_lpos, 0 );
	se_edadr( se_lpos, se_cpos( se_current, se_bpos) );
	refresh( );
	}

/**********************************************************************

	FUNCTION:  se_dlback()

	DESCRIPTION:  This function deletes a character immediately
	before the current cursor position.

	INPUT:  None.

	RETURNS:  The function returns SS_ERROR if it is unable to
	delete the character (i.e., the last LF in a file);
	otherwise returns SS_NOMINAL.

**********************************************************************/

se_dlback()
	{
	register int bpointer, cpointer;
	struct se_line *prevline;

	/***    Is it the first character of a line?  If so, this
		line and the previous will have to be joined together  ***/

	if ( se_bpos == 0 )
		{

		/***    But if this is the first line of the file,
			there is no previous line, so we must exit  ***/

		if ( se_current->prev == se_btop )
			{
			return SS_ERROR;
			}

		prevline = se_current->prev;
		se_bpos = bpointer = se_llen( prevline->characters );
		for ( cpointer = 0; se_current->characters[ cpointer ] != LF;
			++cpointer )
			{
			prevline->characters[ bpointer ] =
				se_current->characters[ cpointer ];
			++bpointer;
			}

		/***    Terminate the line  ***/

		prevline->characters[ bpointer ] = LF;

		/***    Readjust two pointers to delete current line  ***/

		prevline->next = se_current->next;
		prevline->next->prev = prevline;

		/***    Assign se_current to the previous line  ***/

		se_current = prevline;

		/***    Prepare to update the screen ***/

		if ( se_lpos != 0 )
			{
			--se_lpos;
			}
		--se_alc;
		se_screen( se_current, se_lpos );
		se_edadr( se_lpos, se_cpos( se_current, se_bpos ) );
		se_lc();
		refresh();
		return SS_NOMINAL;
		}

	/***    No. So we are ready to adjust the line, in the
		buffer removing the deleted character and
		adjusting all other characters on the line
		up one position.  ***/

	--se_bpos;
	for ( bpointer = 0; se_current->characters [ se_bpos + bpointer ]
		!= LF; ++bpointer )
		{
		se_current->characters[ se_bpos + bpointer ] =
			se_current->characters[ se_bpos + bpointer + 1 ];
		}

	/***    Now the adjusted line is to be displayed on
		the screen.  ***/

	se_displ( se_current, se_lpos, 0 );
	se_edadr( se_lpos, se_cpos( se_current, se_bpos) );
	se_lc();
	refresh();
	}

/**********************************************************************

	FUNCTION:  se_dlword()

	DESCRIPTION:  This function deletes characters until a 
	whitespace is encountered.  It is called by entering 
	^T from the editor.

	INPUT:  None.  The word to be deleted is taken to be
	that at the current position.

	RETURNS:  None.

**********************************************************************/

se_dlword()
	{

	while( isspace( se_current->characters[ se_bpos ] ) == 0 )
		{
		se_dlch();
		}
	while( isspace( se_current->characters[ se_bpos ] ) != 0 )
		{
		se_dlch();
		}

	}

/**********************************************************************

	FUNCTION:  se_dlline()

	DESCRIPTION:  This function deletes the line on which the
	current character is located.  Pointers are reassigned, and
	the screen is updated.

	INPUT:  None.  The line to be deleted is presumed to be
	se_current.

	RETURNS:  None.  If the current line is in fact the last
	line of the file, the line itself is not erased, but all
	characters in the line are removed.

**********************************************************************/

se_dlline()
	{
	static struct se_line *prevline, *nextline;

	/***    If the current line is actually the last line
		in the file, we don't want to actually delete
		it, just remove all the characters from it         ***/

	if ( se_current->next == se_bbot )
		{
		se_current->characters[ 0 ] = LF;
		se_screen( se_current, se_lpos );
		}

	/***    To delete the current line, we must first
		reset the pointers associated with it.              ***/

	else
		{
		prevline = se_current->prev;
		nextline = se_current->next;
		prevline->next = se_current->next;
		nextline->prev = se_current->prev;
		se_current = nextline;

		se_screen( se_current, se_lpos );
		}

	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	return SS_NOMINAL;
	}

/**********************************************************************

	FUNCTION:  se_newline()

	DESCRIPTION:  This function begins a new line, and is called
	when ^M (RETURN) or ^J is typed while editing.  It utilizes
	two pointers to the se_line structure:  oldline points to
	the previously current line, and nextline points to the
	next line in the chain.

	INPUT:  None.  The new line is begun at the current character
	position.

	RETURNS:  If memory cannot be allocated for the new line,
	SS_ERROR is returned, otherwise SS_NOMINAL is returned.

**********************************************************************/

se_newline()
	{
	struct se_line *nextline, *oldline;
	unsigned int oldbpos, oldcpos, savecpos;

	/***    Save old line and old pointer to next line         ***/
	oldline  = se_current;
	nextline = se_current->next;

	/***    Get memory for new line  ***/

	if ( ( se_current = (struct se_line *) calloc( 1, sizeof ( struct se_line ))) == 0 )
		{
		se_current = oldline;
		error( "Out of memory for editing." );
		return SS_ERROR;
		}

	/***    Reassign pointers for doubly-linked list.  ***/

	se_current->next  = nextline;
	se_current->prev  = oldline;
	oldline->next     = se_current;
	nextline->prev    = se_current;

	/***    Save current buffer and column positions.  ***/

	oldbpos = se_bpos;
	savecpos = oldcpos = se_cpos( se_current, se_bpos ) ;

	/***    Set up the new buffer.  ***/

	se_bpos = 0;
	++se_lpos;
	++se_alc;
	se_current->characters[ se_bpos ] = LF;

	/***    Move remainder of oldline to new line.  ***/

	while( oldline->characters[ oldbpos ] != LF )
		{
		se_current->characters[ se_bpos ] =
			oldline->characters[ oldbpos ];
		++oldbpos;
		++se_bpos;
		}

	/***    Terminate the old and new lines with LF.  ***/

	oldline->characters[ savecpos ]   = LF;
	se_current->characters[ se_bpos ] = LF;
	se_bpos = 0;

	/***	If the current line is the last line on the screen, 
		it has to be handled differently than other lines, 
		because the screen will have to be scrolled down from
		the top to add a new line at the bottom for the 
		new line.  The following conditional takes care
		of this case.  ***/

	if ( se_lpos == se_lines )
		{
		--se_lpos;
		se_screen( se_top(), 0 );
		}


	/***	If the line is not the last on the screen, then 
	only the portion of the screen from the current line 
	down has to be redrawn.					   ***/

	else
		{
		se_screen( se_current->prev, se_lpos-1 );
		}

	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	se_lc();
	return SS_NOMINAL;

	}

/********************************************************************

	FUNCTION:  se_displ()

	DESCRIPTION:  This function displays a line from the buffer
	at a specified line in the edit screen.  This function is
	called repeatedly by se_screen to draw or redraw the screen,
	and is called singly by some other functions.

	INPUT:  <lstruct> is a pointer to an se_line structure
	for the line to be drawn.  <line> is the line in the
	visible screen where the line is to be drawn.  <bpos>
	is the buffer position from which the line is to be
	drawn.

	RETURNS:  None.

********************************************************************/

se_displ( lstruct, line, bpos )
	struct se_line *lstruct;
	int line, bpos;
	{
	static char tbuf[ LINESIZE ];
	register unsigned int pointer;
	unsigned int col_counter, start_pos, ctarget;
	char last_char;
	pointer = bpos;
	col_counter = 0;

	if ( bpos > 0 )
		{
		last_char = lstruct->characters[ bpos - 1 ];
		}
	else
		{
		last_char = 0;
		}

	/***    Place the line in the temporary buffer:
		<pointer> points to position in the se_line
		structure, <col_counter> keeps the position
		in the display buffer (<tbuf>)  ***/

	while( ( lstruct->characters[ pointer ] != LF ) &&
		( se_cpos( lstruct, pointer ) < se_cols - 1 ) )
		{
		if ( lstruct->characters[ pointer ] == TAB )
			{
			if ( last_char == TAB )
				{
				ctarget += TAB_INTERVAL;
				}
			else
				{
				ctarget = ( col_counter - ( col_counter
					% TAB_INTERVAL ) ) + TAB_INTERVAL - 1;
				}
			while ( col_counter < ctarget )
				{
				tbuf[ col_counter ] = ' ';
				++col_counter;
				}
			--col_counter;
			}
		else
			{
			tbuf[ col_counter ] = lstruct->characters[ pointer ];
			}
		++col_counter;
		++pointer;
		last_char = lstruct->characters[ pointer ];
		}
	tbuf[ col_counter ] = 0;

	/***    Display the line  ***/

	if ( bpos == 0 )
		{
		start_pos = 0;
		}
	else
		{
		start_pos = se_cpos( lstruct, bpos );
		}
	se_castr( line + LINEOFFSET, start_pos + COLOFFSET, tbuf, SE_NORMAL );

	/***    Fill out blanks at end of line  ***/

#ifdef  FAST_VIDEO
	se_cfill( line + LINEOFFSET, se_cpos( lstruct, pointer),
		line + LINEOFFSET, se_cols - 1, ' ' );
#else
	clrtoeol();
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
#endif

	}

/********************************************************************

	FUNCTION:  se_top()

	DESCRIPTION:  This function returns a pointer to the se_line
	structure for the line that is at the top of the screen.
	This is essentially a utility function called by various
	other functions that have to identify the first line of the
	screen.

	INPUT:  None.  The top of the screen is calculated relative
	to se_current and se_line.

	RETURNS:  the function returns a pointer to an se_line
	structure for the first line on the visible screen.


********************************************************************/

struct se_line *
se_top()
	{
	return se_traverse( se_current, 0 - se_lpos );
	}

/********************************************************************

	FUNCTION:  se_traverse()

	DESCRIPTION:  This function returns a pointer to a line a
	specified number of lines below (positive) or above
	(negative) the current line.  If the specified distance
	would go beyond the top or bottom of the file, the value
	for the top or bottom is returned.

	INPUT:  <startline> is a pointer to an se_line structure
	for the line from which the counting begins.  <howfar>
	gives how many lines up below (positive) or above (negative)
	the starting line one wants to go.

	RETURNS:  There are actually two returns from this function:
	(a) the function itself returns a pointer to an se_line
	structure as far as the function;  (b) the integer <se_trct>
	is set to the actual number of lines traversed.  Calls to
	the se_traverse() function should carefully check se_trct
	to be sure the specified number of lines were in fact
	traversed.

********************************************************************/

int se_trct;                  /* lines actually traversed          */

struct se_line *
se_traverse( startline, howfar )
	struct se_line *startline;
	int howfar;
	{
	register int counter;
	struct se_line *lpointer;
	lpointer = startline;
	counter = howfar;
	se_trct = 0;

	if ( howfar < 0 )
		{
		while( ( counter < 0 ) && ( lpointer->prev != se_btop ) )
			{
			lpointer = lpointer->prev;
			++counter;
			--se_trct;
			}
		}
	else
		{
		while( ( counter > 0 ) && ( lpointer->next != se_bbot ))
			{
			lpointer = lpointer->next;
			--counter;
			++se_trct;
			}
		}

	return lpointer;
	}


/********************************************************************

	FUNCTION:  se_scup()

	DESCRIPTION:  This function scrolls the screen up (lines
	move towards the top of the screen, cursor moves towards
	the end or bottom of the file) a specified number of lines.
	The cursor is moved to the same column (if possible) in the
	line the specified number of lines before the current
	line.   In the following two routines, the current
	visible line number ( se_lpos ) is subtracted from zero
	to get the offset to the top line of the screen.  From
	that point, the number of lines is added or subtracted
	to obtain the correct starting line.

	INPUT:  <lines> is the number of lines to be scrolled.

	RETURNS:  None.

********************************************************************/

se_scup( lines )
	unsigned int lines;
	{
	se_screen( se_traverse( se_top(), 0 - lines), 0 );
	se_current = se_traverse( se_current, se_trct );
	se_alc = se_alc + se_trct;
	se_bpos = 0;
	se_edadr( se_lpos, se_cpos( se_current, se_bpos ) );
	se_lc();
	}

/********************************************************************

	FUNCTION:  se_scdn()

	DESCRIPTION:  This function scrolls the screen down (toward
	the bottom or end of the file) a specified number of lines.
	(See the notes on se_scup() ).

	INPUT:  <lines> is the number of lines to be scrolled.

	RETURNS:  None.

*********************************************************************/

se_scdn( lines )
	unsigned int lines;
	{
	se_screen( se_traverse( se_top(), lines ), 0 );
	se_current = se_traverse( se_current, se_trct );
	se_alc += se_trct;
	se_lpos -= ( lines - se_trct );
	se_bpos = 0;
	se_edadr( se_lpos, se_cpos( se_current, se_bpos ) );
	se_lc();
	}

/**********************************************************************

	FUNCTION:  se_save()

	DESCRIPTION:  This function saves the whole edit buffer into
	the path and file stored in se_pname and se_fname.  It is
	called by both ^KS (temporary) and ^KD (save and return) while
	editing.

	INPUT:  None.

	RETURNS:  The function returns SS_ERROR if the write
	fails.  Otherwise it returns SS_NOMINAL  (these are
	passed from se_write() ).

**********************************************************************/

se_save()
	{
	sprintf( se_tbuf, "Saving file %s; please wait.", se_file );
	message( se_tbuf );
	return se_write();
	}

/**********************************************************************

	FUNCTION:  se_write()

	DESCRIPTION:  This function writes the edit buffer out to the
	disk file indicated by se_file.  The current file is
	overwritten.  This routine is called from se_save().

	INPUT:  None.

	RETURNS:  The function returns SS_ERROR on any failure,
	otherwise SS_NOMINAL (successful write).

**********************************************************************/

se_write()
	{
	register unsigned int bcounter;
	struct se_line *lpointer;

	/***    Open the file                                      ***/
	if ( ( se_fp = fopen( se_file, "wb" )) == NULL )
		{
		error( "Cannot open file for save..." );
		return SS_ERROR;
		}

	/***    Write the buffer out to the file                   ***/
	lpointer = se_btop->next;

	while( lpointer != se_bbot )
		{
		bcounter = 0;

		while( ( lpointer->characters[ bcounter ] != LF )
			&& ( bcounter < LINESIZE ) )
			{
			fputc( lpointer->characters[ bcounter ], se_fp );
			++bcounter;
			}
#ifdef USECR
		fputc( CR, se_fp );
#endif
		fputc( LF, se_fp );
		lpointer = lpointer->next;
		}
#ifdef USECR
		fputc( 0x1a, se_fp );		/* CP/M EOF Marker */
#endif

	/***    Close the file                                     ***/
	if ( fclose( se_fp ) != 0 )
		{
		error( "Could not close file" );
		}
	}

/**********************************************************************

	FUNCTION:  se_edadr()

	DESCRIPTION:  This function addresses the cursor within the
	editing window for Simple EDIT.  Its coordinates are relative
	to the editing window.

	INPUT:  <line> and <column> ae the coordinates relative to
	the visible editing screen at which the cursor is to be
	placed.

	RETURNS:  None.

**********************************************************************/

se_edadr( line, column )
	unsigned int line, column;
	{
	move( LINEOFFSET + line, COLOFFSET + column );
	refresh();
	}

/**********************************************************************

	FUNCTION:  se_lc()

	DESCRIPTION:  This function displays the current line and column
	numbers at the top of the screen while editing.

	INPUT:  None.  The current coordinates for the cursor are
	presumed to be those indicated by <se_alc> and <se_cpos( se_current, se_bpos ) >.

	RETURNS:  None.

**********************************************************************/

se_lc()
	{
	static char tbuf[ 30 ];
	sprintf( tbuf, " Line: %4d ", se_alc + 1 );
	se_castr( 0, 68, tbuf, SE_SPECIAL );
	sprintf( tbuf, "  Col: %4d ", se_cpos( se_current, se_bpos )  + 1 );
	se_castr( 1, 68, tbuf, SE_SPECIAL );
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	refresh();
	}

/**********************************************************************

	FUNCTION:  se_im()

	DESCRIPTION:  This function displays the insert mode on the
	top line of the edit screen.

	INPUT:  None.  The current insert status is taken to be that
	indicated by <se_imode>.

	RETURNS:  None.

**********************************************************************/

se_im()
	{
	if ( se_imode == 1 )
		{
		se_castr( 0, 56, " INSERT ON ", SE_SPECIAL );
		}
	else
		{
		se_castr( 0, 56, "           ", SE_NORMAL );
		}
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	}

/**********************************************************************

	FUNCTION:  se_llen()

	DESCRIPTION:  This function returns the length of a character
	line terminated by the Simple EDIT LF character.  The returned
	length does not include the LF character.

	INPUT:  <string> is a pointer to a character string terminated
	by the LF character.

	RETURNS:  The function returns the length of the string, not
	including the LF character.

**********************************************************************/

se_llen( string )
	char string[];
	{
	char *pointer;
	register unsigned int counter;
	pointer = string;
	counter = 0;
	while ( *pointer != LF )
		{
		++counter;
		++pointer;
		}
	return counter;
	}

/**********************************************************************

	FUNCTION:  se_str()

	DESCRIPTION:  This function gets a character string from
	the console.

	INPUT:  <string> is a pointer to a buffer for the input
	string.

	RETURNS: none.

**********************************************************************/

se_str( string )
     char *string;
     {
     char character, *pointer;
     character = 0;
     pointer = string;
     while( ( character != CR ) && ( character != LF ) )
	  {
	  *pointer = character = kb_ci();
#ifdef  FAST_VIDEO
	  putchar( character );
	  fflush( stdout );
#else
	  addch( character );
	  refresh();
#endif
	  ++pointer;
	  }
     --pointer;
     *pointer = 0;
     }

/**********************************************************************

	FUNCTION:  kb_dmenu()

	DESCRIPTION:  This is the default menu for keyboard
	responses.  In this case, it is a menu for Control sequences.

	INPUT:  none.

	RETURNS:  none.

**********************************************************************/

kb_dmenu()
	{
	se_castr( 1, 0, " CTL ", SE_SPECIAL );
	se_castr( 1, 6, "Back char Forward char Prev line Next line          ", SE_NORMAL );
	se_castr( 2, 6, "Delete char                     ESC - menu          ", SE_NORMAL );
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	refresh();
	}

/**********************************************************************

	FUNCTION:  kb_emenu()

	DESCRIPTION:  This is a menu for escape sequences from the
	keyboard.

	INPUT:  none.

	RETURNS:  none.

**********************************************************************/

kb_emenu()
	{
	se_castr( 1, 0, " ESC ", SE_SPECIAL );
	se_castr( 1, 6, "Back word Forward word Prev screen Next screen      ", SE_NORMAL );
	se_castr( 2, 6, "Save  Quit  eXit  Match  Replace  Home  End  Insert ", SE_NORMAL );
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	refresh();
	}

/**********************************************************************

	FUNCTION:  se_castr()

	DESCRIPTION:  This function position a string of characters
	on the screen at a designated location with a designated
	video attribute.

	INPUT:  <line> and <column> give the location at which
	the string will be addressed; <mes> is a pointer to the
	string; <attribute> is the attribute for the string, where
	0 = normal and 1 = special attribute.

	RETURNS: none.

**********************************************************************/

se_castr( line, column, mes, attribute )
	int line, column, attribute;
	char *mes;
	{
	unsigned int _att;
#ifdef  FAST_VIDEO
	if ( attribute == 1 )
		{
		_att = FV_SPEC1;
		}
	else
		{
		_att = FV_NORMAL;
		}
	fv_castr( line, column, mes, _att );
#else
	move( line, column );
	if ( attribute == 1 )
		{
		standout();
		}
	printw( mes );
	if ( attribute == 1 )
		{
		standend();
		}
	refresh();
#endif
	}

/**********************************************************************

	FUNCTION:  message()

	DESCRIPTION:  This function displays a message at a predefined
	message area, then returns the cursor to its position in the
	edit window.

	INPUT:  <message> is a pointer to a null-terminated character
	string.

	RETURNS:  None.

**********************************************************************/

message( mes )
	char mes[];
	{
	se_castr( 0, 0, "                                       ", SE_NORMAL );
	se_castr( 0, 0, mes, SE_NORMAL );
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	}

se_help()
	{
	message( "SE HELP module: " );
	move( 0, 16 );
	refresh();
	kb_ci();
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	message( " " );
	}

/**********************************************************************

	FUNCTION:  error()

	DESCRIPTION:  This function displays an error message in
	message area, waits for a key to be struck, and then
	returns the cursor to its position in the edit screen.

	INPUT:  <message>  is a pointer to a null-pointed character
	string describing the message.

	RETURNS:  None.

**********************************************************************/

error( mes )
	char mes[];
	{
	char tbuf[ 41 ];
	sprintf( tbuf, "ERROR: %s", mes );
	message( tbuf );
	kb_ci();
	message( " " );
#ifndef FAST_VIDEO
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	refresh();
#endif
	}

/***************************************************************

	FUNCTION:  se_cfill()

	DESCRIPTION:  This is a utility function that fills a
	rectangular screen area with a specified character.
	It is used for blanking areas of the screen.

	INPUT:  <top>, <left>, <bottom>, and <right> define
	the area to be filled in absolute screen coordinates.
	<character> is the character with which the area
	should be filled.

***************************************************************/

se_cfill( top, left, bottom, right, character )
	int top, left, bottom, right;
	char character;
	{
#ifndef  FAST_VIDEO
	register int sandinistas, contras;
#endif

#ifdef  FAST_VIDEO
	fv_cfill( top, left, bottom, right, character, FV_NORMAL );
#else
	sandinistas = top;
	contras = left;

	move( sandinistas, contras );
	while( sandinistas <= bottom )
		{
		while( contras < right )
			{
			addch( character );
			++contras;
			}
		++sandinistas;
		contras = left;
		move( sandinistas, contras );
		}
	refresh( );
#endif
	}


