/***********************************************************************
*
* sim945.c - TILINE 945 VDT IO processing for the TI 990 Simulator.
*
* Changes:
*   11/18/03   DGP   Original.
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (_WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#endif

#if defined(UNIX)
#include <sys/time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pthread.h>
#endif

#include "simdef.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern uint16 lights;	/* The panel lights */

extern int run;
extern int idle;
extern int fault;
extern int devcnt;
extern int model;
extern int mapenabled;
extern int pancount;
extern uint32 memlen;
extern char view[MAXVIEW][81];

extern uint8 memory[MEMSIZE];
extern Device devices[MAXDEVICES];

/*
** Command words access macros
*/

#define GETW0(d)    GETMEM0((d)->devaddr+TPCSSTART)
#define GETW7(d)    GETMEM0((d)->devaddr+TPCSSTART+14)

#define PUTW0(d,v)  PUTMEM0((d)->devaddr+TPCSSTART,v)
#define PUTW7(d,v)  PUTMEM0((d)->devaddr+TPCSSTART+14,v)

#define GETLEN(d)   GETMEM0((d)->devaddr+TPCSSTART+8)
#define PUTLEN(d,v) PUTMEM0((d)->devaddr+TPCSSTART+8,v)

#define GETADDR(d) \
(GETMEMB0((d)->devaddr+TPCSSTART+13)<<16)|GETMEM0((d)->devaddr+TPCSSTART+10)
#define GETCMD(d)   GETMEMB0((d)->devaddr+TPCSSTART+12) & 0xF
#define GETUNIT(d)  (GETMEMB0((d)->devaddr+TPCSSTART+12) >> 4) & 0xF

/*
** Command word control bits
*/

#define OLBIT	0x8000		/* Off Line */
#define TOBIT	0x0100		/* Command Time Out */

#define IDLEBIT	0x8000		/* Idle */
#define COMPBIT	0x4000		/* Complete */
#define ERRBIT	0x2000		/* Error */
#define INTBIT	0x1000		/* Interrupt Enable */
#define sp0BIT	0x0800		/* spare */
#define sp1BIT	0x0400		/* spare */
#define ABNBIT	0x0100		/* Abnormal Completion */
#define IPEBIT	0x0080		/* Interface Parity error */
#define ECEBIT	0x0040		/* Error Correction enabled */
#define HWEBIT	0x0020		/* Hard Error */
#define TMRBIT	0x0010		/* Tiline memory read error */
#define TTEBIT	0x0008		/* Tiline Timing error */
#define TTOBIT	0x0004		/* Tiline Timout error */

/***********************************************************************
* v945open - Open 945.
***********************************************************************/

int 
v945open (Device *dev)
{
#ifdef DEBUGV945
   fprintf (stderr, "v945open: ENTERED\n");
   fprintf (stderr, "   file = %s\n", dev->file);
#endif

   if (dev->fd == DEV_NULL)
   {
      PUTW0 (dev, OLBIT);
      PUTW7 (dev, IDLEBIT); /* Mark idle */
      return (0);
   }

   PUTW7 (dev, IDLEBIT); /* Mark idle */

   return (0);
}

/***********************************************************************
* v945putmem - Put data into memory.
***********************************************************************/

static void
v945putmem (uint32 ma, uint16 v)
{
   if (ma < memlen)
   {
      PUTMEM0 (ma, v);
   }
}

/***********************************************************************
* v945getmem - Get data from memory.
***********************************************************************/

static uint16
v945getmem (uint32 ma)
{
   if (ma >= memlen) return (0);

   return (GETMEM0 (ma));
}

/***********************************************************************
* v945reset - Reset device
***********************************************************************/

void
v945reset (Device *dev)
{
#ifdef DEBUGV945
   fprintf (stderr, "v945reset: ENTERED\n");
#endif

   if (dev->fd == DEV_NULL)
   {
      PUTW0 (dev, OLBIT);
   }

   PUTW7 (dev, IDLEBIT | ERRBIT | ABNBIT ); 

   return;
}

/***********************************************************************
* v945docmd - Do 945 VDT command
***********************************************************************/

int 
v945docmd (Device *dev)
{
   Device *udev;
   int cmd;
   int i;
   int len;
   uint32 addr, saddr;
   uint16 W0, W7;
   uint8 unit;

   /*
   ** Get common command word values
   */

   W0 = GETW0 (dev);
   W7 = GETW7 (dev);

   cmd = GETCMD (dev);
   unit = GETUNIT (dev);
   saddr = addr = ((GETADDR (dev)) + 1) & 0x1FFFFE;
   len = GETLEN(dev) /*(GETLEN (dev) + 1) & 0xFFFE*/;

#ifdef DEBUGV945
   fprintf (stderr,
	    "v945docmd: ENTRY: cmd = %d, unit = %02X, addr = %06X, len = %d\n",
	    cmd, unit, addr, len);
   fprintf (stderr, "   pc = %04X, wp = %04X, st = %04X\n",
	    pcreg, wpreg, statreg);
   for (i = 0; i < 16; i+=2)
      fprintf (stderr, "   ctl[%04X] %04X\n",
	    dev->devaddr+TPCSSTART+i, GETMEM0 (dev->devaddr+TPCSSTART+i));
#endif

   /*
   ** Find selected unit
   */

   for (i = 0; i < devcnt; i++) /* find selected unit */
      if (devices[i].devaddr == dev->devaddr && devices[i].unit == unit)
	 break;
   if (i == devcnt)
   {
#ifdef DEBUGV945
      fprintf (stderr, "v945docmd: unit = %d: Unit not selected\n", unit);
#endif
      /* Drive not ready */
      W0 |= OLBIT;
      W7 |= ERRBIT | IDLEBIT;
      PUTW0 (dev, W0);
      PUTW7 (dev, W7);
      return(0);
   }
   udev = &devices[i];

   if (udev->fd == DEV_NULL)
   {
#ifdef DEBUGV945
      fprintf (stderr, "v945docmd: unit = %d: Unit is NULL\n", unit);
#endif
      PUTW0 (udev, OLBIT );
      PUTW7 (udev, IDLEBIT);
      return (0);
   }

   /*
   ** Set initial sate
   */

   udev->select = TRUE;
   udev->intenabled = FALSE;

   /*
   ** Process command
   */

   switch (GETCMD (udev))
   {

   }

   if (W7 & INTBIT)
      udev->intenabled = TRUE;
   W7 |= COMPBIT | IDLEBIT;
   PUTW0 (udev, W0);
   PUTW7 (udev, W7);
#ifdef DEBUGV945
   fprintf (stderr, "returns: intenabled = %s\n",
	 udev->intenabled ? "TRUE" : "FALSE");
   for (i = 0; i < 16; i+=2)
      fprintf (stderr, "   ctl[%04X] %04X\n",
	    udev->devaddr+TPCSSTART+i, GETMEM0 (udev->devaddr+TPCSSTART+i));
#endif

   return (udev->intenabled ? udev->intlvl : 0);
}

