Module EBuffer;

{ ******************** Allocate buffers for Ethernet IO ******************** }
{
    EB_Alloc allocates memory for the number of sets of buffers, and
    makes them unmovable etc.  It also sets AdRec to point to one
    properly aligned EtherAdRec.  (Presumably you only need one EtherAdRec.)

    EB_Buffers is then called repeatedly, once for each set of buffers
    you need, as specified to EB_Alloc.  Pointers to properly aligned
    unmovable buffers are returned.  If you call EB_Buffers more than
    the number of buffers you specified to EB_Alloc, nil pointers are
    returned.

    Example:
    EB_Alloc (4, pAdRec);
    EB_Buffers (head1, buf1, stat1);
    EB_Buffers (head2, buf2, stat2);
    EB_Buffers (head3, buf3, stat3);
    EB_Buffers (head4, buf4, stat4);
}

Exports

Imports Ether10IO from Ether10IO;

Procedure EB_Alloc (NumBuffers: Integer; Var AdRec: pEtherAdRec);

Procedure EB_Buffers (Var Header: pEtherHeader;
                Var Buffer: pEtherBuffer; Var Status: pEtherStatus);

Private

Imports AlignMemory from AlignMemory;

        { Each buffer has the following:
        {
        {     EtherBuffer,  750 words,  1024 word alignment
        {     EtherHeader,  8 words,    8 word alignment
        {     EtherStatus,  2 words,    Double word alignment
        {
        { Also, there must be one of the following:
        {     EtherAdRec,  4 words,   1 word alignment
        {
        { All of these must be locked in memory at least some of the time.
        { Therefore we allocate them all from one segment we create and make
        { unmovable, using procedure NewBuffer from module AlignMemory.
        { }

Const   HeadOffset = ((Wordsize (EtherBuffer) div 8) + 1) * 8;
        { First multiple of 8 greater than size of EtherBuffer }

        StatOffset = (((HeadOffset + WordSize (EtherHeader)) div 2) + 1) * 2;
        { First multiple of 2 greater than HeadOffset + size of EtherHeader }

        AdRecOffset = StatOffset + WordSize (EtherStatus) + 1;
        { First multiple of 1 greater than StatOffset + size of EtherStatus }

        EntireOffset = 1024;
        { First multiple of 1024 greater than AdRecOffset + size of AdRec }

{$ifc AdRecOffset + WordSize (EtherAdRec) > EntireOffset then}
{$message You need to enlarge the buffer you are allocating in EBuffer. }
End.
{$endc}

{$ifc HeadOffset mod 8 <> 0 then}
{$message You need to fix up HeadOffset in EBuffer. }
End.
{$endc}

{$ifc StatOffset mod 2 <> 0 then}
{$message You need to fix up StatOffset in EBuffer. }
End.
{$endc}

Var     NumBufsLeft, Segment, Base_Offset: Integer;

Procedure EB_Alloc (NumBuffers: Integer; Var AdRec: pEtherAdRec);

        Type    BullShit = Record Case Integer of
                0: (Offset:  Integer;
                    Segment: Integer);
                1: (AP: AlignedPointer);
                End;

        Var     ABuf: AlignedPointer;
                BS: BullShit;

        Begin
        if NumBuffers < 1 then
            begin
            AdRec := nil;
            Exit (EB_Alloc);
            end;
        NewBuffer (ABuf, (NumBuffers * EntireOffset) div 256, 4);
                { 4 means aligned on 1K boundary }

        BS . AP := ABuf;
        Segment := BS . Segment;
        Base_Offset := BS . Offset;

        AdRec := MakePtr (Segment, Base_Offset + AdRecOffset + 8, pEtherAdRec);

        NumBufsLeft := NumBuffers;
        End;

Procedure EB_Buffers (Var Header: pEtherHeader;
                Var Buffer: pEtherBuffer; Var Status: pEtherStatus);

        Begin
        if NumBufsLeft < 1 then
            begin
            Header := nil;
            Buffer := nil;
            Status := nil;
            Exit (EB_Buffer);
            end;
        NumBufsLeft := NumBufsLeft - 1;

        Buffer := MakePtr (Segment, Base_Offset, pEtherBuffer);
        Header := MakePtr (Segment, Base_Offset + HeadOffset + 8, pEtherHeader);
        Status := MakePtr (Segment, Base_Offset + StatOffset + 8, pEtherStatus);
        Base_Offset := Base_Offset + EntireOffset;
        End.
