program List;

imports PascalInit from PascalInit;
imports CmdParse from CmdParse;
imports Sapph from SapphUser;
imports WindowUtils from WindowUtils;
imports PathName from PathName;
imports Spice_String from Spice_String;
imports Spawn from Spawn;

imports AccInt from AccentUser;
imports Except from Except;
imports AccCall from AccCall;
imports SaphEmrServer from SaphEmrServer;
imports SaphEmrExceptions from SaphEmrExceptions;
{imports PM from PMUser;}

const
  FontFName = '';
  NumSwtchs = 1;

  HSeparator = 10;
  VSeparator = 2;
  
  debug = true;

var
  UserVP: Viewport;
  WinLeftX, WinTopY, WinWidth, WinHeight: Integer;
  VPWidth, VPHeight: Integer;
  KPort, DPort: Port;

  Font: Viewport;
  FontFile: String;
  FontName: String;
  FontHeight: Integer;
  RowHeight: Integer;

  WildName: Path_Name;
  DirName: APath_Name;                  {Name of directory listed}

  keyEv: KeyEvent;
  
  changingSize: Boolean;

procedure ParseCmdLine;

  var
    Inputs, Outputs: pArgRec;
    Switches: pSwitchRec;
    Err: String;
    Swtchs: CmdArray;
    swtch: Integer;

  begin {ParseCmdLine}
    (* Parse command line arguments *)
    Swtchs[1] := 'FONT';

    if ParseCmdArgs( Inputs, Outputs, Switches, Err ) then begin
      while Switches <> nil do begin
        with Switches^ do begin
          ConvUpper( switch );
          swtch := UniqueCmdIndex( switch, Swtchs, NumSwtchs );
          case swtch of
            1:
              if arg = '' then
                writeln( 'Font switch must have argument.' )
              else
                fontFile := arg;

            NumSwtchs+1:
              Writeln( 'Invalid switch: ', switch );

            NumSwtchs+2:
              Writeln( 'Ambiguous switch: ', switch );
          end;
        end;
        Switches := Switches^.next;
      end;
    end
    else
      writeln( 'Syntax error in command line.' );
  end {ParseCmdLine};

procedure GetFont( var fontFile: String; var font: Viewport );
  var
    dum: Integer;
    b: Boolean;
  begin {GetFont}
    if fontFile = '' then
      font := GetSysFont( UserVP )
    else begin
      if FindExtendedFileName( fontFile, '.kst;', '', false ) <> success then begin
        writeln( 'Font file ''', fontFile, ''' not found.' );
        font := GetSysFont(UserVP);
      end
      else begin
        font := LoadFont( UserVP, fontFile );
        if font = NULLViewport then begin
          writeln( 'File ''', fontFile, ''' is not a font file.' );
          font := GetSysFont(UserVP);
        end;
      end;
    end;
    FontSize( font, FontName, dum, dum, dum, dum, FontHeight, dum, dum, b, b );
    RowHeight := FontHeight + VSeparator;
    {$ifc debug then}
      writeln( 'Font is ', FontHeight:1, ' bits high.' );
    {$endc}
  end {GetFont};

procedure ListFiles;

  var
    name: String;
    EntryList: Entry_List;                {List of names}
    EntryListCount: Long;
    FoundInFirst: Boolean;

    gr: GeneralReturn;
    i: Integer;
    lastch: Integer;
    width, height: Integer;
    row, col: Integer;

    MaxWidth: Integer;
    NewHeight: Integer;
    NumCols, NumRows, ColWidth, Offset: Integer;

{$RANGE-}
  begin {ListFiles}
    (* Get the list of files *)
    gr := FindWildPathNames( WildName, '', false, Entry_All, 0,
                             FoundInFirst, DirName, EntryList, EntryListCount );
  
    (* Find the widest name in the given font *)
    MaxWidth := 0;
    for i := 0 to shrink(EntryListCount) - 1 do
      with EntryList^[i] do begin
        FontStringWidthVector( Font, EntryName, 1, DONTCARE, width, height ); 
        if width > MaxWidth then
          MaxWidth := width;
      end;
    
    MaxWidth := MaxWidth + HSeparator;
    {$ifc debug then}
      writeln( 'The maximum width is ', MaxWidth:1, ' bits.' );
    {$endc}
    
    (* Determine number and width of columns *)
    if MaxWidth > VPWidth then begin
      NumCols := 1;
      NumRows := shrink(EntryListCount);
      ColWidth := VPWidth;
      Offset := 0;
    end
    else begin
      NumCols := VPWidth div MaxWidth;
      NumRows := (shrink(EntryListCount) + NumCols - 1) div NumCols;
      ColWidth := VPWidth div NumCols;
      Offset := (VPWidth - ColWidth*NumCols) div 2 + (ColWidth - MaxWidth) div 2;
    end;

    {$ifc debug then}
      writeln( shrink(EntryListCount):1, ' entries give ', NumRows:1, ' rows of ', NumCols:1, ' entries each.' );
    {$endc}

    (* Resize window to fit closely if necessary *)
    changingSize := true;
    NewHeight := NumRows*RowHeight + RowHeight div 2;
    if VPHeight <> NewHeight then begin
      {$ifc debug then}
        writeln( 'Disabling emergency messages' );
      {$endc}
      gr := SoftEnable( false, false );                 {Disable Interrupt}
      WinHeight := WinHeight - (VPHeight - NewHeight);
      VPHeight := NewHeight;
      ModifyWindow( UserWindow,
                    UNCHANGED, UNCHANGED,
                    UNCHANGED, WinHeight,
                    UNCHANGED );
      {$ifc debug then}
        writeln( 'Enabling emergency messages' );
      {$endc}
      gr := SoftEnable( false, true );                  {Reenable interrupts}
    end;

    VPColorRect( UserVP, RectWhite, 0, 0, VPWidth, VPHeight );

    {$ifc debug then}
      writeln( 'Writing directory entries' );
    {$endc}

    (* Now, print out the results *)
    for row := 0 to NumRows-1 do begin
      for col := 0 to NumCols-1 do begin
        lastch := DONTCARE;
        if col * NumRows + row < shrink(EntryListCount) then
          VPPutString( UserVP, Font, RRpl,
                       Offset + col*ColWidth, (row+1) * RowHeight,
                       EntryList^[col*NumRows+row].EntryName,
                       1, lastch );
      end;
    end;
    changingSize := false;
  end {ListFiles};
{$RANGE+}

(* Emergency message handling definitions *)
const
  MaxMsgSize = 2048;

type
  Space = Array[0..MaxMsgSize div 2 - 1] of Integer;
  pDumMsg = ^DumMsg;
  DumMsg = record
             head: Msg;
             retType: TypeType;
             retCode: Integer;
             body: Space;
           end;

var
  pMsg: pDumMsg;
  pReplyMsg: pDumMsg;

handler EViewPtChanged( vp: Viewport; newX, newY, newW, newH, rank: Integer );
  var
    dumI: Integer;
    dumB: Boolean;
    dumPS: ProgStr;
    dumTS: TitStr;
  begin {EViewPtChanged}
    {$ifc debug then}
      writeln( 'Viewport changed size' );
    {$endc}
    FullWindowState( UserWindow, WinLeftX, WinTopY, WinWidth, WinHeight, dumI, dumB, dumB, dumB, dumPS, dumTS );
    VPWidth := newW;
    VPHeight := newH;
    {$ifc debug then}
      writeln( 'Window = (', WinLeftX:1, ',', WinTopY:1, ') [', WinWidth:1, ',', WinHeight:1, ']' );
      writeln( 'Viewport = [', VPWidth:1, ',', VPHeight:1, ']' );
    {$endc}
    if not changingSize then begin
      {$ifc debug then}
        writeln( 'ReWriting file list.' );
      {$endc}
      ListFiles;
    end
    else begin
      {$ifc debug then}
        writeln( 'ChangingSize true.' );
      {$endc}
    end;
  end {EViewPtChanged};

handler EViewPtExposed( vp: Viewport; ra: pRectArray; numRectangles: Long );
  var
    gr: GeneralReturn;
  begin {EViewPtExposed}
    {$ifc debug then}
      write( 'Viewport exposed: ' );
    {$endc}
    if not changingSize then begin
      {$ifc debug then}
        writeln( 'ReWriting file list.' );
      {$endc}
      ListFiles;
    end
    else begin
      {$ifc debug then}
        writeln( 'ChangingSize true.' );
      {$endc}
    end;
    gr := InvalidateMemory( KPort, recast(ra,VirtualAddress), wordsize(ra)*2 );
  end {EViewPtExposed};

procedure HandleEmergMsgs;
  var
    b: Boolean;
  begin {HandleEmergMsgs}
    b := SaphEmrServer( pMsg, pReplyMsg );
  end {HandleEmergMsgs};

handler EmergMsg;
  var
    gr: GeneralReturn;
  begin {EmergMsg}
    {$ifc debug then}
      writeln( 'Received emergency message' );
    {$endc}

    pMsg^.head.MsgSize := MaxMsgSize;
    if Receive( pMsg^.head, 0, AllPts, ReceiveIt ) = success then begin
      HandleEmergMsgs;
    end;
    {$ifc debug then}
      writeln( 'Reenabling emergency messages' );
    {$endc}
    gr := SoftEnable( false, true );
  end {EmergMsg};

procedure NewWin;
  var
    FullWindow: Window;
    progName: progStr;
    enable: Boolean;
    dumI: Integer;
    dumB: Boolean;
    dumPS: ProgStr;
    dumTS: TitStr;
  begin {NewWin}
    KPort := KernelPort;
    DPort := DataPort;

    progName := 'List';
    if UserWindowShared then begin
      if Split( KPort, DPort ) = IsParent then
        exit( List );
      FullWindow := GetFullWindow( SapphPort );
      WinLeftX := ASKUSER;
      WinTopY := ASKUSER;
      WinWidth := ASKUSER;
      WinHeight := ASKUSER;
      UserWindow := CreateWindow( FullWindow, false, WinLeftX, WinTopY,
                                  false, WinWidth, WinHeight,
                                  true, true, '', progName, true, UserVP );
      EnableWinListener( UserWindow, DPort, '', 0 );
    end
    else begin
      FullWindowState( UserWindow, WinLeftX, WinTopY, WinWidth, WinHeight, dumI, dumB, dumB, dumB, dumPS, dumTS );
      SetWindowName( UserWindow, progName );
    end;

{    PMRegisterProcess( PMPort, KPort, DPort, progName, UserWindow, NullPort );}

    (* Initialize the emergency message handler *)
    changingSize := false;
    new(pMsg);
    new(pReplyMsg);

    WindowViewport( UserWindow, UserVP, VPWidth, VPHeight );
    EnableNotifyExceptions( UserVP, DPort, true, true );
    ShowPathAndTitle( 'Type ''q'' to quit' );
    {$ifc debug then}
      writeln( 'Viewport = [', VPWidth:1, ',', VPHeight:1, ']' );
    {$endc}

    {$ifc debug then}
      writeln( 'Enabling emergency messages' );
    {$endc}
    if SoftEnable( false, true ) <> success then begin
      writeln( 'Cannot enable emergency message' );
      exit(List);
    end;
  end {NewWin};

(***** MAIN BODY *****)
begin
  (* Set up default values *)
  FontFile := FontFName;
  WildName := '*';
  DirName := '';
  
  (* Parse the command line arguments *)
  ParseCmdLine;

  (* Split off asynchronously and get new window if necessary *)
  NewWin;

  (* Get the font *)
  GetFont( fontFile, Font );

  (* Wait until we are asked to quit *)
  repeat
    (* List the files *)
    ListFiles;

    keyEv := GetEvent( UserVP, KeyWaitEvent );
  until keyEv.ch = 'q';
end.

