module PaneGlobs;

(***********************************************************************
 *
 *    AUTHOR: Lee Iverson
 *    ABSTRACT:
 *      This module is the internal definition module for the Pane
 *      structures.  This should be private to the Pane implementation
 *      modules.  All definitions exported from Panes are included in
 *      the exports section of the module Panes.
 *
 *    -----------
 *    CHANGE LOG:
 *      Jun 27 1984     Lee Iverson     ** Created **
 *
 ***********************************************************************)

exports

imports PaneDefs from PaneDefs;
imports Popup from Popup;

const
  (* Window picture parameters *)
  BorderWidth = 5;
  TitleHIndent = 10;
  TitleVBorder = 2;
  TitleVIndent = TitleVBorder + SysFontHeight;
  TitleHeight = TitleVBorder + TitleVIndent + 1;

  TitleVMiddle = BorderWidth + (TitleHeight div 2);
  TitleVLineSpace = 4;
  
  InsXOffset = BorderWidth;
  InsYOffset = BorderWidth + TitleHeight - 1;
  InsWOffset = 2*BorderWidth;
  InsHOffset = 2*BorderWidth + TitleHeight - 1;

  (* Region handling constants *)
  NumRegions = 150;
  Scrtch1Reg = 2;
  Scrtch2Reg = 3;
  FirstRegion = 4;
  None = -1;

  comDebug = true;

type
  (* Data structure representing all windows *)
  pPane = ^PaneRecord;
  PaneRecord = record
                 paneNum: Pane;                 {The number used by the user}
                 isIcon: Boolean;               {True if it is iconized}
                 outer, inner: PnRectangle;     {The defining rectangles}
                 name: String;                  {Name of window}
                 specRegion: Integer;           {The region representing title}
                 bodyRegion: Integer;           {The body of the pane}
                 regionList: Integer;           {The regions of the window}
                 prev, next: pPane;             {The links}
                 ul, ur, dl, dr: pPane;         {Structure describing}
                 lt, lb, rt, rb: pPane;         { current tiling}
               end;
  
  (* Data structure representing all regions *)
  Region = FirstRegion..NumRegions;
  RegionRecord = record
                   special: Boolean;            {Is it a special region?}
                   pn: pPane;                   {Parent window}
                   userRegion: Integer;         {The user's region number}
                   next, prev: Integer;         {Formed in a chain}
                 end;
  RegionArray = Array[Region] of RegionRecord;
  
  (* Pane commands *)
  PaneComms = ( cmdCreate, cmdDelete );

var
  (* The parent viewport for everything *)
  PaneViewport: Viewport;
  PaneWidth, PaneHeight: Integer;
  
  (* The array of panes *)
  PaneArr: Array[Pane] of record
                           case Integer of
                             1: ( panePtr: pPane );
                             2: ( nextFree: Pane );
                           end;

  (* The system font *)
  SysFont: Viewport;

  (* The window list *)
  FirstPane: pPane;
  FreePane, LastUsedPane: Pane;
  IconPane: pPane;
  ListenerPane: pPane;

  (* The region array *)
  Regions: RegionArray;
  FreeRegion, LastUsedRegion: Integer;

  (* The popup menu *)
  PanePopup: PopList;

  (* Default means of pane creation *)
  DefaultVal: Defaults;
  
  (* Dummy variables *)
  dummyInt: Integer;
  dummyStr: String;
  dummyBool: Boolean;

function
  abs(
    x: Integer
  ): Integer;

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Calculate the absolute value of an integer.
 *
 ***********************************************************************)

procedure
  background(
    left: Integer;
    top: Integer;
    width: Integer;
    height: Integer
  );

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Replace the rectangular area with the background.
 *
 ***********************************************************************)

procedure
  borderPane(
    Pn: pPane
  ); 

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Draw the border and title line for the window.
 *
 ***********************************************************************)

procedure
  clearPane(
    pn: pPane
  );

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Clear the inside region of the pane.
 *
 ***********************************************************************)

procedure
  clipLine(
    var ln: PnLine;
    var rect: PnRectangle
  );

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Clip the given line against the rectangle.
 *
 ***********************************************************************)

procedure
  clipRectangle(
    var rect: PnRectangle;
    var clipRect: PnRectangle
  );

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Clip the given rectangle against the other rectangle.
 *
 ***********************************************************************)

procedure
  clipString(
    var clipRect: PnRectangle;
        font: Viewport;
        adj: AdjustFunct;
    var dx: Integer;
    var dy: Integer;
    var str: String;
    var first: Integer;
    var last: Integer
  );

(***********************************************************************
 *
 *    AUTHOR: Lee Iverson
 *    ABSTRACT:
 *      Clip the given string against the rectangle.
 *
 ***********************************************************************)

{$ifc comDebug then}
  procedure
    DescribeAllPanes;

  procedure
    DescribePane(
      pn: pPane
    ); 
{$endc}

procedure
  disposeRegion(
    reg: Region
  ); 

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Free the region number for the region specified.
 *
 ***********************************************************************)

procedure
  disposePane(
    Pn: pPane
  ); 

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Dispose of the storage for a previously allocated Pane.
 *
 ***********************************************************************)

procedure
  drawArrow(
        lf: LineFunct;
    var src: PnPoint;
    var dest: PnPoint
  );

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Draw an arrow from the source to the destination.
 *
 ***********************************************************************)
 
procedure
  drawBox(
    lf: LineFunct;
    lx: Integer;
    ty: Integer;
    rx: Integer;
    by: Integer
  );

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Draw a box with the given line function.
 *
 ***********************************************************************)
 
function
  findAndLoadCursors(
        name: String;
    var numCursors: Integer
  ): CursorSet;

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Load the named cursor set and return it.  Also set the numCursors
 *        variable to reflect the number of cursors available in the file.
 *
 ***********************************************************************)

function
  findAndLoadFont(
    name: String
  ): Viewport;

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Load the named font, and return a viewport which specifies it.
 *
 ***********************************************************************)

function
  getAnEvent(
    howWait: KeyHowWait
  ): KeyEvent;

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Get an event, waiting as described, from the Pane viewport.
 *        This deals with the problem of emergency messages interrupting
 *        the event in a hopefully robust way.
 *
 ***********************************************************************)

procedure
  handleEmergMsg;

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Handle an emergency message.  Should be called from an emergency
 *        message exception handler.
 *
 ***********************************************************************)
  
procedure
  initBackground;

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Load the background for the Panes window.
 *
 ***********************************************************************)

procedure
  initPaneRegion(
    Pn: pPane
  ); 

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Create the title region and the window region for the Pane.
 *
 ***********************************************************************)

procedure
  initPopup
  ; 

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Initialize the data structure for the Pane manager's popup
 *        menu.
 *
 ***********************************************************************)

procedure
  listenTo(
    pn: pPane
  );

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Set the event handler to listen to the given pane.
 *        This also handles the visual cues needed to show this.
 *
 ***********************************************************************)

function
  max(
    a: Integer;
    b: Integer
  ): Integer;

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Calculate the maximum of two integers.
 *
 ***********************************************************************)

function
  min(
    a: Integer;
    b: Integer
  ): Integer;

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Calculate the minimum of two integers.
 *
 ***********************************************************************)

function
  newPane
  : pPane; 
  
(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Create a new Pane and install it in the window list.
 *
 ***********************************************************************)

function
  newRegion
  : Integer; 

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Get a free region from array.
 *
 ***********************************************************************)

procedure
  PaneMenu;

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Popup the pane manager's system menu and execute the operation
 *        selected.
 *
 ***********************************************************************)

procedure
  PushARegion(
    reg: Integer;
    left: Integer;
    top: Integer;
    width: Integer;
    height: Integer
  );
  
(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Push the region into the pane manager.  Initializes the cursor
 *        and parameters to the natural defaults.
 *
 ***********************************************************************)

procedure
  redrawPane(
    Pn: pPane
  ); 
  
(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Clear the area covered by the pane and then redraw the border
 *        and ask the user to refresh the inside of the pane.
 *
 ***********************************************************************)

procedure
  rePosition(
        Pn: pPane;
    var newPos: PnRectangle
  ); 

(***********************************************************************
 *
 *      AUTHOR: Lee Iverson
 *      ABSTRACT:
 *        Reposition the affected pane at the position and size described.
 *        This will call for the pane to be refreshed.
 *
 ***********************************************************************)

{$ifc comDebug then}
  procedure
    writePaneName(
      pn: pPane;
      newLine: Boolean
    );
{$endc}

private

{$include PaneGlobs.impl}.

