PROGRAM Find(input,output);

{-----------------------------------------------------------------------------}
{    ff.pas -- look for a file in the directory heirarchy.
{
{    Created by Steve Romig on November 18, 1981.
{
{---------------------------
{
{ HISTORY:
{       12-Apr-82  SMR  Added /HELP, /FIRST, /ALL.
{       11-Apr-82  SMR  Added pattern matching.  Uses UtilProgress stuff.
{       18-Nov-81  SMR  Created.
{
{-----------------------------------------------------------------------------}


IMPORTS fileUtils FROM Fileutils;
IMPORTS PERQ_STRING FROM PERQ_STRING;
IMPORTS CmdParse FROM CmdParse;                
IMPORTS UtilProgress FROM UtilProgress;
IMPORTS system FROM system;
IMPORTS allocDisk FROM AllocDisk;
IMPORTS PMatch FROM PMatch;

CONST
    Version     = 'Find V1.2';
    NumCmds     = 3;
    IdxHelp     = 1;
    IdxFirst    = 2;
    IdxAll      = 3;
    IdxNotFound = 4;
    IdxNotUnique = 5;

VAR
    StartWhere  : PathName;             { Where we start looking...}
    FindFile    : PathName;             { What we are looking for. }
    Wild        : BOOLEAN;
    FoundOne    : BOOLEAN;                                          
    FindAll     : BOOLEAN;
    CmdTable    : CmdArray;

{-----------------------------------------------------------------------------}
{ ABSTRACT:
{       Converts two names to uppercase before comparing them for equality.
{
{-----------------------------------------------------------------------------}

FUNCTION UpperEqual(name1, name2 : SimpleName) : BOOLEAN;

    BEGIN
    CnvUpper(name1);
    CnvUpper(Name2);
    UpperEqual := name1 = name2;
    END;

{-----------------------------------------------------------------------------}
{ ABSTRACT:
{       Initialize everything.
{
{----------------------------------------------------------------------------}

PROCEDURE InitFind;

    BEGIN
    StartWhere := '';
    FindAll := TRUE;
    FindFile := '';

    CmdTable[IdxHelp] := '/HELP';
    CmdTable[IdxFirst] := '/FIRST';
    CmdTable[IdxAll] := '/ALL';
    END;

{----------------------------------------------------------------------------}
{ ABSTRACT:
{       Give a nice message, and quit.
{
{----------------------------------------------------------------------------}

PROCEDURE DoHelp;

    BEGIN
    WriteLn;
    WriteLn('    FF is a program which looks for files in the PERQ''s directory');
    WriteLn('    tree.  To find the location of the file ''ff.pas'', you type');
    WriteLn('    ''FF ff.pas''.  FF will then run along, reporting all the');
    WriteLn('    places it found the file.  You can also give FF a pattern to');
    WriteLn('    look for, such as ''*.seg''.  ');
    WriteLn;
    WriteLn('    FF recognizes three special switches: /HELP, /FIRST and /ALL.');
    WriteLn('        /HELP prints this text.');
    WriteLn('        /FIRST causes FF to stop after finding the first occurance');
    WriteLn('          the file being searched for.');
    WriteLn('        /ALL causes FF to report all occurances of a file.');
    WriteLn('          This is the default action.');
    WriteLn;
    WriteLn('    The complete form of a call to FF is ''FF file directory''.');
    WriteLn('    The directory defaults to ''SYS:''.  Switches may appear anywhere,');
    WriteLn('    in any order.  ');
    WriteLn;
    RAISE ExitProgram;
    END;


{----------------------------------------------------------------------------}
{ ABSTRACT:
{       Depth first search of a directory and its children.
{
{----------------------------------------------------------------------------}

PROCEDURE SearchDirectory(where : PathName);

    VAR
        Name, TryName   : PathName;
        idummy  : INTEGER;
        fid     : FileID;
        Ending  : String[2];
        ScanPtr : PtrScanRecord;
        test    : BOOLEAN;

    BEGIN
    NEW(ScanPtr);
    ScanPtr^.InitialCall := TRUE;
    ScanPtr^.DirName := where;

    WHILE FSScan(ScanPtr, Name, idummy) DO
        BEGIN
        ShowProgress(1);
        IF UpperEqual(SubStr(name, Length(Name)-2, 3), '.DR')
        THEN BEGIN
            TryName := SubStr(name, 1, Length(Name)-3);
            AppendChar(TryName, '>');
            SearchDirectory(Concat(where, tryname));
            END
        ELSE BEGIN
            IF Wild
              THEN test := PattMatch(Name, FindFile, TRUE)
              ELSE test := UpperEqual(Name, FindFile);
            IF test
            THEN BEGIN    
                FoundOne := TRUE;
                name := concat(where, name);
                WriteLn(name);
                END;
            END;
        END;
    DISPOSE(ScanPtr);
    END;


{-----------------------------------------------------------------------------}
{ ABSTRACT:
{       For each partition on the device named, do SearchDirectory to
{       search all over.
{
{-----------------------------------------------------------------------------}

PROCEDURE SearchPartitions(where : PathName);

    VAR
        i, disk : INTEGER;
        Name    : PathName;
 
    LABEL
        1;

    BEGIN
    FOR i := 0 TO MAXDISKS-1 DO
        IF UpperEqual(DiskTable[i].RootPartition, 
                      substr(where, 1, length(where)-1))
            THEN BEGIN
                disk := i;
                GOTO 1;
                END;
    WriteLn;
    WriteLn(CHR(7), 'Bad device ', where, ' -- not found.');
    QuitProgress;
    EXIT(find);

1:
    FOR i := 1 TO MAXPARTITIONS DO
        IF PartTable[i].PartInUse AND (PartTable[i].PartDevice = disk)
            THEN BEGIN
                name := PartTable[i].PartName;
                AppendChar(name, '>');                
                SearchDirectory(concat(where, name));
                END;
    END;


{-----------------------------------------------------------------------------}
{ ABSTRACT:
{       Decide whether we are starting with a device or a directory,
{       and start the search with SearchPartitiions or SearchDirectory as
{       appropriate.
{
{-----------------------------------------------------------------------------}

PROCEDURE Search;

    BEGIN
    LoadBusy;
    FoundOne := FALSE;
    IF StartWhere[length(StartWhere)] = ':'     { Device name ends in : }
        THEN SearchPartitions(StartWhere)
        ELSE SearchDirectory(StartWhere);
    QuitProgress;
    IF NOT FoundOne
    THEN BEGIN
        WriteLn;
        WriteLn(CHR(7), FindFile, ' -- not found.');
        END;
    END;


{-----------------------------------------------------------------------------}
{ ABSTRACT:
{       Interpret UsrCMDLine and set the values for StartWhere and FindFile.
{       Decides whether FindFile is a pattern or not and sets WILD accordingly.
{
{-----------------------------------------------------------------------------}

PROCEDURE ReadCmdLine;
 
    VAR
        idummy        : INTEGER;
        Token, BrokeOn        : String;

    BEGIN
    RemDelimiters(UsrCMDLine, ' ', BrokeOn);            { Kill FIND }
    GetSymbol(UsrCMDLine, Token, ' ', BrokeOn);
    RemDelimiters(UsrCMDLine, ' ', BrokeOn);            { Next token }
    GetSymbol(UsrCMDLine, Token, ' ', BrokeOn);

    WHILE Token <> '' DO                { For all the tokens...}
        BEGIN
        IF Token = '?'                  { If ? then help!! }
          THEN BEGIN
              DoHelp;
              END
        ELSE IF Token[1] = '/'          { Switches start with / }
          THEN BEGIN
              ConvUpper(Token);
              CASE UniqueCMDIndex(Token, CmdTable, NumCmds) OF

                  IdxHelp       : DoHelp;

                  IdxFirst      : FindAll := FALSE;
        
                  IdxAll        : FindAll := TRUE;

                  IdxNotFound   : StdError(ErBadSwitch, token, TRUE);

                  IdxNotUnique  : StdError(ErSwNotUnique, token, TRUE);

                  OTHERWISE     : StdError(ErAnyError, 'I''m confused!!', TRUE);

              END;
              END
        ELSE BEGIN                      { Either set FindFile or StartWhere }
            IF FindFile = ''
              THEN FindFile := token    { Set FindFile before StartWhere }
              ELSE IF StartWhere = ''
                THEN StartWhere := Token        { Already set FindFile }
                ELSE StdError(ErCmdParam, Token, TRUE); 
            END;

        RemDelimeters(UsrCMDLine, ' ', BrokeOn);        { Next token }
        GetSymbol(UsrCMDLine, Token, ' ', BrokeOn);
        END;

    IF (StartWhere = '') AND (FindFile = '')    { Prompt for values...}
    THEN BEGIN
        Write('File to search for: ');
        ReadLn(Token);
        RemDelimeters(Token, ' ', BrokeOn);
        GetSymbol(Token, FindFile, ' ', BrokeOn);

        Write('Start looking at [SYS:]: ');
        ReadLn(Token);
        RemDelimeters(Token, ' ', BrokeOn);
        GetSymbol(Token, StartWhere, ' ', BrokeOn);
        IF StartWhere = ''
          THEN StartWhere := 'SYS:';
        END
    ELSE BEGIN                                  { FindFile was given, but }
        IF StartWhere = ''                      { not StartWhere. }
          THEN StartWhere := 'SYS:';
        END;

    Wild := IsPattern(FindFile);
                                
{ Fix up StartWhere, and check to see that it exists...}

    IF (StartWhere[length(StartWhere)] <> '>') AND      
        (StartWhere[length(StartWhere)] <> ':')
      THEN AppendChar(StartWhere, '>');
    IF (StartWhere[Length(StartWhere)] <> ':') AND 
        (FSLocalLookUp(StartWhere, idummy, idummy) = 0)
    THEN BEGIN
        WriteLn;
        WriteLn(StartWhere, ' -- non-existant directory or device.');
        RAISE ExitProgram;
        END;
    END;


{-----------------------------------------------------------------------------}
{ ABSTRACT:
{       This is the MAIN program.
{       Interpret the command line, and search.
{
{-----------------------------------------------------------------------------}

    BEGIN
    InitFind;
    FSAddToTitleLine(version);
    ReadCmdLine;
    Search;
    END.

