using System;
using System.Collections.Generic;
using System.Text;

namespace PERQDisk
{

    /// <summary>
    /// Represents a DeviceInformationBlock (DIB) structure, which is the "ground zero entry point"
    /// into a physical disk's file structure.
    /// 
    /// It contains a table of boot file mappings and the partition table.
    /// 
    /// The structure of this table is described in Section 2.2 of the "File System Data Structures" documentation in the 
    /// POS manual.
    /// </summary>
    public class DeviceInformationBlock
    {
        public DeviceInformationBlock(Sector dibSector)
        {
            if (dibSector == null)
            {
                throw new ArgumentNullException("dibSector");
            }

            _bootTable = new uint[52];
            _interpreterTable = new uint[52];
            _pibEntries = new uint[64];

            BuildData(dibSector);
        }

        private void BuildData(Sector dibSector)
        {            
            //Read boot and interpreter tables
            for (int i = 0; i < 52; i++)
            {
                _bootTable[i] = dibSector.ReadDWord(BootTableStart + i * 4);
                _interpreterTable[i] = dibSector.ReadDWord(InterpreterTableStart + i * 4);
            }

            _deviceName = dibSector.ReadString(228, 8);
            _deviceStart = dibSector.ReadDWord(236);
            _deviceEnd = dibSector.ReadDWord(240);

            //Read partition table
            for (int i = 0; i < 64; i++)
            {
                _pibEntries[i] = dibSector.ReadDWord(PartitionTableStart + i * 4);
            }

            _deviceRoot = dibSector.ReadDWord(500);
            _deviceType = dibSector.ReadWord(504);
        }

        public uint[] BootTable
        {
            get { return _bootTable; }
        }

        public uint[] InterpreterTable
        {
            get { return _interpreterTable; }
        }

        public string DeviceName
        {
            get { return _deviceName; }
        }

        public uint DeviceStart
        {
            get { return _deviceStart; }
        }

        public uint DeviceEnd
        {
            get { return _deviceEnd; }
        }

        public uint[] PibEntries
        {
            get { return _pibEntries; }
        }

        public uint DeviceRoot
        {
            get { return _deviceRoot; }
        }

        public ushort DeviceType
        {
            get { return _deviceType; }
        }    

        // Constants
        private const int BootTableStart = 20;
        private const int InterpreterTableStart = 124;
        private const int PartitionTableStart = 244;
        
        private uint[] _bootTable;
        private uint[] _interpreterTable;
        private string _deviceName;
        private uint _deviceStart;
        private uint _deviceEnd;
        private uint[] _pibEntries;
        private uint _deviceRoot;
        private ushort _deviceType;
    }    

    /// <summary>
    /// Represents a file system volume, an abstraction that sits on top of a 
    /// PhysicalDisk.
    /// </summary>
    public class Volume
    {
        public Volume(PhysicalDisk disk)
        {
            if (disk == null)
            {
                throw new ArgumentNullException("disk");
            }

            if (!disk.Loaded)
            {
                throw new InvalidOperationException("Supplied PhysicalDisk has not been loaded!");
            }

            _disk = disk;

            ReadVolumeInfo();
        }

        /// <summary>
        /// Dumps interesting info from the Volume's DIB to screen.
        /// </summary>
        public void PrintDIBInfo()
        {

            Console.WriteLine(" Device Name: '{0}'", DIB.DeviceName); 
            Console.WriteLine(" Device Type: {0}", DIB.DeviceType);

            Console.WriteLine(" Start of Disk: {0}", _disk.LDAToLogicalBlock(DIB.DeviceStart));
            Console.WriteLine(" End of Disk: {0}", _disk.LDAToLogicalBlock(DIB.DeviceEnd));
            Console.WriteLine(" Device Root: {0}", DIB.DeviceRoot);

            Console.WriteLine("Boot Table:");

            for (int i = 0; i < DIB.BootTable.Length; i++)
            {
                if (DIB.BootTable[i] != 0)
                {
                    Console.WriteLine(" {0} - {1}", (char)( (i < 26 ? 'a' : 'A'-26) + (char)i), _disk.LDAToLogicalBlock(DIB.BootTable[i]));
                }
            }

            Console.WriteLine("Interpreter Table:");
            for (int i = 0; i < DIB.InterpreterTable.Length; i++)
            {
                if (DIB.InterpreterTable[i] != 0)
                {
                    Console.WriteLine(" {0} - {1}", (char)((i < 26 ? 'a' : 'A' - 26) + (char)i), _disk.LDAToLogicalBlock(DIB.InterpreterTable[i]));
                }
            }            
        }


        public DeviceInformationBlock DIB
        {
            get { return _dib; }
        }

        public List<Partition> Partitions
        {
            get { return _partitions; }
        }

        private void ReadVolumeInfo()
        {
            //Read the DIB from the first sector on Cyl 0, Track 1.
            _dib = new DeviceInformationBlock(_disk.GetSector(0, 1, 0));

            ReadPartitionInfo();           
        }

        private void ReadPartitionInfo()
        {
            _partitions = new List<Partition>(16);

            for (int i = 0; i < DIB.PibEntries.Length; i++)
            {
                if (DIB.PibEntries[i] != 0)
                {
                    // This is a valid Partition entry -- read it in.
                    PartitionInformationBlock pib = new PartitionInformationBlock(_disk.GetSector(_disk.LDAToLogicalBlock(DIB.PibEntries[i])));
                    _partitions.Add(new Partition(_disk, pib));                                       
                }
            }
        }

                 
        private DeviceInformationBlock _dib;  //not to be confused with ZIM.
        private List<Partition> _partitions;
        private PhysicalDisk _disk;
    }    
}
