Program C100;
{--------------------------------------------------------------------}
{
{          Written by S. P. Harbison at CMU, May 1981.
{
{ Abstract:
{    Concept-100 terminal emulator.
{
{    This program allows a Perq to be attached to a host computer over
{    the RS232 interface and to emulate a Concept-100 terminal.
{    (Human Designed Systems, Inc.)
{
{    C100 also includes special support for the pointing device
{    for use with ZOG at CMU; it causes the cursor/mouse to mimic
{    a Carroll Mfg. Co. touch screen.  Most users should ignore this.
{
{ Use:
{
{    Invoke the program from the executive.  Characters typed at the
{    keyboard will be sent over the RS232 line, and characters received
{    over that line will be interpreted according to C100 protocols.
{  
{    When using the mouse; the position data is sent to the host
{    when the mouse button is RELEASED.
{
{    One keyboard character is handled in a special way:
{
{       Typing ^^ at the keyboard will exit the emulator and allow the
{       user to execute some meta-commands such as setting the baud rate.
{       Typing a second ^^ results in sending a single ^^ to the host.
{
{ Restrictions:
{    Not all Concept-100 functions are emulated, although enough are to
{    permit using the "Emacs"-style display editors popular at CMU.
{
{    The current version works fine at 1200 baud; it is too slow at
{    9600 baud, and misses incoming characters.  The default is 1200.
{
{ Extensions:
{    1. Extend functionality; implement all C-100 functions.
{
{    2. The program should be speeded up to allow operation at 9600 baud.
{
{    3. The use of the Screen package is minimal; the remaining 
{    references to it should be replaced with local routines to do the
{    work.  (SPutChr and the cursor routines; createwindow, etc.)
{    
{    4. Add customization files so that users can tailor keys to
{       their taste (like changing the backspace key to send rubout)
{
{    5. Look at all eight bits of the keyboard, and optionally (see 4)
{       allow meta codes through (ctrl+shift=>meta)
{
{--------------------------------------------------------------------}
{
{ Change log:
{
{ 16 Nov 82 steve  Combined jag's c100 with latest c100 and d100 from
{                  the Spice vax.  Added DefCrsChar const.
{                  Implemented mcCurUnd & mcCurBlk.
{                  From d100:  large's fix to ignore HELP key.
{                  From c100:  sph's DefaultBaud const.
{
{ 20 May 82 jag    Increased input buffer size.  Added code for block
{                  insert/delete line.  Minor tweaks.
{
{ 22 Apr 82 Chiron Major modifications to allow use on CMUC. 
{                  When host sets length to 24, interpret this
{                  as meaning 60, as Perqs are longer than 24 lines.
{                  Fixed various problems with Line Insert/Delete and 
{                  Clear to End of Frame.  This version should work
{                  better when talking to CMUC.
{
{ 24 Nov 81 (ggr)  Fixed bug in cursor support for ZOG.
{
{ 21 Nov 81 (rpf)  Installed sphs direct cursor addressing bug fix.
{
{ 24 Aug 81 (sph)  Added code to save a transcript file.
{
{ 18 Jul 81 (rpf)  Tweaked so that it would compile under 3crocks
{                  system D.4.  (Renamed ToggleCursor to ToggelCursor
{                  to avoid name conflicts.)  Added ability to change
{                  cursor character.
{
{ 30 Jun 81 (rpf)  Minor tweaks.  Keep squashing CtrlSPending to avoid
{                  getting wedged.  Avoid toggling cursor all the time.
{                  Unpack AttrRec.  Update attributes when they are
{                  changed rather than on main path.  Changed
{                  meta-character to ^^ so I can do reverse search.
{                  Replaced most 3Crocks Screen calls with home brews.
{                  Only ones left are CreateWindow, ChangeWindow,
{                  GetFont and ScreenReset.  Screen.p has to stick
{                  around for defns of Font, etc. anyway.
{                  Not sure it can do an honest 9600, but it seems to
{                  work ok most of the time.  Now default 9600 baud rate.
{                  Added host input buffer ring.  Now looks ahead to
{                  try to avoid scrolling.  (Scrolling is very slow.)
{                  Also tried optimizing insert characters, but it
{                  makes little difference, so I took it out again.
{
{ 20 May 81 (sph)  V 1.2 Made mouse use optional; added option for
{                  increasing the number of lines displayed.
{
{ 16 May 81 (sph)  Fixed wraparound bug in Xp1; set default baud rate
{                  to 1200.
{
{ 14 May 81 (ggr)  Added cursor support for ZOG.
{
{ 10 May 81 (sph)  Created.
{
{--------------------------------------------------------------------}

imports IO from IO;
imports IOErrors from IOErrors;
imports System from System;
imports Raster from Raster;
imports Screen from Screen;
imports RS232Baud from RS232Baud;

{$R-}

const
      debug = false;
      DefaultBaud = '4800';
      DefCrsChar = '_';

const
      chNul     = chr(0);
      chBel     = chr(7);
      chBS      = chr(8);
      chTab     = chr(9);
      chLF      = chr(10);
      chCR      = chr(13);
      chFF      = chr(12);
      chEsc     = chr(27);
      chUnder   = chr(95);
      chDel     = chr(127);
      
      chAPL     = chr(14);  { Apl mode - alternate to mcAPL }
      chASCII   = chr(15);  { ASCII mode - alternate to mcASCII }
      chHostMC  = chESC;    { Character signalling C100 escape sequence }
      chKbdMC   = chDEL;    { Character to act like MULTICODE on C100 kbd }
      chExit    = chr(30);  { C100 simulation escape character (^^) }

{ Escape/Multi-Code sequences (2nd character) }

const
      mcReset   = ',';      { Reset all functions }
      mcStatus  = '+';      { Toggle Status line }
      mcDepend  = '3';      { Set device dependent }
      mcIndep   = '#';      { Set device independent }
      mcChgMsg  = 'o';      { Change message character }
      
      mcAPL     = 'O';      { APL mode }
      mcASCII   = ')';      { ASCII mode }
      mcUser    = 'u';      { User mode }
      mcProg    = 'U';      { Programmer mode }
      mcText    = 'f';      { Text mode }
      mcForm    = 'F';      { Form mode }
      mcScroll  = 's';      { Scroll mode }
      mcPage    = 'S';      { Page mode }
      mcChar    = '7';      { Character mode }
      mcBlock   = '&';      { Block mode }
      mcULC     = '5';      { Upper/Lower case mode }
      mcCaps    = '%';      { Caps lock mode }
      mcFDupl   = '8';      { Full duplex mode }
      mcHDupl   = '*';      { Half duplex mode }
      mcRemote  = '9';      { Remote mode }
      mcLocal   = '(';      { Local mode }
      mcTransp  = 'T';      { Transparent mode on }
      mcOpaque  = 't';      { Transparent mode off }
      mcAutLF   = 'L';      { Auto linefeed on }
      mcManLF   = 'l';      { Auto linefeed off }
      mcAutTab  = 'B';      { Auto tabs on }
      mcManTab  = 'b';      { Auto tabs off }

      mcBTab    = chr(39);  { (') Back tab }
      mcStTab   = ']';      { Tab set }
      mcClTab   = chr(95);  { (_) Tab clear }
      mcCurUp   = ';';      { Cursor up }
      mcCurDown = '<';      { Cursor down }
      mcCurRight= '=';      { Cursor right }
      mcCurLeft = '>';      { Cursor left }
      mcHome    = '?';      { Home }
      mcWrAddr  = 'a';      { Write cursor address }
      mcRdAddr  = 'A';      { Read cursor address }
      mcEOText  = 'p';      { End of text }
      mcCurUnd  = 'w';      { Set cursor to blink underline }
      mcCurBlk  = 'W';      { Set cursor to blink block }
      
      mcInsMd   = chr(16);  { Insert mode on (^P) }
      mcXInsMd  = chr(0);   { Insert mode off (^@) }
      mcDlChLin = chr(17);  { Delete character in line/field (^Q) }
      mcDlChWin = chr(1);   { Delete character in window }
      mcInsLin  = chr(18);  { Insert line (^R) }
      mcDelLin  = chr(2);   { Delete line }
      mcClUnLin = chr(19);  { Clear unprotected to end of line/field (^S) }
      mcClUnWin = chr(3);   { Clear unprotected to end of window (^C) }
      mcClAlLin = chr(21);  { Clear all to end of line/field (^U) }
      mcClAlWin = chr(5);   { Clear all to end of window }

      mcBlink   = 'C';      { Blink on }
      mcXBlink  = 'c';      { Blink off }
      mcReverse = 'D';      { Reverse video on }
      mcXReverse= 'd';      { Reverse video off }
      mcHalf    = 'E';      { Half bright on }
      mcXHalf   = 'e';      { Half bright off }
      mcUnder   = 'G';      { Underline on }
      mcXUnder  = 'g';      { Underline off }
      mcNonDis  = 'H';      { Nondisplay on }
      mcXNonDis = 'h';      { Nondisplay off }
      mcProtect = 'I';      { Protection on }
      mcXProtect= 'i';      { Protection off }
      mcSlChSet = 'j';      { Select character set }
      mcRvScreen= 'k';      { Reverse screen video }
      mcNmScreen= 'K';      { Normal screen video }
      mcHfBrPr  = 'M';      { Half bright protected fields }
      mcFuBrPr  = 'm';      { Full bright protected fields }
      mcStAttr  = 'N';      { Set attribute word }
      mcRdAttr  = 'n';      { Read attribute word }
      mcRptHor  = 'r';      { Repeat character horizontal }
      mcRptVer  = 'R';      { Repeat character vertical }
      mcStBlAttr= 'J';      { Set attributes of block }
      
      mcStBaud  = 'O';      { Set baud rate }
      mcStParity= 'P';      { Set parity }
      mcStSOT   = '1';      { Set start of print/transmit }
      mcTrLin   = chr(20);  { Transmit line/field (^T) }
      mcTrWin   = chr(4);   { Transmit window (^D) }
      mcTrAlLin = chr(22);  { Transmit all line (^V) }
      mcTrAlWin = chr(6);   { Transmit all window (^F) }
      
      mcStOutNet= 'Y';      { Set output network }
      mcRdOutNet= 'y';      { Read output network }
      mcFunRoute= 'Q';      { Function route }
      mcPrWin   = '{';      { Print window }
      mcPrLin   = '|';      { Print line }
      mcAttPr   = '}';      { Attach printer }
      mcDetPr   = '~';      { Detach printer }
      mcPrFF    = 'Z';      { FF prior to print on }
      mcXPrFF   = 'z';      { FF prior to print off }
      mcAttTap  = '@';      { Attach tape }
      mcDetTap  = '^';      { Detach tape }
      mcTapMsg  = chr(96);  { Message to tape (') }

      mcSetMouse= chr(29);  { Set the mouse on or off }
      mcInsBlk  = chr(30);  { Insert a block of lines }
      mcDelBlk  = chr(31);  { Delete a block of lines }
      mcDefWin  = 'v';      { Define window }
      mcTieWin  = 'q';      { Tie window }
      mcPagUp   = '.';      { Page up }
      mcPagDn   = '-';      { Page down }
      mcScrUp   = '\';      { Scroll up }
      mcScrDn   = 'L';      { Scroll down }
      mcStaScr  = 'V';      { Start of screen }
      
      mcDefPrg  = '4';      { Program }
      mcRstAll  = '$';      { Reset all (function keys) }
      mcTrFun   = 'X';      { Set function pad to transmit mode }
      mcExFun   = 'x';      { Set function pad to execute mode }
 
{ Cursor }

var
      CrsChr    : char;

type
      JiffyType = packed record
         case integer of
         0 : (
            Dbl     : Double );
         1 : (
            Pad0    : 0..   #7;
            Toggle  : Boolean;      { flips every ~1/8 sec }
            Pad1    : 0..#7777 );
         end;

var
      CurX, CurY: integer;
      CursorOn  : boolean;
      JiffyTime : JiffyType;

{ KSet }

const
      KSetSLen  = 48;       { same as SScreenW }

var
      KSet      : FontPtr;
      KSetRaster: FontPtr;  { KSet+#404 }

{ Current window dimensions }

var
      WinTopX, WinTopY, WinBotX, WinBotY: integer;
      WinWidth, WinHeight : Integer;

{ Attribute words }

type AttrRec = record
       Charset  : 0..3;
       Reverse  : boolean;
       Half     : boolean;
       Protect  : boolean;
       NonDis   : boolean;
       Blink    : boolean;
       Under    : boolean;
       end;

var
     Attr       : AttrRec;  { Current character attributes }
            
 { Mode variables }
 
 var
       mdRemote,
       mdFDupl,
       mdChar,
       mdULC,
       mdDepend,
       mdUser,
       mdText,
       mdScroll,
       mdTransp,
       mdAutLF,
       mdAutTab: boolean;

var
       AmInserting: boolean;
       ChrFunc  : integer;  { Current RasterOp for putting characters }

{ The C100 display memory image }

const
      MaxLine   = 59;
      MaxChar   = 79;
      
{ Screen coordinates for rasterop }

var
       LinePos  : array [0..MaxLine] of integer;
       CharPos  : array [0..MaxChar] of integer;

{ Tab stops }

var
      TabStops  : array [0..MaxChar] of boolean;

{ RS232 status }

var
      Stat      : DevStatusBlock;
      
{ Screen coordinates }

const
      CharBase  =  1;
      Char0X    = 20;
      Char0Y    = 40;
      CharWid   =  9;       { Distance between successive characters }
      CharHit   = 13;       { Actual Height of characters }
      LineHit   = 15;       { Distance between successive lines }
      W1Bot     = MaxLine*LineHit+Char0Y+10;  { Bottom of C100 Window }

{ Tablet coordinates }

var
      tabX, tabY: integer;
      tabLine, tabChar: integer;
      WantTablet: boolean;
      EmacsTablet: boolean;
      EmacsKey:char;

{ Input character buffering }

var
      HostInCh  : char;     { Character read from host }
      KbdInCh   : char;     { Character read from keyboard }

var
      QuitDisplay: boolean; { Give someone else a chance }
                       
const
      HostInMask= #3777;     { must be 1 less than a power of 2 }

var
      HostInHead: Integer;  { Inserting end }
      HostInTail: Integer;  { Deleting end  }

var
      HostInBuf : array [0..HostInMask] of char;

{ Salvage procedure declaration order }

procedure ToHost( ch: char );
      forward;
procedure ToDisplay( ch: char );
      forward;

{ misc stats }

var
      MaxInserts: Integer;
      MaxScrolls: Integer;
      MaxBuffer : Integer;

{ Transcript files }

var
    Transfile   : text;
    TFileName   : string;
    Transcript  : boolean;

{--------------------------------------------------------------------}
{
{ Abstract:
{    Debugging procedures
{
{--------------------------------------------------------------------}


procedure DStr( s: string );
begin
ChangeWindow(2);
Writeln;
Write(s);
ChangeWindow(1);
end; { DStr }


procedure DNum( n: integer );
begin
ChangeWindow(2);
Write(n);
ChangeWindow(1);
end; { DStr }

procedure UpdateAttributes;
{--------------------------------------------------------------------}
{
{ Abstract
{    guess
{
{--------------------------------------------------------------------}
begin
if Attr.Reverse
then  ChrFunc := RNot
else  ChrFunc := RRpl;
end; { UpdateAttributes }

procedure FrameWindow(TopX, TopY, BotX, BotY: integer);
{--------------------------------------------------------------------}
{
{ Abstract
{    Records dimensions of window
{
{--------------------------------------------------------------------}
begin
WinTopX   := TopX;
WinTopY   := TopY;
WinBotX   := BotX;
WinBotY   := BotY;
WinWidth  := (WinBotX-WinTopX+1)*CharWid+1;
WinHeight := (WinBotY-WinTopY+1)*LineHit;
end; { FrameWindow }

procedure ToggelCursor;
{--------------------------------------------------------------------}
{
{ Abstract
{    Invert cursor at current location.
{
{--------------------------------------------------------------------}
begin
CursorOn := not CursorOn;
with KSet^.Index[ord(CrsChr)]
do    begin
      RasterOp( RXor,
                Width,
                CharHit,
                CharPos[CurX]-1,
                LinePos[CurY]-1,
                SScreenW,
                SScreenP,
                Offset,
                (Line*CharHit),
                KSetSLen,
                KSetRaster );
      end;
end;

procedure PutChr(ch: char);
{--------------------------------------------------------------------}
{
{ Abstract
{    Write a character at current location
{
{--------------------------------------------------------------------}
begin
with KSet^.Index[ord(ch)]
do    begin
      RasterOp( ChrFunc,
                Width,
                CharHit,
                CharPos[CurX],
                LinePos[CurY],
                SScreenW,
                SScreenP,
                Offset,
                (Line*CharHit),
                KSetSLen,
                KSetRaster );
      end;
if Attr.Under and (ch<>'_') then
with KSet^.Index[ord('_')]
do    begin
      RasterOp( Ror,
                Width,
                CharHit,
                CharPos[CurX],
                LinePos[CurY],
                SScreenW,
                SScreenP,
                Offset,
                (Line*CharHit),
                KSetSLen,
                KSetRaster );
      end;
end;

procedure ClearScreen;
{--------------------------------------------------------------------}
{
{ Abstract
{    Clear C100 screen and home cursor
{
{--------------------------------------------------------------------}
begin
RasterOp( RXor,
          WinWidth,
          WinHeight,
          CharPos[WinTopX],
          LinePos[WinTopY],
          SScreenW,
          SScreenP,
          CharPos[WinTopX],
          LinePos[WinTopY],
          SScreenW,
          SScreenP );
CurX := 0;
CurY := 0;
QuitDisplay := True;
end; { ClearScreen }

procedure ClearRight;
{--------------------------------------------------------------------}
{
{ Abstract:
{    Clear line to the right, beginning with (CurX,CurY); used to
{    delete remainder of line.
{
{--------------------------------------------------------------------}
var
   func:integer;
begin       
if Attr.Reverse then func := RXNor
                else func := RXor;
rasterop( func,
          (MaxChar-CurX+1)*CharWid+1,
          LineHit,
          CharPos[CurX],
          LinePos[CurY],
          SScreenW,
          SScreenP,
          CharPos[CurX],
          LinePos[CurY],
          SScreenW,
          SScreenP );
end; {ClearRight}

procedure ClearEOF;
{--------------------------------------------------------------------}
{
{ Abstract
{    Clear C100 screen to end of frame
{
{--------------------------------------------------------------------}
begin
ClearRight;
RasterOp( RXor,
          WinWidth,
          (WinBotY-(CurY+1))*LineHit,
          CharPos[WinTopX],
          LinePos[CurY+1],
          SScreenW,
          SScreenP,
          CharPos[WinTopX],
          LinePos[CurY+1],
          SScreenW,
          SScreenP );
end; { ClearScreen }

procedure ShiftRight( Count : integer );
{--------------------------------------------------------------------}
{
{ Abstract:
{    Shift Count characters beginning with (CurX,CurY) to the right;
{    Used to make room for characters in insert mode.
{
{--------------------------------------------------------------------}
begin
rasterop( RRpl,
          (MaxChar-(CurX+Count)+1)*CharWid+1,
          LineHit,
          CharPos[CurX+Count],
          LinePos[CurY],
          SScreenW,
          SScreenP,
          CharPos[CurX],
          LinePos[CurY],
          SScreenW,
          SScreenP );
end; {ShiftRight}

procedure ShiftLeft;
{--------------------------------------------------------------------}
{
{ Abstract:
{    Shift all characters to the right of (CurX, CurY) to the left.
{    the character (CurX,CurY) is lost.  Used to delete a character 
{    and close up.
{
{--------------------------------------------------------------------}
begin
rasterop( RRpl,
          (MaxChar-CurX)*CharWid+1,
          LineHit,
          CharPos[CurX],
          LinePos[CurY],
          SScreenW,
          SScreenP,
          CharPos[CurX+1],
          LinePos[CurY],
          SScreenW,
          SScreenP );
end; {ShiftLeft}

procedure ScrollUp( TopY : integer; Count : integer );
{--------------------------------------------------------------------}
{
{ Abstract:
{    Scroll lines (in current window) below TopY up Count lines;
{    Blank bottom Count lines.  Used to delete lines and close up
{    and for general scrolling at screen bottom.
{
{--------------------------------------------------------------------}
begin

{ Scroll up }

rasterop( RRpl,
          WinWidth,
          (WinBotY-(TopY+Count)+1)*LineHit,
          CharPos[WinTopX],
          LinePos[TopY],
          SScreenW,
          SScreenP,
          CharPos[WinTopX],
          LinePos[TopY+Count],
          SScreenW,
          SScreenP );

{ Clear bottom line }

rasterop( RXor,
          WinWidth,
          LineHit*Count,
          CharPos[WinTopX],
          LinePos[WinBotY-Count+1],
          SScreenW,
          SScreenP,
          CharPos[WinTopX],
          LinePos[WinBotY-Count+1],
          SScreenW,
          SScreenP );

QuitDisplay := True;
end; {ScrollUp}

procedure ScrollDown (CurY:integer; Count:integer);
{--------------------------------------------------------------------}
{
{ Abstract:
{    Scroll lines (in current window) at CurY and below down one line;
{    set line CurY to blanks.
{
{--------------------------------------------------------------------}
begin

{ Scroll down }

rasterop( RRpl,
          WinWidth,
          (WinBotY-(CurY+Count-1))*LineHit,
          CharPos[WinTopX],
          LinePos[CurY+Count],
          SScreenW,
          SScreenP,
          CharPos[WinTopX],
          LinePos[CurY],
          SScreenW,
          SScreenP );

{ Clear current line }
rasterop( RXor,
          WinWidth,
          LineHit*Count,
          CharPos[WinTopX],
          LinePos[CurY],
          SScreenW,
          SScreenP,
          CharPos[WinTopX],
          LinePos[CurY],
          SScreenW,
          SScreenP );

QuitDisplay := True;
end; {ScrollDown}

procedure Yp1;
{--------------------------------------------------------------------}
{ 
{ Abstract:
{    Advance cursor by 1 line, taking care to wrap around properly
{    For pedagogic purposes only -- actually expanded in line.
{
{--------------------------------------------------------------------}
begin
if (CurY >= WinBotY)
then  CurY := WinTopY
else  CurY := CurY + 1
end; { Yp1 }

procedure Xw1;
{--------------------------------------------------------------------}
{ 
{ Abstract:
{    Handle wrap around when advancing cursor past end of line
{
{--------------------------------------------------------------------}
begin
CurX := WinTopX;
if (CurY >= WinBotY)
then  begin
      ScrollUp(WinTopY, 1);
      CurY := WinBotY
      end
else  CurY := CurY + 1
end; { Xw1 }

procedure Xp1;
{--------------------------------------------------------------------}
{ 
{ Abstract:
{    Advance cursor by 1 char, taking care to wrap around properly
{    For pedagogic purposes only -- actually expanded in line.
{
{--------------------------------------------------------------------}
begin
if (CurX >= WinBotX)
then  Xw1
else  CurX := CurX + 1
end; { Xp1 }

procedure Ym1;
{--------------------------------------------------------------------}
{ 
{ Abstract:
{    Retard cursor by 1 line, taking care to wrap around properly
{    For pedagogic purposes only -- actually expanded in line.
{
{--------------------------------------------------------------------}
begin
if (CurY <= WinTopY)
then  CurY := WinBotY
else  CurY := CurY - 1
end;

procedure Xm1;
{--------------------------------------------------------------------}
{ 
{ Abstract:
{    Retard cursor by 1 character, taking care to wrap around properly
{    For pedagogic purposes only -- actually expanded in line.
{
{--------------------------------------------------------------------}
begin
if (CurX<=WinTopX)
then  begin
      CurX := WinBotX;
      if (CurY <= WinTopY)
      then  CurY := WinBotY
      else  CurY := CurY - 1
      end
else  CurX := CurX - 1
end; { Xm1 }

procedure ResetInit;
{--------------------------------------------------------------------}
{ Abstract:
{     Called from main program (power-on) and DoMC (mcReset) to
{     initialize the C100 state.
{
{--------------------------------------------------------------------}
type PtrSegOfs = record
      case integer of
      0 : ( Ptr : FontPtr );
      1 : ( Seg, Ofs : integer );
      end;
var
      i,j,n: integer;
      Trix: PtrSegOfs;
begin
writeln( chFF );

{ Set tab stops }

for i := 0 to MaxChar
do    TabStops[i] := ( (i mod 8)=0 );

{ Set up screen coordinates }

n := Char0Y - LineHit + CharBase;
for i := 0 to MaxLine
do    begin
      LinePos[i] := n;
      n := n + LineHit;
      end;

n := Char0X;
for i := 0 to MaxChar 
do    begin
      CharPos[i] := n;
      n := n + CharWid;
      end;

{ Default window settings }

FrameWindow(0, 0, MaxChar, 59);
CurX := 0;
CurY := 0;

AmInserting := false;

{ Default modes }

Attr.Blink      := false;
Attr.Reverse    := false;
Attr.Half       := false;
Attr.Under      := false;
Attr.NonDis     := false;
Attr.Protect    := false;
Attr.Charset    := 0;

KSet            := GetFont;
Trix.Ptr        := KSet;
KSetRaster      := MakePtr(Trix.Seg, Trix.Ofs+#404, FontPtr);
ChrFunc         := RRpl;
CursorOn        := False;

end; {ResetInit}

function GetNextCh( FromHost: boolean ): char;
{--------------------------------------------------------------------}
{
{ Abstract:
{    Wait for and return the next character from host or keyboard.
{
{ Parameters:
{    FromHost:  if true, wait for host character; else wait for keyboard
{
{ Return value:
{    The next character
{
{--------------------------------------------------------------------}
var
      ch: char;
      HostInScan: integer;
begin
if FromHost
then  begin
      HostInScan := LAnd(HostInTail+1, HostInMask);
      if (HostInScan = HostInHead)
      then  begin
            repeat until IOCRead( RS232In, ch ) = IOEIOC;
            ch := chr( LAnd(ord(ch),#177)  );
            end
      else  begin
            ch := HostInBuf[HostInScan];
            HostInTail := HostInScan
            end
      end
else  begin
      repeat until IOCRead( TransKey, ch ) = IOEIOC;
      CtrlCPending := false;
      CtrlSPending := false;
      end;
GetNextCh := ch;
end; {GetNextCh}

procedure DoMC( FromHost: boolean );
{--------------------------------------------------------------------}
{ Abstract:
{    Process a multi-code command from keyboard or host
{
{ Parameters:
{    FromHost: true if host send MC; false if keyboard sent MC;
{
{--------------------------------------------------------------------}
type
      bitoverlay = packed record
        case Integer OF
        0 : (
            bit0   : boolean;
            bit1   : boolean;
            bit2   : boolean;
            bit3   : boolean;
            bit4   : boolean;
            bit5   : boolean;
            bit67  : 0..3 );
        1 : (
            chr0   : char )
        end;
var
      c, ch: char;
      OldX, OldY: integer;
      i,n: integer;
      TopX, TopY, BotX, BotY: integer;
      b : bitoverlay;
label 1;
      
procedure UnImpl( n: integer; FromHost: boolean );
    var i: integer;
    begin
    ChangeWindow(2);
    if ( (ch >= ' ') and (ch < chr(#175)) )
    then  Write( '{MC ''', ch, '''')
    else  Write( '{MC ', ord(ch):3 );
    for i := 1 to n
    do    begin
          ch := GetNextCh( FromHost );
          Write( ord(ch):4 );
          end;
    Write('}');
    ChangeWindow(1);
    end; {UnImpl}

begin
{ Wait for the control code }
ch := GetNextCh( FromHost );

{ Pass two consecutive chKbdMC's to host }

if (not FromHost) and (ch=chKbdMC)
then  begin
      ToHost( chKbdMC );
      goto 1;
      end;

{ Decode the MC code }

case ch of

  mcReset:
      begin
      ResetInit;
      if debug then DStr('Reset');
      UpdateAttributes;
      ClearScreen;
      end;
  
  mcCurUp:
      begin
      if (CurY <= WinTopY)
      then  CurY := WinBotY
      else  CurY := CurY - 1;
      if debug then DStr('CurUp');
      end;

  mcCurDown:
      begin
      if (CurY >= WinBotY)
      then  CurY := WinTopY
      else  CurY := CurY + 1;
      if debug then DStr('CurDown');
      end;

  mcCurLeft:
      begin
      if (CurX <= WinTopX)
      then  begin
            CurX := WinBotX;
            if (CurY <= WinTopY)
            then  CurY := WinBotY
            else  CurY := CurY - 1
            end
      else  CurX := CurX - 1;
      if debug then DStr('CurLeft');
      end;

  mcCurRight:
      begin
      if (CurX >= WinBotX)
      then  Xw1
      else  CurX := CurX + 1;
      if debug then DStr('CurRight');
      end;

  mcHome:
      begin
      if debug then DStr('Home');
      CurX := 0;
      CurY := 0;
      end;
  
  mcInsMd:
      begin
      if debug then DStr('InsMd');
      AmInserting := true;
      end;
  
  mcXInsMd:
      begin
      if debug then DStr('XInsMd');
      AmInserting := false;
      end;
    
  mcWrAddr:
      begin
      { NOTE:  coordinates are supposed to be window-relative }
      CurY := ord(GetNextCh(FromHost)) - 32 + WinTopY;
      CurX := ord(GetNextCh(FromHost)) - 32 + WinTopX;
      
      if debug 
      then  begin
            DStr('WrAddr');
            DNum( CurY );
            DNum( CurX );
            end;
      end;
  
  mcRptHor:
      begin
      c := GetNextCh( FromHost );
      n := ord( GetNextCh( FromHost ) );
      if debug
      then  begin
            DStr('RptHor');
            DNum( ord(c) );
            DNum( n );
            end;
      for i := 1 to n-32
      do    begin
            HostInBuf[HostInTail] := c;
            HostInTail := LAnd(HostInTail-1, HostInMask);
            end;
      end;
  
  mcStAttr:
      begin
      c := GetNextCh( FromHost );
      if debug then 
            begin
            DStr('StAttr');
            DNum( ord(c) );
            end;
      b.chr0 := c;
      Attr.NonDis  := b.bit0;
      Attr.Blink   := b.bit1;
      Attr.Under   := b.bit2;
      Attr.Protect := not b.bit3;
      Attr.Half    := b.bit4;
      Attr.Reverse := b.bit5;
      Attr.Charset := b.bit67;
      UpdateAttributes;
      end;
  
  mcBlink:
      begin
      Attr.Blink := true;
      end;
  
  mcXBlink:
      begin
      Attr.Blink := false;
      end;
  
  mcReverse:
      begin
      Attr.Reverse := true;
      ChrFunc := RNot;
      end;
  
  mcXReverse:
      begin
      Attr.Reverse := false;
      ChrFunc := RRpl;
      end;
  
  mcHalf:
      begin
      Attr.Half := true;
      end;
  
  mcXHalf:
      begin
      Attr.Half := false;
      end;
  
  mcUnder:
      begin
      Attr.Under := true;
      end;
  
  mcXUnder:
      begin
      Attr.Under := false;
      end;
  
  mcNonDis:
      begin
      Attr.NonDis := true;
      end;
  
  mcXNonDis:
      begin
      Attr.NonDis := false;
      end;
  
  mcProtect:
      begin
      Attr.Protect := true;
      end;
  
  mcXProtect:
      begin
      Attr.Protect := false;
      end;
  
  mcSlChSet:
      Attr.Charset := ord( GetNextCh(FromHost) ) - 32;
  
  mcClAlLin,mcClUnLin:     { Clear to end of line }
      begin
      ClearRight;
      end;
  
  mcClalWin,mcClUnWin:     { Clear to end of frame }
      Begin
      ClearEOF;
      end;

  mcDlChLin,mcDlChWin:  { Delete character in middle of line }
      begin
      if debug then DStr('DlChLin/DlChWin');
      ShiftLeft;
      end;
  
  mcDelBlk:
      begin
      ScrollUp(CurY, ord(GetNextCh(FromHost))-32);
      end;

  mcInsBlk:
      begin
      ScrollDown(CurY, ord(GetNextCh(FromHost))-32);
      end;

  mcSetMouse:
      if GetNextCh(FromHost) = '0' then  
        begin
        WantTablet := false;
        IOSetModeTablet(OffTablet);
        IOCursorMode(OffCursor);
        end
      else  
        begin
        WantTablet := true; 
        EmacsTablet := true;
        IOCursorMode(TrackCursor);
        IOSetModeTablet(relTablet);
        end;

  mcDelLin:
      begin      
      if debug then DStr('DelLin');
      ScrollUp(CurY, 1);
      end;
  
  mcInsLin:
      begin
      if debug then DStr('InsLin');
      ScrollDown (CurY, 1);
      end;
          
  mcDefWin:
      begin
      TopY := ord(GetNextCh(FromHost)) - 32;
      TopX := ord(GetNextCh(FromHost)) - 32;
      BotY := TopY + ord(GetNextCh(FromHost)) - 33;
      BotX := TopX + ord(GetNextCh(FromHost)) - 33;

      if BotY=23 then BotY:=59;    {********* QUESTIONABLE!!!!! *********}

      FrameWindow(TopX, TopY, BotX, BotY);
      if debug
      then  begin
            ChangeWindow(2);
            Writeln;
            Write('DefWin',TopY, TopX, BotY, BotX );
            ChangeWindow(1);
            end;
      CurX := TopX;
      CurY := TopY;
      end;

  mcCurUnd:
      begin
      if debug then DStr ('CurUnd');
      if CursorOn then ToggelCursor;
      CrsChr := chUnder;
      end;

  mcCurBlk:
      begin
      if debug then DStr ('CurBlk');
      if CursorOn then ToggelCursor;
      CrsChr := chDel;
      end;
  
  mcChar, mcULC, mcFDupl, mcRemote,     { No-ops in this implementation }
  mcOpaque, mcManLF, mcManTab, mcScroll,
  mcProg, mcExFun, mcTrFun, mcRstAll:
      begin
      if Debug
      then  begin
            DStr('Mode');
            DNum( ord(ch) );
            end;
      end;

  
  mcStBlAttr:                           { Unimplemented w/4 arguments }
      UnImpl(4,FromHost);
    
  mcChgMsg, mcRptVer:                   { Unimplemented w/2 arguments }
       UnImpl(2,FromHost);
  
  mcStBaud,                             { Unimplemented w/1 arguments }
  mcStParity, mcStOutNet, mcTieWin,
  mcStaScr:
      UnImpl(1,FromHost);
  
  otherwise:                            { Unimplemented w/0 arguments }
      UnImpl(0,FromHost);
  end; {case}

1:
end; {DoMC}

procedure InsertBlock( ch: char );
{--------------------------------------------------------------------}
{
{ Abstract:
{    Sends a run of ordinary characters to the C100 display when it
{    is in insert mode.
{    No longer used -- made no noticeable difference.
{
{--------------------------------------------------------------------}
label 1;
var
      HostInScan : integer;
      Count : integer;
      i : integer;
begin
HostInScan := HostInTail;
Count := 0;
while true
do    begin
      HostInScan := LAnd(HostInScan+1, HostInMask);
      Count := Count + 1;
      if (HostInScan = HostInHead) then goto 1;
      case HostInBuf[HostInScan] of
         chNul :
            Count := Count - 1;
         chBel,
         chBS,
         chTab,
         chLF,
         chFF,
         chCR,
         chHostMC,
         chDel :
            goto 1;
         otherwise :
            if (CurX+Count > WinBotX) then goto 1;
         end;
      end;
1:
{ HostInScan is sitting over first bad character }
if (Count > MaxInserts) then MaxInserts := Count;
ShiftRight(Count);
PutChr( ch );
for i := 2 to Count
do    begin
      HostInTail := LAnd(HostInTail+1, HostInMask);
      ch := HostInBuf[HostInTail];
      if (ch <> chNul) then PutChr(ch);
      end;
end;  { InsertBlock }

function NumScrolls: integer;
{--------------------------------------------------------------------}
{
{ Abstract:
{    Counts number of times we are sure we can scroll by looking at
{    HostInBuf.
{
{--------------------------------------------------------------------}
label 1;
var
      HostInScan : Integer;
      Count : Integer;
begin
HostInScan := HostInTail;
Count := 1;
if (Count >= WinBotY-WinTopY) then goto 1;
while true
do    begin
      HostInScan := LAnd(HostInScan+1, HostInMask);
      if (HostInScan = HostInHead) then goto 1;
      case HostInBuf[HostInScan] of
         chBS,
         chFF,
         chHostMC :
            goto 1;
         chLF:
            begin
            Count := Count + 1;
            if (Count >= WinBotY-WinTopY) then goto 1;
            end
         end;
      end;
1:
{ fell off the end }
if (Count > MaxScrolls) then MaxScrolls := Count;
NumScrolls := Count
end; { NumScrolls }

procedure ToDisplay(*ch: char*);
{--------------------------------------------------------------------}
{
{ Abstract:
{    Sends a character to the C100 display.
{
{--------------------------------------------------------------------}
var Scrolls: integer;
begin
CtrlCPending := false;
if Transcript then
    if ch<>chHostMC then
        write(Transfile, ch);

case ch of
   chNul,
   chDel :
      { nothing to do};

   chBel:
      IOBeep;

   chTab:
      repeat
        PutChr(' ');
        if (CurX >= WinBotX)
        then  Xw1
        else  CurX := CurX + 1;
      until TabStops[ CurX ];

   chBS:
      begin
      if (CurX <= WinTopX)
      then  begin
            CurX := WinBotX;
            if (CurY <= WinTopY)
            then  CurY := WinBotY
            else  CurY := CurY - 1
            end
      else  CurX := CurX - 1
      end;

   chLF:
      begin
      if CurY >= WinBotY
      then  begin
            Scrolls := NumScrolls;
            ScrollUp(WinTopY, Scrolls);
            CurY := WinBotY - Scrolls + 1;
            end
      else  
            CurY := CurY + 1;
      end;

   chFF:
      ClearScreen;

   chCR:
      begin
      CurX := WinTopX;
      end;

   chHostMC :
      DoMC( true );

   otherwise:
      begin
            
      { Shift line right if in insert mode }

(*
 *  code for attempting block inserts removed because block size
 *  was never > 1
 *
 *    if AmInserting
 *    then  InsertBlock( ch )
 *    else  PutChr( ch );
 *)

      if AmInserting then ShiftRight(1);

      PutChr( ch );

      { Advance cursor }

      if (CurX >= WinBotX)
      then  Xw1
      else  CurX := CurX + 1;

      end;

   end; {case}

end; {ToDisplay}

procedure Aside;
{--------------------------------------------------------------------}
{
{ Abstract:
{    Called from main program when user types 'chExit' character.
{    This routine allows the user to execute some Meta-commands.
{
{--------------------------------------------------------------------}
var
    Ans: char;
    B: string[10];
    Now: Double;
    Started: Double;
    i, j : integer;
    DT1, DT2: Integer;
    c : char;

procedure SpinUp;   { find the beginning of a jiffy }
    begin
    IOGetTime(Started);
    repeat
          IOGetTime(Now);
      until (Now[0] <> Started[0]);
    end;

begin
if CursorOn then ToggelCursor;
ChangeWindow(2);
CtrlCPending := false;   { Don't let IO trap ^C }
writeln(chFF, 'Type:');
  write('  Q to quit C100            ');
  write('  C to continue C100        ');
writeln('  E to expand the window');
  write('  B to set the baud rate    ');
  write('  K to set the cursor char  ');
writeln('  T to start/stop tablet');
  write(' ^^ to send a ^^            ');
  write('  F to start/stop transcript');
    
Ans := GetNextCh( false );
Writeln(chFF,'Type ^^ to escape from Concept-100 simulation');
    
case Ans of
  'Q','q':
      begin
      ScreenReset;
      if Transcript then
          close(Transfile);
      exit(C100);
      end;
  'C','c':
      Writeln('    Continuing');
  'E','e':
      begin
      FrameWindow(0, 0, MaxChar, MaxLine);
      Writeln('    Expanded to maximum window');
      end;
  'K','k':
      begin
      Write('    Cursor character: ');
      CrsChr := GetNextCh(false);
      Writeln(CrsChr);
      end;
  'T','t':
      begin
      if WantTablet
      then  
        begin
        WantTablet := false;
        IOSetModeTablet(OffTablet);
        IOCursorMode(OffCursor);
        Writeln('    Tablet disabled');
        end
      else  
        begin
        WantTablet := true;
        IOCursorMode(TrackCursor);
        IOSetModeTablet(relTablet);
        Writeln('    Tablet enabled')
        end;
      EmacsTablet := Ans='T';
      end;
  chExit:
      begin
      CtrlCPending := false;   { Don't let IO trap ^C }
      ToHost( chExit );
      writeln('    ^^ passed to host');
      end;
  'F','f':
      begin
      if Transcript then
          begin
          Writeln('    Transcript off');
          writeln(Transfile);
          close(transfile);
          Transcript := FALSE;
          end
      else
          begin
          Write('    Transcript file (<CR> to abort): ');
          Readln(TFileName); 
          if TFileName<>'' then
              begin
              rewrite(Transfile, TFileName );
              Transcript := TRUE;
              end
          else
              writeln(chBel,'    ** Aborted');
          end;
      end;
  'B','b':
      begin
      Write('    New baud rate: ');
      Readln(B);
      if ( (B='9600') or
           (B='4800') or
           (B='2400') or
           (B='1200') or
           (B= '600') or
           (B= '300') or
           (B= '150') or
           (B= '110') )
      then  begin
            SetBaud(B,true);
            Writeln('    New baud rate: ', B);
            end
      else Writeln('    Bad baud rate: ', B);
      end;
  'V','v':
      Writeln('    Version of 16-Nov-82');
  '0':
      begin
      with KSet^.Index[ord(' ')]
      do    begin
            writeln('    MaxScrolls: ', MaxScrolls:1,
(*                       ' MaxInserts: ', MaxInserts:1, *)
                       ' MaxBuffer: ',  MaxBuffer:1 );
            end;
      MaxInserts := 0;
      MaxScrolls := 0;
      MaxBuffer  := 0;
      end;
(*
 *'1': { FIXME -- THIS IS WRONG -- FITZ }
 *    begin
 *    ChangeWindow(1);
 *    ToggelCursor;
 *    FrameWindow(0, 0, MaxChar, MaxLine);
 *    CurX := 0;
 *    CurY := MaxLine;
 *    SpinUp;
 *    for i := 1 to 2048
 *    do    begin
 *          c := 'a';
 *          for j := 1 to 62 do ToDisplay( c );
 *          c := chCR;
 *          ToDisplay( c );
 *          c := chLF;
 *          ToDisplay( c );
 *          end;
 *    IOGetTime( Now );
 *    DT1 := Now[0] - Started[0];
 *    SpinUp;
 *    for i := 1 to 2048
 *    do    begin
 *          c := 'a';
 *          for j := 1 to 62 do;
 *          c := chCR;
 *          c := chLF;
 *          end;
 *    IOGetTime( Now );
 *    DT2 := Now[0] - Started[0];
 *    ToggelCursor;
 *    ChangeWindow(2);
 *    Writeln( '    DT1: ', DT1:1,
 *                ' DT2: ', DT2:1,
 *                ' DT: ', (DT1-DT2):1,
 *                ' Baud: ', (75*(#77777 div ((DT1-DT2) div 32))):1);
 *    end;
 *'2':
 *    begin
 *    ChangeWindow(1);
 *    ToggelCursor;
 *    FrameWindow(0, 0, MaxChar, MaxLine);
 *    CurX := 0;
 *    CurY := MaxLine;
 *    SpinUp;
 *    for i := 0 to #7777 do ScrollUp( WinTopY, 1 );
 *    IOGetTime( Now );
 *    DT1 := Now[0] - Started[0];
 *    SpinUp;
 *    for i := 0 to #7777 do;
 *    IOGetTime( Now );
 *    DT2 := Now[0] - Started[0];
 *    ToggelCursor;
 *    ChangeWindow(2);
 *    Writeln( '    Full Screen scrolling: ',
 *                ' DT1: ', DT1:1,
 *                ' DT2: ', DT2:1,
 *                ' DT: ', (DT1-DT2):1,
 *                ' Baud: ', (20*(#10000 div ((DT1-DT2) div 30))):1);
 *    end;
 *'3':
 *    begin
 *    ChangeWindow(1);
 *    ToggelCursor;
 *    FrameWindow(0, 0, MaxChar, MaxLine);
 *    CurX := 0;
 *    CurY := MaxLine;
 *    SpinUp;
 *    for i := #100000 to 0 do PutChr('a');
 *    IOGetTime( Now );
 *    DT1 := Now[0] - Started[0];
 *    SpinUp;
 *    for i := #100000 to 0 do;
 *    IOGetTime( Now );
 *    DT2 := Now[0] - Started[0];
 *    ToggelCursor;
 *    ChangeWindow(2);
 *    Writeln( '    Single Character RasterOp: ',
 *                ' DT1: ', DT1:1,
 *                ' DT2: ', DT2:1,
 *                ' DT: ', (DT1-DT2):1,
 *                ' KBaud: ', ((((#77777 div (DT1-DT2)))*6) div 10):1,
 *                 '.', ((((#77777 div (DT1-DT2)))*6) mod 10):1);
 *    end;
 *)
  otherwise:
      begin
      if ( (Ans >= ' ') and (Ans < chr(#175)) )
      then  Writeln(chBel, chBel, '    ??? ''', Ans, '''')
      else  Writeln(chBel, chBel, '    ??? chr(#', ord(Ans):1:8, ')')
      end;
end;
ChangeWindow(1);
end; {Aside}

procedure ToHost(*ch: char*);
{--------------------------------------------------------------------}
{
{ Abstract:
{    Sends a character to the host computer.
{
{--------------------------------------------------------------------}
begin
CtrlCPending := false;
if IOCWrite( RS232Out, ch ) <> IOEIOC
then  writeln('** RSOut error **');
end; {ToHost}

handler HelpKey(var retstr:sys9s);
{--------------------------------------------------------------------}
{
{ Abstract:
{    Provides the text to insert when the help key is pressed.
{    Currently, we just want to kill the default  '/HELP'.
{
{--------------------------------------------------------------------}
   begin
   retstr := '';
   end;

{--------------------------------------------------------------------}
{    MAIN PROGRAM
{--------------------------------------------------------------------}
begin
reset(input);
rewrite(output);
write(chFF);

createwindow(1,0,0,768,W1Bot,
    'Siemens Concept-100 emulator V1.7 (16 Nov 82)      Type ^^ to suspend');
createwindow(2,0,W1Bot+1,768,1022-W1Bot,'');
ChangeWindow(2);
writeln(chFF,'Type ^^ to escape from Concept-100 emulation');
ChangeWindow(1);

{ Initialize RS232 connection }

with Stat
do    begin
      ByteCnt := 1;
      RSRcvEnable := true;
      end;
IOPutStatus( RS232In, Stat );

{ 1200 is correct default for CMU front end }
{ 4800 is most that can be relied upon      }
{ 9600 works most of the time               }
SetBaud(DefaultBaud, true);

{ Initialize C100 state }

ResetInit;
UpdateAttributes;
CrsChr := DefCrsChar;
ToggelCursor;

{ Full screen window }

FrameWindow(0, 0, MaxChar, MaxLine);

{ Initialize cursor }

IOCursorMode(TrackCursor);
IOLoadCursor(DefaultCursor, Char0X, Char0Y);
IOSetCursorPos(Char0X*2, Char0Y*2);
IOCursorMode(OffCursor);
WantTablet := false;

{ Initialize Host input buffer }

HostInHead := 0;
HostInTail := 0;

{ Initialize stats }
MaxInserts := 0;
MaxScrolls := 0;
MaxBuffer  := 0;

{ Initialize transcripting }

Transcript := FALSE;

{ Go into a loop acting like a C100 }

while true
do    begin
      
      CtrlCPending := false;
      CtrlSPending := false;

      if (HostInHead <> HostInTail)
      then  begin
            if CursorOn then ToggelCursor;

            repeat
                  QuitDisplay := False;
                  ToDisplay( HostInBuf[HostInTail] );
                  HostInTail := LAnd(HostInTail+1, HostInMask);
               until ( (HostInHead = HostInTail) or QuitDisplay );
            end;

      while IOCRead( RS232In, HostInCh ) = IOEIOC
      do    begin
            HostInBuf[HostInHead] := chr(LAnd(ord(HostInCh),#177));
            HostInHead := LAnd(HostInHead+1, HostInMask);
            if (HostInHead = HostInTail)
            then  begin
                  ChangeWindow(2);
                  Write('{HostInOverrun}');
                  ChangeWindow(1);
                  end
            end;

      if (MaxBuffer < LAnd(HostInHead-HostInTail, HostInMask))
      then  MaxBuffer := LAnd(HostInHead-HostInTail, HostInMask);

      if (HostInHead = HostInTail) 
      then  begin
            IOGetTime(JiffyTime.Dbl);
            if (CursorOn <> JiffyTime.Toggle)
            then  ToggelCursor;
            end;

      if IOCRead( TransKey, KbdInCh ) = IOEIOC
      then  begin
            CtrlCPending := false;
            CtrlSPending := false;
            if debug then DStr('User');
            if KbdInCh=chExit
                 then  Aside
                 else  ToHost( KbdInCh );
            end;
      
      if WantTablet
      then if TabSwitch then
            begin
            if TabYellow then     EmacsKey := '0'
            else if TabWhite then EmacsKey := '1'
            else if TabBlue then  EmacsKey := '2'
            else                  EmacsKey := '3';
            While TabSwitch do ;
            IOReadTablet(tabX, tabY);
            tabLine := ((tabY-Char0Y) div LineHit) - 1;
            tabChar := ((tabX-Char0X) div CharWid) - 1;
            if (TabLine < 65) and (TabLine > 0)
            and (TabChar < 81) and (TabChar > 0)
            then if EmacsTablet then
                  begin
                  ToHost(chr(#33));
                  ToHost('M');
                  ToHost(EmacsKey);
                  ToHost (chr(TabChar+33));
                  ToHost (chr(TabLine+33));
                  end
                 else 
                  begin
                  ToHost(chr(#36));
                  ToHost(chr(#37 + tabChar));
                  ToHost(chr(#36 + (tabLine*2)));
                  ToHost(chr(#35));
                  end;
            end;
      
      end;
      
end.

