;**********************************************************************
;	   R E S T R I C T E D   R I G H T S   L E G E N D 
;**********************************************************************

;**********************************************************************
;	Use, duplication or disclosure by the Government is
; subject to restrictions as set forth in paragraph (b) (3) (B)
; of the Rights in Technical Data and Computer Software clause
; in DAR 7-104.9(a). Contractor/manufacturer is Zenith Data
; Systems of Hilltop Road, St. Joseph, Michigan  49085.
;**********************************************************************


	PAGE	,132
	TITLE	Z-150 Video Drivers

;**********************************************************************
;
;                           --------------
;--------------------------- Video Driver -----------------------------
;                           --------------
;
;		Copyright (C) 1983, by Zenith Data Systems
;
;**********************************************************************

MONITOR_SEGMENT SEGMENT WORD PUBLIC

	ASSUME	CS:MONITOR_SEGMENT, DS:ROM_DATA, SS:NOTHING, ES:NOTHING

	INCLUDE	../ROM/ROM.LIT
	INCLUDE	../ROM/IO.LIT
	INCLUDE	../ROM/INTR.LIT
	INCLUDE ../SYS/SYS.EXT
	INCLUDE	VIDEO.LIT
	INCLUDE	VIDEO.EXT

	EXTRN	CASE:NEAR, GET_INTR_PTR:NEAR, BEEP:NEAR, READ_DIP_SWITCHES:NEAR
	EXTRN	DATA_SEGMENT:WORD


;**********************************************************************
; VIDEO_IO_INTERRUPT: (FUNCTION, PARAMETERS)
;
;	Video_IO_Interrupt is the entry point for all of the
; video Input/Output functions.  Called by performing an interrupt
; number 10H (16 decimal), it expects a function number in AH -
; this number is then used to call the appropriate function.
; The function calls supported by this interrupt are as follows:
;
; Function Code 0 - Set Video Mode
;    Input:
;	AL: New video mode, as follows:
;	    0 - 40 by 25 display in black and white
;	    1 - 40 by 25 display in color
;	    2 - 80 by 25 display in black and white
;	    3 - 80 by 25 display in color
;	    4 - 320 by 200 pixel graphics in color
;	    5 - 320 by 200 pixel graphics in black and white
;	    6 - 640 by 200 pixel graphics in black and white
;	    7 - 80 by 25 display on the monochrome board
;
; Function Code 1 - Set Cursor Style
;    Input:
;	CH: Starting scan line for cursor (from 0-31)
;	CL: Ending scan line for cursor (from 0-31)
;	    Note that CH must not be greater than CL!
;
; Function Code 2 - Set Cursor Position
;    Input:
;	DX: Alphanumeric cursor position (DH:row, and DL:column)
;	    The 'Home' position is the upper-left hand corner of the
;	    screen, and is defined as (0:0)
;	BH: Page number, as appopriate for the current video mode.
;	    See the Set Display Page function for more information
;
; Function Code 3 - Read Cursor Position
;    Input:
;	BH: Page number
;    Output:
;	DX: Cursor position (DH:row, and DL:column)
;	CX: Cursor style (CH:starting scan line, and CL:ending line)
;
; Function Code 4 - Read Light Pen Status
;    Output:
;	AH: Light pen status.  If AH is 0, then the light pen is
;	    either not pressed to the screen, or is not triggered.
;	    If AH = 1, then the following registers are valid:
;	DX: Position of light pen (DH:row, and DL:column) as
;	    an ALPHANUMERIC position.
;	CH: Vertical scan line number (from 0-199).
;	BX: Approximate pixel column (from 0-319, or 0-639, depending
;	    on the currently selected video mode).
;
; Function Code 5 - Select Display Page
;    Input:
;	AL: Page number to display.  For the 40 by 25 display modes
;	    the page may range from 0 to 7;  for 80 by 25 it may
;	    be 0-3, and only 0 is acceptable for the graphics modes.
;
; Function Code 6 - Scroll Currently Displayed Page Up
;    Input:
;	AL: Number of lines to scroll the selected window.  If
;	    AL is 0, the entire scroll window is erased.
;	CX: Upper-left hand scroll window position (CH:row, and
;	    CL:column).
;	DX: Lower-right hand scroll window position (DH:row, and
;	    DL:column).
;	BH: Attribute byte used for erased lines - ignored in
;	    graphics modes.
;
; Function Code 7 - Scroll Currently Displayed Page Down
;    Input:
;	AL: Number of lines to scroll.	If AL = 0, then the
;	    window is erased.
;	CX: Upper-left hand scroll window position (CH:row, and
;	    CL:column).
;	DX: Lower-right hand scroll window position (DH:row, and
;	    DL:column).
;	BH: Attribute byte for erased lines - ignored while in
;	    graphics modes.
;
; Function Code 8 - Read Character and Attribute at Cursor Position
;    Input:
;	BH: Page number.
;    Output:
;	AL: Ascii value of character read.
;	AH: Attribute byte for this character (except while graphics
;	    modes are in effect).
;
; Function Code 9 - Write Character and Attribute at Cursor Position
;    Input:
;	BH: Page number.
;	AL: Ascii value of the character to display.
;	BL: Attribute for this character (for alphanumeric modes).
;	CX: Repeat count (see the Display_Character routine for more
;	    information).
;
; Function Code 10 - Write Character ONLY at Current Cursor Position
;    Input:
;	BH: Page number.
;	AL: Ascii value of the character to display.
;	CX: Repeat count (see the Display_Character routine).
;
; Function Code 11 - Set Colors
;    Input:
;	BH: Color register selection.  If BH = 0, then the color value
;	    in BL is used for the background color (for alphanumeric
;	    modes) - legal values are from 0-15 (bits 0-3), with bit 4
;	    accepted as the high-intensity flag.  This may also be used 
;	    for medium-resolution graphics mode, except that the high-
;	    intensity flag is ignored.
;	    If BH = 1, then the value in BL will contain the 'pallette'
;	    used for medium-resolution graphics (320 by 200).
;	BL: Value to use for color register.  In medium-resolution
;	    graphics mode, two different pallettes are used:
;	    If BL=0, then:
;			Color 0 is the background color
;			Color 1 is green
;			Color 2 is red
;			Color 3 is yellow
;	    If BL=1, then:
;			Color 0 is the background color
;			Color 1 is cyan
;			Color 2 is magenta
;			Color 3 is white
;
; Function Code 12 - Write Graphics Pixel
;    Input:
;	DX: Vertical position (row number, from 0-199)
;	CX: Horizontal position (column, from 0-319, or 0-639)
;	AL: Color Value (0-1 for high-resolution graphics, or
;	    0-3 for medium-resolution graphics).  If the most-
;	    significant bit of AL is set, then the pixel value
;	    will be XORed with the existing screen contents.
;
; Function Code 13 - Read Graphics Pixel
;    Input:
;	DX: Vertical position (row number, from 0-199).
;	CX: Horizontal position (column, from 0-319, or 0-639).
;    Output:
;	AL: Pixel value at the specified location.
;
; Function Code 14 - Terminal Display
;    Input:
;	AL: Character to display / control character.  Carriage
;	    return, line feed, backspace and bell are handled
;	    as commands, not displayable characters.
;	BL: Foreground color (if in medium-resolution graphics mode).
;
; Function Code 15 - Determine Video Status
;    Output:
;	AL: Currently selected video mode (see Set Video Mode)
;	AH: Screen width in characters.
;	BH: Currently displayed page.
;
; Function Code 100 - Set Scroll Mode
;    Input:
;	AL: Scroll Mode, as follows:
;		AL=0:   'Normal' software scroll mode (used for
;			compatibility with existing software).
;		AL=1:	'Jump' scroll mode.  Much faster than
;			normal scroll mode - appears the same.
;	        AL=2:   'Smooth' scroll mode.  Slower than
;			jump scrolling, but smooth scrolling is
;			performed by scrolling a scan line at a
;			time - hence, the display is much more
;			readable.
;
; Function Code 101 - Set Interlace Mode
;    Input:
;	AL: Interlace Mode, as follows:
;		AL=0:	'Normal' non-interlaced 25 line display.
;		AL=1:	'Fat-character' display;  25 lines of
;			high-density characters.
;		AL=2:	Interlace Mode.  This mode only works in
;			text display modes, but it provides a 50
;			line display.
; 
; Function Code 102 - Set Ultra-compatible display mode
;    Input:
;	AL: 00H to enable standard display, or 0FFH to enable
;	    ultra-compatible display.  In this mode, all low-level
;	    functions in the terminal display routines are vectored
;	    back through the video interrupt.
;
;**********************************************************************
VIDEO_IO_INTERRUPT PROC FAR
	PUBLIC	VIDEO_IO_INTERRUPT
	PUSHREG	<BP,DS>
	STI				;Enable interrupts
	CLD				;Set forward byte moves
	MOV	VIDEO_ACTIVE,TRUE	;Flag that video is active
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	MOV	BP,AX			;No, save the input parameters
	MOV	VIDEO_SEGMENT,SCREEN_SEGMENT ;Assume the color card is in use
	MOV	AL,BYTE PTR IO_CONFIG	;Get the I/O configuration 
	AND	AL,VIDEO_MODE_MASK	;Mask out all but the video select bits
	CMP	AL,VIDEO_MODE_MONO	;Is the monochrome card selected?
	JNE	VI1			;No, continue
	MOV	VIDEO_SEGMENT,MONO_SEGMENT ;Yes - write the mono video segment
;
; Check for POKEs into NUMBER_OF_COLUMNS and VIDEO_MODE
;
VI1:	MOV	AL,BYTE PTR NUMBER_OF_COLUMNS ;Get the number of columns
	CMP	AL,OLD_COLUMNS		;Was the screen width changed?
	JNE	VI7			;Yes - change the page specifications
	MOV	AL,VIDEO_MODE		;Get the current video
	CMP	AL,OLD_VIDEO_MODE	;Has a new mode been poked in to memory?
	JNE	VI5			;Yes - set up for the new mode
;
; Check for special fast video routines
;
VI2:	CMP	AH,14			;Do special fast-video-routine check
	JNE	VI3			;Not Terminal_Display - continue
	CMP	SUPER_COMPATIBLE,FALSE	;Can we run using the fast display routines?
	JNE	VI3			;No, must use the very slow routine
	AND	AX,07H			;Make sure that it is in range
	SHL	AX,1			;Set up for word accesses
	XCHG	BP,AX			;Point BP to routine, AL=char
	CALL	CS:FAST_ROUTINE[BP]	;Display the char in fast mode
	JMP	SHORT VI4		;Return to the caller
VI3:	CMP	AH,VMAX			;Is this a valid request?
	JAE	VI8			;Not in table... may be extended function
	MOV	AL,AH			;Get the mode in AX
	XOR	AH,AH			;Extend it to a word
	SHL	AX,1			;Mult by two for word accesses
	XCHG	AX,BP			;Set BP = routine, AX = char & function
	CALL	CS:VIDEO_ROUTINE[BP]	;Perform the requested video function
VI4:	MOV	VIDEO_ACTIVE,FALSE	;Flag that video code is done
	POPREG	<DS,BP>			;Restore the user registers
	IRET				;Return to the caller
;
; Update video variables if the mode was POKEd into memory
;
VI5:	MOV	OLD_VIDEO_MODE,AL	;Yes - set it so vars are consistent
	CALL	SET_VIDEO_VARS		;Initialize the other video variables
	JMP	SHORT VI2		;Continue processing video request
;
; Handle POKEs into NUMBER_OF_COLUMNS
;
VI7:	CALL	SET_SCREEN_WIDTH	;Set the new screen width
	JMP	SHORT VI1		;Try it all over again
;
; Check for Extended Video Control Functions
;
VI8:	CMP	AH,100			;Is this an extended-function request?
	JB	VI4			;No, ignore the out-of-range request
	CMP	AH,103			;Is this a valid extended function?
	JAE	VI4			;No, ignore it
	PUSH	AX			;Yes - save register AX
	MOV	AX,BP			;Get the video code again
	SUB	AH,100			;Map function calls to CASE range
	CALL	CASE			;Perform extended video function
	JMP	NEAR PTR VI9		;All functions return here (MUST BE NEAR!)
	DW	SET_SCROLL_MODE		;If AH = 100, Set the scroll mode
	DW	SET_ILACE_MODE		;If AH = 101, Set the interlace mode
	DW	SET_SUPER_COMPATIBLE	;If AH = 102, Set Super-Compatibility
VI9:	POP	AX			;Restore register AX
	JMP	VI4			;...and return
;
; Jump table for video routines
;
VIDEO_ROUTINE EQU THIS WORD
	DW	SET_VIDEO_MODE		;If AH = 0, Set video mode
	DW	SET_CURSOR_STYLE	;If AH = 1, Set cursor style
	DW	SET_CURSOR_POSITION	;If AH = 2, Set cursor position
	DW	GET_CURSOR_POSITION	;If AH = 3, Get the cursor position
	DW	GET_LPEN_STATUS		;If AH = 4, Read the light pen status
	DW	SET_DISPLAY_PAGE	;If AH = 5, Select a page for display
	DW	SCROLL_UP		;If AH = 6, Scroll up page
	DW	SCROLL_DOWN		;If AH = 7, Scroll down page
	DW	READ_CHAR		;If AH = 8, Read character at cursor
	DW	DISP_CHAR_ATTR		;If AH = 9, Write char + attribute
	DW	DISP_CHAR		;IF AH = 10, Write character only
	DW	SET_COLOR		;If AH = 11, Set color register
	DW	DISPLAY_PIXEL		;If AH = 12, Write graphics pixel
	DW	READ_PIXEL		;If AH = 13, Read graphics pixel
	DW	TERMINAL_DISPLAY	;If AH = 14, Display as terminal
	DW	GET_VIDEO_STATUS	;If AH = 15, Get the video status
VMAX	EQU	($-VIDEO_ROUTINE)/2
;
; Special Fast Display Routines for Terminal-Display-Mode
;
FAST_ROUTINE EQU THIS WORD
	DW	TERMINAL_DISPLAY	;If Mode=0, use standard video routine
	DW	TERMINAL_DISPLAY	;If Mode=1, use standard video routine
	DW	TERMINAL_DISPLAY_8025	;If Mode=2, use fast 80x25 routine
	DW	TERMINAL_DISPLAY_8025	;If Mode=3, use fast 80x25 routine
	DW	TERMINAL_DISPLAY	;If Mode=4, use standard video routine
	DW	TERMINAL_DISPLAY	;If Mode=5, use standard video routine
	DW	TERMINAL_DISPLAY_HRES	;If Mode=6, use fast high-res routine
	DW	TERMINAL_DISPLAY_MONO	;If mode=7, use fast monochrome routine
VIDEO_IO_INTERRUPT ENDP



	INCLUDE FASTHRES.ASM		;High-Resolution terminal mode display
	INCLUDE FAST8025.ASM		;Fast Text Mode Terminal Display
	INCLUDE	DISPWRT.ASM		;Standard Write Char & Pixel
	INCLUDE DISPRD.ASM		;Standard Read Char & Pixel
	INCLUDE SCROLL.ASM		;Scroll Routines
	INCLUDE	ILACE.ASM		;Interlace Routines



;***********************************************************************
; SET_SUPER_COMPATIBLE:
;
;	Set_Super_Compatible is called to enable and disable the
; super-compatible display mode.  When this mode is enabled, all
; terminal display functions vector their low-level actions back
; through the video interrupt, permitting right-to-left display, etc.
;
;
; Input:
;	AL: 00H if standard display, 0FFH for Super-Compatible display.
;***********************************************************************
SET_SUPER_COMPATIBLE PROC NEAR
	PUBLIC	SET_SUPER_COMPATIBLE
	MOV	SUPER_COMPATIBLE,AL	;Set the new compatibility flag
	RET				;...and return!
SET_SUPER_COMPATIBLE ENDP



;**********************************************************************
; SET_VIDEO_MODE: (MODE)
;
;	Set_Video_Mode is called to set up the ROM for a new
; screen format.  All display memory is cleared, and the CRT
; controller is programmed for the new display format.  In
; addition, the screens are cleared, page 0 is displayed, and 
; the ROM variables are set to a value appropriate for the new
; format.
;
; Input:
;	AL: Mode value
;**********************************************************************
SET_VIDEO_MODE PROC NEAR
	PUBLIC	SET_VIDEO_MODE
	PUSHREG	<AX,BX,CX,DX,SI,ES>
	MOV	AH,BYTE PTR IO_CONFIG	;Get the current I/O configuration
	AND	AH,VIDEO_MODE_MASK	;Mask off the video select bits
	CMP	AH,VIDEO_MODE_MONO	;Is the monochrome board active?
	JNE	SVM0			;No, continue with the mode set
	MOV	AL,7			;Yes - force mono for compatibility(?)
SVM0:	CALL	SET_VIDEO_VARS		;Initialize the video variables
	CALL	WAIT_VERTICAL_SYNC	;Update screen hardware during VSYNC
	MOV	AL,DISABLE_VIDEO_COLOR	;Get command for color video-disabled
	CMP	VIDEO_MODE,7		;Is this monochrome?
	JNE	SVM1			;No, continue
	MOV	AL,DISABLE_VIDEO_MONO	;Yes - get monochrome video disable
SVM1:	MOV	DX,VIDEO_BASE		;Point DX to the CRT mode-select port
	ADD	DX,CRT_MODE_SELECT
	OUT	DX,AL			;Set the video-disabled mode
	MOV	AL,0			;Get the video hardware mode variable
	CALL	OUT_VIDEO_MODE		;Disable the CRT video
	CALL	GET_VIDEO_PARMS		;Point ES:SI to the video parameters
	MOV	DX,VIDEO_BASE		;Point to the base reg of the CRT
	MOV	BL,0			;Keep register number in BL
	MOV	CX,SIZE VID_PARMS	;Get the size of the video parameters
SVM2:	MOV	AL,BL			;Get the current register number
	OUT	DX,AL			;Send it to the CRT controller
	INC	BL			;Point to the next register
	LODS	BYTE PTR ES: [SI]	;Get a video parameter
	INC	DX			;Point to the parameter register
	OUT	DX,AL			;Output it to the CRT controller
	DEC	DX			;Point back to the control reg
	LOOP	SVM2			;Loop until completed
	CALL	CLEAR_VIDEO_RAM		;Erase all screens for this card
	MOV	AL,CURRENT_COLORS	;Get the current color values
	ADD	DX,CRT_COLOR_SELECT	;Point to the color selection port
	OUT	DX,AL			;Set the default colors
	CALL	WAIT_VERTICAL_SYNC	;Wait for vertical sync to occur
	CALL	WAIT_VERTICAL_SYNC	;Wait for another sync pulse
	MOV	AL,0			;Get page number 0
	CALL	SET_DISPLAY_PAGE	;Display page 0
	MOV	AL,VIDEO_HW_MODE	;Get the correct hardware mode
	ADD	DX,CRT_MODE_SELECT-CRT_COLOR_SELECT ;Point DX to mode select
	OUT	DX,AL			;Enable the video hardware!
	POPREG	<ES,SI,DX,CX,BX,AX>
	RET
SET_VIDEO_MODE ENDP



;**********************************************************************
; SET_VIDEO_VARS:
;
;	Set_Video_Vars is called when a new video mode has been
; programmed (via the Set_Video_Mode call), or when a mode change
; has been detected.  It initializes the global variables for a
; new video display format.
;
; Input:
;	AL: New mode value
;**********************************************************************
SET_VIDEO_VARS PROC NEAR
	PUBLIC	SET_VIDEO_VARS
	PUSHREG	<AX,BX,CX,SI,DI,ES>
	CMP	AL,7			;Is this a valid mode?
	JBE	SVV0			;Yes - change the variables to new values
	JMP	SVV6			;No, ignore it (must be 3rd party stuff)
SVV0:	MOV	CURSOR_FLAG,NOT CURSOR_ENABLED	;Turn the software cursor off
	MOV	VIDEO_MODE,AL		;Set the global display variable
	MOV	OLD_VIDEO_MODE,AL	;Set the 'old mode' to the new one
	MOV	BX,COLOR_CARD		;Set I/O addr as default to color card
	MOV	VIDEO_SEGMENT,SCREEN_SEGMENT ;Point to the color card RAM
	CMP	VIDEO_MODE,7		;Is this a monochrome set-mode?
	JNE	SVV1			;No, have correct base address
	MOV	BX,MONO_CARD		;Yes - point to the monochrome card
	MOV	VIDEO_SEGMENT,MONO_SEGMENT   ;Point to the mono card RAM
SVV1:	MOV	VIDEO_BASE,BX		;Update the global variable
	CBW				;Convert mode into a word
	MOV	BX,AX			;Copy mode into BX
	SHR	AX,1			;Turn mode into a page type
	CMP	VIDEO_MODE,7		;Is this a monochrome set mode?
	JNE	SVV2			;No, all set
	INC	AX			;Yes - use special vars for mono
SVV2:	MOV	CL,SIZE VID_VARS	;Get the size of the CRT variables
	MUL	CL			;Generate offset of the right set
	PUSH	DS			;Save the current data segment
	PUSH	CS			;Point DS to the ROM's code segment
	POP	DS
	MOV	ES,DATA_SEGMENT		;Point to the current data segment
	LEA	SI,ROM_VARS		;Point to the start of the ROM vars
	ADD	SI,AX			;Now point to the right set
	LODSB				;Read the number of pages
	MOV	NUMBER_OF_PAGES,AL	;Place it in the global variable
	LODSW				;Write the global page size
	MOV	ES: PAGE_SIZE,AX
	DEC	AX			;Make it into a page size mask
	MOV	TEXT_SCREEN_MASK,AX	;Save it in the global variable
	LODSW		 		;Update the page width
	MOV	ES: NUMBER_OF_COLUMNS,AX
	MOV	OLD_COLUMNS,AL		;...and save the 'old' copy for POKE checks
	LODSW				;Write the column increment
	MOV	COLUMN_INCREMENT,AX
	LODSW				;... and the row increment
	MOV	ROW_INCREMENT,AX
	LODSB				;Finally, get the color select
	MOV	ES: CURRENT_COLORS,AL
	POP	DS			;Restore the data segment
	MOV	AL,CS:CRT_MODE[BX]	;Get the right mode select
	MOV	VIDEO_HW_MODE,AL	;Place it in memory for updates
	MOV	AX,OFFSET SCREEN	;Point to the base of the screen RAM
	MOV	BX,0			;Use BX as word pointer to PAGE_BASE
	MOV	CL,NUMBER_OF_PAGES	;Get the number of valid pages
	MOV	CH,0
SVV3:	CMP	VIDEO_MODE,7		;Entering monochrome mode?
	JNE	SVV4			;No, set normal base addresses
	MOV	PAGE_START[8 * 2],0	;Yes - set monochrome page start
	MOV	PAGE_BASE[8 * 2],0	;Set monochrome page base
	JMP	SHORT SVV5		;Only set monochrome info!
SVV4:	MOV	PAGE_START[BX],AX	;Set a page's start address
	MOV	PAGE_BASE[BX],AX	;Set the page's current top-of-page
SVV5:	MOV	CURSOR_POSITION[BX],0	;Home this page's cursor
	ADD	AX,PAGE_SIZE		;Point to the next page
	ADD	BX,TYPE PAGE_BASE	;Point to the next PAGE_BASE
	LOOP	SVV3			;Set base addresses for all pages
	MOV	AX,DEFAULT_CURSOR_STYLE	;Get the default cursor style
	MOV	CURSOR_STYLE,AX		;Initialize the current cursor style
	MOV	AL,NUMBER_OF_ROWS-1	;Get the number of the last line
	MOV	LAST_ROW,AL		;Set it up for global access
	MOV	VIDEO_ACTIVE,FALSE	;Flag that no scrolling is happening
SVV6:	POPREG	<ES,DI,SI,CX,BX,AX>
	RET
SET_VIDEO_VARS ENDP



;***********************************************************************
; SET_SCREEN_WIDTH:
;
;	Set_Screen_Width is called to set up the new screen width for
; a program.  It is called with the new number of columns, and sets
; up the global variables START_OF_PAGE and END_OF_PAGE.
;
; Input:
;	AL: Number of columns to be displayed.
;***********************************************************************
SET_SCREEN_WIDTH PROC NEAR
	PUBLIC	SET_SCREEN_WIDTH
	PUSHREG	<AX,CX,DX>
	MOV	PAGE_START[0],0		;Assume start of page is at 0000H
	MOV	OLD_COLUMNS,AL		;Save the new number of columns
	MOV	AH,0			;Extend screen width to a word
	SHL	AX,1			;Include chars & attributes in size
	MOV	ROW_INCREMENT,AX	;Store as offset to next line
	MOV	CX,NUMBER_OF_ROWS	;Get the number of rows in DX
	MUL	CX			;Make this an address for end of page
	MOV	PAGE_SIZE,AX		;Save the number of bytes/page
	DEC	AX			;Find address of last byte on page
	MOV	END_OF_PAGE,AX		;Save the end of page address
	MOV	TEXT_SCREEN_MASK,SCREEN_BYTES-1 ;Use all of RAM for accesses
	MOV	SUPER_COMPATIBLE,TRUE	;Force use of slow video routines
	POPREG	<DX,CX,AX>
	RET
SET_SCREEN_WIDTH ENDP



;**********************************************************************
; GET_VIDEO_STATUS:
;
;	Get_Video_Status is used to read the current status of the
; video sub-system.  It returns the currently displayed page, CRT mode,
; screen width, and the current scrolling mode.
;
; Output:
;	AL: Currently selected video mode (see Set Video Mode)
;	AH: Screen width in characters.
;	BH: Currently displayed page.
;**********************************************************************
GET_VIDEO_STATUS PROC NEAR
	PUBLIC	GET_VIDEO_STATUS
	MOV	AL,VIDEO_MODE		;Get the current video mode
	MOV	AH,BYTE PTR NUMBER_OF_COLUMNS	;Get the page width
	MOV	BH,DISPLAYED_PAGE	;... and the displayed page
	RET
GET_VIDEO_STATUS ENDP



;**********************************************************************
; SET_SCROLL_MODE: (MODE)
;
;	Set_Scroll_Mode is called to set up the ROM variables for a
; particular scroll mode (software, jump or smooth scrolling). 
; Due to the nature of the Z-150 hardware, not all combinations of video
; modes and scroll modes are correct.  The following table represents
; the acceptable combinations:
;
;		40 Column Displays: Software scrolling only
;
;		80 Column Displays: Software scrolling
;		 (including mono)   'Jump' scrolling
;
;		Graphics Displays:  Software scrolling
;				    'Jump' scrolling
;				    Smooth scrolling
;
; NOTE:  If an 'invalid' scroll mode is used, the ROM will automatically
;	 default to a scroll mode which is valid for the selected video
;	 mode.
;	 Also, setting the scroll mode affects both the monochrome AND
;	 color boards!!!
;
; Input:
;	AL: Scroll mode.  0 = Software scrolling
;			  1 = Hardware jump scrolling
;			  2 = Hardware smooth scrolling
;
; Output:
;	Global variable SCROLL_MODE.
;**********************************************************************
SET_SCROLL_MODE PROC NEAR
	PUBLIC	SET_SCROLL_MODE
	PUSHREG	<AX,BX,CX>
	MOV	SCROLL_MODE,AL		;Save scroll mode as current mode
	MOV	AX,OFFSET SCREEN	;Point to the base of the screen RAM
	MOV	BX,0			;Use BX as word pointer to PAGE_BASE
	MOV	CL,NUMBER_OF_PAGES	;Get the number of valid pages
	MOV	CH,0
SSM1:	MOV	PAGE_START[BX],AX	;Set a page's start address
	MOV	PAGE_BASE[BX],AX	;Set the page's current top-of-page
	MOV	CURSOR_POSITION[BX],0	;Home this page's cursor
	ADD	AX,PAGE_SIZE		;Point to the next page
	ADD	BX,TYPE PAGE_BASE	;Point to the next PAGE_BASE
	LOOP	SSM1			;Set base addresses for all pages
	MOV	PAGE_START[8 * 2],0	;Force new start of mono page
	MOV	PAGE_BASE[8 * 2],0	;...and new base of mono page
	PUSH	VIDEO_SEGMENT		;Save the current video segment
	MOV	VIDEO_SEGMENT,SCREEN_SEGMENT ;Point to the color video board
	CALL	CLEAR_VIDEO_RAM		;Clear the color card
	MOV	VIDEO_SEGMENT,MONO_SEGMENT   ;Point to the monochrome board
	CALL	CLEAR_VIDEO_RAM		;Clear the monochrome card
	POP	VIDEO_SEGMENT		;Restore the original video segment
	PUSH	VIDEO_BASE		;Save the current I/O address
	MOV	VIDEO_BASE,COLOR_CARD	;Point to the base of the color card
	MOV	AL,DISPLAYED_PAGE	;Get the currently displayed page
	CALL	SET_DISPLAY_PAGE	;Set up scroll hardware for color card
	MOV	VIDEO_BASE,MONO_CARD	;Point to base I/O addr of mono board
	CALL	SET_DISPLAY_PAGE	;Set up scroll hardware for mono board
	POP	VIDEO_BASE		;Restore the correct video base
	POPREG	<CX,BX,AX>
	RET
SET_SCROLL_MODE ENDP



;**********************************************************************
; SET_DISPLAY_PAGE: (PAGE)
;
; 	Set_Display_Page is called to determine what page will
; be shown on the monitor.
;
; Input:
;	AL: Page number to be displayed.
;
; Output:
;	Sets the correct cursor position for this page,
;	including global variables CURRENT_PAGE, DISPLAYED_PAGE,
;	and CRT_START_ADDRESS.
;**********************************************************************
SET_DISPLAY_PAGE PROC NEAR
	PUBLIC	SET_DISPLAY_PAGE
	PUSHREG	<AX,BX,DX,SI>
	MOV	BH,AL			;Get page number in BH
	CALL	SET_IO_PAGE		;Set up I/O variables for page
	MOV	DISPLAYED_PAGE,AL	;Keep track of page being displayed
	MOV	OLD_DISPLAYED_PAGE,AL	;Keep copy of old displayed page
	CBW				;Convert page number to a word
	MOV	BX,AX			;Get page number in BX
	SHL	BX,1			;Make BX into a word offset
	MOV	SI,PAGE_BASE[BX]	;Get the current page bage
	CMP	VIDEO_MODE,7		;Are we in monochrome mode?
	JNE	SDP1			;No, have the correct address
	MOV	SI,PAGE_BASE[8 * 2]	;Yes - treat monochrome as page 8
SDP1:	SHR	SI,1			;Make it into a hardware address
	MOV	CRT_START_ADDRESS,SI	;Save display address
	MOV	DX,CURSOR_POSITION[BX]	;Get the current cursor position
	PUSH	BX			;Save the page pointer
	MOV	BH,DISPLAYED_PAGE	;Get the new display page number
	CALL	SET_CURSOR_POSITION	;Update the hardware cursor
	POP	BX			;Restore the page pointer
	CMP	VIDEO_MODE,2		;Are we in 80-column mode?
	JB	SDP2			;No, in 40 column mode
	CMP	VIDEO_MODE,3
	JA	SDP2			;No, in graphics or monochrome mode
	CMP	SCROLL_MODE,0		;80 column display: jump scroll mode?
	JE	SDP2			;No, use standard CRT addressing
	CMP	LAST_ROW,NUMBER_OF_ROWS-1;Are we in interlace mode?
	JNE	SDP2			;Yes - use all of video RAM
	MOV	AX,PAGE_START[BX]	;No, hardware 80 column scroll mode
	SHL	AX,1			;Get the hardware start for this page
	SHL	AX,1			;...in the right page select bits
	AND	AH,HSCROLL_PAGE_MASK	;Mask out all but page select
	OR	AH,VIDEO_ON OR HSCROLL	;Set 80-column scroll enable bit
	MOV	AL,AH
	JMP	SHORT SDP3		;...and enable video
SDP2:	MOV	AL,VIDEO_ON		;Turn on the video
SDP3:	CALL	OUT_VIDEO_MODE		;Set up the CRT for the right mode
	MOV	BX,SI			;Get the CRT start address
	MOV	AL,CRT_START_PORT	;Get the CRT start addr port number
	CALL	SEND_VIDEO_WORD		;Display the right page!
	POPREG	<SI,DX,BX,AX>
	RET
SET_DISPLAY_PAGE ENDP



;**********************************************************************
; SET_IO_PAGE: (PAGE)
;
;	Set_IO_Page is called prior to reading, writing or scrolling
; information on a page.  It sets global variables CURRENT_PAGE,
; START_OF_PAGE and END_OF_PAGE, so that calling routines will properly 
; wrap-around when information is displayed.
;
; Input:
;	BH: Page to be displayed.
;**********************************************************************
SET_IO_PAGE PROC NEAR
	PUBLIC	SET_IO_PAGE
	PUSHREG	<BX>			;Save page number on stack
	CMP	VIDEO_MODE,3		;In alphanumeric mode?
	JBE	SIOP1			;Yes - use page number passed
	MOV	BH,0			;No, force 0 (only 1 in graphics & mono!)
SIOP1:	MOV	CURRENT_PAGE,BH		;Record page number globally
	MOV	BX,0			;Assume PAGE_START for mono mode
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	SIOP2			;Yes - have the correct PAGE_START
	MOV	BL,CURRENT_PAGE		;No, place the page number in BX
	MOV	BH,0
	SHL	BX,1			;Make page number a word offset
	MOV	BX,PAGE_START[BX]	;Point to the start of this page
SIOP2:	MOV	START_OF_PAGE,BX	;Save start-of-page address
	ADD	BX,PAGE_SIZE		;Point to last byte of page
	DEC	BX
	MOV	END_OF_PAGE,BX		;Write last address to global var
	POPREG	<BX>			;Restore page number
	RET
SET_IO_PAGE ENDP



;**********************************************************************
; SET_CURSOR_POSITION: (PAGE, VERT, HOR)
;
;	Set_Cursor_Position is called to set up the ROM
; variables for a screen coordinate. Also updates the hardware 
; cursor if the page specified is the currently active page.
;
; Input:
;	BH: Page number
;	DL: Column number
;	DH: Row number
;
; Output:
;	Sets global variables CURRENT_PAGE and CURSOR_POSITION.
;**********************************************************************
SET_CURSOR_POSITION PROC NEAR
	PUBLIC	SET_CURSOR_POSITION
	PUSHREG	<AX,BX,DX>
	MOV	CURRENT_PAGE,BH		;Set the page for future I/O
	MOV	BL,BH			;Get the page as a word in BX
	MOV	BH,0
	SHL	BX,1			;Make it a word offset
	MOV	CURSOR_POSITION[BX],DX	;Update the cursor position variable
	MOV	AL,CURRENT_PAGE		;Get the new page number
	CMP	AL,DISPLAYED_PAGE	;Are we displaying this page?
	JNE	SCP3			;No, finished with update
	MOV	AL,DH			;Get the row number
	MUL	BYTE PTR NUMBER_OF_COLUMNS ;Generate offset of start of row
	ADD	AL,DL			;Add column to get right offset
	ADC	AH,0
	ADD	AX,CRT_START_ADDRESS	;Add in start of current page
	CMP	AX,SCREEN_BYTES / 2	;Wrapped around?
	JBE	SCP1			;No, all is OK
	SUB	AX,SCREEN_BYTES		;Yes - wrap-around to rest of screen
SCP1:	CMP	VIDEO_MODE,6		;In high-resolution graphics mode?
	JNE	SCP2			;No, update hardware cursore
	MOV	AH,TRUE			;Yes - set update-cursor flag
	CALL	UPDATE_CURSOR_HRES	;Update the software cursor
	JMP	SHORT SCP3		;Return
SCP2:	MOV	BX,AX			;Store CRT address in BX
	MOV	DX,VIDEO_BASE		;Point DX to the CRT chip
	MOV	AL,CRT_CURSOR_START	;Point to the CRT cursor address reg
	OUT	DX,AL			;Select M.S. byte of cursor addr
	INC	DX			;Now, point to the parameter port
	MOV	AL,BH			;Get the M.S. cursor addr byte
	OUT	DX,AL			;Output it...
	DEC	DX			;Point back to the control port
	MOV	AL,CRT_CURSOR_START+1
	OUT	DX,AL			;Select L.S. byte of cursor address
	INC	DX			;Point to parameter port again
	MOV	AL,BL			;Output L.S. byte of address
	OUT	DX,AL
SCP3:	POPREG	<DX,BX,AX>
	RET
SET_CURSOR_POSITION ENDP



;**********************************************************************
; GET_CURSOR_POSITION: (PAGE)
;
; 	Get_Cursor_Position is called to read the cursor's
; current position from the global variables.  In addition,
; the starting and ending line defining the hardware cursor
; are returned.
;
; Input:
;	BH: Current page number
;
; Output:
;	DL: Column number
;	DH: Row number
;	CH: Cursor's starting scan line
;	CL: Cursor's ending scan line
;**********************************************************************
GET_CURSOR_POSITION PROC NEAR
	PUBLIC	GET_CURSOR_POSITION
	PUSHREG	<BX>
	MOV	BL,BH			;Get the page number
	MOV	BH,0			;Extend it to a word
	SHL	BX,1			;Make it a word offset
	MOV	DX,CURSOR_POSITION[BX]	;Get the cursor's position
	MOV	CX,CURSOR_STYLE		;Get the current cursor style
	POPREG	<BX>
	RET
GET_CURSOR_POSITION ENDP



;**********************************************************************
; SET_CURSOR_STYLE: (START_SCAN_LINE, END_SCAN_LINE)
;
;	Set_Cursor_Style sets up the CRT controller chip for a
; particular cursor style.  The cursor will start at 'START_SCAN_LINE'
; and end at 'END_SCAN_LINE'.
;
; Input:
;	CH: Starting scan line
;	CL: Ending scan line
;**********************************************************************
SET_CURSOR_STYLE PROC NEAR
	PUBLIC	SET_CURSOR_STYLE
	PUSHREG	<AX,BX>			;Save registers
	MOV	CURSOR_STYLE,CX		;Place the cursor style in memory
	MOV	AL,CRT_CURSOR_STYLE	;Get the cursor style port	
	MOV	BX,CX			;Place start & end scan lines in BX
	CALL	SEND_VIDEO_WORD		;Tell CRT controller
	POPREG	<BX,AX>			;Restore registers
	RET
SET_CURSOR_STYLE ENDP



;**********************************************************************
; SET_COLOR: (PALLETTE, COLOR)
;
;	Set_Color is called to set the screen colors used.  It can
; select background colors, as well as the foreground colors in
; medium-resolution graphics mode.
;
; Input:
;	BH: Color register selection.  If BH = 0, then the color value
;	    in BL is used for the background color (for alphanumeric
;	    modes) - legal values are from 0-15 (bits 0-3), with bit 4
;	    accepted as the high-intensity flag.  This may also be used 
;	    for medium-resolution graphics mode, except that the high-
;	    intensity flag is ignored.
;	    If BH = 1, then the value in BL will contain the 'pallette'
;	    used for medium-resolution graphics (320 by 200).
;	BL: Value to use for color register.  In medium-resolution
;	    graphics mode, two different pallettes are used:
;	    If BL=0, then:
;			Color 0 is the background color
;			Color 1 is green
;			Color 2 is red
;			Color 3 is yellow
;	    If BL=1, then:
;			Color 0 is the background color
;			Color 1 is cyan
;			Color 2 is magenta
;			Color 3 is white
;**********************************************************************
SET_COLOR PROC NEAR
	PUBLIC	SET_COLOR
	PUSHREG	<AX,BX,CX,DX>
	MOV	AL,CURRENT_COLORS	;Get the current color values in AL
	TEST	BH,BH			;Setting background colors?
	JNZ	SCLR1			;No, continue
	AND	AL,NOT BACKGROUND_COLOR_MASK ;Yes - mask out old background
	AND	BL,BACKGROUND_COLOR_MASK;Mask out unused bits in new value
	JMP	SHORT SCLR2
SCLR1:	AND	AL,NOT FOREGROUND_COLOR_MASK ;Mask out old foreground color
	MOV	CL,5			;Shift count to put bit in right posn
	SHL	BL,CL			;Place pallette select in right bit
	AND	BL,FOREGROUND_COLOR_MASK;Mask out unused bits of new value
SCLR2:	OR	AL,BL			;Insert new color value
	MOV	DX,VIDEO_BASE		;Point to the color selection port
	ADD	DX,CRT_COLOR_SELECT
	OUT	DX,AL			;Output new colors
	MOV	CURRENT_COLORS,AL	;Save new value
	POPREG	<DX,CX,BX,AX>
	RET
SET_COLOR ENDP



;**********************************************************************
; GET_LPEN_STATUS:
;
;	Get_LPen_Status reads the light pen status.  It
; determines if the light pen has been activated - if so, the
; screen coordinates (both pixel & alphanumeric) are returned.
;
; Output:
;	AH: Light pen status.  If AH is 0, then the light pen is
;	    either not pressed to the screen, or is not triggered.
;	    If AH = 1, then the following registers are valid:
;	DX: Position of light pen (DH:row, and DL:column) as
;	    an ALPHANUMERIC position.
;	CH: Vertical scan line number (from 0-199).
;	BX: Approximate pixel column (from 0-319, or 0-639, depending
;	    on the currently selected video mode).
;**********************************************************************
GET_LPEN_STATUS PROC NEAR
	PUBLIC	GET_LPEN_STATUS
	PUSHREG	<SI,ES>
	MOV	BH,DISPLAYED_PAGE	;Point to the currently displayed page
	CALL	SET_IO_PAGE		;Set up displayed page for I/O
	MOV	DX,VIDEO_BASE		;Point to the video status register
	ADD	DX,CRT_STATUS_PORT
	IN	AL,DX			;Read the light pen's status
	TEST	AL,LPEN_SWITCH		;Is the light pen's switch pressed?
	MOV	AH,0			;[Assume that it isn't]
	JNZ	GLPS4			;No, reset trigger and return not ready
	TEST	AL,LPEN_TRIGGER		;Is the light pen triggered?
	JZ	GLPS5			;No, return not ready
	MOV	AL,CRT_LPEN_POSITION	;Yes - point to the lpen position port
	CALL	RECV_VIDEO_WORD		;Read the light pen position from 6845
	MOV	CX,AX			;Save lpen position in CX
	MOV	BL,VIDEO_MODE		;Get the current video mode
	MOV	BH,0			;Convert it to a word
	SUB	CL,CS:LPEN_ADJUST[BX]	;Adjust the lpen position for this mode
	SBB	CH,0			;Take care of any carry from subtract
	SUB	CX,CRT_START_ADDRESS	;Generate offset of start of page
	MOV	AX,PAGE_SIZE		;Get the software page size
	SHR	AX,1			;Make it a hardware page size
	DEC	AX			;Make it a bit mask
	AND	CX,AX			;Keep the page offset in range
GLPS1:	CALL	GET_VIDEO_PARMS		;Point ES:SI to the video parameters
	MOV	AX,CX			;Get the screen offset in AX
	DIV	BYTE PTR ES:[SI].H_DISP	;Divide by characters per line
	MOV	CL,3			;Set divide/multiply by 8 shift count
	XCHG	AL,AH			;Put column # in AL, row # in AH
	MOV	CH,AH			;Place vertical position in CH
	CMP	BYTE PTR ES:[SI].MAX_SCAN,1 ;In graphics mode?
	JNE	GLPS2			;No, have correct row number
	SHL	CH,1			;Yes - generate graphics row #
	SHR	AH,1			;Also, divide alpha row number by 4...
	SHR	AH,1			;...to account for 2-scan line chars
	CMP	VIDEO_MODE,6		;In high-resolution graphics mode?
	JNE	GLPS3			;No, all set
	SHL	AL,1			;Yes - get actual character column #
	JMP	SHORT GLPS3		;Set horizontal position
GLPS2:	SHL	CH,CL			;Generate graphics row # (in text mode)
GLPS3:	MOV	DX,AX			;Place alpha screen posn in DX
	MOV	BL,AL			;Get horiz. posn in BX as a word
	MOV	BH,0
	SHL	BX,CL			;Generate graphics horiz. position
	MOV	AH,1			;Flag that light pen was activated
GLPS4:	PUSH	DX			;Save the character position
	MOV	DX,VIDEO_BASE		;Point DX to the base video I/O addr
	ADD	DX,LPEN_CTRL		;Point to the light pen control port
	OUT	DX,AL			;Write to port clears trigger status
	POP	DX			;Restore the alpha screen position
GLPS5:	POPREG	<ES,SI>
	RET
GET_LPEN_STATUS ENDP



;**********************************************************************
; GET_VIDEO_PARMS:
;
;	Get_Video_Parms returns a pointer to the video parameter 
; block associated with the current video mode.
;
; Output:
;    ES:SI: Pointing to the currently active video parameters
;**********************************************************************
GET_VIDEO_PARMS PROC NEAR
	PUBLIC	GET_VIDEO_PARMS
	PUSHREG	<AX,CX>
	PUSH	DS			;Save the monitor data segment
	MOV	AL,VIDEO_PARMS_INTR	;Get intr # of the video parameters
	CALL	GET_INTR_PTR		;Point to the video parameter tables
	PUSH	DS			;Save pointer to video parms segment
	POP	ES			;Point ES to the video parms segment
	POP	DS			;Restore pointer to data segment
	MOV	AL,VIDEO_MODE		;Get the video mode again
	CBW				;Convert mode into a word
	SHR	AX,1			;Turn mode into a page type
	CMP	VIDEO_MODE,6		;Entering high-res graphics mode?
	JNE	GVP1			;No, pointing to right parameters
	DEC	AX			;Yes - use common graphics parameters
GVP1:	MOV	CL,SIZE VID_PARMS	;Get the size of the video parameters
	MUL	CL			;Generate offset of correct table
	ADD	SI,AX			;Now point to the right table
	POPREG	<CX,AX>
	RET
GET_VIDEO_PARMS ENDP



;**********************************************************************
; SET_SCREEN_ADDRESS:
;
;	Set_Screen_Address is called to update the global
; cursor pointer with the address of the current character.
; This routine will return a pointer to a CHARACTER, not
; a pixel.
;
; Input:
;	    Global variables CURRENT_PAGE, CURSOR_POSITION
;
; Output:
;    ES:DI: Pointing to current screen character
;**********************************************************************
SET_SCREEN_ADDRESS PROC NEAR
	PUBLIC	SET_SCREEN_ADDRESS
	PUSHREG	<BX,CX>
	MOV	BL,CURRENT_PAGE		;Get the current page in BX
	MOV	BH,0
	SHL	BX,1			;Make it a word offset
	MOV	CX,CURSOR_POSITION[BX]	;Get the current cursor position
	CALL	SET_SCREEN_PTR		;Set ES:DI to the right address
	POPREG	<CX,BX>
	RET
SET_SCREEN_ADDRESS ENDP



;**********************************************************************
; SET_SCREEN_PTR: (POSITION)
;
;	Set_Screen_Ptr is similar to Set_Screen_Address,
; except that it permits a variable screen position to be
; used.
;
; Input:
;	CX: Screen position (CH: row, and CL: column)
;
; Output:
;    ES:DI: Pointing to the correct screen location
;**********************************************************************
SET_SCREEN_PTR PROC NEAR
	PUBLIC	SET_SCREEN_PTR
	PUSHREG	<AX,CX,DX>
	MOV	DL,CH			;Place row number in DX
	MOV	DH,0
	MOV	CH,DH			;Extend row and column to words
	MOV	AH,TRUE			;Indicate character mode
	CALL	VIDEO_PTR		;Point to the right location
	POPREG	<DX,CX,AX>
	RET
SET_SCREEN_PTR ENDP



;**********************************************************************
; VIDEO_PTR: (Alpha_Flag, Row, Column)
;
;	Video_Ptr is called to transfer a screen coordinate 
; into an absolute screen address.
;
; Input:
;	AH: If set (non-zero) then an alphanumeric mode
;	    address will be returned.  Otherwise, a
;	    graphics address will be returned.
;	CX: Column number (or horizontal position for graphics)
;	DX: Row number (or vertical position)
;
; Output:
;    ES:DI: Pointing to the actual screen address for this
;	    coordinate.
;	AX: If a graphics address was requested, then AH
;	    will contain the pixel number (0-3, or 0-7 for
;	    medium and high resolution graphics modes,
;	    respectively), and AL will contain the number
;	    of bits/pixel.
;**********************************************************************
VIDEO_PTR PROC NEAR
	PUBLIC	VIDEO_PTR
	PUSHREG	<BX,CX,DX,BP>
;
; Generate offset of start of line
;
	TEST	AH,AH			;Are we in alphanumeric mode?
	JNE	VDP1			;Yes - continue
	SHR	DX,1			;No, divide vertical position by 2
	PUSHF				;Save Even/Odd scan line status on stack
VDP1:	MOV	AL,0			;Place only alpha mode flag in AX
	MOV	BP,AX			;Save alpha mode flag in BP
	MOV	AX,DX			;Get the row number
	TEST	BP,BP			;In alpha mode?
	JE	VDP2			;No, generate graphics address
	MUL	ROW_INCREMENT		;Yes - mult by bytes between rows
	JMP	VDP3
VDP2:	PUSH	CX			;No, save horizontal position
	MOV	CL,SCAN_LINE_INCREMENT	;Get the offset between scan lines
	MUL	CL			;Multiply by bytes between scan lines
	POP	CX
VDP3:	MOV	DX,AX			;Save start-of-line offset in DX
	TEST	BP,BP			;Are we in graphics mode?
	JNE	VDP5			;No, generate alphanumeric address
	MOV	AH,CL			;Yes - get the horizontal position
;
; Generate pixel number for graphics mode & adjust horizontal position
;
	SHR	CX,1			;Divide horiz position by pixels/byte
	SHR	CX,1			;Assume medium resolution...
	CMP	VIDEO_MODE,6		;Are we in high-resolution mode?
	JNE	VDP4			;No, use 2 bits/pixel
	AND	AH,8 - 1		;Yes - have 1 bit/pixel (8 / byte)
	MOV	AL,1			;Set return parameter to 1 bit/pixel
	SHR	CX,1			;Adjust horiz position for high-res
	JMP	SHORT VDP6
VDP4:	AND	AH,4 - 1		;Get medium resolution pixel number
	MOV	AL,2			;Set return parm as 2 bits/pixel
	JMP	SHORT VDP6
;
; Add line offset, and create screen pointer
;
VDP5:	CMP	COLUMN_INCREMENT,1	;Is the column increment > 1?
	JLE	VDP6			;No, already have line offset
	SHL	CX,1			;Yes - get offset in the current row
VDP6:	ADD	DX,CX			;Now have screen offset in DX
	MOV	BL,CURRENT_PAGE		;Get the currently active page
	MOV	BH,0			;Extend it to a word
	SHL	BX,1			;Make it a word offset
	MOV	BX,PAGE_BASE[BX]	;Point to start of the current page
	CMP	VIDEO_MODE,7		;In monochrome text mode?
	JNE	VDP65			;No, have correct page base
	MOV	BX,PAGE_BASE[8 * 2]	;Yes - treat monochrome as page 8
VDP65:	ADD	DX,BX			;Add in offset - now have pointer
	CMP	VIDEO_MODE,3		;Is the hardware in graphics mode?
	JBE	VDP7			;No, continue
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	VDP7			;Yes - cannot use graphics!
	TEST	BP,BP			;Generating graphics address?
	JE	VDP7			;Yes - use full screen addressing
	CMP	DX,OFFSET END_OF_PART	;Alpha/graphics - past end of part 1?
	JBE	VDP8			;No, continue
	SUB	DX,SCREEN_BYTES SHR 1	;Yes - wrap to start of partition
	JMP	SHORT VDP8
VDP7:	CMP	DX,END_OF_PAGE		;Beyond end of screen?
	JBE	VDP8			;No, continue
	SUB	DX,PAGE_SIZE		;Yes - wrap-around
VDP8:	TEST	BP,BP			;In alphanumeric mode?
	JNE	VDP9			;Yes - return
	POPF				;Restore Even/Odd scan line flag
	JNC	VDP9			;On even scan line in graphics mode
	XOR	DX,SCREEN_PARTITION_OFS	;On odd scan line - point to it
VDP9:	MOV	ES,VIDEO_SEGMENT	;Prepare return address...
	MOV	DI,DX			;...in ES:DI
	POPREG	<BP,DX,CX,BX>
	RET
VIDEO_PTR ENDP



;**********************************************************************
; EXPAND_FONT: (FONT_BYTE)
;
;	Expand_Font is called in the medium-resolution
; graphics mode to expand a font character (byte) into
; a word suitable for display.  This is accomplished by
; doubling each bit in the byte.
;
; Input:
;	AL: Font byte
;
; Output:
;	AX: Expanded font word
;**********************************************************************
EXPAND_FONT PROC NEAR
	PUBLIC	EXPAND_FONT
	PUSHREG	<BX,CX>
	MOV	CX,8			;Repeat for each bit of the font byte
EF1:	SHL	BX,1			;Shift dest left for next bit
	SHL	BX,1
	SHL	AL,1			;Get a bit of the font into carry
	JNC	EF2			;Bit not set - continue
	OR	BX,11B			;Bit set - set correct bits in dest
EF2:	LOOP	EF1			;Continue until whole word expanded
	MOV	AX,BX			;Place result in AX
	POPREG	<CX,BX>
	RET
EXPAND_FONT ENDP



;**********************************************************************
; COMPRESS_FONT: (FONT_WORD)
;
;	Compress_Font takes a medium-resolution graphics word and
; compacts it into a single byte.  Of the 8 pixels in the graphics
; word, any pixel with a color other than the background color is
; considered 'ON'.
;
; Input:
;	AX: Graphics word
;
; Output:
;	AL: Packed font byte
;	AH: will not be informative
;**********************************************************************
COMPRESS_FONT PROC NEAR
	PUBLIC	COMPRESS_FONT
	PUSHREG	<BX,CX,DX>
	MOV	BX,1100000000000000B	;Mask for a single pixel
	MOV	CX,8			;Number of pixels to work on
CF1:	SHL	DL,1			;Align dest for next bit
	PUSH	AX			;Save font value
	AND	AX,BX			;Mask out a single pixel
	JZ	CF2			;Pixel not set - continue
	OR	DL,1			;Pixel on - set dest bit
CF2:	POP	AX			;Restore word value
	SHR	BX,1			;Shift BX for new pixel
	SHR	BX,1
	LOOP	CF1			;Loop till all 8 bits gotten
	MOV	AL,DL			;Get return parameter
	POPREG	<DX,CX,BX>
	RET
COMPRESS_FONT ENDP



;**********************************************************************
; INCREMENT_POSITION: (DIRECTION, INCREMENT, POINTER1, POINTER2)
;
;	Increment_Position is used to advance a pair of
; screen pointers either forward or backward by a fixed
; amount.  This routine will detect and correct overflow
; problems.
;
; Input:
;	AH: Direction flag.  If AH is set (non-zero), then
;	    the increment will be subtracted from the
;	    pointers.  Otherwise, the increment will be
;	    added to them.
;	DX: Increment to add to pointers
;	SI: Screen pointer
;	DI: Screen pointer
;
; Output:
;	SI: Updated screen pointer
;	DI: Updated screen pointer
;**********************************************************************
INCREMENT_POSITION PROC NEAR
	PUBLIC	INCREMENT_POSITION
	PUSHREG	<BX,CX,BP>
	MOV	CX,START_OF_PAGE	;Point to the start of this page
	MOV	BX,PAGE_SIZE		;Get the page size, too
	MOV	BP,END_OF_PAGE		;...as well as the end of the page
	CMP	VIDEO_MODE,3		;In alphanumeric mode?
	JBE	IP2			;Yes - parameters set correctly
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	IP2			;Yes - parameters are OK
	MOV	BX,SCREEN_BYTES	SHR 1	;Get the graphics partition size
	CMP	SI,OFFSET END_OF_PART	;Are we in the first partition?
	JA	IP1			;No, set second partition parms
	MOV	BP,OFFSET END_OF_PART	;Partition 1 - set end of partition
	JMP	SHORT IP2		;Continue
IP1:	MOV	CX,OFFSET END_OF_PART+1	;Partition 2 - set start address
IP2:	TEST	AH,AH			;Are we adding the increment?
	JNE	IP4			;No, subtract it
	ADD	SI,DX			;Yes - advance pointer1
	CMP	SI,BP			;Beyond end of page on pointer1?
	JBE	IP3			;No, continue
	SUB	SI,BX			;Yes - wrap-around
IP3:	ADD	DI,DX			;Advance pointer2
	CMP	DI,BP			;Pointer2 beyond end of page?
	JBE	IP6			;No, return
	SUB	DI,BX			;Yes - wrap-around pointer1
	JMP	SHORT IP6
IP4:	SUB	SI,DX			;Subtract increment from pointer1
	CMP	SI,CX			;Below start of page?
	JGE	IP5			;No wrap-around required
	ADD	SI,BX			;Wrap-around pointer1
IP5:	SUB	DI,DX			;Subtract increment from pointer2
	CMP	DI,CX			;Below start of page?
	JGE	IP6			;No wrap required - return
	ADD	DI,BX			;Wrap pointer2 around
IP6:	POPREG	<BP,CX,BX>
	RET
INCREMENT_POSITION ENDP



;**********************************************************************
; CLEAR_VIDEO_RAM:
;
;	Clear_Video_RAM is called when a mode change has occurred - 
; it clears all of the pages.
;**********************************************************************
CLEAR_VIDEO_RAM PROC NEAR
	PUBLIC	CLEAR_VIDEO_RAM
	PUSHREG	<AX,CX,DI,ES>
	MOV	ES,VIDEO_SEGMENT	;Point ES to the screen segment
	LEA	DI,SCREEN		;Point to the base screen address
	MOV	AX,(DEFAULT_ATTRIBUTE SHL 8) OR ' '	;Assume alpha mode
	CMP	VIDEO_SEGMENT,MONO_SEGMENT ;Is this the monochrome card?
	JE	CVR1			;Yes - have right data to fill with
	CMP	VIDEO_MODE,3		;Are we in graphics mode?
	JBE	CVR1			;No, fill screen with blanks
	XOR	AX,AX			;Color card graphics mode - fill w/00H
CVR1:	MOV	CX,SCREEN_WORDS		;Get the screen size
	REP	STOS WORD PTR ES: [DI]	;Clear screen
	POPREG	<ES,DI,CX,AX>
	RET
CLEAR_VIDEO_RAM ENDP



;**********************************************************************
; CLEAR_LINE: (LINE_TYPE, LINE_START_ADDRESS, ATTRIBUTE, LENGTH)
;
;	Clear_Line is used to erase a line of information
; from the screen.  Either individual scan lines, or char-
; acter lines may be erased.  The 'line' is actually a
; string of sequential characters of arbitrary length.
; Also, the attribute for the new line may be set to any
; desired value.
;
; Input:
;	AL: Line-Type flag.  If set (non-zero), only a single
;	    scan line is cleared.  Otherwise (if AL is 0),
;	    an entire character line is cleared.
;    ES:DI: Start address of line
;	BH: Attribute byte for the line
;	CX: Length of the line in characters
;**********************************************************************
CLEAR_LINE PROC NEAR
	PUBLIC	CLEAR_LINE
	PUSHREG	<AX,BX,CX,DX,SI,DI,BP>
	MOV	BP,1			;Assume 1 scan line / char line
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	CL1			;Yes - clear text lines
	CMP	VIDEO_MODE,3		;Is graphics mode enabled?
	JBE	CL1			;No, continue
	TEST	AL,AL			;Clearing a single scan line?
	JNE	CL1			;Yes - count already set to 1
	MOV	BP,FONT_HEIGHT		;No, get the number of scan lines
CL1:	PUSH	DI			;Save the line's start address
	PUSH	CX			;Save the line length, too
CL2:	CMP	COLUMN_INCREMENT,1	;Are the characters bytes or words?
	JE	CL3			;Bytes - have the right count already
	SHL	CX,1			;Words - include attrib byte in count
CL3:	MOV	AH,FALSE		;Clear the wrap-around flag
	MOV	SI,DI			;Point to area to be cleared
	ADD	SI,CX			;Add in length of line
	CMP	VIDEO_MODE,7		;Are we in monochrome mode?
	JE	CL4			;Yes - don't bother with graphics
	CMP	VIDEO_MODE,3		;Are we in graphics mode?
	JBE	CL4			;No, continue
	CMP	DI,OFFSET END_OF_PART	;Are we in the second partition?
	JBE	CL4			;No, check first partition
	CMP	SI,OFFSET END_OF_SCREEN	;Yes - will the clear wrap around?
	JBE	CL6			;No, continue
	MOV	DX,OFFSET END_OF_SCREEN+1 ;Wrap will occur - get part. length
	JMP	SHORT CL5		;Calculate clear parameters
CL4:	CMP	SI,END_OF_PAGE		;Will the clear wrap-around screen?
	JBE	CL6			;No, continue
	MOV	DX,END_OF_PAGE		;Yes - get the last screen offset
	INC	DX			;Make it the addr of the next page
CL5:	SUB	DX,DI			;Find how many bytes can be cleared
	XCHG	DX,CX			;Place new count in CX, save old
	SUB	DX,CX			;Subtract from orig. count # bytes
	MOV	AH,TRUE			;Set the wrapped-around flag
CL6:	PUSH	AX			;Save the line-type flag
	MOV	AH,BH			;Get attribute (if used)
	MOV	AL,' ' 			;Assume space for fill character
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	CL7			;Yes - have correct fill character
	CMP	VIDEO_MODE,3		;In graphics mode?
	JBE	CL7			;No, continue
	MOV	AX,0			;Yes - set zero as fill character
CL7:	SHR	CX,1			;Make byte count a word count
	JNC	CL8			;Moving even number of words
	STOS	BYTE PTR ES: [DI]	;Moving odd count - move the odd byte
CL8:	REP	STOS WORD PTR ES: [DI]	;Clear a scan line of words
	POP	AX			;Restore the line-type flag
	TEST	AH,AH			;Did we wrap-around?
	JZ	CL9			;No, go on to next scan line
	MOV	CX,DX			;Yes - get remaining byte count
	MOV	DI,START_OF_PAGE	;Point to first byte of page
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	CL3			;Yes - have correct screen start
	CMP	VIDEO_MODE,3		;In graphics mode?
	JBE	CL3			;No, continue
	CMP	SI,OFFSET END_OF_SCREEN	;Was the wrap past partition 2?
	JBE	CL3			;No, finish clearing scan line
	MOV	DI,OFFSET END_OF_PART+1	;Yes - point to start of partition 2
	JMP	SHORT CL3		;Clear rest of this scan line
CL9:	POP	CX			;Restore line length
	POP	DI			;Restore start-of-line pointer
	DEC	BP			;Cleared all of character line yet?
	JZ	CL11			;Yes - return
	XOR	DI,SCREEN_PARTITION_OFS	;No, pnt to the (even/odd) scan line
	TEST	BP,1			;Did we just clear an odd scan line?
	JNE	CL10			;No, pointing to an odd scan line now
	ADD	DI,SCAN_LINE_INCREMENT	;Yes - point to the even scan line
	CMP	DI,OFFSET END_OF_PART	;Did we go past the grafix partition?
	JBE	CL10			;No, go on to erase next scan line
	SUB	DI,SCREEN_BYTES SHR 1	;Yes - wrap-around to next line
CL10:	JMP	CL1			;Go on to the next scan line
CL11:	POPREG	<BP,DI,SI,DX,CX,BX,AX>
	RET
CLEAR_LINE ENDP



;**********************************************************************
; CLEAR_SCREEN:
;
;	Clear_Screen erases the contents of the currently displayed
; page, and homes the cursor.  The attribute used in text modes is the
; usual normal-intensity white characters on black background.
;**********************************************************************
CLEAR_SCREEN PROC NEAR
	PUBLIC	CLEAR_SCREEN
	PUSHREG	<AX,BX,CX,DX,DI,ES>	;Save registers
	MOV	CX,(0 SHL 8) OR 0	;Set position of upper-left hand corner
	MOV	DH,LAST_ROW		;Set position of lower right-hand
	MOV	DL,BYTE PTR NUMBER_OF_COLUMNS	;...corner of screen in DX
	DEC	DL
	MOV	BH,DEFAULT_ATTRIBUTE	;Set BH to the default attribute
	CALL	SCROLL_UP		;Erase the entire page
	MOV	DX,CX			;Now, set DX to the home position
	MOV	BH,DISPLAYED_PAGE	;Set the current display page
	CALL	SET_CURSOR_POSITION	;Home the logical and physical cursors
	POPREG	<ES,DI,DX,CX,BX,AX>	;Restore registers
	RET
CLEAR_SCREEN ENDP



;**********************************************************************
; OUT_VIDEO_MODE: (MODE)
;
;	Out_Video_Mode outputs a mode value to the video hardware.
; It takes the mode set by the user, and adds to the mode the
; current hardware alternate font selection.
;
; Input:
;	AL: New mode value to be output.
;**********************************************************************
OUT_VIDEO_MODE PROC NEAR
	PUBLIC	OUT_VIDEO_MODE
	PUSHREG	<AX,CX,DX>
	MOV	CH,AL			;Save the value to be output
	MOV	DX,VIDEO_BASE		;Point to the correct I/O board
	CMP	DX,MONO_CARD		;Is this the monochrome card?
	JE	OVM1			;Yes - don't have control port!!!
	ADD	DX,CRT_STATUS_PORT	;No, point to the CRT status port
	IN	AL,DX			;Read the CRT status
	MOV	CL,3			;Shift font select to right position
	SHR	AL,CL
	AND	AL,CRT_FONT_SELECT	;Mask out all but font select bits
	OR	AL,CH			;OR in font select with new mode
	OUT	DX,AL			;Select right mode, font
	MOV	CRT_CONTROL,AL		;Save the value written to the control
OVM1:	POPREG	<DX,CX,AX>
	RET
OUT_VIDEO_MODE ENDP



;**********************************************************************
; SEND_VIDEO_WORD: (START_PORT, VALUE)
;
;	Send_Video_Word sends a word value to the 6845 CRT controller
; chip.  The high byte is sent first, then the low byte is sent to the
; next register.
;
; Input:
;	AL: First register number in the 6845
;	BX: Value to be output
;**********************************************************************
SEND_VIDEO_WORD PROC NEAR
	PUBLIC	SEND_VIDEO_WORD
	PUSHREG	<AX,DX>
	MOV	DX,VIDEO_BASE		;Point to the 6845 registers
	OUT	DX,AL			;Select correct register
	PUSH	AX			;Save register number
	MOV	AL,BH			;Get the high byte to be sent
	INC	DX			;Point to the parameter port
	OUT	DX,AL			;Output high byte
	POP	AX			;Restore register number
	INC	AL			;Point to the next register
	DEC	DX			;Point DX back to the control reg
	OUT	DX,AL			;Select the second register
	INC	DX			;Point back to parameter register
	MOV	AL,BL			;Get the low byte to send
	OUT	DX,AL			;Send the low byte
	POPREG	<DX,AX>			;Restore registers
	RET
SEND_VIDEO_WORD ENDP



;**********************************************************************
; RECV_VIDEO_WORD: (START_PORT)
;
;	Recv_Video_Word reads a word value from a sequential register
; pair in the 6845 CRT controller chip.
;
; Input:
;	AL: Starting port number.
;
; Output:
;	AX: Value read
;**********************************************************************
RECV_VIDEO_WORD PROC NEAR
	PUBLIC	RECV_VIDEO_WORD
	PUSHREG	<CX,DX>
	MOV	DX,VIDEO_BASE		;Point DX to the controller chip
	OUT	DX,AL			;Select the first register
	MOV	CL,AL			;Copy register number into CL
	INC	DX			;Point to the parameter port
	IN	AL,DX			;Read high byte
	MOV	AH,AL			;Place in high byte of AX
	MOV	AL,CL			;Get the starting register
	INC	AL			;Make it the second register number
	DEC	DX			;Point back to the control port
	OUT	DX,AL			;Select the second register
	INC	DX			;Point to the parameter port
	IN	AL,DX			;Read low byte from controller
	POPREG	<DX,CX>
	RET
RECV_VIDEO_WORD ENDP



;**********************************************************************
; SET_DEFAULT_VIDEO_MODE:
;
;	Set_Default_Video_Mode polls the system configuration DIP
; switches to determine what mode should be used following power-up
; or reset, and then sets the system video to that selected.  This
; routine also forces the scroll mode to 0 (compatible mode).
;**********************************************************************
SET_DEFAULT_VIDEO_MODE PROC NEAR
	PUBLIC	SET_DEFAULT_VIDEO_MODE
	PUSHREG	<AX>
	MOV	SCROLL_MODE,0		;Force software scrolling mode
	CALL	READ_DIP_SWITCHES	;Read the system DIP switches
	IF	CHEAP			;If cost-reduced Z-150...
	CMP	RESET_FLAG,1234H	;Is this a warm boot?
	JE	SDVM0			;Yes - have logical switch values
	OR	AH,VIDEO_MODE_80	;No, synthesize 80 column mono/color
	ENDIF				;...switch settings for setup check
SDVM0:	AND	AH,VIDEO_MODE_MASK	;Mask out all but video mode
	MOV	AL,3			;Assume 80 by 25 color...
	CMP	AH,VIDEO_MODE_MONO	;Is this monochrome mode?
	JNE	SDVM1			;No, try next mode
	MOV	AL,7			;Yes - set correct video mode
SDVM1:	CMP	AH,VIDEO_MODE_40	;Is 40 column output requested?
	JNE	SDVM2			;No, the selected mode
	MOV	AL,1			;Yes - set 40 by 25 color mode
SDVM2:	MOV	AH,0			;Set AH to set-video-mode
	MOV	OLD_VIDEO_MODE,AL	;Ignore old video mode completely
	MOV	VIDEO_MODE,AL		;Force skip of old-video-mode check
	INT	VIDEO_IO_INTR		;Set the default video mode
	MOV	SUPER_COMPATIBLE,FALSE	;Start up in FAST mode
	POPREG	<AX>
	RET
SET_DEFAULT_VIDEO_MODE ENDP



;**********************************************************************
;	 C R T   M O D E   V A R I A B L E   T E M P L A T E S
;**********************************************************************
	 PUBLIC	  ROM_VARS
ROM_VARS VID_VARS <8,2048,40,2,80,30H>	;Templates for 40 by 25
         VID_VARS <4,4096,80,2,160,30H>	;Templates for 80 by 25
         VID_VARS <1,8192,40,2,320,30H>	;Templates for Med-res
	 VID_VARS <1,8192,80,1,320,3FH>	;Templates for High-res
	 VID_VARS <1,4096,80,2,160,30H>	;Templates for monochrome card



;**********************************************************************
;	     C R T   H A R D W A R E   M O D E   T A B L E
;**********************************************************************
		PUBLIC	CRT_MODE
CRT_MODE	DB	101100B		;40 by 25 B & W mode
		DB	101000B		;40 by 25 color mode
		DB	101101B		;80 by 25 B & W mode
		DB	101001B		;80 by 25 color mode
		DB	101010B		;320 by 200 color graphics mode
		DB	101010B		;320 by 200 B & W graphics mode
		DB	011110B		;640 by 200 B & W graphics mode
		DB	101001B		;80 by 25 mono card mode


;**********************************************************************
;	     L I G H T   P E N   A D J U S T   T A B L E
;**********************************************************************
		PUBLIC	LPEN_ADJUST
LPEN_ADJUST	DB	04H, 04H, 05H, 05H, 04H, 04H, 04H, 04H ;By video mode...


;**********************************************************************
;		     P O I N T E R   T O   F O N T
;**********************************************************************
		PUBLIC	FONT_PTR
FONT_PTR	DD	FONT		;Pointer to standard ROM font


;**********************************************************************
;		    G R A P H I C S   T A B L E S
;**********************************************************************
; Mask for high-resolution graphics pixels
			PUBLIC	HIGH_RES_MASKS
HIGH_RES_MASKS		DB	10000000B		;Pixel 0
			DB	01000000B		;Pixel 1
			DB	00100000B		;Pixel 2
			DB	00010000B		;Pixel 3
			DB	00001000B		;Pixel 4
			DB	00000100B		;Pixel 5
			DB	00000010B		;Pixel 6
			DB	00000001B		;Pixel 7
; Mask for medium-resolution graphics pixels
			PUBLIC	MED_RES_MASKS
MED_RES_MASKS		DB	11000000B		;Pixel 0
			DB	00110000B		;Pixel 1
			DB	00001100B		;Pixel 2
			DB	00000011B		;Pixel 3
			DB	11000000B		;Pixel 0
			DB	00110000B		;Pixel 1
			DB	00001100B		;Pixel 2
			DB	00000011B		;Pixel 3

; Table to spread a color value across a word
			PUBLIC	MED_RES_COLOR
MED_RES_COLOR		DW	0000000000000000B	;Color 0
			DW	0101010101010101B	;Color 1
			DW	1010101010101010B	;Color 2
			DW	1111111111111111B	;Color 3



MONITOR_SEGMENT ENDS

	END

