;**********************************************************************
; TERMINAL_DISPLAY_HRES: (CHAR)
;
;	Terminal_Display_Hres is used as a special case routine for
; very high-speed display of characters while in high-resolution
; graphics mode.  Terminal_Display_Hres, like Terminal_Display, 
; cepts an ASCII character, and outputs it to the display.  The 
; ntrol characters CR (Carriage Return), LF (Line Feed), BS (Backspace) 
; and BELL are processed as commands, rather than being displayed.
;
;	One important note is that the structure of this routine is
; rather unorthodox - this was done to optimize execution time.
;
; Input:
;	AL: Character to be displayed / control character
;**********************************************************************
TERMINAL_DISPLAY_HRES PROC NEAR
	PUBLIC	TERMINAL_DISPLAY_HRES
	TEST	BL,80H			;Displaying char in XOR mode?
	JZ	TDH0			;No, display char as is
	JMP	TERMINAL_DISPLAY	;Yes - use general purpose routine
TDH0:	PUSHREG	<AX,BX,DX,SI,ES>	;Save registers
	MOV	DX,SCREEN_SEGMENT	;Point ES to the screen segment
	MOV	ES,DX
	MOV	DX,CURSOR_POSITION[0]	;Place the ROW in DH, COLUMN in DL
	MOV	BL,DH			;Get the current row number
	MOV	BH,0			;Extend it to a word offset
	SHL	BX,1
	MOV	BX,CS:GRAPHICS_LINE_OFFSET[BX] ;Pick up the offset to this line
	ADD	BX,PAGE_BASE[0]		;Add in address of start of page
	ADD	BL,DL			;Now point to the current character
	ADC	BH,0
	AND	BX,GRAPHICS_SCREEN_MASK	;Ensure that the ptr is valid
;
; Remove software cursor from its present location (if it is on)
;
	MOV	CURSOR_ACCESS,TRUE	;Flag that cursor is being updated
	CMP	CURSOR_FLAG,CURSOR_ENABLED OR CURSOR_ON ;Cursor on?
	JNE	TDH1			;No, cursor already off
	MOV	AH,CURSOR_SAVE		;Write old screen data at cursor posn
	MOV	ES:[BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS],AH
;
; Check for possible control character
;
TDH1:	CMP	AL,' '			;Might this be a control char?
	JAE	TDH2			;No, display the character
	JMP	TDH7			;Just might be - see if it really is!
;
; Process extended-font characters
;
TDH1A:	PUSH	AX			;Save the character to be displayed
	MOV	AL,USER_FONT_INTR	;Yes - get the user font vector #
	CALL	GET_INTR_PTR		;Point to the user's extended font
	POP	AX			;Restore the character
	AND	AL,7FH			;Mask out the extended-font sel bit
	JMP	SHORT TDH25		;Display the character
;
; Display the character using high-res graphics mode
;
TDH2:	PUSH	DS			;Save the data segment
	TEST	AL,80H			;Is this an extended character?
	JNZ	TDH1A			;Yes - use the extended font code
	LDS	SI,CS:FONT_PTR		;Point to the standard character font
TDH25:	MOV	AH,0			;Extend char to a word
	SHL	AX,1			;Mult char by size of font entry
	SHL	AX,1			;(x 4)
	SHL	AX,1			;(x 8)
	ADD	SI,AX			;Point to start of this char's font
	MOV	AX,BX			;Get the pointer to the character
	ADD	AX,SCAN_LINE_INCREMENT*3;Add in size of character
	TEST	AX,SCREEN_PARTITION_OFS	;Is this char going to wrap-around?
	JNZ	TDH6			;Yes - must check for each byte
	LODS	WORD PTR [SI]		;Get and display scan lines 0 and 1
	MOV	ES: [BX],AL
	MOV	ES: [BX+SCREEN_PARTITION_OFS],AH
	LODS	WORD PTR [SI]		;Display the 3rd and 4th scan lines
	MOV	ES: [BX+SCAN_LINE_INCREMENT*1],AL
	MOV	ES: [BX+SCAN_LINE_INCREMENT*1+SCREEN_PARTITION_OFS],AH
	LODS	WORD PTR [SI]		;Display the 5th and 6th scan lines
	MOV	ES: [BX+SCAN_LINE_INCREMENT*2],AL
	MOV	ES: [BX+SCAN_LINE_INCREMENT*2+SCREEN_PARTITION_OFS],AH
	LODS	WORD PTR [SI]		;Display the last two scan lines
	MOV	ES: [BX+SCAN_LINE_INCREMENT*3],AL
	MOV	ES: [BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS],AH
TDH3:	POP	DS			;Restore the ROM's data segment
	INC	BX			;Point to the next character
	INC	DL			;Increment column number
	CMP	DL,MAX_COLUMNS		;Beyond end of line?
	JAE	TDH11			;Yes - advance to next line
;
; Display the software cursor at the new character position, and update vars
;
TDH4:	AND	BX,GRAPHICS_SCREEN_MASK	;Keep cursor ptr in screen RAM
	CMP	CURSOR_FLAG,CURSOR_ENABLED OR CURSOR_ON ;Should cursor be on?
	JNE	TDH5			;Turn cursor on only if enabled
	MOV	AL,ES:[BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS]
	MOV	CURSOR_SAVE,AL		;Save screen data at cursor position
	MOV	BYTE PTR ES:[BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS],CURSOR
TDH5:	MOV	CURSOR_PTR,BX		;Write pointer to cursor
	MOV	CURSOR_POSITION[0],DX	;Update cursor position
	MOV	CURSOR_ACCESS,FALSE	;Flag that cursor is 'available'
	POPREG	<ES,SI,DX,BX,AX>
	RET
;
; Short jump to alternate display routine
;
TDH6:	JMP	SHORT TDH15
;
; Process possible control characters
;
TDH7:	CMP	AL,CR			;Is the char a carriage return?
	JE	TDH9			;Yes - go to start of line
	CMP	AL,LF			;Is this a line-feed?
	JE	TDH12			;Yes - advance to next line
	CMP	AL,BELL			;Is the character a BELL?
	JE	TDH10			;Yes - beep!
	CMP	AL,BS			;Is the char a backspace?
	JE	TDH8			;Yes - back up the cursor
	JMP	TDH2			;No, display character as usual
;
; Process Backspace
;
TDH8:	TEST	DL,DL			;At the start of the line already?
	JZ	TDH5			;Yes - return
	DEC	DL			;No, decrement the column number
	DEC	BX			;Point to the previous character
	JMP	SHORT TDH4		;Set cursor position and return
;
; Process Carriage Return
;
TDH9:	SUB	BL,DL			;Subtract horizontal offset...
	SBB	BH,0			;...to point to start of line
	MOV	DL,0			;Now, set column to 0
	JMP	SHORT TDH4		;Set cursor ptr to start of line
;
; Process Bell character (BEEP)
;
TDH10:	CALL	BEEP			;Beep!
	JMP	SHORT TDH4		;Update cursor and return
;
; Handle Automatic CR-LF at end of line
;
TDH11:	SUB	BL,DL			;Point back to start of line
	SBB	BH,0
	MOV	DL,0			;Set position to start of line
;
; Process Line Feed
;
TDH12:	CMP	DH,NUMBER_OF_ROWS-1	;On the last line already?
	JNE	TDH13			;No, continue
	CALL	FAST_SCROLL_UP_HRES	;Yes, scroll the display up a line
	CMP	SCROLL_MODE,0		;Did the display RAM advance?
	JE	TDH4			;No, continue with same screen addr
	JMP	SHORT TDH14		;Yes - advance to next line & return
TDH13:	INC	DH			;Go on to the next ROW
TDH14:	ADD	BX,SCAN_LINE_INCREMENT*4;Advance cursor ptr as well as posn
	JMP	SHORT TDH4		;Set cursor position and return
;
; Display character - called when a char will wrap-around the partition
;
TDH15:	PUSH	BX			;Save cursor pointer
	LODSW				;Get the first two scan lines
	MOV	ES:[BX],AL		;Write the even scan line
	MOV	ES:[BX+SCREEN_PARTITION_OFS],AH	;Write the odd scan line
	ADD	BX,SCAN_LINE_INCREMENT	;Go on to the next scan line
	AND	BX,GRAPHICS_SCREEN_MASK	;Keep the character pointer in range
	LODSW				;Display the second two scan lines
	MOV	ES:[BX],AL		;Write the even scan line
	MOV	ES:[BX+SCREEN_PARTITION_OFS],AH	;Write the odd scan line
	ADD	BX,SCAN_LINE_INCREMENT	;Go on to the next scan line
	AND	BX,GRAPHICS_SCREEN_MASK
	LODSW				;Get the first two scan lines
	MOV	ES:[BX],AL		;Write the even scan line
	MOV	ES:[BX+SCREEN_PARTITION_OFS],AH	;Write the odd scan line
	ADD	BX,SCAN_LINE_INCREMENT	;Go on to the next scan line
	AND	BX,GRAPHICS_SCREEN_MASK
	LODSW				;Get the first two scan lines
	MOV	ES:[BX],AL		;Write the even scan line
	MOV	ES:[BX+SCREEN_PARTITION_OFS],AH	;Write the odd scan line
	POP	BX			;Restore cursor pointer
	JMP	TDH3			;Continue processing
TERMINAL_DISPLAY_HRES ENDP



;**********************************************************************
; FAST_SCROLL_UP_HRES:
;
;	Fast_Scroll_Up_Hres scrolls the screen up a single line when 
; using the Terminal_Display_Hres routine.  If the 'jump' scroll mode 
; is active, then Fast_Scroll_Up will actually perform the scroll.  If 
; not, the normal Scroll_Up routine will be called.
;
; NOTE: Register AX is used, but NOT saved, for efficiency.
;**********************************************************************
FAST_SCROLL_UP_HRES PROC NEAR
	PUBLIC	FAST_SCROLL_UP_HRES
	PUSHREG	<CX,DI>
	CMP	SCROLL_MODE,1		;In jump scroll mode?
	JE	FSUH			;Yes - perform special fast scroll
	MOV	AL,1			;No, set scroll count to 1
	MOV	CX,(0 SHL 8) OR 0	;Set upper-left hand position
	PUSH	DX			;Save register DX
	MOV	DX,(24 SHL 8) OR 79	;Set lower-right hand position
	CALL	SCROLL_UP		;Scroll the screen up a line
	POP	DX			;Restore register DX
	JMP	FSUH3			;Return
FSUH:	MOV	DI,PAGE_BASE[0]		;Get the start address of the screen
	MOV	AX,DI			;Get the starting erase address
	ADD	AX,SCAN_LINE_INCREMENT*4;Add in number of bytes to erase
	CMP	AX,OFFSET END_OF_PART	;Will the erase wrap around?
	JBE	FSUH1			;No, continue
	MOV	CX,OFFSET END_OF_PART+1	;Yes - get the address of end of part
	SUB	CX,DI			;Subtract to get OK-to-clear count
	SHR	CX,1			;Make clear count a word count
	PUSH	CX			;Save this number on the stack
	PUSH	CX			;Save first pass clear count
	PUSH	DI			;Save pointer to area to clear
	XOR	AX,AX			;Set 0 for graphics fill character
	REP	STOSW			;Clear to end of partition
	POP	DI			;Restore pointer to area to clear
	ADD	DI,SCREEN_PARTITION_OFS	;Point to the odd scan lines now
	POP	CX			;Get clear count again
	REP	STOSW			;Clear to end of partition here, too
	POP	CX			;Restore original byte count
	SUB	CX,SCAN_LINE_INCREMENT*2;Determine how much isn't yet cleared
	NEG	CX
	PUSH	CX			;Save clear count
	MOV	DI,0			;Set pointer to start of partition
	REP	STOSW			;Clear rest of first partition
	MOV	DI,SCREEN_PARTITION_OFS	;Point to second partition
	POP	CX			;Get the clear count again
	REP	STOSW			;Clear rest of second partition
	JMP	SHORT FSUH2		;Whew!  Advance start address
FSUH1:	MOV	CX,SCAN_LINE_INCREMENT*2;Get length of line as word counter
	XOR	AX,AX			;Going to be clearing to 0
	REP	STOSW			;Erase even scan lines
	ADD	DI,SCREEN_PARTITION_OFS-SCAN_LINE_INCREMENT*4 ;Point to odd
	MOV	CX,SCAN_LINE_INCREMENT*2;Set CX to # of words in odd lines
	REP	STOSW			;Erase odd scan lines
FSUH2:	CALL	WAIT_VERTICAL_SYNC	;Wait till OK to update start address
	MOV	CX,PAGE_BASE[0]		;Get the start address of this page
	ADD	CX,SCAN_LINE_INCREMENT*4;Point to the next character line
	AND	CX,GRAPHICS_SCREEN_MASK ;Keep start addr in 1st partition
	MOV	PAGE_BASE[0],CX		;Write new page start address
	SHR	CX,1			;Generate a hardware address
	MOV	CRT_START_ADDRESS,CX	;Update start address for next pass
	MOV	DI,DX			;Save the cursor position in DI
	MOV	DX,COLOR_CARD + CRT_BASE_REGISTER ;Point DX to the CRT chip
	MOV	AL,CRT_START_PORT	;Point to the CRT start addr register
	OUT	DX,AL			;Select M.S. byte of start addr
	INC	DX			;Now, point to the parameter port
	MOV	AL,CH			;Get the M.S. start addr byte
	OUT	DX,AL			;Output it...
	DEC	DX			;Point back to the control port
	MOV	AL,CRT_START_PORT+1
	OUT	DX,AL			;Select L.S. byte of start address
	INC	DX			;Point to parameter port again
	MOV	AL,CL			;Output L.S. byte of address
	OUT	DX,AL
	MOV	DX,DI			;Restore cursor position
FSUH3:	POPREG	<DI,CX>
	RET
FAST_SCROLL_UP_HRES ENDP



;**********************************************************************
; UPDATE_CURSOR_HRES:
;
;	Update_Cursor_Hres is called to update the special
; 'software' cursor used for the high resolution video mode.
; If the routine is told to leave the cursor off, it will leave
; the cursor in protected mode.
;
; Input:
;	AH: Flag:  TRUE (Turn on new cursor), FALSE: (Leave cursor off)
;	DX: New cursor position (DH:row, and DL:column)
;**********************************************************************
UPDATE_CURSOR_HRES PROC NEAR
	PUBLIC	UPDATE_CURSOR_HRES
	PUSHREG	<AX,BX,ES>
	CMP	VIDEO_MODE,6		;Are we in high-res graphics mode?
	JNE	UCH3			;No, don't bother with the cursor!
	PUSH	DX			;Save the new cursor position
	MOV	DX,SCREEN_SEGMENT	;Point ES to the screen segment
	MOV	ES,DX
	MOV	DX,CURSOR_POSITION[0]	;Place the ROW in DH, COLUMN in DL
	MOV	BL,DH			;Get the existing row number
	MOV	BH,0			;Extend it to a word offset
	SHL	BX,1
	MOV	BX,CS:GRAPHICS_LINE_OFFSET[BX] ;Pick up the offset to this line
	ADD	BX,PAGE_BASE[0]		;Add in address of start of page
	ADD	BL,DL			;Now point to the old character
	ADC	BH,0
	AND	BX,GRAPHICS_SCREEN_MASK	;Ensure that the ptr is valid
	MOV	CURSOR_ACCESS,TRUE	;Flag that cursor is being updated
	CMP	CURSOR_FLAG,CURSOR_ENABLED OR CURSOR_ON ;Cursor on?
	JNE	UCH1			;No, can't erase it!
	MOV	AL,CURSOR_SAVE		;Write old screen data at cursor posn
	MOV	ES:[BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS],AL
UCH1:	POP	DX			;Restore the new cursor position
	TEST	AH,AH			;Are we to turn the cursor back on?
	JZ	UCH3			;No, leave the cursor off
	MOV	BL,DH			;Get the current row number
	MOV	BH,0			;Extend it to a word offset
	SHL	BX,1
	MOV	BX,CS:GRAPHICS_LINE_OFFSET[BX] ;Pick up the offset to this line
	ADD	BX,PAGE_BASE[0]		;Add in address of start of page
	ADD	BL,DL			;Now point to the new cursor posn
	ADC	BH,0
	AND	BX,GRAPHICS_SCREEN_MASK	;Ensure that the ptr is valid
	CMP	CURSOR_FLAG,CURSOR_ENABLED OR CURSOR_ON ;Should cursor be on?
	JNE	UCH2			;Turn cursor on only if enabled
	MOV	AL,ES:[BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS]
	MOV	CURSOR_SAVE,AL		;Save screen data at cursor position
	MOV	BYTE PTR ES:[BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS],CURSOR
UCH2:	MOV	CURSOR_PTR,BX		;Write pointer to new cursor position
	MOV	CURSOR_ACCESS,FALSE	;Flag that cursor is 'available'
UCH3:	POPREG	<ES,BX,AX>
	RET
UPDATE_CURSOR_HRES ENDP



