/*
    DETECT - This program will detect bad sectors on a winchester disk.
	It first loads in the bad sector table from the disk and
	verifies it has a good copy. bad sectors are added to this
	table if any are found else a new table is created.
    
    Author: RJM
    
    Date: 11/3/83

    Version 1.76 - Modify for 8 winchesters
    Version 1.77 - Use new bad sector table with interleave included.
    Version 1.90 - Update utilities to V1.90
    Version 2.00 - Update utilities to V2.00
    Version 2.01 - Put interleave in bst
    Version 2.02 - Turn off retries on detection pass.
    Version 2.03 - Allow sector per track information to come from
		   the controller.
    Version 2.04 - Correct problem with bad sector detection
    Version 2.05 - Take out 32 meg drive limit
*/

/*
		RESTRICTED RIGHTS LEGEND
		------------------------
	
	    "Use, duplication, or disclosure by the
	Government is subject to restrictions as set forth
	in paragraph (b) (3) (B) of the Rights in Technical
	Data and Computer Software clause in DAR
	7-104.9(a).  Contractor/manufacturer is Zenith
	Data Systems Corporation of Hilltop Road, St.
	Joseph, Michigan 49085.
*/

#include "stdio.h"
#include "conio.h"
#include "ctype.h"
#include "z150rom.h"
#include "fixed.h"
#include "bst.h"

#define	TRUE	1
#define	FALSE	0


/* Flags for different signons */
#define	ZENITH	TRUE
#define	NBI	FALSE
#define	SYNTREX	FALSE

/* Define the version and release numbers for DETECT */
#define VERSION 2
#define RELEASE 5

/* Define a bad sector table */
struct badsec_table bst;

#define CARRY	0x0001		/* Carry flag position in flag word */

#define DOSF_INSTR 0xa

unsigned user_cyl, bs_cyl;
char max_hds;
unsigned max_spt;

unsigned head, cyl, sector;

char disk_buf[SECTOR_SIZE], command, drive;

char buf1[2*SECTOR_SIZE];	/* Buffer for ensuring no DMA errors */
char *buf;

int bad_count;			/* Count of bad sectors in bst */
    
char bad_flag = FALSE;		/* Flag for bad sectors detected */

unsigned dpb_segment, dpb_offset;

struct
{
    struct dpb dpbt[4];
} dpb_table;

/* Define the externals used in the z150int() module */
extern char ral, rah, rbl, rbh, rcl, rch, rdl, rdh;
extern unsigned rax, rbx, rcx, rdx, rsi, rdi, rsp, rbp;
extern unsigned rcs, rds, res, rss, flags;
    
main()
{
    init();
    signon();			/* Greet the user */
    if (!setup()) exit(0);	/* Make sure the controller is there */
    bad_count = init_bst();	/* Read in/set up bad sector table */
    bad_sector();		/* Do bad sector detection */
    put_tables();		/* Write the default tables to the disk */
    exit(0);
}

init()
{
    res = mydsreg();		/* Set up ES register to this data seg. */
    get_buf();
    return;
}

/*
    Initialize the bad sector table to either the old one or to
    no bad sectors.
*/
init_bst()
{
    int i;
    long bst_sec;
    unsigned check_sum(), check;
    
    /* Attempt to get the old bad sector table */
    bst_sec = (long)bs_cyl * max_hds * max_spt;
    for (i = 0; i < 2; ++i)
    {
	set_sector(bst_sec);
	/* If the sector can be read, verify the check sum */
	rbx = &bst;
	if (get_sector())
	{
	    check = bst.check_sum;
	    bst.check_sum = 0;
	
	    /* If good check sum then return */
	    if (check == check_sum(mydsreg(), &bst, SECTOR_SIZE)) return(cbs());
	}
	/* Try next bad sector table */
	bst_sec += max_spt;
    }

    /* No bad sector table exists, make a new one */
    for (i = 0; i < MAX_BAD; ++i) 
    {
	bst.bad_secs[i].head = 0;
	bst.bad_secs[i].cyl = 0;
	bst.bad_secs[i].sec = 0;
    }
    bst.interleave = INTERLEAVE;
    bst.check_sum = 0;
    return(0);				/* No bad sectors */
}

/*
    cbs() - This routine will count the number of bad sectors
	in a bad sector table.
*/
cbs()
{
    int count;

    count = 0;
    while (bst.bad_secs[count].head != 0 ||
	    bst.bad_secs[count].cyl != 0 ||
	    bst.bad_secs[count].sec != 0)   ++count;
    return(count);
}


/*
    Greet the user and tell him about DETECT.
*/
signon()
{
    float a;
    a = VERSION + (float)RELEASE / 100;

#if ZENITH
    printf("               DETECT version %3.2f\r\n", a);
    puts("Copyright(C) 1984, Zenith Data Systems Corporation\r\n");
#endif

#if NBI
    printf("    DETECT version %3.2f\r\n", a);
    puts(  "(C)Copyright NBI, Inc. 1984\r\n");
#endif

#if SYNTREX
    printf("      DETECT version %3.2f\r\n", a);
    puts(  "(C)Copyright SYNTREX, Inc. 1984\r\n");
#endif

    puts("\r\nThe DETECT utility helps you to:\r\n\n");
    puts("    * Locate sectors that have failed since you last ran PREP\r\n");
    return;
}


/*
    get_buf() - This routine will prepare the "buf1" buffer and
	the "buf" pointer so it doesn't cross a 64K memory boundry.
*/
get_buf()
{
    unsigned address;
    
    buf = &buf1[0];
    address = (((unsigned)mydsreg()) << 4) + (unsigned)&buf1[0];	/* Get physical address of buffer */
    address += SECTOR_SIZE;		/* Bump address by SECTOR_SIZE */

    /* Modify buffer address if the current buffer crosses boundry */
    if (address < (unsigned) buf) buf += SECTOR_SIZE;
    return;
}


/*
    This routine will ensure that the user wants to DETECT the disk
    and that the controller actually exists.
*/
setup()
{
    char c, flag;
    /* Ask the user if he is sure */
    puts("\r\nDo you wish to proceed with DETECT (Y/N)?");
    if (!yes_no()) return(FALSE);
    
    /* Get the drive number */
    printf("\r\nWinchester drive unit number (0-%c): ", NUM_FIXED-1+'0');
    while(TRUE)
    {
	drive = getchar() - '0';
	if (drive >= NUM_FIXED) beep();
	else break;
    }
    printf("%c\r\n", drive+'0');

    /* Verify controller exists (get parameters for drive 0) */
    flag = controller(drive);
    if (!flag || drive >= rdl)
    {
	puts("\r\nCan not communicate with Winchester controller\r\n");
	exit(0);
    }
    
    
    /* Get the maximum number of heads */
    max_hds = rdh + 1;

    /* Get the maximum number of spt */
    max_spt = rcl & 0x3f;
    
    /* Get the last user cylinder and the bad sector table cylinder */
    user_cyl = rch + ((rcl & 0xc0) << 2);
    bs_cyl = user_cyl + 1;	  /* reserved cyl is one after last user */


    /* Reset the disk system */
    disk_reset();

    return(TRUE);		/* All OK */
}

/*
    put_tables() - This routine will put the bad sector tables to disk.
	It does a check sum on the bad sector table and then writes
	two copies to the disk.
*/
put_tables()
{
    unsigned check_sum(), sbcseg();
    struct p_ent *ptr;

    /* Check sum the bad sector table */
    bst.check_sum = check_sum(mydsreg(), &bst, SECTOR_SIZE);
    
    /* Write bad sector table A */
    rbx = &bst;
    set_sector((long)bs_cyl*max_hds*max_spt);
    put_sector();
    
    /* Write bad sector table B */
    rbx = &bst;
    set_sector((long)bs_cyl*max_hds*max_spt + max_spt);
    put_sector();
    
    return;
}

beep()
{
    putchar(7);
    return;
}

yes_no()
{
    char c1;
    
    c1 = getchar();
    if (islower(c1)) c1 = toupper(c1);
    putchar(c1);
    puts("\r\n");
    return(c1 == 'Y');
}

/*
    bad_sector() - This routine will prompt the user for bad sectors
	and mark them in the table. Then a single pass is made across
	the entire disk to locate other bad sectors.
*/
bad_sector()
{
    unsigned get_hex();
    long lsec;

    puts("\r\n");
    do
    {
	puts("Enter bad sector address, or zero to end:");
	lsec = (long)((unsigned)get_hex());
	if (lsec != 0L)
	{
	    set_sector(lsec);
	    if (!flag_sector()) return;
	}
    }
    while (lsec != 0L);

    puts("Beginning verification...");
    retry_off();
    get_pattern();
    retry_on();
    puts("Completed\r\n");
    if (bad_flag) puts("Bad sectors located. Tables modified.\r\n");
    else puts("No bad sectors located\r\n");
    return;
}

/*
	get_pattern() - This routine will read every sector on the disk.
	    If the sector can not be read it is flagged as a bad sector.
*/
get_pattern()
{
    set_sector(0L);
    do
    {
	if (!get_sector())
	{
	    if (!flag_sector()) return;
	}
    }
    while (next_sector());

    return;
}

/*
	This routine will set the variables head, track and sector
    to the next sector value. If the sector exists then TRUE is
    returned, if the end of the disk is hit then FALSE is returned.
*/
next_sector()
{
    if (++sector > max_spt)
    {
	sector = 1;
	if (++head == max_hds)
	{
	    head = 0;
	    if (++cyl == bs_cyl) return(FALSE);
	}
    }
    return(TRUE);
}

/*
	This routine will set a logical sector into the actual
    sector, track and head.
*/
set_sector(sec)
long sec;
{
    sector = sec % max_spt + 1;		/* Get sector (based at 1) */
    sec  = sec / max_spt;		/* Update sec, get number of tracks */
    head = sec % max_hds;		/* Get head */
    cyl  = sec / max_hds;	/* Get cylinder */
    return;
}

/*
    Flag a sector as bad in the bad sector table. If more than MAX_BAD
    sectors are found FALSE is returned. If the sector was sucessfully 
    entered (or if teh sector was already entered) TRUE is returned.
*/
flag_sector()
{
    int i, j;
    long logical_sector();
    struct hcs temp_hcs;
 
    bad_flag = TRUE;
    encode();
    temp_hcs.head = rdh;
    temp_hcs.cyl = rch;
    temp_hcs.sec = rcl;
    
    
    /* See if the sector is already present */
    for (i = 0; i < bad_count; ++i)
    {
	if (logical_sector(&temp_hcs) == logical_sector(&bst.bad_secs[i])) return(TRUE);
	if (logical_sector(&temp_hcs) < logical_sector(&bst.bad_secs[i])) break;
    }
    
    /* Have found bad sector, see if over the limit */
    if (bad_count == (MAX_BAD-1))
    {
	puts("\r\nBad sector count exceeded for this drive.\r\n\n");
	return(FALSE);
    }
    ++bad_count;
    
    /* Shift the other bad sectors down to enter this one in order */
    for (j = bad_count; j > i; --j)
    {
	bst.bad_secs[j].head = bst.bad_secs[j-1].head;
	bst.bad_secs[j].cyl = bst.bad_secs[j-1].cyl;
	bst.bad_secs[j].sec = bst.bad_secs[j-1].sec;
    }
    
    /* Enter the new bad sector */
    bst.bad_secs[i].head = temp_hcs.head;
    bst.bad_secs[i].cyl = temp_hcs.cyl;
    bst.bad_secs[i].sec = temp_hcs.sec;
    
    return(TRUE);
}


/*
    logical_sector() - this routine will take the head cylinder and
	sector passed to it and return the logical sector it corresponds
	to.
*/
long logical_sector(p)
struct hcs *p;
{
    long tmp;
    
    tmp = p->head + (p->cyl + (((unsigned)p->sec & 0xc0) << 2)) * max_hds;
    return((long) (tmp * max_spt + (p->sec & 0x3f) - 1L));
}


/*
    retry_on() - This routine will turn on controller retries of
	disk I/O
*/
retry_on()
{
    putword(0, (DPB_POINTER << 2) + 2, dpb_segment);
    putword(0, DPB_POINTER << 2, dpb_offset);
    rdl = 0x80 + drive;
    rah = DIO_SETPARMS;
    z150int(DISK_IO_INTR);
    return;
}

    
/*
    retry_off() - This routine will turn off controller retries of
	disk I/O
*/
retry_off()
{
    int i;

    dpb_segment = getword(0, (DPB_POINTER << 2) + 2);
    dpb_offset = getword(0, DPB_POINTER << 2);
    movbyte(dpb_segment, dpb_offset, mydsreg(), &dpb_table, sizeof(dpb_table));
    for (i = 0; i < 4; ++i) dpb_table.dpbt[i].control |= 0xc0;
    putword(0, (DPB_POINTER << 2) + 2, mydsreg());
    putword(0, DPB_POINTER << 2, &dpb_table);
    rdl = 0x80 + drive;
    rah = DIO_SETPARMS;
    z150int(DISK_IO_INTR);
    return;
}



/*
    This routine will return a hex number between 0 and FFFF.
*/
unsigned get_hex()
{
#define BUF_LEN 10		/* buffer for input */
    char buf[BUF_LEN], c;
    unsigned val, index;
    unsigned max_sec;
    
    max_sec = bs_cyl * max_hds * max_spt;
    
    buf[0] = BUF_LEN - 2;
    
    while(TRUE)
    {
	/* Get the user string */
	bdos(DOSF_INSTR, &buf[0]);
    
	val = 0;
	index = 2;		/* Start processing at the second byte */
	while(TRUE)
	{
	    c = buf[index++];
	    if (c == '\r') break;		/* if Return then done */
	    if (islower(c)) c = toupper(c);	/* Map to upper */
	
	    /* check for valid hex digit */
	    if ((c >= '0' && c <= '9') || (c >= 'A' && c <= 'F'))
	    {
		if (c < 'A') c -= '0';	/* Bias character to get digit */
		else c -= 'A' - 10;

		if (val & 0xf000)	/* Ensure number is not too large */
		{
		    beep();
		    puts("\r\nInvalid HEX value, Try again: ");
		    break;
		}
		else val = (val << 4) + c;	/* Update value */
	    }
	    else	/* Invalid hex digit */
	    {
		beep();
		puts("\r\nInvalid HEX value, Try again: ");
		break;
	    }
	}
	if (val >= max_sec)
	{
	    beep();
	    puts("\r\nInvalid HEX value, Try again: ");
	}
	else if (c == '\r') break;		/* If return then done */
    }
    puts("\r\n");
    return(val);
}

unsigned check_sum(seg, off, len)
unsigned seg, off, len;
{
    unsigned sum;
    
    sum = 0;
    do
    {
	sum += getbyte(seg, off++);
    }
    while(--len > 0);
    return(~sum);
}

controller()
{
    rah = DIO_GETPARMS;
    rdl = drive + 0x80;
    z150int(DISK_IO_INTR);
    return(!(flags & CARRY));
}

encode()
{
    rcl = ((cyl & 0x0300) >> 2) | sector;
    rch = cyl & 0x00ff; 
    rdh = head;
    return;
}


/*
    disk_reset() - This routine will reset the disk system.
*/
disk_reset()
{
    char temp;
    
    rah = DIO_RESET;
    temp = rdl;
    rdl = 0x80;
    z150int(DISK_IO_INTR);
    rdl = temp;
    return(!(flags & CARRY));
}

/*
    get_sector() - This routine will read a sector from the winchester
	disk. It assumes all other parameters except rax have been set
	previously. (i.e. the externals head, sector, cyl and transfer
	address (res:rbx) are set). If the operation is sucessful TRUE
	is returned else FALSE is returned.
*/
get_sector()
{
    unsigned temp;
    int i;
    char flg;
    
    encode();			/* Set the head, sector and cylinder */
    temp = rbx;
    rbx = buf;
    rdl = 0x80+drive;
    for (i = 0; i < NUM_RETRY; ++i)
    {
	rah = DIO_READ;			/* Read function (restore in case of reset)*/
	ral = 1;			/* do 1 sector */
	z150int(DISK_IO_INTR);
	flg = !(flags & CARRY);
	if (flg) break;
	disk_reset();
    }
    if (flg)
    {
	movbyte(mydsreg(), buf, mydsreg(), temp, SECTOR_SIZE);
    }
    return(flg);	/* Return status */
}

put_sector()
{
    int i;
    char flg;

    movbyte(mydsreg(), rbx, mydsreg(), buf, SECTOR_SIZE);
    
    encode();
    rbx = buf;
    rdl = 0x80 + drive;
    for (i = 0; i < NUM_RETRY; ++i)
    {
	rah = DIO_WRITE;	/* Restore these in case of reset */
	ral = 1;
	z150int(DISK_IO_INTR);
	flg = !(flags & CARRY);
	if (flg) break;
	disk_reset();
    }
    return(flg);
}
