page
;----------------------------------------------------------------------
Scroll Proc Near; (17_Jul_85)
;----------------------------
;	1) Scroll is called to scroll the screen upwards or downwards
;	   (upwards means that the existing screen lines are moved up).
;	2) Arbitrary windows may be scrolled
;	3) Note that Scroll actually contains two entry points:
;		Scroll_Up
;		Scroll_Down.
;	4) Entry Register Values:
;		AL -> Number of lines to scroll (usually 1).  If the scroll
;		      count is zero, then the entire window is cleared.
;		CX -> Row and column of the upper-left hand window corner.
;		DX -> Row/column of the lower-right hand window corner.
;		BH -> Contains the attribute byte to be used on blank lines.
;
Public	Scroll_Up

Scroll_Up:
	push	ax
	mov	ah,False		;Set upwards-scroll flag
	jmp	Short SC0

Scroll_Down:
	push	ax
	mov	ah,True			;Set downwards-scroll flag

SC0:	PushReg	<bx,cx,dx,si,di,es>

	push	bx
	push	ax
	push	dx
	mov	al,Displayed_Page	;AL:= New display page
	cmp	al,Old_Displayed_Page	;If (Display page has NOT been poked)
	jne	SC05

	mov	ax,Crt_Start_Address	;	AX:= current CRT start address
	shl	ax,1			;	Make it a software address
	mov	dx,0			;	Convert to double-word for div.

	div	Page_Size		;	AX:= Active Page


;*** START OF PAGING CHANGES

	extrn	page_map:byte

	push	bx
	mov	bl,al
	mov	bh,0
	mov	al,page_map[bx]
	pop	bx

;*** END OF PAGING CHANGES





SC05:	mov	bh,al			;BH:= Current Page
	call	Set_IO_Page		;Set this page to be scrolled
	pop	dx
	pop	ax
	pop	bx

	call	Set_Scroll_Addresses	;Set source and dest scroll addresses
	test	al,al			;If (NOT scrolling the whole screen)
	je	SC4

	call	Software_Scroll		;	Scroll screen by moving text
SC4:	test	al,al			;If (scrolling the whole screen)
	jne	SC5

	mov	al,ch			;	AL:= Line Count
SC5:	mov	ch,0			;CX:= Character count

SC6:	push	ax			;Repeat
	mov	al,False		;	Set erase-character-line mode
	call	Clear_Line		;	Erase a line of text
	pop	ax

	mov	dx,Row_Increment	;	DX:= increment to next line
	call	Increment_Position	;	Point to the next line
	dec	al			;	Decrement Line Counter
	jne	SC6			;Until (all lines cleared)

SC7:	PopReg	<es,di,si,dx,cx,bx>	;Restore registers
	pop	ax
	ret				;Return

Scroll	EndP

page
;----------------------------------------------------------------------
Set_Scroll_Addresses Proc Near
;-----------------------------
;	1) Set_Scroll_Addresses is called when 'software scrolling' is to
;	   be performed.  It translates a pair of window coordinates and a
;	   scroll count into source and destination addresses for text moves.
;	2) Entry Register Values:
;		AH -> Scroll direction flag.  If zero, then upwards
;		      scrolling is assumed.  Otherwise, downwards scroll
;		      addresses are generated.
;		AL -> Number of lines to scroll (usually 1).
;		CX -> Row/column of the upper-left  hand window corner.
;		DX -> Row/column of the lower-right hand window corner.
;	3) Exit Register Values:
;	     ES:SI -> Pointing to the 'source' address for screen moves
;	     ES:DI -> Pointing to the 'dest' address for screen moves
;		CH -> Number of lines to be scrolled
;		CL -> Number of characters per scroll line
;
Public	Set_Scroll_Addresses

	push	ax
	push	ax			;Save the scroll count (ie, distance)
	test	ah,ah			;If (Scrolling Up)
	jne	SA1

	call	Set_Screen_Ptr		;	ES:DI:= Starting Address
	mov	si,di			;	SI:= Destination Address
	jmp	SA2

SA1:	push	cx			;Else
	mov	ch,dh			;	Get starting character position
	call	Set_Screen_Ptr		;	ES:DI:= Starting Address
	mov	si,di			;	SI:= Destination Address
	pop	cx

SA2:	test	al,al			;If (scroll count ne 0)
	jz	SA6

SA3:	test	ah,ah			;	Repeat	If (Scrolling Up)
	jne	SA4

	add	si,Row_Increment	;			SI:= Next Line
	cmp	si,End_of_Page		;			If (SI gt End)
	jbe	SA5

	sub	si,Page_Size 		;				Wrap
	jmp	SA5

SA4:	sub	si,Row_Increment	;		Else 	SI:= Last Line
	cmp	si,Start_of_Page	;			If (SI lt Start)
	jge	SA5

	add	si,Page_Size		;				Wrap
SA5:	dec	al			;		Dec Scroll Counter
	jne	SA3			;	Until (Scroll Count eq 0)

SA6:	sub	cx,dx
	neg	cx			;CX:= Number of Rows/Columns to Scroll
	add	cx,(1 Shl 8) or 1	;(Adjust counts for zero offset)
	pop	ax

	sub	ch,al			;CH:= Scroll Row Count - Lines Cleared
	pop	ax			;Restore registers
	ret

Set_Scroll_Addresses EndP

page
;**********************************************************************
; SOFTWARE_SCROLL: (DIRECTION, SOURCE, DEST, LINE_COUNT, SCROLL_COUNT)
;
;	Software_Scroll performs a scroll of a specified segment
; of the screen by moving bytes around on the screen.
;
; Input:
;	AH: Scroll direction.  If AH is 0, upwards scrolling is
;	    used - otherwise the screen is scrolled downwards.
;	CH: Number of lines to be scrolled
;	CL: Number of characters per scroll line
;    ES:SI: Pointing to the 'source' address for screen moves
;    ES:DI: Pointing to the 'dest' address for screen moves
;
; Output:
;    ES:DI: Pointing to first line to be cleared
;**********************************************************************
SOFTWARE_SCROLL PROC NEAR
PUBLIC	SOFTWARE_SCROLL

	PUSHREG	<AX,CX,DX,SI,BP>	;Save registers
SWS1:	MOV	BP,1			;Assume one line of bytes / char
	CMP	VIDEO_MODE,3		;In graphics mode?
	JBE	SWS2			;No

	CMP	VIDEO_MODE,7		;In the monochrome text mode?
	JE	SWS2			;Yes - chars only one byte-line tall

	MOV	BP,FONT_HEIGHT		;No, have FONT_HEIGHT lines / char
SWS2:	PUSH	SI			;Save source pointer
	PUSH	DI			;Save dest pointer
	PUSH	CX			;Save line count
	MOV	CH,0			;Extend char count to a word
	CMP	COLUMN_INCREMENT,1	;Are attribute bytes involved?
	JE	SWS3			;No, have correct byte count

	SHL	CX,1			;Yes - include attributes in count
SWS3:	CALL	MOVE_SCREEN_BYTES	;Move a line of bytes (maybe chars)
	DEC	BP			;Moved all of the character yet?
	JE	SWS4			;Yes - go on to next line

	XOR	SI,SCREEN_PARTITION_OFS	;No, point to the next scan line
	XOR	DI,SCREEN_PARTITION_OFS
	TEST	BP,1			;Did we just move an odd scan line?
	JNE	SWS3			;No, already pointing correctly

	PUSH	AX			;Save direction flag
	MOV	AH,0			;Force pointers to next scan line
	MOV	DX,SCAN_LINE_INCREMENT	;Want to inc ptrs to next scan line
	CALL	INCREMENT_POSITION	;Point src & dest to next scan line
	POP	AX			;Restore direction flag

	JMP	SHORT SWS3		;Continue till done w/character
SWS4:	POP	CX			;Restore line count
	POP	DI			;Restore dest pointer
	POP	SI			;Restore source pointer
	MOV	DX,ROW_INCREMENT	;Get offset to next character line
	CALL	INCREMENT_POSITION	;Point to start of next char line
	DEC	CH			;Moved all lines yet?
	JNZ	SWS1			;No, loop until all moved

	POPREG	<BP,SI,DX,CX,AX>	;Yes - restore registers
	RET

SOFTWARE_SCROLL ENDP

page
;**********************************************************************
; MOVE_SCREEN_BYTES: (SOURCE, DEST, COUNT)
;
;	Move_Screen_Bytes is used to move a byte string from one screen
; location to another.  It checks to see whether or not the move will
; wrap around at the end of the screen - if so, the move is split into
; two phases, so that the information is transferred correctly.
; 	Note that it is possible for either the source or dest to wrap
; around - it is impossible for BOTH to wrap around simultaneously!
;
; Input:
;	ES: Screen segment
;	SI: Pointer to source string
;	DI: Pointer to destination string
;	CX: Byte count
;**********************************************************************
MOVE_SCREEN_BYTES PROC NEAR
PUBLIC	MOVE_SCREEN_BYTES

	PUSHREG	<AX,BX,CX,DX,SI,DI,BP>
	MOV	BP,OFFSET END_OF_SCREEN	;Assume alpha mode for end-screen
	CMP	VIDEO_MODE,3		;In alphanumeric mode?
	JBE	MSB1			;Yes - end-screen addr set correctly

	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	MSB1			;Yes - screen end all set

	CMP	SI,OFFSET END_OF_PART	;Graphics - in the first partition?
	JA	MSB1			;No, end-of-screen correct

	MOV	BP,OFFSET END_OF_PART	;Partition 1 - point to end-screen
MSB1:	MOV	DH,0			;Clear the wrap-around flag
	MOV	AX,SI			;Get the source pointer
	ADD	AX,CX			;Add in length of string to move
	CMP	AX,BP			;Going to wrap around on source?
	JBE	MSB2			;No, continue

	MOV	BX,CX			;Yes - save original byte count
	MOV	AX,BP			;Point to the end of screen
	SUB	AX,SI			;Determine how many can be moved
	INC	AX			;Include last byte in count
	MOV	CX,AX			;Save as new pass-one count
	SUB	BX,CX			;Set BX to remaining bytes to move
	MOV	DH,1			;Set the wrap-around flag to src
	JMP	SHORT MSB3

MSB2:	MOV	AX,DI			;Get the dest pointer
	ADD	AX,CX			;Add in number of bytes to move
	CMP	AX,BP			;Going to wrap around on dest?
	JBE	MSB3			;No, continue

	MOV	BX,CX			;Yes - save original byte count
	MOV	AX,BP			;Point to end of screen
	SUB	AX,DI			;See how many can be moved first pass
	INC	AX			;Include last byte in count
	MOV	CX,AX			;Save as pass-one count
	SUB	BX,CX			;Determine pass two count
	MOV	DH,2			;Set the wrapped-around flag for dest

MSB3:	PUSH	DS			;Save the data segment
	PUSH	ES			;Copy screen segment
	POP	DS
	SHR	CX,1			;Make byte count a word count
	JNC	MSB4			;Moving an even number of bytes
	MOVSB				;Moving odd count - move the odd byte
MSB4:	REP	MOVSW			;Copy a passes worth of words
MSB5:	POP	DS			;Restore data segment
	TEST	DH,DH			;Wrapped around?
	JZ	MSB8			;No, done!

	MOV	CX,BX			;Yes - get the pass two byte count
	MOV	AX,OFFSET SCREEN	;Assume start of screen
	CMP	VIDEO_MODE,3		;Are we in graphics mode?
	JBE	MSB6			;No, continue

	CMP	VIDEO_MODE,7		;In the monochrome mode?
	JE	MSB6			;Yes - try for second pass

	CMP	BP,OFFSET END_OF_SCREEN	;Graphics - in second partition?
	JB	MSB6			;No, continue

	MOV	AX,OFFSET END_OF_PART+1	;Set base addr of partition 2
MSB6:	CMP	DH,1			;Did the source wrap around?
	JNE	MSB7			;No, it was the dest ptr

	MOV	SI,AX			;Yes - point src to start of screen
	JMP	SHORT MSB1		;Go for next pass

MSB7:	MOV	DI,AX			;Wrap dest around to start of screen
	JMP	SHORT MSB1		;Copy the next chunk of data!

MSB8:	POPREG	<BP,DI,SI,DX,CX,BX,AX>
	RET

MOVE_SCREEN_BYTES ENDP

