page
;**********************************************************************
; TERMINAL_DISPLAY: (CHAR, FOREGROUND_COLOR, PAGE)
;
;	Terminal_Display emulates a 'dumb' terminal for output.
; It recognizes backspace, carriage-return, line-feed, and
; bell.  Unlike the other display routines, it also advances
; the cursor automatically - it will wrap-around at the end
; of a line, and scroll if a line-feed is received on the last
; line (or if a character is displayed at the last character
; position of the last line).
;	When the screen is scrolled, the Terminal_Display
; routine must know what attribute to use on the newly
; blanked scroll line;  in graphics mode, color 0 is used,
; but in alphanumeric mode, the attribute used is taken from
; the character pointed to by the cursor just prior to the
; scroll.
;
; Input:
;	AL: Character to display in Ascii.  As mentioned
;	    above, BS, CR, LF & BELL are handled as special
;	    cases, and are not displayed, but rather, are
;	    processed as commands.
;	BL: Foreground color (in graphics modes only)
;**********************************************************************
TERMINAL_DISPLAY PROC NEAR
PUBLIC	TERMINAL_DISPLAY

	PUSHREG	<AX,BX,CX,DX,SI>
	MOV	BH,DISPLAYED_PAGE	;Get the currently active page

	PUSH	BX			;Save the character attribute
	MOV	AH,3			;Get code to read cursor position
	INT	VIDEO_IO_INTR		;Read the cursor's position
	POP	BX			;Restore page number and attribute

	CMP	AL,BELL			;Was the character a bell?
	JNE	TD1

	CALL	BEEP			;Yes - ring the bell
	JMP	SHORT TD8

TD1:	CMP	AL,BS			;Was it a Backspace?
	JNE	TD2

	TEST	DL,DL			;Yes - at start of line already?
	JE	TD8			;At start of line - don't backspace

	DEC	DL			;Backspace to the previous position
	JMP	SHORT TD7		;Set the cursor position

TD2:	CMP	AL,CR			;Was it a carriage return?
	JNE	TD3

	MOV	DL,0			;Yes - set start of line position
	JMP	SHORT TD7

TD3:	CMP	AL,LF			;Was it a line-feed?
	JE	TD4			;Yes - go to next line/scroll

	MOV	CX,1			;Set repeat count parameter to 1
	MOV	AH,10			;Get code to disp char w/o attribute
	INT	VIDEO_IO_INTR		;Display the character on the screen

	INC	DL			;Increment horizontal position
	CMP	DL,BYTE PTR NUMBER_OF_COLUMNS	;At end of line?
	JNE	TD7			;No, set cursor position and return

	MOV	DL,0			;Yes - go to the start of the next line
TD4:	CMP	DH,LAST_ROW		;At end of screen already?
	JE	TD5			;Yes - scroll up a line

	INC	DH			;No, go to next line
	JMP	SHORT TD7		;Set the cursor position

TD5:	CMP	VIDEO_MODE,7		;In monochrome text mode?
	JE	TD55			;Yes - read old screen attribute

	CMP	VIDEO_MODE,3		;Are we in graphics mode?
	JA	TD6			;Yes - have attribute all set

TD55:	MOV	AH,2			;Get code to set the cursor position
	INT	VIDEO_IO_INTR		;Set the cursor position

	MOV	AH,8			;Get code to read character + attribute
	INT	VIDEO_IO_INTR		;Read the char + attribute from the screen

	MOV	BL,AH			;Get the attribute for the scroll
TD6:	MOV	SI,DX			;Save the cursor position in SI
	MOV	AL,1			;Scroll up one line
	MOV	CX,(0 SHL 8) OR 0	;Set upper-left hand scroll position
	MOV	DH,LAST_ROW		;Set lower-right hand scroll position
	MOV	DL,BYTE PTR NUMBER_OF_COLUMNS
	DEC	DL
	XCHG	BH,BL			;Place attribute byte in BH
	MOV	AH,6			;Get code to scroll page up
	INT	VIDEO_IO_INTR		;Scroll the screen upwards!

	XCHG	BL,BH			;Restore attrib and page number
	MOV	DX,SI			;Restore the cursor position
TD7:	MOV	AH,2			;Get code to set cursor posn again
	INT	VIDEO_IO_INTR		;Set the new cursor position

TD8:	POPREG	<SI,DX,CX,BX,AX>	;Restore registers
	RET

TERMINAL_DISPLAY ENDP

page
;**********************************************************************
; DISP_CHAR_8025: (CHAR, PAGE, ATTRIB_FLAG, ATTRIB, REPEAT)
; DISP_CHAR_MONO: (CHAR, PAGE, ATTRIB_FLAG, ATTRIB, REPEAT)
;
;	Display_Character is a special fast character display 
; routine used when in 80 x 25 display mode on the color display
; adapter card.  The character may optionally be displayed 
; more than once, and attributes are set for the new character.
;
;	Disp_Char_Mono is an alternate entry point used when the
; monochrome card is the currently selected video board.
;
; Input:
;	AL: Character to display in ASCII.
;	BH: set to the page number to display on.
;	BL: Attribute byte (displayed if DH is true)
;	AH: Flag to modify attribute byte (True/False).
;	CX: is a repeat count - it determines how many
;	    times the character should be displayed.
;
; NOTE: This routine is JMPed to by Display_Char with the contents
; 	of AX and DX pushed on to the stack!!! (See below)
;**********************************************************************
PUBLIC	DISP_CHAR_MONO
DISP_CHAR_MONO:

	PUSHREG	<SI>
	MOV	SI,8 * 2		;Treat monochrome card as page 8
	MOV	DX,CURSOR_POSITION[0]	;...but use page 0's cursor position
	JMP	SHORT DC80		;Display char on the monochrome card

DISP_CHAR_8025 PROC NEAR
PUBLIC	DISP_CHAR_8025

	PUSHREG	<SI>
	MOV	DL,BH			;Get the page in DX as a word
	MOV	DH,0
	MOV	SI,DX			;Place it in SI
	SHL	SI,1			;Set up for word accesses
	MOV	DX,CURSOR_POSITION[SI]	;Place the ROW in DH, COLUMN in DL
DC80:	PUSHREG	<BX,CX,DI,ES>		;Save rest of user registers

	XCHG	BL,AH			;Place attribute in AH, disp flag in BL
	MOV	DI,AX			;Save attribute and character
	MOV	AL,DH			;Get the current row number
	MOV	BH,DL			;Save the horizontal position in BH
	XOR	AH,AH			;Extend row number to a word
	MUL	ROW_INCREMENT		;Point to the right line
	ADD	AX,PAGE_BASE[SI]	;Add in address of start of page
	SHL	BH,1			;Mult horiz offset by 2 for chars & attrs
	ADD	AL,BH			;Now point to the current character
	ADC	AH,0
	XCHG	AX,DI			;Get pointer in DI, char in AH, attrib in AL
	MOV	ES,VIDEO_SEGMENT	;Point ES to the screen segment
DC81:	AND	DI,TEXT_SCREEN_MASK	;Keep the pointer in range!
	ADD	DI,PAGE_START[SI]	;Point to the right page in memory
	STOSB				;Write a character on the screen
	TEST	BL,BL			;Are attributes being displayed?
	JZ	DC82			;No, continue

	MOV	BYTE PTR ES:[DI],AH	;Yes - write the attribute byte
DC82:	DEC	CX			;Displayed enough of this character?
	JZ	DC83			;Yes - return

	INC	DI			;No, advance to the next character
	JMP	SHORT DC81		;Loop until enough chars displayed

DC83:	POPREG	<ES,DI,CX,BX,SI>	;Restore the user's registers
	POPREG	<DX,AX>			;Restore the user's AX register
	RET

DISP_CHAR_8025 ENDP

page
;**********************************************************************
; DISPLAY_CHARACTER: (CHAR, PAGE, ATTRIB_BYTE, REPEAT)
;
;	Display_Character displays a character on the screen.  The
; character may optionally be displayed more than once, and
; attributes may optionally be set. This routine takes care of
; both graphics mode, and alphanumeric mode displays.
;	Note that Display_Character will actually be called as two
; different routines:  Disp_Char and Disp_Char_Attr.  Disp_Char
; will write the character only to the display - existing
; attributes are unchanged.  Disp_Char_Attr, on the other
; hand, uses the attribute byte in BL.
;
; Input:
;	AL: Character to display in ASCII.
;	BH: set to the page number to display on.
;	BL: Attribute byte (or color number from 0-3 for
;	    medium-resolution graphics mode).  If the
;	    high-order bit of BL is set and the display is in
;	    graphics mode, then the character to be displayed 
;	    will be XORed with the existing screen contents, 
;	    regardless of whether or not attributes are being 
;	    displayed.
;	CX: is a repeat count - it determines how many
;	    times the character should be displayed.
;	    Note that in graphics mode, wrap-around
;	    at the end of a line may not work correctly.
;	Global variables CURRENT_PAGE, DISPLAYED_PAGE,
;	    CURSOR_POSITION.
;**********************************************************************
DISPLAY_CHAR PROC NEAR
PUBLIC	DISP_CHAR
PUBLIC	DISP_CHAR_ATTR

DISP_CHAR:
	PUSH	AX			;Save register AX
	MOV	AH,FALSE		;Set character-only flag
	JMP	SHORT DCH

DISP_CHAR_ATTR:
	PUSH	AX			;Save register AX
	MOV	AH,TRUE			;Set char + attributes flag

DCH:	PUSH	DX			;Save the DX register
	CMP	VIDEO_MODE,7		;In monochrome alpha mode?
	JE	DISP_CHAR_MONO		;Yes - use fast monochrome routine

	CMP	VIDEO_MODE,2		;Is this 80x25 color mode?
	JE	DISP_CHAR_8025		;Yes - use fast output routine

	CMP	VIDEO_MODE,3		;Is this 80x25 black and white?
	JE	DISP_CHAR_8025		;Yes - use the special routine

	PUSHREG	<BX,CX,SI,DI,BP,ES>	;No, save the user's registers
	MOV	DH,AH			;Save display flag in AH
	CALL	SET_IO_PAGE		;Do I/O on the right page!
	CALL	SET_SCREEN_ADDRESS	;Point ES:DI to the right screen loc
	CMP	VIDEO_MODE,7		;In monochrome text mode?
	JE	DCH1			;Yes - display char as is

	CMP	VIDEO_MODE,3		;Are we in color alpha mode?
	JG	DCH4			;No, perform graphics routine
;
; Display characters in alphanumeric mode
;
DCH1:	MOV	ES: SCREEN[DI],AL	;Display the character
	TEST	DH,DH			;Are we to display attributes?
	JE	DCH2			;No, continue

	MOV	ES: SCREEN[DI+1],BL	;Yes - place attribute on screen
DCH2:	DEC	CX			;Displayed enough chars yet?
	JCXZ	DCH6			;Return if done

	ADD	DI,COLUMN_INCREMENT	;Point to the next character
	CMP	DI,END_OF_PAGE		;Reached end of screen yet?
	JBE	DCH3			;No, continue

	SUB	DI,PAGE_SIZE		;Yes - wrap-around to rest of screen
DCH3:	JMP	DCH1			;Loop until enough chars displayed
;
; Display characters in graphics mode
;
DCH4:	MOV	BP,COLUMN_INCREMENT	;Store col. incr. as parameter
	MOV	BH,VIDEO_MODE		;Place video mode in BH

	PUSH	DS			;Save pointer to data segment
	MOV	SI,MONITOR_SEGMENT	;Point DS to the font in ROM
	MOV	DS,SI
	MOV	SI,OFFSET FONT		;Assume that the ROM font is used
	TEST	AL,80H			;Using the user-defined font?
	JE	DCH5			;No, continue using ROM font

	PUSH	AX			;Yes - save the character
	MOV	AL,USER_FONT_INTR	;Get the user font ptr intr #
	CALL	GET_INTR_PTR		;Point DS:SI to the user font
	POP	AX			;Restore the char to be displayed

	AND	AL,7FH			;Mask out user-defined-font flag
DCH5:	CBW				;Convert the char to a word
	SHL	AX,1			;Multiply char * font size
	SHL	AX,1
	SHL	AX,1
	ADD	SI,AX			;Point to this char's font
	CALL	DISPLAY_GRAPHICS_CHAR	;Display the character in graphics mode
	POP	DS			;Restore data segment

DCH6:	POPREG	<ES,BP,DI,SI,CX,BX>
	POPREG	<DX,AX>
	RET

DISPLAY_CHAR ENDP

page
;**********************************************************************
; DISPLAY_GRAPHICS_CHAR: (VIDEO_MODE, ATTRIB_BYTE, FONT_PTR, SCREEN_PTR,
;			  COLUMN_INCREMENT, REPEAT)
;
;	Display_Graphics_Char is used to display a character
; on the screen while in graphics mode.
;
; Input:
;	BH: Current video mode
;	BL: Attribute byte (or color number from 0-3 for
;	    medium-resolution graphics mode).  If the
;	    high-order bit of BL is set, then the character
;	    to be displayed will be XORed with the existing
;	    screen contents, regardless of whether or not
;	    attributes are being displayed.
;    DS:SI: Pointer to this character's font.
;    ES:DI: Pointer to first byte of screen RAM for char.
;	BP: Column increment (between characters)
;	CX: is a repeat count - it determines how many
;	    times the character should be displayed.
;	    Programs should not assume that wrap-around at
;	    the end of a line or the screen will work
;	    correctly.
;
; NOTE: For efficiency, this routine does not save registers
;	AX, BX, CX, SI or DI (though they are used)!!!
;**********************************************************************
DISPLAY_GRAPHICS_CHAR PROC NEAR
PUBLIC	DISPLAY_GRAPHICS_CHAR

	PUSHREG	<DX>
	MOV	AH,BH			;Keep video mode in AH
DGC1:	MOV	BH,FONT_HEIGHT		;Use BH as a scan-line counter

	PUSH	DI			;Save pointer to char start
DGC2:	LODS	BYTE PTR [SI]		;Get a character of the font
	CMP	AH,6			;Is the display in high-res mode?
	JNE	DGC4			;No, use medium-resolution mode

	TEST	BL,80H			;High-res - displaying normally?
	JNE	DGC3			;No, display char in XOR mode
	MOV	BYTE PTR ES: [DI],AL	;Place the character on the screen
	JMP	SHORT DGC6		;Go on to next scan line

DGC3:	XOR	BYTE PTR ES: [DI],AL	;XOR character with old data
	JMP	SHORT DGC6

DGC4:	PUSH	BX			;Save attribute byte & scan line
	AND	BX,11B			;Mask out all but color select
	SHL	BX,1			;Make the color a word offset
	MOV	DX,CS:MED_RES_COLOR[BX]	;Spread the color across the word
	PUSH	AX			;Save the video mode

	CALL	EXPAND_FONT		;Turn 8-bit font into 16-bit font
	AND	DX,AX			;Set color bits only for 'on' pixels
	XCHG	DH,DL			;Swap bytes to correspond to hardware
	POP	AX			;Restore the video mode
	POP	BX			;Restore attribute and scan counter

	TEST	BL,80H			;Should the colors be XORed in?
	JE	DGC5			;No, continue
	XOR	WORD PTR ES: [DI],DX	;Yes - XOR in new colors with old
	JMP	SHORT DGC6

DGC5:	MOV	WORD PTR ES: [DI],DX	;Place colors into screen RAM
DGC6:	XOR	DI,SCREEN_PARTITION_OFS	;Point to the even/odd scan line
	TEST	BH,1			;Did we just display an odd line?
	JE	DGC7			;No, continue

	ADD	DI,SCAN_LINE_INCREMENT	;Yes - point to the next scan line
	CMP	DI,OFFSET END_OF_PART	;Beyond end of screen?
	JBE	DGC7			;No, continue

	SUB	DI,SCREEN_BYTES	SHR 1	;Yes - wrap-around
DGC7:	DEC	BH			;Sent all of a character yet?
	JNE	DGC2			;No, loop until char completed

	POP	DI			;Restore ptr to char start
	DEC	CX			;Done yet?
	JCXZ	DGC10			;Yes - return directly

DGC8:	SUB	SI,FONT_HEIGHT		;Point back to start of font char
	ADD	DI,BP			;Point to start of next display char
	CMP	DI,OFFSET END_OF_PART	;Beyond end of screen RAM?
	JBE	DGC9			;No, continue

	SUB	DI,SCREEN_BYTES	SHR 1	;Yes - wrap-around to rest of screen
DGC9:	JMP	DGC1			;Jump back to display next character

DGC10:	POPREG	<DX>
	RET

DISPLAY_GRAPHICS_CHAR ENDP

page
;**********************************************************************
; DISPLAY_PIXEL: (VERT, HOR, PIXEL_VALUE)
;
;	Display_Pixel is called in graphics mode to set or 
; reset individual pixels on the display screen. Optionally,
; pixels may be XORed into the display RAM.
;
; Input:
;	AL: Pixel value (1 or 2 bits, depending on
;	    whether high or medium resolution
;	    graphics mode is in effect).  If the
;	    most-significant bit of AL is set, then
;	    the data written will be XORed with the
;	    existing screen contents.
;	CX: Horizontal position (0-319, or 0-639)
;	DX: Vertical position (0-199)
;**********************************************************************
DISPLAY_PIXEL PROC NEAR
PUBLIC	DISPLAY_PIXEL

	PUSHREG	<AX,BX,CX,DI,ES>
	PUSH	AX			;Save pixel value on stack
	MOV	AH,FALSE		;Want to generate graphics address
	CALL	VIDEO_PTR		;Point to the correct byte location
	MOV	BL,AH			;Point BX to the correct mask byte
	MOV	BH,0			;...for this pixel location
	DEC	AL			;Is there one bit/pixel?
	JE	DP1			;Yes - no need to adjust bit number

	SHL	AH,1			;No, get the starting bit number
	INC	AH			;Increment to include 2nd pixel bit
DP1:	MOV	CL,7			;Make count go from 7-0, 'sted 0-7
	SUB	CL,AH			;...since M.S. bit is displayed first
	POP	AX			;Restore pixel value from stack

	TEST	AL,80H			;Are we in XOR mode?
	PUSHF				;Save XOR mode status as zero flag
	AND	AL,11B			;Mask out all but pixel
	SHL	AL,CL			;Place pixel in right bit position(s)
	MOV	CL,AL			;Save pixel value in CL
	POPF				;Restore XOR mode flag into Zero Flag
	JE	DP2			;In normal 'draw' mode

	XOR	BYTE PTR ES: [DI],CL	;XOR mode - merge pixel with screen
	JMP	SHORT DP4

DP2:	MOV	AL,CS:HIGH_RES_MASKS[BX];Get the high-resolution bit mask
	CMP	VIDEO_MODE,6		;In high-resolution mode?
	JE	SHORT DP3		;Yes - all set

	MOV	AL,CS:MED_RES_MASKS[BX]	;No, Get the medium-res bit mask
DP3:	NOT	AL			;Want to mask OUT old pixel
	AND	AL,BYTE PTR ES: [DI]	;Get all of old byte, clear old pixel
	OR	AL,CL			;Merge in new pixel
	MOV	BYTE PTR ES: [DI],AL	;Replace graphics byte on screen
DP4:	POPREG	<ES,DI,CX,BX,AX>
	RET

DISPLAY_PIXEL ENDP

