/*
 *		Synchronous Data Set Adapter driver
 *
 *	- uses Autodriver Channel for output
 *
 */

#include "../h/param.h"
#include "../h/conf.h"
#include "../h/tty.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/proc.h"
#include "../h/ccb.h"

extern int	ndsa;		/* number of DSA'S */
extern char	dsaaddr[];	/* DSA addresses */
extern struct tty dsa[];	/* common tty structure */

#ifdef ACD
int	DOBUF = 0;	/* patch here to turn off acd */
extern struct ccb *dsaccb;		/* output channel control blocks */
extern short isp[];			/* interrupt service ptr table */
#endif

#define LASTC	DKCALL		/* one character left in this input block */

int dsastart(), ttrstrt();

/*
 * DSA commands & status bits
 */

#define	DIS	0xc0
#define	ENB	0x40
#define PAR	0x20
#define SYNSCH	0x10
#define	DTR	0x08
#define	WR	0x02
#define RD	0x01

#define OV		0x80
#define PARF		0x40
#define SYNC		0x20
#define BSY		0x08
#define	EX		0x04
#define	CARR_OFF	0x02
#define DU		0x01

/*
 * open routine:
 *	called each time a process opens a DSA as a character file
 *
 *	- set up the initial status
 *	  and arm interrupts
 */
dsaopen(dev)
{
	register struct tty *tp;

	if (minor(dev) >= ndsa) {
		u.u_error = ENXIO;
		return;
	}
	tp = &dsa[minor(dev)];
	if (tp->t_state & ISOPEN) {
		u.u_error = ENXIO;
		return;
	}
	tp->t_state = ISOPEN;
	tp->t_dev = dev;
	tp->t_flags = RAW;
	tp->t_oproc = dsastart;
#ifdef ACD
	if (DOBUF)
		isp[dsaaddr[minor(dev)] + 1] = (short)&dsaocb[minor(dev)] + 1;
#endif

	dsaenab(tp);
	spl4();
	while (!(tp->t_state&CARR_ON))
		sleep(&tp->t_rawq, TTIPRI);
	spl0();
	ttyopen(dev, tp);
}

/*
 * close routine:
 */
dsaclose(dev)
{
	register struct tty *tp;
	tp = &dsa[minor(dev)];
	ttyclose(tp);
	dsadisab(dev);
}

/*
 * read, write, ioctl routines:
 */
dsaread(dev)
{
	register struct tty *tp;
	register c;

	tp = &dsa[minor(dev)];
	spl4();
	while (tp->t_rawq.c_cc == 0) {
		if ((tp->t_state & CARR_ON) == 0)
			return(0);
		sleep(&tp->t_rawq, TTIPRI);
	}
	spl0();

	while ((c = getc(&tp->t_rawq)) >= 0) {
		if (passc(c) < 0)
			break;
	}
}

dsawrite(dev)
{
	register struct tty *tp;
	register c;

	tp = &dsa[minor(dev)];
	if ((tp->t_state & CARR_ON) == 0)
		return(0);
	while ((c = cpass()) >= 0)
		putc(c, &tp->t_outq);
	dsastart(tp);
}

dsaioctl(dev, cmd, addr, flag)
caddr_t addr;
{
	register struct tty *tp;

	tp = &dsa[minor(dev)];
	u.u_error = ENOTTY;
	return;
}

/*
 * dsastart routine - called when there might be something to send
 *   to the DSA
 */

dsastart(atp)
struct tty *atp;
{
	register struct tty *tp;
	register c, waddr;
	register struct ccb *ccb;

	tp = atp;
	waddr = dsaaddr[minor(tp->t_dev)] + 1;
trace(8<<16, "dsastart", waddr);
	if ((tp->t_state&(BUSY|CARR_ON)) != CARR_ON
	    || tp->t_outq.c_cc <= 0)
		return;
	tp->t_state |= BUSY;
	wd(waddr, -1);
	oc(waddr, ENB | PAR | DTR | WR);
trace(8<<16, "wrt", ss(waddr));
#ifdef ACD
	if (DOBUF) {
		ccb = &dsaccb[minor(tp->t_dev)];
		if (dsaputc(tp, ccb))
			ccb->cc_ccw = 0x85;
	}
#endif
}

dsastop(tp)
register struct tty *tp;
{
	register struct ccb *ccb;
	register n;

trace(8<<16, "dsstop", tp->t_dev);
#ifdef ACD
	if (!DOBUF)
		return;
	ccb = &dsaccb[minor(tp->t_dev)];
	if ((n = ccb->cc_cnt1 + ccb->cc_cnt0 - 1) > 0)
		ndflush(&tp->t_outq, n);
	ccb->cc_cnt1 = 0;
	ccb->cc_ccw = 0;
#endif
}

#ifdef ACD
/*
 * Set up ccb to continue output
 */
dsaputc(tp, ccb)
register struct tty *tp;
register struct ccb *ccb;
{
	register n;

trace(8<<16, "dsaputc", ccb->cc_buf1);
	if ((n = ccb->cc_cnt1 = ndqb(&tp->t_outq, 0200)) <= 0)
		return(0);
	n--;
	ccb->cc_cnt0 = -n;
	ccb->cc_buf0 = tp->t_outq.c_cf + n;
trace(8<<16, "ccount", n);
	return(1);
}
#endif

dsarint(dev, stat)
{
	register int raddr;
	register struct tty *tp;
	register char c;

	tp = &dsa[dev];
	raddr = dsaaddr[dev];

	if (!(tp->t_state & ISOPEN))
		return;
trace(8<<16, "rint", raddr);
trace(8<<16, "stat", stat);

	/*
	 * SYNC interrupt - nothing to do
	 */
	if (stat & (SYNC|BSY) == (SYNC|BSY))
		return;

	/*
	 * OV interrupt - resynchronise
	 */
	if (stat & OV) {
		oc(raddr, ENB | PAR | SYNSCH | DTR);
		rd(raddr);
		return;
	}

	/*
	 * Loss of DSR - hangup
	 */

	if (stat & DU) {
		if (tp->t_state & CARR_ON) {
			signal(tp->t_pgrp, SIGHUP);
			flushtty(tp);
		}
		tp->t_state &= ~CARR_ON;
		return;
	}

	c = rd(raddr)&0177;
trace(8<<16, "char", c);
	if (!(tp->t_state & CARR_ON)) {
		tp->t_state |= CARR_ON;
		wakeup(&tp->t_rawq);
		return;
	}
	if (tp->t_rawq.c_cc > 512)
		flushtty(tp);
	putc(c, &tp->t_rawq);
	wakeup(&tp->t_rawq);
	if (tp->t_state & LASTC) {
		tp->t_state &= ~LASTC;
		oc(raddr, ENB | PAR | SYNSCH | DTR | RD);
	} else if (c == '\03')
		tp->t_state |= LASTC;
}

dsaxint(dev, stat)
{
	register int waddr;
	register struct tty *tp;
	register c;

	tp = &dsa[dev];
	waddr = dsaaddr[dev] + 1;
	if (!(tp->t_state & ISOPEN))
		return;
trace(8<<16, "wint", waddr);
trace(8<<16, "stat", stat);
	if ((c = getc(&tp->t_outq)) < 0) {
		tp->t_state &= ~BUSY;
		oc(waddr, DIS | DTR | WR);
		return;
	}
	wd(waddr, c);
trace(8<<16, "wrt", ss(waddr));
#ifdef ACD
	if (DOBUF) {
		ccb = &dsaccb[minor(tp->t_dev)];
		if (dsaputc(tp, ccb))
			ccb->cc_ccw = 0x85;
	}
#endif

}

#ifdef ACD
dsaxbint(dev, stat);
{
	register int waddr;
	register struct tty *tp;
	register struct ccb *ccb;

	tp = &dsa[dev];
	waddr = dsaaddr[dev] + 1;
	if (!(tp->t_state & ISOPEN))
		return;
trace(8<<16, "bint", waddr);
trace(8<<16, "stat", stat);
	ccb = &dsaccb[minor(tp->t_dev)];
	ndflush(&tp->t_outq, ccb->cc_cnt1);
	if ((tp->t_state&TTSTOP) || dsaputc(tp, ccb) == 0)
		isp[waddr] = (short)&dsax;
}
#endif

/*
 * Arm interrupts from DSA and set reading
 */
dsaenab(atp)
struct tp *atp;
{
	register struct tp *tp;
	register radd;
	register stat;

	tp = atp;
	radd = dsaaddr[minor(tp->t_dev)];

	oc(radd, ENB | PAR | SYNSCH | DTR);
	stat = ss(radd);
	if ((stat&DU) == 0)
		tp->t_state |= CARR_ON;

trace(8<<16, "dsaenab", stat);
	rd(radd);
}

/*
 * Disarm interrupts from DSA
 */
dsadisab(dev)
{
	register radd, wadd;

	radd = dsaaddr[minor(dev)];
	wadd = radd + 1;

	oc(radd, DIS);
	oc(wadd, DIS);

trace(8<<16, "dsadisab", ss(radd));
}
