/***********************************************************************
*
* asmpass0 - Pass 0 for the IBM 7090 assembler.
*
* Changes:
*   12/12/04   DGP   Original.
*   02/02/05   DGP   Correct Macro/IRP expansions.
*   02/03/05   DGP   Added JOBSYM.
*   02/09/05   DGP   Added FAP mode.
*	
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "asmdef.h"

extern int pc;			/* the assembler pc */
extern int symbolcount;		/* Number of symbols in symbol table */
extern int absolute;		/* In absolute section */
extern int radix;		/* Number scanner radix */
extern int linenum;		/* Source line number */
extern int exprtype;		/* Expression type */
extern int pgmlength;		/* Length of program */
extern int absmod;		/* In absolute module */
extern int monsym;		/* Include IBSYS Symbols (MONSYM) */
extern int jobsym;		/* Include IBJOB Symbols (JOBSYM) */
extern int termparn;		/* Parenthesis are terminals (NO()) */
extern int genxref;		/* Generate cross reference listing */
extern int addext;		/* Add extern for undef'd symbols (!absolute) */
extern int addrel;		/* ADDREL mode */
extern int qualindex;		/* QUALifier table index */
extern int fapmode;		/* FAP assembly mode */
extern int headcount;		/* Number of entries in headtable */

extern char inbuf[MAXLINE];	/* The input buffer for the scanners */
extern char deckname[MAXSYMLEN+2];/* The assembly deckname */
extern char lbl[MAXLBLLEN+2];	/* The object label */
extern char titlebuf[TTLSIZE+2];/* The assembly TTL buffer */
extern char qualtable[MAXQUALS][MAXSYMLEN+2];/* The QUALifier table */
extern char headtable[MAXHEADS];/* HEAD table */

extern SymNode *symbols[MAXSYMBOLS];/* The Symbol table */

static long curpos;		/* Current file position */
static int asmskip;		/* In skip line mode IFF/IFT */
static int gotoskip;		/* In GOTO mode */
static int lblgennum;		/* LBL generated number */
static int macrocount;		/* Number of macros defined */
static int ifcont;		/* IFF/IFT continued */
static int ifrel;		/* IFF/IFT continue relation OR/AND */
static int ifskip;		/* IFF/IFT prior skip result */
static int eofflg = FALSE;	/* EOF on input file */

static char cursym[MAXSYMLEN+2];/* Current label field symbol */
static char gotosym[MAXSYMLEN+2];/* GOTO target symbol */
static char srcbuf[MAXSRCLINE];	/* Source line buffer */
static char etcbuf[MAXSRCLINE];	/* ETC (continuation) buffer */
static char errtmp[256];	/* Error print string */

static MacLines *freemaclines = NULL;/* resusable Macro lines */
static MacDef macdef[MAXMACROS];/* Macro template table */

/***********************************************************************
* maclookup - Lookup macro in table
***********************************************************************/

static MacDef *
maclookup (char *name)
{
   MacDef *mac = NULL;
   int i;

   for (i = 0; i < macrocount; i++)
   {
      mac = &macdef[i];
      if (!strcmp (mac->macname, name))
      {
	 return (mac);
      }
   }
   return (NULL);
}

/***********************************************************************
* p0mactoken - Process macro tokens
***********************************************************************/

static char *
p0mactokens (char *cp, int field, int index, MacDef *mac)
{
   char *token;
   char *ocp;
   int tokentype;
   int val;
   int j;
   char term;
   char targ[32];
   char largs[MAXSRCLINE];

#ifdef DEBUGMACRO
   printf ("p0mactokens: field = %d, index = %d, cp = %s",
	    field, index, cp);
#endif

   /*
   ** Check for NULL operands 
   */

   if (field == 2)
   {
      ocp = cp;
      while (isspace (*cp))
      {
         cp++;
	 if (cp - ocp > (NOOPERAND))
	    return (cp);
      }
   }

   /*
   ** Substitute #%02d for each macro parameter. The number corresponds to
   ** the paramter number.
   */

   term = '\0';
   largs[0] = '\0';
   do {

      targ[0] = '\0';
      cp = tokscan (cp, &token, &tokentype, &val, &term);
#ifdef DEBUGMACRO
      printf (
	    "   token1 = %s, tokentype = %d, val = %o, term = %c(%x)\n",
	    token, tokentype, val, term, term);
#endif
      /*
      ** If operator in input stream, copy it out
      */

      if (token[0] == '\0')
      {
	 if (tokentype == ASTER)
	    strcpy (targ, "*");
	 else
	    sprintf (targ, "%c", term);
      }

      /*
      ** If subsitution field, marked with apostophe, process.
      */

      else if (term == '\'')
      {
	 for (j = 0; j < mac->macargcount; j++)
	 {
	    if (!strcmp (token, mac->macargs[j]))
	    {
	       sprintf (token, "#%02d", j);
	       break;
	    }
	 }
	 strcat (largs, token);

	 cp++;
	 cp = tokscan (cp, &token, &tokentype, &val, &term);
#ifdef DEBUGMACRO
	 printf (
	    "   token2 = %s, tokentype = %d, val = %o, term = %c(%x)\n",
	    token, tokentype, val, term, term);
#endif
	 if (token[0])
	 {
	    for (j = 0; j < mac->macargcount; j++)
	    {
	       if (!strcmp (token, mac->macargs[j]))
	       {
		  sprintf (targ, "#%02d", j);
		  break;
	       }
	    }
	    if (j == mac->macargcount)
	       strcpy (targ, token);
	 }
	 if (term == '\'')
	    cp++;
         else if (!isspace (term))
	 {
	    strncat (targ, &term, 1);
	    if (!strchr (",+-/*", term)) cp++;
	 }
      }

      /*
      ** If token is a parameter, process
      */

      else
      {
	 for (j = 0; j < mac->macargcount; j++)
	 {
	    if (!strcmp (token, mac->macargs[j]))
	    {
	       sprintf (targ, "#%02d", j);
	       break;
	    }
	 }
	 if (j == mac->macargcount)
	    strcpy (targ, token);
	 if (!isspace(term))
	 {
	    strncat (targ, &term, 1);
	    if (term != ',') cp++;
	 }
      }
      strcat (largs, targ);
   } while (!isspace(term) && !isspace(*cp));

#ifdef DEBUGMACRO
   printf ("   largs = %s\n", largs);
#endif

   /*
   ** Place processed field into macro template
   */

   switch (field)
   {
   case 0:
      strcat (mac->maclines[index]->label, largs);
      break;
   case 1:
      strcat (mac->maclines[index]->opcode, largs);
      break;
   case 2:
      strcat (mac->maclines[index]->operand, largs);
      break;
   }
   return (cp);
}

/***********************************************************************
* p0macro - Process macro templates
***********************************************************************/

static void
p0macro (char *bp, FILE *infd, FILE *outfd)
{
   MacDef *mac;
   char *token;
   char *cp;
   int tokentype;
   int val;
   int i;
   int done;
   int genmode;
   int oldtermparn;
   char term;
   char genline[MAXSRCLINE];

   oldtermparn = termparn;
   termparn = TRUE;

   mac = &macdef[macrocount];

   strcpy (mac->macname, cursym);

   /*
   ** Scan off the parameters
   */

   i = 0;
   do {
      bp = tokscan (bp, &token, &tokentype, &val, &term);
      strcpy (mac->macargs[i], token);
      i++;
   } while (!isspace(term));
   mac->macargcount = i;

#ifdef DEBUGMACRO
   printf ("\nMACRO0: name = %s\n", mac->macname);
   for (i = 0; i < mac->macargcount; i++)
      printf ("   arg%02d = %s\n", i, mac->macargs[i]);
#endif

   /*
   ** Read the source into the template until ENDM.
   */

   done = FALSE;
   i = 0;
   while (!done)
   {
      /*
      ** Write source line to intermediate file with MACDEF mode
      */

      genmode = MACDEF;
      if (etcbuf[0])
      {
         strcpy (genline, etcbuf);
	 etcbuf[0] = '\0';
      }
      else
	 fgets (genline, MAXSRCLINE, infd);
      fwrite (&genmode, 1, 4, outfd);
      fputs (genline, outfd);
      linenum++;

      /*
      ** If END, we're done
      */

      if (fapmode)
      {
	 if (!strncmp (&genline[OPCSTART], "END", 3))
	    done = TRUE;
      }
      else
      {
	 if (!strncmp (&genline[OPCSTART], "ENDM", 4))
	    done = TRUE;
      }

      /*
      ** If not a comment, process template line
      */

      if (!done && genline[0] != COMMENTSYM)
      {

	 /*
	 ** Allocate line template
	 */

	 if (freemaclines)
	 {
	    mac->maclines[i] = freemaclines;
	    freemaclines = mac->maclines[i]->next;
	 }
	 else if ((mac->maclines[i] = (MacLines *)malloc(sizeof(MacLines))) == NULL)
	 {
	    fprintf (stderr, "asm7090: Unable to allocate memory\n");
	    exit (ABORT);
	 }
	 memset (mac->maclines[i], '\0', sizeof(MacLines));

	 /*
	 ** If label field process label
	 */

	 cp = genline;
	 if (strncmp (cp, "      ", 6))
	 {
	    while (isspace (*cp)) cp++;
	    if (isalnum(*cp) || *cp == '.')
	    cp = p0mactokens (cp, 0, i, mac);
#ifdef DEBUGMACRO
	    printf ("   label = %s\n", mac->maclines[i]->label);
#endif
	 }

	 /*
	 ** Process opcode field
	 */

	 cp = p0mactokens (cp, 1, i, mac);
#ifdef DEBUGMACRO
	 printf ("   opcode = %s\n", mac->maclines[i]->opcode);
#endif

	 /*
	 ** Process operand field
	 */

	 cp = p0mactokens (cp, 2, i, mac);
#ifdef DEBUGMACRO
	 printf ("   operand = %s\n", mac->maclines[i]->operand);
#endif
	 i++;
      }
   }
   mac->maclinecount = i;
   macrocount++;

   termparn = oldtermparn;
}

/***********************************************************************
* p0pop - Process Pseudo operation codes.
***********************************************************************/

static int
p0pop (OpCode *op, char *bp, FILE *infd, FILE *outfd, int inmacro)
{
   SymNode *s;
   OpCode *addop;
   char *token;
   char *cp;
   int tokentype;
   int relocatable;
   int val;
   int i, j;
   int genmode;
   int savenmode;
   char term;
   char genline[MAXSRCLINE];

   savenmode = FALSE;
   switch (op->opvalue)
   {

   case ABS_T:			/* ABSoulute assembly (FAP mode) */
      if (fapmode)
      {
	 addext = FALSE;
         absolute = TRUE;
         absmod = TRUE;
      }
      break;

   case CALL_T:			/* Process CALL */
      {
	 int argcnt = 0;
	 int retcnt = 0;
	 int calline;
	 int oldtermparn;
	 char name[MAXSYMLEN+2];
	 char lclargs[MAXMACARGS][MAXFIELDSIZE];
	 char retargs[MAXMACARGS][MAXFIELDSIZE];

	 oldtermparn = termparn;
	 termparn = TRUE;
	 calline = linenum;

	 /*
	 ** Get name of called routine
	 */

	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 strcpy (name, token);

	 /*
	 ** If arguments specified, then scan them off 
	 */

	 if ((fapmode && term == ',') || term == '(')
	 {
	    bp++;
	    cp = bp;
	    while (*bp != ')' && !isspace (*bp)) bp++;
	    if (!fapmode && *bp != ')')
	    {
	       logerror ("Missing ')' in CALL");
	       return(FALSE);
	    }
	    bp++;
	    do {
	       cp = tokscan (cp, &token, &tokentype, &val, &term);
	       strcpy (lclargs[argcnt], token);
	       argcnt++;
	    } while (term != ')');
	    if (!isspace(*bp)) goto GET_RETURNS;
	 }

	 /*
	 ** Scan off return fields
	 */

	 else if (term == ',')
	 {
      GET_RETURNS:
	    do {
	       bp = tokscan (bp, &token, &tokentype, &val, &term);
	       strcpy (retargs[argcnt], token);
	       retcnt++;
	    } while (term == ',');
	    if (term == '\'')
	    {
	       bp++;
	       bp = tokscan (bp, &token, &tokentype, &val, &term);
	       calline = val;
	    }
	 }
	 
	 /*
	 ** Expand the CALL
	 */

	 genmode = GENINST | CALL;

	 sprintf (genline, "%-6.6s %-7.7s %s,4\n",
		  "", "TSX", name);
	 pc++;
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
	 
	 if (!fapmode)
	 {
	    sprintf (genline, "%-6.6s %-7.7s *+2+%d+%d,,%d\n",
		     "", "TXI", argcnt, retcnt, argcnt);
	    pc++;
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    sprintf (genline, "%-6.6s %-7.7s %d,,..LDIR\n",
		     "", "PZE", calline);
	    pc++;
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	 }

	 for (i = 0; i < argcnt; i++)
	 {
	    sprintf (genline, "%-6.6s %-7.7s %s\n",
		     "", fapmode ? "TSX" : "PZE", lclargs[i]);
	    pc++;
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	 }

	 for (i = retcnt; i > 0; i--)
	 {
	    sprintf (genline, "%-6.6s %-7.7s %s\n",
		     "", "TRA", retargs[i]);
	    pc++;
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	 }

	 termparn = oldtermparn;
      }
      break;

   case DUP_T:			/* DUPlicate lines */
      {
         int dupin, dupout;
	 char *duplines[MAXDUPLINES];

	 genmode = DUPINST;

	 /*
	 ** Scan off DUP input line count
	 */

	 bp = exprscan (bp, &dupin, &term, &relocatable, 1, FALSE, 0);
	 if (dupin > MAXDUPLINES)
	 {
	    logerror ("Unsupported opcode addition");
	 }
	 else
	 {
	    /*
	    ** Scan off DUP output line count
	    */

	    bp = exprscan (bp, &dupout, &term, &relocatable, 1, FALSE, 0);

	    /*
	    ** Read the DUP input lines
	    */

	    for (i = 0; i < dupin; i++)
	    {
	       if (etcbuf[0])
	       {
		  strcpy (genline, etcbuf);
		  etcbuf[0] = '\0';
	       }
	       else
		  fgets (genline, MAXSRCLINE, infd);

	       if ((duplines[i] = (char *) malloc (MAXSRCLINE)) == NULL)
	       {
		  fprintf (stderr, "asm7090: Unable to allocate memory\n");
		  exit (ABORT);
	       }
	       strcpy (duplines[i], genline);
	    }

	    /*
	    ** Write out the DUP'd lines
	    */

	    for (i = 0; i < dupout; i++)
	    {
	       for (j = 0; j < dupin; j++)
	       {
		  pc++;
		  fwrite (&genmode, 1, 4, outfd);
		  fputs (duplines[j], outfd);
		  if (strncmp (duplines[j], "      ", 6))
		  {
		     strncpy (duplines[j], "       ", 6);
		  }
		  linenum++;
	       }
	    }
	    for (i = 0; i < dupin; i++)
	       free (duplines[i]);
	 }
      }
      break;

   case END_T:			/* END */
      /*
      ** Because of ETC read ahead we reset the input file back one.
      */

      if (!eofflg)
      {
	 if (fseek (infd, curpos, SEEK_SET) < 0)
	    perror ("asm7090: Can't reset input temp file");
      }
      return (TRUE);

   case ENDQ_T:			/* ENDQ */
      if (!fapmode)
      {
	 qualindex --;
	 if (qualindex < 0) qualindex = 0;
      }
      break;

   case EQU_T:			/* EQU */
      bp = exprscan (bp, &val, &term, &relocatable, 1, TRUE, 0);

DO_EQU:
      if (cursym[0])
      {
	 if (!(s = symlookup (cursym, fapmode ? "" : qualtable[qualindex],
			      FALSE, FALSE)))
	 {
	    if (fapmode)
	    {
	       if (headcount && (strlen(cursym) < MAXSYMLEN))
	       {
	          int i;

		  for (i = 0; i < headcount; i++)
		  {
		     char temp[32];

		     sprintf (temp, "%c", headtable[i]);
		     s = symlookup (cursym, temp, TRUE, FALSE);
		     if (s)
		     {
			s->value = val;
			s->relocatable = relocatable;
		     }
		  }
	       }
	       else
	       {
		  s = symlookup (cursym, "", TRUE, FALSE);
	       }
	    }
	    else
	    {
	       s = symlookup (cursym, qualtable[qualindex], TRUE, FALSE);
	    }
	 }
	 if (s)
	 {
	    s->value = val;
	    s->relocatable = relocatable;
	 }
      }
#ifdef DEBUGEQU
      printf ("p0pop: EQU: cursym = '%s', val = %d\n", cursym, val);
#endif
      break;

   case EVEN_T:			/* EVEN */
      if (absolute & (pc & 00001)) pc++;
      break;

   case GOTO_T:			/* GOTO */

      /*
      ** In this pass, only process if in a macro
      */
      
      if (inmacro)
      {
	 /*
	 ** Scan off target symbol
	 */

	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (strlen(token) > MAXSYMLEN) token[MAXSYMLEN] = '\0';
	 strcpy (gotosym, token);
#ifdef DEBUGGOTO
         printf ("asmpass0: GOTO: gotosym = '%s'\n", gotosym);
#endif
	 gotoskip = TRUE;
      }
      break;

   case HEAD_T:			/* HEAD */
      if (fapmode)
      {
	 headcount = 0;
	 do {
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
	    headtable[headcount++] = token[0];
	    if (headcount >= MAXHEADS)
	    {
	       break;
	    }
	 } while (term == ',');
	 if (headtable[0] == '0') headcount = 0;
      }
      break;

   case IFF_T:			/* IF False */

      /*
      ** In this pass, only process if in a macro
      */
      
      if (inmacro)
      {
	 /*
	 ** Scan the conditional expression and get result
	 */

	 bp = condexpr (bp, &val, &term);
	 if (val >= 0)
	 {
	    int skip;

	    skip = val;
	    asmskip = 0;

	    /*
	    ** If continued, use last result here
	    */

	    if (ifcont)
	    {
#ifdef DEBUGIF
	       printf ("p0IFF: ifrel = %s, ifskip = %s, skip = %s",
		  ifrel == IFOR ? "OR" : "AND",
		  ifskip ? "TRUE" : "FALSE",
		  skip ? "TRUE" : "FALSE");
#endif
	       ifcont = FALSE;
	       if (ifrel == IFAND)
	       {
		  if (!(ifskip && skip))
		     asmskip = 1;
	       }
	       else if (ifrel == IFOR)
	       {
		  if (!(ifskip || skip))
		     asmskip = 1;
	       }
#ifdef DEBUGIF
	       printf ("   asmskip = %d\n", asmskip);
#endif
	    }

	    /*
	    ** If not continued, check for relation
	    */

	    else if (term == ',')
	    {
	       ifcont = TRUE;
	       ifskip = skip;
	       if (!(strcmp (bp, "OR")))
		  ifrel = IFOR;
	       else if (!(strcmp (bp, "AND")))
		  ifrel = IFAND;
	    }

	    /*
	    ** Neither, just do it
	    */

	    else if (val)
	    {
	       asmskip = 1;
	    }
	 }
      }
      break;

   case IFT_T:			/* IF True */

      /*
      ** In this pass, only process if in a macro
      */

      if (!fapmode && inmacro)
      {

	 /*
	 ** Scan the conditional expression and get result
	 */

	 bp = condexpr (bp, &val, &term);
	 if (val >= 0)
	 {
	    int skip;

	    skip = val;
	    asmskip = 0;

	    /*
	    ** If continued, use last result here
	    */

	    if (ifcont)
	    {
#ifdef DEBUGIF
	       printf ("p0IFT: ifrel = %s, ifskip = %s, skip = %s",
		  ifrel == IFOR ? "OR" : "AND",
		  ifskip ? "TRUE" : "FALSE",
		  skip ? "TRUE" : "FALSE");
#endif
	       ifcont = FALSE;
	       if (ifrel == IFAND)
	       {
		  if (!(ifskip && skip))
		     asmskip = 1;
	       }
	       else if (ifrel == IFOR)
	       {
		  if (!(ifskip || skip))
		     asmskip = 1;
	       }
#ifdef DEBUGIF
	       printf ("   asmskip = %d\n", asmskip);
#endif
	    }

	    /*
	    ** If not continued, check for relation
	    */

	    else if (term == ',')
	    {
	       ifcont = TRUE;
	       ifskip = skip;
	       if (!(strcmp (bp, "OR")))
		  ifrel = IFOR;
	       else if (!(strcmp (bp, "AND")))
		  ifrel = IFAND;
	    }

	    /*
	    ** Neither, just do it
	    */

	    else if (!val)
	    {
	       asmskip = 1;
	    }
	 }
      }
      break;

   case LBL_T:			/* LBL */
      if (fapmode)
      {
	 if (!deckname[0])
	 {
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
	    if (strlen(token) > MAXLBLLEN) token[MAXLBLLEN] = '\0';
	    strcpy (deckname, token);
	    strcpy (lbl, token);
	 }
      }
      break;

   case LDIR_T:			/* Linkage DIRector */
      if (!fapmode)
      {
	 genmode = GENINST;
	 sprintf (genline, "%-6.6s %-7.7s %s\n", "..LDIR", "PZE", "**");
	 pc++;
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
	 sprintf (genline, "%-6.6s %-7.7s 1,%s\n", "", "BCI", deckname);
	 pc++;
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
      }
      break;

   case MACRO_T:		/* Macro */
      if (inmacro)
	 logerror ("MACRO defintion in MACRO");
      else
	 p0macro (bp, infd, outfd);
      break;

   case NULL_T:			/* NULL */
   	val = pc;
	goto DO_EQU;

   case OPSYN_T:		/* OP SYNonym */
      /*
      ** Scan off opcode
      */

      bp = tokscan (bp, &token, &tokentype, &val, &term);
#ifdef DEBUGOPSYN
      printf (
      "OPSYN0: cursym = %s, token = %s, tokentype = %d, val = %o, term = %c\n",
	      cursym, token, tokentype, val, term);
#endif
      /*
      ** Delete any previous definition and add
      */

      if ((addop = oplookup (token)) != NULL)
      {
	 opdel (cursym);
	 opadd (cursym, addop->opvalue, addop->opmod, addop->optype);
      }

      /*
      ** Synonym opcode not found, error
      */

      else
      {
	 MacDef *oldmac;
	 MacDef *newmac;

	 if ((oldmac = maclookup (token)) != NULL)
	 {
	    newmac = &macdef[macrocount];
	    memcpy (newmac, oldmac, sizeof (MacDef));
	    strcpy (newmac->macname, cursym);
	    macrocount++;
	 }
	 else
	 {
	    sprintf (errtmp, "Invalid opcode for OPSYN: %s", token);
	    logerror (errtmp);
	 }
      }
      break;

   case QUAL_T:			/* QUALified section */
      if (!fapmode)
      {
	 qualindex ++;
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (strlen(token) > MAXSYMLEN) token[MAXSYMLEN] = '\0';
	 strcpy (qualtable[qualindex], token);
      }
      break;

   case RETURN_T:		/* RETURN */
      if (!fapmode)
      {
	 char name[MAXSYMLEN+2];

	 genmode = GENINST | RETURN;
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 strcpy (name, token);
	 if (term == ',')
	 {
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
	    sprintf (genline, "%-6.6s %-7.7s %d,4\n", "", "AXT", val);
	    pc++;
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    sprintf (genline, "%-6.6s %-7.7s %s,4\n", "", "SXD", name);
	    pc++;
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	 }
	 sprintf (genline, "%-6.6s %-7.7s %s+1\n", "", "TRA", name);
	 pc++;
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
      }
      break;

   case SAVEN_T:		/* SAVEN cpu context */
      savenmode = TRUE;

   case SAVE_T:			/* SAVE cpu context */
      if (!fapmode)
      {
	 int oldtermparn;
	 int regs[8];
	 int modes[4];
	 char label[MAXSYMLEN+2];
	 char operand[MAXFIELDSIZE];

	 oldtermparn = termparn;
	 termparn = TRUE;
	 for (i = 0; i < 8; i++) regs[i] = FALSE;
	 for (i = 0; i < 4; i++) modes[i] = FALSE;

	 /*
	 ** Scan off SAVE args and regs. Order regs.
	 */

	 do
	 {
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
	    if (term == '(' || term == ')') term = ',';
	    if (tokentype == DECNUM)
	    {
	       if (val < 1 || val > 7)
	       {
		  sprintf (errtmp, "Invalid register for %s: %d",
			   savenmode ? "SAVEN" : "SAVE", val);
		  logerror (errtmp);
		  return(FALSE);
	       }
	       regs[val] = TRUE;
	    }
	    else if (tokentype == SYM)
	    {
	       if (!strcmp (token, "I")) modes[0] = TRUE;
	       else if (!strcmp (token, "D")) modes[1] = TRUE;
	       else if (!strcmp (token, "E")) modes[2] = TRUE;
	       else 
	       {
		  sprintf (errtmp, "Invalid mode for %s: %s",
			   savenmode ? "SAVEN" : "SAVE", token);
		  logerror (errtmp);
		  return(FALSE);
	       }
	    }
	 } while (term == ',');

	 /*
	 ** Expand the SAVE[N] 
	 */

	 genmode = GENINST | SAVE;

	 /*
	 ** If symbol, generate ENTRY
	 */

	 if (cursym[0])
	 {
	    if (!savenmode) /* No ENTRY for SAVEN mode */
	    {
	       sprintf (genline, "%-6.6s %-7.7s %s\n", "", "ENTRY", cursym);
	       fwrite (&genmode, 1, 4, outfd);
	       fputs (genline, outfd);
	       linenum++;
	    }
	    sprintf (operand, "..%04d,,0", lblgennum+3);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", cursym, "TXI", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	 }
	 else 
	 {
	    sprintf (errtmp, "Label required for %s",
		     savenmode ? "SAVEN" : "SAVE");
	    logerror (errtmp);
	    return(FALSE);
	 }

	 /*
	 ** Generate E mode, Error return
	 */

	 if (modes[2])
	 {
	    sprintf (operand, "%s,%d", cursym, 4);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "LDC", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	    sprintf (operand, "%s,%d", "*+5", 4);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "SXD", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	    sprintf (operand, "..%04d,%d", lblgennum+1, 4);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "LAC", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	    sprintf (operand, "%s,%d", "*+1,4", 1);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "TXI", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	    sprintf (operand, "%s,%d", "*+1", 4);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "SXA", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	    sprintf (operand, "%s,%d", "**", 4);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "LXA", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	    sprintf (operand, "%s,%d", "*+1,4", 0);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "TXI", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	    sprintf (operand, "..%04d,%d", lblgennum+2, 4);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "SXA", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	 }

	 /*
	 ** Generate index register loads
	 */

	 for (i = 7; i > 0; i--)
	 {
	       if (i == 4) continue;
	       if (regs[i])
	       {
		  sprintf (operand, "%s,%d", "**", i);
		  sprintf (genline, "%-6.6s %-7.7s %s\n", "", "AXT", operand);
		  fwrite (&genmode, 1, 4, outfd);
		  fputs (genline, outfd);
		  linenum++;
		  pc++;
	       }
	 }
	 
	 sprintf (label, "..%04d", lblgennum+1);
	 sprintf (operand, "%s,%d", "**", 4);
	 sprintf (genline, "%-6.6s %-7.7s %s\n", label, "AXT", operand);
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
	 pc++;
	 
	 /* 
	 ** Generate Indicator load
	 */

	 if (modes[0])
	 {
	    sprintf (operand, "..%04d+1", lblgennum+2);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "LDI", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	 }

	 /*
	 ** Generate Trap enables
	 */

	 if (modes[1])
	 {
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "NZT", ".TRPSW");
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "ENB*", ".TRAPX");
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	 }
	 sprintf (label, "..%04d", lblgennum+2);
	 if (modes[2])
	 {
	    strcpy (operand, "**");
	 }
	 else
	 {
	    strcpy (operand, "1,4");
	 }
	 sprintf (genline, "%-6.6s %-7.7s %s\n", label, "TRA", operand);
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
	 pc++;

	 if (modes[0])
	 {
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "PZE", "**");
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	 }
	 sprintf (label, "..%04d", lblgennum+3);
	 sprintf (genline, "%-6.6s %-7.7s %s\n", label, "EQU", "*");
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
	 pc++;
	 if (modes[1])
	 {
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "XEC", "SYSDSB");
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	 }

	 /*
	 ** Gerate Indicator save
	 */

	 if (modes[0])
	 {
	    sprintf (operand, "..%04d+1", lblgennum+2);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "STI", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	 }

	 /*
	 ** Generate Trap save
	 */

	 if (modes[2])
	 {
	    sprintf (operand, "%s,%d", cursym, 0);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "SXD", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	 }
	 sprintf (operand, "%s,%d", "SYSLOC", 4);
	 sprintf (genline, "%-6.6s %-7.7s %s\n", "", "SXA", operand);
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
	 pc++;

	 /*
	 ** Generate Linkage Director, if not SAVEN
	 */

	 if (!savenmode)
	 {
	    sprintf (operand, "%s,%d", "..LDIR", 4);
	    sprintf (genline, "%-6.6s %-7.7s %s\n", "", "SXA", operand);
	    fwrite (&genmode, 1, 4, outfd);
	    fputs (genline, outfd);
	    linenum++;
	    pc++;
	 }

	 /*
	 ** Generate save registers
	 */

	 sprintf (operand, "..%04d,%d", lblgennum+1, 4);
	 sprintf (genline, "%-6.6s %-7.7s %s\n", "", "SXA", operand);
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
	 pc++;
	 j = 1;
	 for (i = 1; i < 8; i++)
	 {
	    if (i == 4) continue;
	    if (regs[i])
	    {
	       sprintf (operand, "..%04d-%d,%d", lblgennum+1, j, i);
	       sprintf (genline, "%-6.6s %-7.7s %s\n", "", "SXA", operand);
	       fwrite (&genmode, 1, 4, outfd);
	       fputs (genline, outfd);
	       linenum++;
	       pc++;
	       j++;
	    }
	 }
	 lblgennum += 3;
	 termparn = oldtermparn;
      }
      break;

   case SET_T:			/* SET value immediately */
      bp = exprscan (bp, &val, &term, &relocatable, 1, FALSE, 0);
      if (cursym[0])
      {
	 if (!(s = symlookup (cursym, "", FALSE, FALSE)))
	    s = symlookup (cursym, "", TRUE, FALSE);
	 if (s)
	 {
	    s->value = val;
	    s->relocatable = FALSE;
	    s->setvar = TRUE;
	 }
      }
      break;

   case SST_T:
      if (fapmode)
      {
	 monsym = TRUE;
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
#ifdef DEBUGMONSYM
	 printf ("SST %s\n", token);
#endif
	 definemonsyms(0);
	 if (!strcmp (token, "FORTRAN"))
	 {
	    definemonsyms(1);
	 }
      }
      break;

   default: ;
   }

   return (FALSE);
}

/***********************************************************************
* p0expand - Expand macros
***********************************************************************/

static void
p0expand (MacDef *mac, char *bp, FILE *infd, FILE *outfd)
{
   OpCode *op;
   MacDef *lclmac;
   char *token;
   char *cp;
   char *ocp;
   int tokentype;
   int inirp;
   int irpline;
   int irpnum;
   int val;
   int i, j;
   int done;
   int genmode;
   int argcnt;
   int oldtermparn;
   char term;
   char genline[MAXSRCLINE];
   char lclargs[MAXMACARGS][MAXFIELDSIZE];
   char irpargs[MAXFIELDSIZE];
   char irparg[MAXFIELDSIZE];


#ifdef DEBUGMACROEXP
   printf ("p0expand: macro = %s, args = %s", mac->macname, bp);
#endif

   oldtermparn = termparn;
   termparn = TRUE;

   for (i = 0; i < MAXMACARGS; i++) lclargs[i][0] = '\0';

   /*
   ** Scan off MACRO arguments
   */

   argcnt = 0;
   do {
      ocp = bp;
      bp = tokscan (bp, &token, &tokentype, &val, &term);
#ifdef DEBUGMACROEXP
      printf (
	    "   token = %s, tokentype = %d, val = %o, term = %c(%x)\n",
	    token, tokentype, val, term, term);
#endif
      if (token[0] == '\0' && term == '(')
      {
	 ocp = bp;
	 i = 1;
         while (i > 0 && !isspace(*bp))
	 {
	    if (*bp == '(')
	    {
	       i++;
	       bp++;
	    }
	    else if (*bp == ')')
	    {
	       if (i)
	       {
		  i--;
		  if (i) bp++;
	       }
	    }
	    else bp++;
	 }
#ifdef DEBUGMACROEXP
         printf ("   *bp = %c(%x) %s, pocp = %s\n", *bp, *bp, bp, ocp);
#endif
	 if (*bp == ')')
	 {
	    *bp++ = '\0';
	    strcat (lclargs[argcnt], ocp);
	 }
	 else
	 {
	    logerror ("Missing ')' in macro");
	 }
	 term = *bp;
	 if (term == ',') bp++;
	 else if (isalnum(term) || term == '.')
	    term = ',';
      }
      else if (token[0] == '\0' && tokentype == NULSYM)
      {
	 strcat (lclargs[argcnt], "**");
      }
      else if (token[0] == '\0' && tokentype == ASTER)
      {
	 strcat (lclargs[argcnt], "*");
      }
      else if (token[0] == '\0')
      {
         strncat (lclargs[argcnt], &term, 1);
      }
      else
      {
	 strcat (lclargs[argcnt], token);
      }
      if (isspace (term) || term == ',') argcnt++;
   } while (!isspace(term));

   termparn = oldtermparn;

#ifdef DEBUGMACROEXP
   for (i = 0; i < argcnt; i++)
      printf ("   arg%02d = %s\n", i, lclargs[i]);
#endif

   inirp = FALSE;

   /*
   ** Go through MACRO template and replace parameters with the
   ** user arguments.
   */

   for (i = 0; i < mac->maclinecount; i++)
   {
      char genlabel[MAXFIELDSIZE];
      char genopcode[MAXFIELDSIZE];
      char genoperand[MAXFIELDSIZE];
      char genindex[3];

      /*
      ** Process label field
      */

      done = FALSE;
      genmode = 0;
      genlabel[0] = '\0';
      token = genlabel;
      cp = mac->maclines[i]->label;

      while (*cp && !done)
      {
	 if ((ocp = strchr(cp, '#')) != NULL)
	 {
	    if (ocp > cp)
	       strncat (token, cp, ocp-cp);
	    ocp++;
	    strncpy (genindex, ocp, 2);
	    ocp += 2;
	    cp = ocp;
	    genindex[2] = '\0';
	    j = atoi (genindex);
	    if (inirp && j == irpnum)
	    {
	       strcat(token, irparg);
	    }
	    else
	    {
	       strcat (token, lclargs[j]);
	    }
	 }
	 else
	 {
	    strcat (token, cp);
	    done = TRUE;
	 }
      }
      if (strlen(genlabel) > MAXSYMLEN) genlabel[MAXSYMLEN] = '\0';
      strcpy (cursym, genlabel);
#ifdef DEBUGMACROEXP
      printf ("   genlabel = %s\n", genlabel);
#endif

      /*
      ** Process opcode field
      */

      done = FALSE;
      genopcode[0] = '\0';
      token = genopcode;
      cp = mac->maclines[i]->opcode;

      while (*cp && !done)
      {
	 if ((ocp = strchr(cp, '#')) != NULL)
	 {
	    if (ocp > cp)
	       strncat (token, cp, ocp-cp);
	    ocp++;
	    strncpy (genindex, ocp, 2);
	    ocp += 2;
	    cp = ocp;
	    genindex[2] = '\0';
	    j = atoi (genindex);
	    if (inirp && j == irpnum)
	    {
	       strcat(token, irparg);
	    }
	    else
	    {
	       strcat (token, lclargs[j]);
	    }
	 }
	 else
	 {
	    strcat (token, cp);
	    done = TRUE;
	 }
      }
#ifdef DEBUGMACROEXP
      printf ("   genopcode = %s\n", genopcode);
#endif

      /*
      ** Process operand field
      */

      done = FALSE;
      genoperand[0] = '\0';
      token = genoperand;
      cp = mac->maclines[i]->operand;

      while (*cp && !done)
      {
	 if ((ocp = strchr(cp, '#')) != NULL)
	 {
	    if (ocp > cp)
	       strncat (token, cp, ocp-cp);
	    ocp++;
	    strncpy (genindex, ocp, 2);
	    ocp += 2;
	    cp = ocp;
	    genindex[2] = '\0';
	    j = atoi (genindex);
	    if (inirp && j == irpnum)
	    {
	       strcat(token, irparg);
	    }
	    else
	    {
	       strcat (token, lclargs[j]);
	    }
	 }
	 else
	 {
	    strcat (token, cp);
	    done = TRUE;
	 }
      }
      strcat (genoperand, "\n");
#ifdef DEBUGMACROEXP
      printf ("   genoperand = %s", genoperand);
#endif
      sprintf (genline, "%-6.6s %-7.7s %s", genlabel, genopcode, genoperand);
#ifdef DEBUGMACROEXP
      printf ("%s", genline);
#endif
      
      /*
      ** Lookup opcode for macro expansion control
      */

      op = oplookup (genopcode);

      /*
      ** If not skipping, IF[F/T], process
      */

      if (!asmskip)
      {
	 /*
	 ** Process GOTO in macro
	 */

	 if (gotoskip)
	 {
	    if (cursym[0] && !strcmp(cursym, gotosym))
	    {
#ifdef DEBUGGOTO
	       printf ("   cursym = '%s', gotosym = '%s'\n", cursym, gotosym);
#endif
	       gotoskip = FALSE;
	       goto GOTO_ENDS;
	    }
	    else
	    {
#ifdef DEBUGGOTO
	       printf ("   genopcode = '%s'\n", genopcode);
#endif
	       if (op && op->optype == TYPE_P && op->opvalue == IRP_T)
	       {
	          gotoskip = FALSE;
		  genmode = MACEXP | MACCALL;
	       }
	       else
	       {
		  genmode = MACEXP | SKPINST;
	       }
	       fwrite (&genmode, 1, 4, outfd);
	       fputs (genline, outfd);
	       linenum++;
	    }
	 }

	 /*
	 ** Not in GOTO, check for other controls
	 */

	 else
	 {
	 GOTO_ENDS:
	    /*
	    ** Check Pseudo ops that control macro
	    */

	    if (op && op->optype == TYPE_P)
	    {
	       char psopline[MAXSRCLINE];

	       switch (op->opvalue)
	       {
	       case IFF_T:
	       case IFT_T:
	       case GOTO_T:
		  genmode = MACCALL;
	       case SET_T:
		  genmode |= MACEXP;
		  fwrite (&genmode, 1, 4, outfd);
		  fputs (genline, outfd);
		  linenum++;
		  strcpy (psopline, genoperand);
		  cp = psopline;
		  p0pop (op, cp, infd, outfd, TRUE);
		  break;

	       case IRP_T:

		  /*
		  ** Process IRP here
		  */

		  genmode = (MACEXP | MACCALL);
		  fwrite (&genmode, 1, 4, outfd);
		  fputs (genline, outfd);
		  linenum++;

		  /*
		  ** If in IRP block, get next IRP arg
		  */

		  if (inirp)
		  {
		     irparg[0] = '\0';
		     strcpy (genoperand, irpargs);
		     cp = genoperand;

		     /*
		     ** If another IRP args, scan it off and loop
		     */

		     if (*cp)
		     {
			if (*cp == '(')
			{
			   int ii;

			   cp++;
			   ocp = cp;
			   ii = 1;
			   while (ii > 0 && !isspace(*cp))
			   {
			      if (*cp == '(')
			      {
				 ii++;
				 cp++;
			      }
			      else if (*cp == ')')
			      {
				 if (ii)
				 {
				    ii--;
				 }
			      }
			      else cp++;
			   }
#ifdef DEBUGMACROEXP
			printf ("   *cp = %c(%x), pocp = %s\n", *cp, *cp, ocp);
#endif
			   if (*cp == ')')
			   {
			      *cp++ = '\0';
			      if (*cp == ',') cp++;
			   }
			   strcpy (irparg, ocp);
			}
			else do {
			   cp = tokscan (cp, &token, &tokentype, &val, &term);
			   strcat (irparg, token);
			   if (term == ',') break;
			} while (*cp);
			strcpy (irpargs, cp);
#ifdef DEBUGMACROEXP
			printf ("   irpargs1 = %s\n", irpargs);
			printf ("   irparg1  = %s\n", irparg);
#endif
			i = irpline;
		     }
		     /*
		     ** No more args, exit loop
		     */
		     else
		     {
			inirp = FALSE;
		     }
		  }

		  /*
		  ** New IRP, get first arg and start loop
		  */

		  else
		  {
		     inirp = TRUE;
		     irpline = i;
		     irpnum = j;
		     irparg[0] = '\0';
		     cp = strchr (genoperand, '\n');
		     *cp = '\0';
		     cp = genoperand;
		     if (*cp == '(')
		     {
			int ii;

		        cp++;
			ocp = cp;
			ii = 1;
			while (ii > 0 && !isspace(*cp))
			{
			   if (*cp == '(')
			   {
			      ii++;
			      cp++;
			   }
			   else if (*cp == ')')
			   {
			      if (ii)
			      {
				 ii--;
			      }
			   }
			   else cp++;
			}
#ifdef DEBUGMACROEXP
		     printf ("   *cp = %c(%x), pocp = %s\n", *cp, *cp, ocp);
#endif
			if (*cp == ')')
			{
			   *cp++ = '\0';
			   if (*cp == ',') cp++;
			}
			strcpy (irparg, ocp);
		     }
		     else do {
			cp = tokscan (cp, &token, &tokentype, &val, &term);
			strcat (irparg, token);
		     } while (*cp && term != ',');
		     strcpy (irpargs, cp);
#ifdef DEBUGMACROEXP
		     printf ("   irpargs0 = %s\n", irpargs);
		     printf ("   irparg0  = %s\n", irparg);
#endif
		  }
		  break;

	       default:
		  genmode = MACEXP;
		  fwrite (&genmode, 1, 4, outfd);
		  fputs (genline, outfd);
		  linenum++;
	       }
	    }

	    /*
	    ** Check if macro is called in macro
	    */

	    else if ((lclmac = maclookup (genopcode)) != NULL)
	    {
	       genmode = (MACEXP | MACCALL);
	       fwrite (&genmode, 1, 4, outfd);
	       fputs (genline, outfd);
	       linenum++;
	       p0expand (lclmac, genoperand, infd, outfd);
	    }

	    /*
	    ** None of the above, send it on to next pass
	    */

	    else
	    {
	       genmode = MACEXP;
	       fwrite (&genmode, 1, 4, outfd);
	       fputs (genline, outfd);
	       linenum++;
	    }
	 }
      }

      /*
      ** In skip, decrement skip count
      */

      else
      {
	 genmode = MACEXP | SKPINST;
	 fwrite (&genmode, 1, 4, outfd);
	 fputs (genline, outfd);
	 linenum++;
	 asmskip--;
      }
   }

   asmskip = 0;
   gotoskip = FALSE;
}

/***********************************************************************
* p0ibsys - process IBSYS cards.
***********************************************************************/

static void
p0ibsys (char *bp, FILE *outfd)
{
   char *token;
   int genmode;
   int tokentype;
   int val;
   int oldtermparn;
   char term;

#ifdef DEBUGCTLCARD
   printf ("p0ibsys: %s", bp);
#endif

   /*
   ** Send control card to next pass
   */

   genmode = CTLCARD;
   fwrite (&genmode, 1, 4, outfd);
   fputs (bp, outfd);

   oldtermparn = termparn;
   termparn = FALSE;

   bp++;
   /*
   ** If not a comment, process
   */

   if (*bp != COMMENTSYM)
   {
      /*
      ** Scan off command
      */

      bp = tokscan (bp, &token, &tokentype, &val, &term);
#ifdef DEBUGCTLCARD
      printf (
	    "   token1 = %s, tokentype = %d, val = %o, term = %c(%x)\n",
	    token, tokentype, val, term, term);
#endif
      /*
      ** If $TITLE then scan off the title
      */

      if (!strcmp (token, "TITLE"))
      {
	 char *cp;

	 bp = &inbuf[VARSTART];
	 while (*bp && isspace (*bp)) bp++;
	 cp = bp;
	 while (*bp != '\n')
	 {
	    if (bp - inbuf > RIGHTMARGIN) break;
	    if (bp - cp == TTLSIZE) break;
	    bp++;
	 }
	 *bp = '\0';
	 strcpy (titlebuf, cp);
      }

      /*
      ** If $EXECUTE then scan and check if IBSFAP.
      */

      else if (!strcmp (token, "EXECUTE"))
      {
	 bp = &inbuf[VARSTART];
	 bp = tokscan (bp, &token, &tokentype, &val, &term);
	 if (!strcmp (token, "IBSFAP"))
	 {
	    fapmode = TRUE;
	    oldtermparn = FALSE;
	 }
      }

      /*
      ** If $IBMAP then scan and process the options.
      */

      else if (!strcmp (token, "IBMAP"))
      {
         bp = &inbuf[OPCSTART];

	 /*
	 ** Get the deckname
	 */

	 if (!isspace(*bp))
	 {
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
#ifdef DEBUGCTLCARD
	    printf (
	    "   token2 = %s, tokentype = %d, val = %o, term = %c(%x)\n",
	    token, tokentype, val, term, term);
#endif
	    if (strlen (token) > MAXSYMLEN) token[MAXSYMLEN] = '\0';
	    strcpy (deckname, token);
	    if (strlen (token) > 4) token[4] = '\0';
	    strcpy (lbl, token);
	    
	 }

	 /*
	 ** Parse the options
	 */

	 bp = &inbuf[VARSTART];
	 do {
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
	    if (term == '(')
	    {
	       strcpy (token, "(");
	       strncat (token, bp, 3);
	       bp += 3;
	       term = *bp;
	       if (*bp == ',') bp++;
	    }
#ifdef DEBUGCTLCARD
	    printf (
	    "   token3 = %s, tokentype = %d, val = %o, term = %c(%x)\n",
	    token, tokentype, val, term, term);
#endif
	    if (!strcmp (token, "ABSMOD"))
	    {
	       addext = FALSE;
	       absmod = TRUE;
	    }
	    else if (!strcmp (token, "ADDREL"))
	    {
	       addrel = TRUE;
	       addext = TRUE;
	       absmod = FALSE;
	    }
	    else if (!strcmp (token, "RELMOD"))
	    {
	       addext = TRUE;
	       absmod = FALSE;
	    }
	    else if (!strcmp (token, "JOBSYM"))
	    {
	       jobsym = TRUE;
	       definejobsyms();
	    }
	    else if (!strcmp (token, "MONSYM"))
	    {
	       monsym = TRUE;
	       definemonsyms(0);
	    }
	    else if (!strcmp (token, "NO()"))
	    {
	       oldtermparn = TRUE;
	    }
	    else if (!strcmp (token, "()OK"))
	    {
	       oldtermparn = FALSE;
	    }
	    else if (!strcmp (token, "REF"))
	    {
	       genxref = TRUE;
	    }
	 } while (term == ',');
      }
   }

   termparn = oldtermparn;
}

/***********************************************************************
* asmpass0 - Pass 0
***********************************************************************/

int
asmpass0 (FILE *tmpfd0, FILE *tmpfd1)
{
   MacDef *mac;
   char *token;
   int i;
   int done;
   int genmode;
   int status = 0;
   int tokentype;
   int val;
   char term;
   char opcode[MAXSYMLEN+2];

#ifdef DEBUGP0RDR
   printf ("asmpass0: Entered\n");
#endif

   /*
   ** Clear out macro table.
   */

   for (i = 0; i < MAXMACROS; i++)
   {
      memset ((char *)&macdef[i], '\0', sizeof(MacDef));
   }

   lblgennum = 0;
   macrocount = 0;

   /*
   ** Rewind the output.
   */

   if (fseek (tmpfd1, 0, SEEK_SET) < 0)
   {
      perror ("asm7090: Can't rewind temp file");
      return (-1);
   }

   /*
   ** Process the source.
   */

   pc = 0;
   linenum = 0;

   etcbuf[0] = '\0';
   qualindex = 0;
   linenum = 0;
   headcount = 0;
   done = FALSE;

   while (!done)
   {
      char *bp;

      srcbuf[0] = '\0';
      curpos = ftell (tmpfd0);
      if (etcbuf[0])
      {
         strcpy (srcbuf, etcbuf);
	 if (!eofflg && !fgets(etcbuf, MAXSRCLINE, tmpfd0))
	 {
	    etcbuf[0] = '\0';
	    eofflg = TRUE;
	 }
      }
      else
      {
	 if (!eofflg && !fgets(srcbuf, MAXSRCLINE, tmpfd0))
	 {
	    srcbuf[0] = '\0';
	    eofflg = TRUE;
	 }
	 else if (!eofflg && !fgets(etcbuf, MAXSRCLINE, tmpfd0))
	 {
	    etcbuf[0] = '\0';
	    eofflg = TRUE;
	 }
      }

      if (eofflg && !srcbuf[0])
      {
         done = TRUE;
	 status = 1;
	 break;
      }

      linenum++;
      genmode = 0;
      gotoskip = FALSE;
      exprtype = ADDREXPR;
      radix = 10;

      strcpy (inbuf, srcbuf);
#ifdef DEBUGP0RDR
      printf ("P0in = %s", inbuf);
#endif
      bp = inbuf;

      /*
      ** Check if IBSYS control card.
      */

      if (*bp == IBSYSSYM)
      {
         p0ibsys (bp, tmpfd1);
	 linenum--;
      }

      /*
      ** If not a comment, then process.
      */

      else if (*bp != COMMENTSYM)
      {
	 OpCode *op;

	 /*
	 ** If label present, add to symbol table.
	 ** On MAP/FAP the symbol can start in any column up to 6.
	 ** And FAP can have embedded blanks, eg. "( 3.4)"
	 */

	 if (strncmp (bp, "      ", 6))
	 {
	    char *cp;
	    char *dp;
	    char temp[8];

	    strncpy (temp, bp, 6);
	    cp = temp+6;
	    *cp-- = '\0';
	    while (*cp == ' ') *cp-- = '\0';
	    cp = dp = temp;
	    while (*cp)
	    {
	       if (*cp == ' ') cp++;
	       else *dp++ = *cp++;
	    }
	    *dp = '\0';

	    strcpy (cursym, temp);
#ifdef DEBUGCURSYM
	    printf ("asmpass0: cursym = %s\n", cursym);
#endif
	    bp += 6;
	    while (isspace (*bp)) bp++;
	 }
	 else 
	 {
	    cursym[0] = '\0';
	    while (isspace (*bp)) bp++;
	 }

	 /*
	 ** Scan off opcode.
	 */

	 if (!strncmp (&inbuf[OPCSTART], "   ", 3))
	 {
	    strcpy (opcode, "PZE"); /* Make a blank opcode a PZE */
	    bp = &inbuf[10];
	 }
	 else
	 {
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
	    strcpy (opcode, token);
	 }

	 /*
	 ** Check etcbuf for ETC
	 */

	 if (etcbuf[0])
	 {
	    if ((strlen(etcbuf) > OPCSTART+4) &&
	        !strncmp (&etcbuf[OPCSTART], "ETC", 3))
	    {
	       genmode |= CONTINU;
	    }
	 }

	 /*
	 ** Check opcodes
	 */

	 op = oplookup (opcode);

	 /*
	 ** If pseudo op, process
	 */

	 if (op && op->optype == TYPE_P)
	 {
	    /*
	    ** Set MACCALL for CALL, SAVE[N] and RETURN.
	    */

	    switch (op->opvalue)
	    {
	    case CALL_T:
	    case SAVE_T:
	    case SAVEN_T:
	    case RETURN_T:
	       genmode |= MACCALL;
	       break;
	    default: ;
	    }

	    fwrite (&genmode, 1, 4, tmpfd1);
	    fputs (srcbuf, tmpfd1);
	    done = p0pop (op, bp, tmpfd0, tmpfd1, FALSE);
	    if (done) status = 0;
	 }

	 /*
	 ** Else, Check macro table
	 */

	 else if ((mac = maclookup (opcode)) != NULL)
	 {
	    genmode |= MACCALL;
	    fwrite (&genmode, 1, 4, tmpfd1);
	    fputs (srcbuf, tmpfd1);
	    p0expand (mac, bp, tmpfd0, tmpfd1);
	 }

	 /*
	 ** None of the above, send on
	 */

	 else
	 {
	    pc++;
	    fwrite (&genmode, 1, 4, tmpfd1);
	    fputs (srcbuf, tmpfd1);
	 }
      }

      /*
      ** A comment, in FAP there is some meaning.
      */

      else
      {
	 fwrite (&genmode, 1, 4, tmpfd1);
	 fputs (srcbuf, tmpfd1);
	 if (fapmode && !titlebuf[0])
	 {
	    char *cp;

	    bp++;
	    while (*bp && isspace(*bp)) bp++;
	    cp = bp;
	    bp = tokscan (bp, &token, &tokentype, &val, &term);
	    if (!strcmp (token, "FAP")) ;
	    else if (!strcmp (token, "ID")) ;
	    else
	    {
	       while (*bp != '\n')
	       {
		  if (bp - inbuf > RIGHTMARGIN) break;
		  if (bp - cp == TTLSIZE) break;
		  bp++;
	       }
	       *bp = '\0';
	       strcpy (titlebuf, cp);
	    }
	 }
      }

   }

   /*
   ** Clean up
   */

   for (i = 0; i < macrocount; i++)
   {
      int j;
      for (j = 0; j < macdef[i].maclinecount; j++)
      {
         macdef[i].maclines[j]->next = freemaclines;
	 freemaclines = macdef[i].maclines[j];
      }
   }

#ifdef DEBUGMULTIFILE
   printf ("asmpass0: %d lines in file\n", linenum);
#endif

   return (status);
}
