/* ntutl.c */

/*
 *  Includes
 */
#include <stdio.h>
#include <string.h>

#include "debug.h"
#include "dfault.h"
#include "evtdef.h"
#include "hstdef.h"
#include "prodef.h"
#include "tcpdat.h"
#include "ntutl.h"
#include "mapskt.h"
#include "bytprc.h"
#include "cticks.h"
#include "suspnd.h"

/*
 *  transq
 *
 *  Transmit the entire queue (window) to the other host without expecting
 *  any sort of acknowledgement.
 */
int transq(skt)
register struct socket *skt;
{
	register struct window *w;
	register unsigned int i,j;
	int k,n;
	char *tqrdptr;
	unsigned int bytes,qbytes;
	int offset;

	if(skt==NULL || skt==-1) {
		/*
		 * NULL port for trans
		 */
		skposterr(406);
		return(-1);
	}
	w = &skt->out;
	/*
	 *  limit how many bytes we may send to
	 *  what the other side will allow us
	 *  to send (window) or to what we have
	 */
	offset = outqoff(skt);
	qbytes = quedata(w) - offset;
	bytes = w->size - offset;
	if(qbytes<bytes)
		bytes = qbytes;
	/*
	 *  set up the tcp packet for this,
	 *  ACK field is same for all packets
	 */
	skt->tcpout.t.ack = longswap(skt->in.nxt);
	/*
	 * clear push & urgent flags
	 */
	skt->tcpout.t.flags &= ~(TPUSH | TURG);
	/*
	 * if no data to send . . .
	 */
	if((bytes<=0) || skt->state!=SEST) {
		/*
		 * just a retransmission or ACK
		 */
#ifdef	DEBUGOPTION
if(ndebug&0x10) {
	printf("transq: ack\r\n");
}
#endif
		tcpsend(skt,0);
		return(0);
	}
	/*
	 *  we have data to send, get the correct sequence #'s
	 *  To be really real, we should check wraparound
	 *  sequence # in the loop.
	 */
	j = w->rdptr - w->where;
	j += offset;
	if(j >= QUEUESIZE) {
		j -= QUEUESIZE;
	}
	tqrdptr = w->where + j;
	/*
	 * is push indicator on?
	 */
	if(w->push) {
		skt->tcpout.t.flags |=  TPUSH;
	}
	/*
	 * is urgent flag set?
	 */
	if(w->urgent) {
		skt->tcpout.t.flags |=  TURG;
	}
	/*
	 *  in a loop, transmit the entire queue of data
	 *  limited by the maximum packet count per transq
	 *
	 *  (1)	   if the push flag is set then send all that we can
	 *  (2)    else send only full packets
	 */
	if(skt->out.push) {
		k = nnpkt;
	} else {
		k = bytes/skt->sendsize;
		if(k>nnpkt) {
			k = nnpkt;
		}
	}
	for(i=0; i<bytes && k>0; i+=skt->sendsize,--k) {
		n = skt->sendsize;
		if(i+n>bytes)
			n = bytes-i;
		j = tqrdptr - w->where;
		j = QUEUESIZE - j;
		if(j<n) {
			movebytes(skt->tcpout.x.data,tqrdptr,j);
			movebytes((skt->tcpout.x.data+j),w->where,n-j);
			tqrdptr = w->where + n - j;
		} else {
			movebytes(skt->tcpout.x.data,tqrdptr,n);
			if(j==n) {
				tqrdptr = w->where;
			} else {
				tqrdptr += n;
			}
		}
#ifdef	DEBUGOPTION
if(ndebug&0x10) {
	printf("transq: seq=0x%x, cnt=%d\n", w->nxt,n);
}
#endif
		/*
		 * Load urgent data pointer
		 */
		if(skt->tcpout.t.flags & TURG) {
			skt->tcpout.t.urgent = intswap(n);
		} else {
			skt->tcpout.t.urgent = 0;
		}
		/*
		 * send it
		 */
		tcpsend(skt,n);
		w->nxt += n;
	}
#ifdef	DEBUGOPTION
if(ndebug&0x10) {
	printf("transq: done\r\n");
}
#endif
	return(0);
}

/*
 *  ntclose
 *
 *  Start the closing process on socket sknum.
 */
int ntclose(sknum)
register int sknum;
{
	register struct socket *skt;

	skt = (struct socket *) skvalid(sknum);
	if(skt == -1)
		return(-1);

#ifdef	DEBUGOPTION
if(ndebug&0x01) {
	printf("ntclose: sknum=%d, skt->state(in)=%d,", sknum, skt->state);
}
#endif

	switch (skt->state) {
		default:
		case SLISTEN:
		case SSYNS:
			/*
			 * SLISTEN	termination
			 * SSYNS	termination
			 */
			skt->state = SCLOSED;
			break;

		case SSYNR:
		case SEST:
			skt->tcpout.t.flags = TFIN|TACK;
			tcpsend(skt,0);
			skt->state = SFW1;
			break;

		case SFW1:
		case SFW2:
			/*
			 * SFW1		missing FIN & ACK
			 * SFW2		missing FIN
			 */
			if(skt->out.lasttime != 0 &&
			   elapsed(skt->out.lasttime) > WAITTIME) {
				skt->tcpout.t.flags = TFIN|TACK;
				tcpsend(skt,0);
				skt->state = STWAIT;
			}
			break;

		case SCLOSING:
		case STWAIT:
			/*
			 * SCLOSING	missing ACK
			 * STWAIT	timeout
			 */
			if(skt->out.lasttime != 0 &&
			   elapsed(skt->out.lasttime) > WAITTIME) {
				skt->state = SCLOSED;
			}
			break;

		case SCWAIT:
			skt->tcpout.t.flags = TFIN|TACK;
			tcpsend(skt,0);
			skt->state = SLAST;
			break;

		case SLAST:
			if(skt->out.lasttime != 0 &&
			   elapsed(skt->out.lasttime) > LASTTIME) {
				skt->state = SCLOSED;
			}
			break;
	}

#ifdef	DEBUGOPTION
if(ndebug&0x01) {
	printf(" skt->state(out)=%d\r\n", skt->state);
}
#endif

	return(0);
}

/*
 * Compute elapsed time
 */
long elapsed(ltm)
long ltm;
{
	long el;

	el = cticks(NULL)-ltm;
	if(el < 0)
		el += WRAPTIME;
	return(el);
}
                                                                                                                                                                                                                                                                                                                     