/*	bit.h

(See also: bits.c)

Usage:

	int
	tst_bit(bit_array, bit_number)
	int	bit_array[];	Array (field) of bit flags
	int	bit_number;	Number of bit to reference

	int
	set_bit(bit_array, bit_number)

	int
	clr_bit(bit_array, bit_number)

	int
	flip_bit(bit_array, bit_number)

>>> CAUTION <<< No space allowed between macro name and '('.

Description:

These are bit manipulation macros in lieu of support for bit fields.

tst_bit tests the selected bit to see if it is set.
It provides the value of WORD & BIT, where WORD is the word in the
bit array that contains the selected bit, and BIT is a bit mask with only
the selected bit set.  Thus the macro tests TRUE if the selected bit
is set; FALSE otherwise.

set_bit sets the selected bit.
clr_bit clears the selected bit.
flip_bit reverses the sense of the selected bit.

>>> Note <<<  The bit_array must be word aligned (character arrays are
acceptable only if the array starts on a word boundary).

The bit numbering is right to left ordered:
	0 is LSB of the first word of the array,
	15 is the MSB of the first word of the array,
	16 is the LSB of the second word of the array,
	31 is the MSB of the second word of the array,
	etc.

    >>> CAUTION <<< This bit numbering is the opposite of that used in bits.c

******************************************************************************/

/* >>> WARNING <<<  A 16 bit word size is assumed. */

#define	WORD_IN_ARRAY(bit_array, bit_number)	(\
    (int)bit_array[(unsigned)(bit_number) >> 4]\
    )
#define	BIT_IN_WORD(bit_number)			(\
    1 << ((bit_number) & 0xF)\
    )

#define	tst_bit(bit_array, bit_number)		(\
    WORD_IN_ARRAY(bit_array, bit_number) & BIT_IN_WORD(bit_number)\
    )

#define	set_bit(bit_array, bit_number)		(\
    WORD_IN_ARRAY(bit_array, bit_number) |= BIT_IN_WORD(bit_number)\
    )

#define	clr_bit(bit_array, bit_number)		(\
    WORD_IN_ARRAY(bit_array, bit_number) &= ~BIT_IN_WORD(bit_number)\
    )

#define	flip_bit(bit_array, bit_number)		(\
    WORD_IN_ARRAY(bit_array, bit_number) ^= BIT_IN_WORD(bit_number)\
    )

