#
/*
 * PROPRIETARY INFORMATION.  Not to be reproduced, transmitted, or disclosed
 * in any way without permission.
 *
 * UNIX RT-11 SJ Emulator.
 *
 * Produced by Human Computing Resources Corp.,
 * 10 St. Mary Street, Toronto, Ont. Canada
 *
 * Version:	2		Date:	January 1979
 * Author:	Mike Tilson
 * Description:
 *	This routines implement RT-11 channel I/O (i.e. do not include
 *	non-channel TTY I/O.)
 * Revisions:
 */
#include "defns.h"
#include "decls.h"
#include "io.h"
#include "stdaddrs.h"
#include "errors.h"

char rtlib[] "/lib/rt11/";
char tmpsuffix[] ":tmp";
char spooler[] "/bin/opr";
char sy[] "sy";
char dk[] "dk";
char lp[] "lp";
char tt[] "tt";
char nl[] "nl";
char devnull[] "/dev/null";

busychan(channel) {
	return(valid(channel) && (chan[channel].flags&BUSY)==BUSY);
}

valid(channel) {
	return(channel>=0 && channel<=NCHAN);
}

#define EOFCHAR	'\032'	/* rt-11 eof indicator */

/*
 * device info
 *
 * several glitches:
 * 1) SY is always an RK05
 * 2) device size is always huge
 */
struct devinfo devinfo[] {
	sy,	RK05|RANDOM,		MAXSIZE,
	dk,	RK05|RANDOM,		MAXSIZE,
	lp,	WRT_ONLY|LP11,		0,
	tt,	CONSOLE,		0,
	nl,	NLDEV,			0,
	NULL,
};
struct devinfo *dinfo(dev)
	rad50 dev;
{
	register struct devinfo *d;

	convasc(dev, filename);
	filename[0] = lcase(filename[0]);
	filename[1] = lcase(filename[1]);
	filename[2] = lcase(filename[2]);
	doassign();
	for(d=devinfo; d->devnm!=NULL; d++) {
		if(checkdev(d->devnm)) {
			return(d);
		}
	}
	return(NULL);
}


delete(achannel, r0)
	word *r0;
{
	register int channel;

	channel = achannel;
	if(!valid(channel) || busychan(channel)){
		errcode(0);
		return;
	}
	/* opening the file is the easiest way to check it */
	if(!findfile(channel, r0[1])) {
		errcode(1);
		return;
	}
	rtclose(channel);
	/* ascii version of filename left by findfile */
	if(unlink(filename)<0){
		warning(NOUNLINK, filename);
		errcode(2);
	}
}

enter(achannel, r0)
	word *r0;
{
	register int channel;

	channel = achannel;
	if(!valid(channel) || busychan(channel)) {
		errcode(0);
		return;
	}
	entering = YES;
	elen = r0[2];
	if(!findfile(channel, r0[1])) {
		/*
		warning(CANTCREAT, filename);
		*/
		errcode(1);
	}
	entering = NO;
	retval = YES;
	rvalue = chan[channel].fsize;
}

lookup(achannel, r0)
	word *r0;
{
	register int channel;

	channel = achannel;
	if(!valid(channel) || busychan(channel)) {
		errcode(0);
	}
	else if(!findfile(channel, r0[1])) {
		errcode(1);
	}
}

rename(achannel, r0)
	word *r0;
{
	register dblk *d;
	register int i;
	register int channel;
	word f;
	char oldname[sizeof filename];

	channel = achannel;
	if(!valid(channel) || busychan(channel)) {
		errcode(0);
		return;
	}
	d = r0[1]&~01;	/* see comment in findfile re this bit */
	if(!findfile(channel, &d[0])) {
		errcode(1);
		return;
	}
	f = chan[channel].flags;
	copy(filename, oldname);
	rtclose(channel);
	if(f&SYSDISK) {
		/* didn't really find the file, just found it in last resort directory */
		errcode(1);
		return;
	}
	convert(&d[1]);
	/* cross device rename not allowed */
	for(i=0; i<3; i++) {
		if(oldname[i]!=filename[i]) {
			errcode(2);
			return;
		}
	}
	if(filename[4]=='\0') {	/* must not move to a directory */
		errcode(2);
		return;
	}
	if(!move(oldname, filename))
		errcode(2);
}

/* this structure is used to access the block of memory used
 * by savestatus/reopen.  If no programs ever peeked into the
 * memory, we would simply stuff in the filename for future
 * reference in reopen.  however, PIP does look, so we move
 * a few bits around to try and fool it.
 */
struct {
	dblk	filler;
	byte	filler2;
	byte	subdev;
};
#define devindex	device.lobyte
#define SUBDEV	2

savestatus(achannel, r0)
	word *r0;
{
	register dblk *d, *p;
	register int channel;

	channel = achannel;
	if(!busychan(channel)) {
		errcode(0);
		return;
	}
	if(chan[channel].flags&ENTERED) {
		errcode(1);
		return;
	}
	d = r0[1];
	p = &chan[channel].filename;
	d->device = p->device;
	d->fname[0] = p->fname[0];
	d->fname[1] = p->fname[1];
	d->extension = p->extension;
	convert(p);
	d->subdev = filename[SUBDEV]-'0';		/* keep PIP happy */
	d->filler2 = d->devindex;
	d->devindex = (dinfo(p->device)-devinfo)<<1;	/* keep PIP happy */
	rtclose(channel);
}

reopen(achannel, r0)
	word *r0;
{
	register int channel;
	register word d;
	register word sav;

	channel = achannel;
	if(!valid(channel) || busychan(channel)) {
		errcode(0);
		return;
	}
	d = r0[1]&~01;
	sav = d->devindex;
	d->devindex = d->filler2;
	lookup(channel, r0);	/* status was saved so that this works */
	d->devindex = sav;
	if(rt_error) {
		error(NOREOPEN, filename, rt_error);
	}
}

/*
 * RT11 read
 */
rtread(channel, r0)
	word *r0;
{
	rtio(0, channel, r0);
}

/*
 * RT11 write
 */
rtwrite(channel, r0)
	word *r0;
{
	rtio(1, channel, r0);
}

/*
 * rtio -- do a read or write
 */
boolean tty_eof;	/* set if "tt:" driver has seen an EOF */

rtio(wrflg, channel, r0)
	word *r0;
{
	register int len, reqlen;
	register struct chan *c;
	int f;
	byte *p;
	int chr;

	c = &chan[channel];
	if(!(c->flags&BUSY)) {
		errcode(2);
		return;
	}
	/*
	 * since there are fewer available UNIX file descriptors than there
	 * are RT-11 channels, we may have to reopen the file descriptor
	 * associated with this channel.
	 */
	if(!(c->flags&OPEN) && !(c->flags&TTY)) {
		getfdes();
		if(!findfile(channel, &(c->filename))) {
			warning(FILEGONE, filename);
			errcode(1);
			return;
		}
	}
	if(wrflg && c->fsize!=0 && unsgncmp(r0[1],c->fsize)>=0) {
		len = 0;
		errcode(0);
		goto Iodone;
	}
	f = c->fdes;
	if(f==STDIN && wrflg)
		f = STDOUT;
	if((c->flags&CHARDEV) == 0)
		seek(c->fdes, r0[1], 3);
	reqlen = r0[3]*2;
	p = r0[2];
	if(wrflg) {
		/* writing */

		/* check for read-only */
		if(c->flags&DIRECTORY) {
			errcode(0);
			len = 0;
		}
		else if(c->flags&TTY) {
			/* SPR 007 -- pass writes to TT: to putchar */
			/* SPR 014 -- use unsigned compare to control loop */
			for(len=0; unsgncmp(len, reqlen)<0; len++)
				if(p[len])
					putchar(p[len]);
		}
		else
			len = write(f, p, reqlen);
	}
	else {
		/* reading */

		/* check for writeonly devices */
		if(c->flags&PRINTER) {
			errcode(0);
			len = 0;
		}
		else if(c->flags&DIRECTORY) {
			/* fake an rt-11 directory read */
			convert(&c->filename);
			append("/", filename);
			len = diread(f, r0[1], reqlen, p);
		}
		else if(c->flags&TTY) {
			/* read from tt is a special case in order
			 * to simulate undesirable behavior of rt-11
			 */
			len = 0;
			if(!tty_eof)
				putchar('^');
			while(unsgncmp(len,reqlen)<0 && !tty_eof) {
				chr = getchar(f);
				if(chr == EOF)
					tty_eof = YES;
				else if(chr=='^') {
					/* accept ascii ^Z to mean EOFCHAR,
					 * for indirect files
					 */
					getchar(f);	/* ignore next char */
					tty_eof = YES;
				}
				else
					p[len++] = chr;
			}
		}
		else
			len = read(f, p, reqlen);
	}
	if(len == -1)
		errcode(1);
	else if(len==0)
		errcode(0);
	else if(unsgncmp(len,reqlen)<0) {
		if(wrflg)
			errcode(1);
		else
			zero(reqlen-len, p+len);
	}

Iodone:
	if(len == -1)
		len = 0;
	else if((c->flags&CHARDEV) == 0)
		len = (len+511)&~511;
	len = (len>>1)&077777;
	if(r0[4]!=0 && r0[4]!=1)
		completion(r0[4], len==0?EOFBIT:0, channel);
	retval = YES;
	rvalue = len;
}

rtclose(channel)
{
	char name2[sizeof filename];
	register struct chan *c;

	c = &chan[channel];
	if(!busychan(channel))
		return;
	cclose(c);
	if(c->flags&ENTERED) {
		convert(&(c->filename));
		append(tmpsuffix, filename);
		if(c->flags&SYSDISK) {
			insertdev(dk);
			prepend(rtlib, filename);
		}
		else
			insertdev(dk);
		copy(filename, name2);
		name2[length(filename)-4] = '\0';	/* remove :tmp */
		if(!move(filename, name2))
			warning(BAD_RENAME, name2);
	}
	c->flags = 0;
}

cclose(ac)
	struct chan *ac;
{
	register struct chan *c;

	c = ac;
	if(c->flags&OPEN) {
		if(!(c->flags&TTY)) {
			close(c->fdes);
			nopen--;
		}
		else
			/* this does not always work.  tty_eof should
			 * be cleared when the last tt: channel open
			 * for input has been closed, but that is hard
			 * to determine.  this flaw will probably never
			 * be noticed.
			 */
			tty_eof = NO;
		if(c->flags&PRINTER)
			wait();		/* don't allow dead spooler procs
					 * to hang around
					 */
	}
}


purge(channel)
{
	register struct chan *c;
	if(!busychan(channel))
		return;
	c = &chan[channel];
	cclose(c);
	if(c->flags&ENTERED) {
		convert(&c->filename);
		append(tmpsuffix, filename);
		unlink(filename);
	}
	c->flags = 0;
}

rtwait(channel)
{
	/* since all emulated I/O is "instantaneous", there is not much to do */
	if(!busychan(channel))
		errcode(0);
}

purgeall()
{
	register int i;
	for(i=0; i<NCHAN; i++)
		if(!(chan[i].flags&OVERLAY))
			purge(i);
}

/*
 * device name assignment
 */

#define SIZEATAB	4	/* if changed here, change in rtcom as well */
struct atab {
	char aname[3];	/* assigned device name */
	char iname[3];	/* internal device name */
} atab [SIZEATAB];

/*
 * assign a "physical" device name to a logical name
 */
assign(internal, external)
	char *external, *internal;
{
	register char *ext, *in;
	register struct atab *a;

	ext = external;
	in = internal;
	for(a=atab; a!= &atab[SIZEATAB]; a++) {
		if(a->aname[0]==NULL) {
			a->aname[0] = *ext++;
			if(a->aname[1] = *ext++)
				a->aname[2] = *ext++;
			else {
				a->aname[1] = ' ';
				a->aname[2] = '\0';
			}
			a->iname[0] = *in++;
			if(a->iname[1] = *in++)
				a->iname[2] = *in++;
			else {
				a->iname[1] = ' ';
				a->iname[2] = '\0';
			}
			return;
		}
	}
	error(ATABFULL);
}

/*
 * convert filename from logical to "physical" device name
 */
doassign() {
	register struct atab *a;
	register char *f;

	f = filename;
	for(a=atab; a!= &atab[SIZEATAB]; a++) {
		if(a->aname[0]=='\0')
			break;
		if(a->aname[0]==f[0] && a->aname[1]==f[1] &&
		   (a->aname[2]==f[2] || (a->aname[2]=='\0' && (f[2]==' ' || digit(f[2]))))) {
			f[0] = a->iname[0];
			f[1] = a->iname[1];
			if(!digit(f[2])) {
				if(a->iname[2]=='\0')
					f[2] = ' ';
				else
					f[2] = a->iname[2];
			}
			break;
		}
	}
	if(f[2]==' ')
		f[2] = '0';
}



/*
 * fetch/releas
 */
fetch(dev, addr)
	rad50 *dev;
	word addr;
{
	retval = YES;
	rvalue = addr;
	if(dinfo(*dev)==NULL)
		errcode(0);
}

/*
 * device status
 */

dstatus(dev, dstblk)
	word	*dstblk;
	rad50	*dev;
{
	register struct devinfo *d;
	register word *dst;

	dst = dstblk;
	if((d=dinfo(*dev)) == NULL)
		errcode(0);
	else {
		*dst++ = d->devst;
		*dst++ = 2;	/* 2 byte handler size */
		*dst++ = 1;	/* give him odd address fault if he tries
				 * to use device handler address.
				 */
		*dst++ = d->devsize;
	}
}




/*
 * I/O Utilities
 */

/*
 * look up and open a file
 */
findfile(channel, rd)
	word rd;	/* should be "dblk *rd", but must bit twiddle */
{
	register dblk *d;
	register int i;
	register struct chan *c;
	int pipdes[2];

	/* Note:  there is an undocumented feature of rt-11 that
	 * uses the low-order bit as some sort of flag.  I don't
	 * know what it is, but PIP uses it.  If you clear the bit
	 * PIP does not seem to notice, so ????.
	 */
	d = rd&~01;

	c = &chan[channel];
	/* findfile may be called to internally reopen a file, in which
	 * case c==d and we must be careful
	 */
	if(&(c->filename) != d)
		zero(sizeof c->filename, &(c->filename));

	c->filename.device = d->device;
	/* we do not save away the rest of the name until later, since
	 * this saved data may get stuffed into a SAVESTATUS block.  if
	 * we are dealing with a non-file structured device, we will want
	 * the second word of the block to be zero (c->filename.fname[0]).
	 * ordinary users are not supposed to snoop into SAVESTATUS blocks,
	 * but PIP does.  the problem arises because we want to save
	 * completely different information than RT-11 does, but PIP
	 * uses SAVESTATUS as a means of getting info about files.  see
	 * savestatus below.
	 */
	convert(d);
	if(filename[SUBDEV]<'0' || filename[SUBDEV]>'7') {
		/* invalid device (after assignment in convert) */
		return(NO);
	}

	/* the following to make PIP happy, since it insists on
	 * snooping in the system.
	 * The following is essentially the "major" and "minor"
	 * devices, for channel 016 only (but we always set it,
	 * since nobody but pip looks at it.)
	 */
	setval(INTRNCSW, (dinfo(d->device)-devinfo)<<1);
	setval(INTRNCSW+010, (filename[SUBDEV]-'0')<<8);

	if(entering || (c->flags&ENTERED))
		append(tmpsuffix, filename);
	getfdes();
	c->fsize = 0;
	if(checkdev(lp)) {
		/* Line printer -- invoke spooler */
		makpipe(pipdes);
		switch(fork()) {
		case -1:
			warning(NOLPFORK);
			close(pipdes[0]);
			close(pipdes[1]);
			c->fdes = open(devnull, 1);
			break;
		default:	/* parent proc */
			close(pipdes[0]);
			c->fdes = pipdes[1];
			break;
		case 0:		/* child */
			close(0);
			dup(pipdes[0]);
			close(pipdes[0]);
			close(pipdes[1]);
			for(i=3; i<14; i++)
				close(i);
			execl(spooler, spooler, 0);
			error(NOSPOOL);
			exit();
		}
		c->flags =| BUSY|OPEN|PRINTER;
		c->fsize = MAXSIZE;
		nopen++;
		return(YES);
	}
	else if(checkdev(tt)) {
		/* "TTY" -- use standard input or output or tty */
		c->flags =| BUSY|TTY|OPEN;
		c->fdes = terminal;
		c->fsize = MAXSIZE;
		return(YES);
	}
	else if(checkdev(nl)) {
		c->flags =| OPEN|BUSY;
		c->fdes = open(devnull, 2);
		c->fsize = MAXSIZE;
		nopen++;
		return(YES);
	}
	else if(checkdev(sy) || checkdev(dk)) {
		/* have a file structured device, so save away the whole name */
		c->filename.fname[0] = d->fname[0];
		c->filename.fname[1] = d->fname[1];
		c->filename.extension = d->extension;
		insertdev(dk);
		if(openfile(channel))
			return(YES);
		prepend(rtlib, filename);
		if(openfile(channel)) {
			c->flags =| SYSDISK;
			return(YES);
		}
	}
	return(NO);
}

/* some UNIX systems have a bug in pipe.s, which does not save R2.
 * This encapsulates pipe in its own routine.
 */
makpipe(pipdes)
	int *pipdes;
{
	if(pipe(pipdes)<0)
		abend(NOFILES);
}

/*
 * open a given file name 
 */
openfile(channel) {
	register int f;
	register struct chan *c;
	struct statbuf sbuf;

	c = &chan[channel];
	if(entering) {
		/* enter on dk: (non-file-structured) not allowed */
		if(c->filename.fname[0]==0)
			return(NO);
		if((f=creat(filename, 0666)) < 0)
			return(NO);
		close(f);
		if((f=open(filename, 2)) < 0)
			abend(FATALREOPEN, filename);
		c->flags =| ENTERED;
		if(elen==0 || elen == -1)
			c->fsize = MAXSIZE;
		else
			c->fsize = elen;
	}
	else {
		/* try to open for read-write, but allow open for read
		 * only, if that is all that is allowed.  writes will
		 * then give errors.
		 */
		if((f=open(filename, 2))<0 && (f=open(filename, 0))<0)
			return(NO);
		fstat(f, &sbuf);
		c->fsize = sbuf.size1>>9;
		c->fsize =+ (sbuf.size0&0377)<<7;
		if(sbuf.size1&0777)
			c->fsize++;
		if((sbuf.flgs&060000)==040000)
			c->flags =| DIRECTORY;
	}
	c->flags =| OPEN|BUSY;
	c->fdes = f;
	nopen++;
	return(YES);
}

/*
 * make a file descriptor available
 */
getfdes() {
	register struct chan *c;

	if(nopen<NFILES)
		return;
	for(c = &chan[0]; c < &chan[NCHAN]; c++) {
		if((c->flags&OPEN) && !(c->flags&(PRINTER|TTY|OVERLAY))) {
			close(c->fdes);
			c->flags =& ~OPEN;
			nopen--;
			return;
		}
	}
	abend(NOFILES);
}

/*
 * patch in a new "device name" in the filename
 */
insertdev(d)
	char *d;
{
	filename[0] = d[0];
	filename[1] = d[1];
}

/*
 * see if the filename starts with the given "device"
 */
checkdev(d)
	char *d;
{
	if(filename[0] == d[0] && filename[1] == d[1])
		return(YES);
	return(NO);
}

/*
 * convert a radix 50 filename into a unix path name
 */
convert(rr)
	dblk *rr;
{
	register dblk *r;
	register char *a;

	r = rr;
	a = filename;

	convasc(r->device, a);
	a =+ 3;
	*a++ = '/';
	if(r->fname[0]!=0) {
		convasc(r->fname[0], a);
		convasc(r->fname[1], &a[3]);
		a[6] = '\0';
		/* strip trailing blanks */
		while(*a)
			if(*a == ' ')
				break;
			else
				a++;
		*a++ = '.';
		convasc(r->extension, a);
		a[3] = '\0';
		while(*a)
			if(*a == ' ')
				break;
			else
				a++;
	}
	*a = '\0';
	/* convert to lower case */
	for(a=filename; *a != '\0'; a++)
		*a = lcase(*a);
	doassign();
}


/*
 * rename a file
 */
move(fa, fb)
	char *fa, *fb;
{
	register char *filea, *fileb;

	filea = fa;
	fileb = fb;

	if(equal(filea,fileb))
		return(YES);
	unlink(fileb);
	if(link(filea, fileb) < 0)
		return(NO);
	if(unlink(filea)<0)
		return(NO);
	return(YES);
}
