/*
 * PROPRIETARY INFORMATION.  Not to be reproduced or transmitted in any
 * way without permission.
 *
 *
 * RT-11 V3 Command Interpreter for UNIX.
 * Version 2.0, May 1979
 *
 * Produced by:
 *	Mike Tilson
 *	Human Computing Resources Corporation
 *	10 St. Mary Street
 *	Toronto, Ontario, Canada   M4Y 1P9
 *	416-922-1937
 *
 *
 * This is the guts of the command simulator.
 *
 * The general algorithm is:
 *	1.  Parse command to get overall structure into parse tree.
 *
 *	2.  Lookup command name, set up flags and defaults.
 *
 *	3.  Build a CSI input string:
 *		"DEV:FILE1,DEV:FILE2,DEV:FILE3=DEV:FILE4,.../X/Y:VAL/Z"
 *	    We maintain the output files (left hand side), input files,
 *	    and switch values which apply to an entire command line
 *	    separately.  They are merged into one CSI string at the end.
 *
 *	4.  Execute the desired program.   Usually this means calling the
 *	    RT-11 emulator, although some commands are executed internally.
 */

#include "defns.h"
#include "decls.h"

int toplevel;	/* set if we are called as a top level command interpreter */
int flags;	/* command flags */
int optflags;	/* option flags */
int delfiles;	/* output files to be suppressed */
/*
 * genfiles specifies the output files to be generated from
 * input names.  It consists of three 4-bit fields.  The top bit
 * in the field indicates that a file is to be generated.
 * The next three bits indicate which input argument is to be
 * used to generate the output name.
 */
#define GENSHIFT	4
#define GENMASK		017
#define DOGEN		010
#define GENWHICH	007
int genfiles;

/* to save state for multi-line commands */
int savdelfiles;
int savgenfiles;
char savoutstring[STRLEN+1];
char savcomopts[STRLEN+1];

char outstring[STRLEN+1];	/* string of outputs */
char filestr[STRLEN+1];		/* used to build up a file spec */
char instring[STRLEN+1];	/* string of inputs */
char geninstr[STRLEN+1];	/* saved file name for GENINPUT flag */
char comopts[STRLEN+1];		/* CSI switches associated with entire command */
char commstr[20];		/* name of program which executes command */

int ninargs;		/* number of inputs supplied */
int inmax;		/* max number of inputs */
int noutargs;		/* number of outputs */
int outmax;		/* max number of outputs */
int ncsi;		/* number of CSI lines generated:  MACRO A+B,C gives 2 */
#define MAXCSI	10
char *csistr[MAXCSI];

int exambase;	/* base for "E" command */
char *defeditor;	/* for SET EDIT */
int emsg;		/* for SET MSG ERRS extension to command language */
int wmsg;		/* for SET MSG WARN */

/*
 * device name assignment
 */

#define SIZEATAB	4	/* if changed here, change in ru as well */
struct atab {
	char aname[4];	/* assigned device name */
	char iname[4];	/* internal device name */
} atab [SIZEATAB];


struct {
	char lobyte;
	char hibyte;
};



main(argc, argv)
	char **argv;
{
	register c;
	register int f;

	if(argv[0][0]=='-')
		toplevel++;
	while(argc>1 && argv[1][0]=='-') {
		switch(argv[1][1]) {
		case 'd':
			debug = YES;
			break;
		default:
			usage();
		}
		argc--;
		argv++;
	}
	if(argc>1)
		usage();

	initialize();
	/*
	 * execute the startup indirect file
	 */
	if((f=openfile("starts", ".com")) != ERROR)
		doindirect(f);
	/* main loop */
	do {
		prompt(COMMAND);
		while((c=getch())!=EOF && (c==' ' || c=='\t'))
			;
		if(c!='\n' && c!=EOF) {
			ungetch(c);
			yyparse();
			while((c=getch())!=EOF && c!='\n')
				;
			if(errors==0)
				traverse(root);
		}
		cleanup();
	} while(c!=EOF);
	terminate();
}

usage() {
	printf("Usage:  rtcom [-d]\n");
	exit();
}

initialize() {
	savintr = signal(INTR, IGNORE);
	savquit = signal(QUIT, IGNORE);
	if(toplevel) {
		savintr = 0;
		savquit = 0;
		close(2);
		dup(1);
	}
	yyinit();
	cleanup();
}

terminate() {
	if(!debug)
		unlink("core");
	temps(DELTEMPS);
	putchar('\n');
	exit(0);
}

/*
 * cleanup -- called at the end of every command execution.
 */
cleanup() {
	if(errors)
		while(inpsp>0) {
			close(input[inpsp]);
			popinpsp();
		}
	errors = 0;
	longmsg = 0;
	ninargs = 0;
	noutargs = 0;
	ncsi = 0;
	defdevnest = 0;
	release();
	instring[0] = '\0';
	outstring[0] = '\0';
	filestr[0] = '\0';
	comopts[0] = '\0';
	commstr[0] = '\0';
	geninstr[0] = '\0';
	genfiles = 0;
	delfiles = 0;
}


/*
 * traverse the parse tree, and execute the command.
 */

traverse(anode)
	struct node *anode;
{
	register struct node *node;
	register int ret;
	char filename[STRLEN+1];

	if((node=anode) == NULL)
		return(YES);
	switch(node->type) {
	case COMMAND:
		filmode = COMMAND;
		if(traverse(node->left) && traverse(node->right)) {
			if(defdevnest!=0) {
				error("No closing ')'");
				return(NO);
			}
			if((flags&EXACTINPUT) && ninargs != inmax) {
				error("Incorrect arg count");
				return(NO);
			}
			return(execute());
		}
		break;

	case NAME:
		if(lookup(node->left)) {
			commdefaults();
			return(traverse(node->right));
		}
		break;

	case OPTLIST:
		if(traverse(node->left))
			return(traverse(node->right));
		break;

	case OPTION:
		return(option(node->left, node->right));

	case FILSPECS:
		/* if we are doing the LIBR command, we may have to
		 * do all sorts of horrible things before processing the
		 * filspecs, since LIBR is so non-uniform.
		 */
		if(flags&LIBRCOMM)
			librkludge();

		if(getfspec(INPUT, node->left)) {
			if(getfspec(OUTPUT, node->right)) {
				/* having completed scan of filspecs,
				 * see if GENINPUT set.  If so, copy
				 * it over to instring.
				 */
				if(flags&GENINPUT) {
					if(instring[0]!='\0')
						prepend(",", instring);
					prepend(geninstr, instring);
				}
				return(YES);
			}
		}
		break;

	case ',':
		if(traverse(node->left)) {
			if(flags&USEPLUS) {
				if(!nextfile())
					return(NO);
			}
			return(traverse(node->right));
		}
		break;

	case '+':
		if(traverse(node->left)) {
			if(filmode!=INPUT || !(flags&(USEPLUS|PLUSOPT))) {
				error("Invalid use of '+'");
				return(NO);
			}
			concfile();
			return(traverse(node->right));
		}
		break;

	case ')':
		if(traverse(node->left) && traverse(node->right))
			ret = YES;
		else
			ret = NO;
		defdev = seldev();
		defdevnest--;
		if(defdevnest<0) {
			error("Invalid ')'");
			return(NO);
		}
		return(ret);


	case FSPEC:
		if(traverse(node->left))
			return(traverse(node->right));
		break;

	case '(':
		defdevnest++;
		if(defdevnest>1) {
			error("Can't nest device factoring");
			return(NO);
		}
	case FILE:
		if(node->left==NULL) {
			if(!device(defdev))
				return(NO);
		}
		else {
			if(!device(node->left))
				return(NO);
			if(node->type=='(') {
				defdev = node->left;
			}
		}
		return(file(node->right));

	default:
		fatal("bad tree type");	/* "impossible" */
	}
	return(NO);
}

/*
 * Interpret a FILSPECS node.  If null, prompt as required.
 */
getfspec(mode, anode)
	struct node *anode;
{
	register struct node *node;
	register int iomode;
	register c;

	node = anode;
	filmode = iomode = mode;
	if(node==NULL && (flags&(iomode==INPUT?NEEDINPUT:NEEDOUTPUT))) {
		for(;;) {
			filmode = iomode;
			pushtoken();
			prompt(iomode);
			yyparse();
			while((c=getch())!=EOF && c!='\n')
				;
			if(errors || root==NULL)
				return(NO);
			if(root->type == OPTLIST) {
				filmode = COMMAND;
				if(!traverse(root))
					return(NO);
			}
			else {
				node = root;
				break;
			}
		}
	}
	if(iomode==INPUT) {
		/*
		 * At this point we have finished with all options which
		 * pertain to the entire command (as opposed to those
		 * associated with the input or output list.)  We are about
		 * to process the arg lists.  Before doing so, save away
		 * the command info, in case this turns out to be
		 * a multi-line command (e.g. MACRO A+B,C+D.)
		 *
		 * This probably should be handled somewhere else.
		 */
		savgenfiles = genfiles;
		savdelfiles = delfiles;
		copy(outstring, savoutstring);
		copy(comopts, savcomopts);
	}
	return(traverse(node));
}

/*
 * nextfile -- called to handle ',' in multi-csi-line commands, and
 * at end of entire input or output string.
 */
nextfile() {
	register int i;
	char genname[30];

	/* build and save away a CSI command string */
	if(ncsi >= MAXCSI) {
		error("Command too complex");
		return(NO);
	}
	for(i=0; i<3; i++) {
		if(genfiles&(DOGEN<<(i*GENSHIFT))) {
			/*
			 * If we are generating an output name from an
			 * input name, pull out the desired input file
			 * name and call gfile
			 */
			getfile((genfiles>>(i*GENSHIFT))&GENWHICH, instring, genname);
			gfile(genname, i);
		}
		if(delfiles&(1<<i))
			dfile(i);
	}
	if(outstring[0]!='\0')
		append("=", outstring);
	append(instring, outstring);
	append(comopts, outstring);
	csistr[ncsi] = alloc(length(outstring)+1);
	copy(outstring, csistr[ncsi++]);

	/* restore the overall command options to prepare for
	 * a possible next line of command (e.g. MACRO A+B,C+D.)
	 */
	delfiles = savdelfiles;
	genfiles = savgenfiles;
	copy(savoutstring, outstring);
	copy(savcomopts, comopts);
	instring[0] = '\0';
	ninargs = 0;
	return(YES);
}

/*
 * Handle a '+' file separator.  Normally nothing to do, but may have
 * to insert a PIP "/U" option.
 */
concfile() {
	if(flags&PLUSOPT) {
		delopt('U', comopts);
		append("/U", comopts);
	}
}

/*
 * Check "device" string for validity and save it away.  If device is
 * null and we are looking for a file name, insert default device.
 */
device(dev)
	char *dev;
{
	if(!(flags&ARBARG) && dev[0]=='\0')
		copy(defdev, filestr);
	else {
		if(!(flags&ARBARG) && !isdev(dev)) {
			error("Invalid argument");
			return(NO);
		}
		copy(dev, filestr);
	}
	return(YES);
}

/*
 * Check a string for validity as a file name.  Save name away, adding
 * missing names or extensions as required.  E.g. ".*" as an extension.
 */
file(s)
	char *s;
{
	register char *str;
	register char *target;
	register int n;

	str = s;
	if(str!=NULL) {
		if(!(flags&ARBARG) && !isfile(str, flags&ALLOWPAT)) {
			error("Invalid argument");
			return(NO);
		}
		append(str, filestr);
	}
	else if(flags&GENFIL)
		append("*", filestr);
	if(flags&(GENLST|GENEXT) && !match('.', filestr)) {
		if(flags&GENLST)
			append(".LST", filestr);
		else
			append(".*", filestr);
	}
	target = selectinout();
	if(filmode==INPUT) {
		n = ++ninargs;
		if(n > inmax) {
			error("Too many arguments");
			return(NO);
		}
	}
	else {
		n = ++noutargs;
		if(n > outmax) {
			error("Too many arguments");
			return(NO);
		}
	}
	if(n==1) {
		/*
		 * In the normal case, we simply copy filestr to the
		 * target.  However, if LIBRCOMM is set, we may have
		 * to prepend instead of copy, because LIBRCOMM may
		 * have generated a file name, due to use of BACKWARDS.
		 * In addition, if GENINPUT is set (also only for LIBR),
		 * we must save away the file in the input string as well.
		 * This is another instance of the horrible LIBR kludge.
		 */
		if(!(flags&LIBRCOMM) || target[0]=='\0')
			copy(filestr, target);	/* normal case */
		else if(target[0]==',')
			prepend(filestr, target);
		/* else file already specified by LIBR/OBJECT:FILE */

		/* before returning, see if we are generating
		 * an input.  if so, save it away for future use.
		 */
		if(flags&GENINPUT && filmode==INPUT)
			copy(filestr, geninstr);
	}
	else {
		append(",", target);
		append(filestr, target);
	}
	return(YES);
}

/*
 * Return instring or outstring as appropriate.  This handles the
 * BACKWARDS flag.
 */
selectinout() {
	if((filmode==INPUT && !(flags&BACKWARDS))
	  || (filmode!=INPUT && (flags&BACKWARDS)))
		return(instring);
	return(outstring);
}

/*
 * select the default device to be prepended to arguments without
 * a device.  if ARBARG is on, the default is null, since we are not
 * really specifying a file, otherwise default is DK:.
 */
seldev() {
	if(flags&ARBARG)
		return("");
	else
		return("DK:");
}


/*
 * lookup command name, and set up command flags.
 */
lookup(c)
	char *c;
{
	register int i;
	register int nmatch;

	currcomm = 0;
	nmatch = 0;
	for(i=0; command[i].c_name!=NULL; i++) {
		if(equal(c, command[i].c_name)) {
			currcomm = i;
			nmatch = 1;
			break;
		}
		if(matchname(c, command[i].c_name)) {
			currcomm = i;
			nmatch++;
		}
	}
	if(nmatch>1) {
		error("Ambiguous command name");
		return(NO);
	}
	else if(nmatch<1) {
		error("No such command");
		return(NO);
	}
	outmax = command[currcomm].c_outmax;
	inmax = command[currcomm].c_inmax;
	flags = command[currcomm].c_flags;
	if(flags&GENOUT)
		genfiles =| DOGEN;
	defdev = seldev();
	return(YES);
}

/*
 * set up default inputs, outputs, and switches.
 */
commdefaults() {
	register char *s, *p;

	if((s=command[currcomm].c_prog) != NULL) {
		p = commstr;
		while(*s && *s!='*')
			*p++ = *s++;
		*p = '\0';
		if(*s)
			s++;
		p = outstring;
		while(*s && *s!='=')
			*p++ = *s++;
		*p = '\0';
		if(*s)
			s++;
		p = instring;
		while(*s && *s!='/')
			*p++ = *s++;
		*p = '\0';
		copy(s, comopts);
	}
}

/*
 * Process an option and a possible value. (e.g. /COPIES:3)
 * Verify validity, and then process.  There are 4 flavors of option:
 *	SWITCH -	A CSI switch with or without value
 *	FSPECn -	Generates an output file.  May cause an
 *			output file to be generated from an input
 *			name (e.g. /OBJECT) or a default output
 *			file to be suppressed (/NOOBJECT).
 *	SUPPRESS -	Turn off a CSI switch.
 *	IGNORE -	Option is a NOP.
 *
 */
option(opt, val)
	char *opt, *val;
{
	register int i;
	register int nmatch;
	register struct option *optlist;
	int loc;
	struct option *o;

	nmatch = 0;
	if((optlist = command[currcomm].c_opts) != NULL) {
		for(i=0; opt[i]!='\0'; i++)
			opt[i] = lcase(opt[i]);
		for(i=0; optlist[i].o_name!=NULL; i++) {
			if(equal(opt, optlist[i].o_name)) {
				nmatch = 1;
				o = &optlist[i];
				break;
			}
			else if(matchname(opt, optlist[i].o_name)) {
				nmatch++;
				o = &optlist[i];
			}
		}
		if(nmatch>1) {
			error("Ambiguous option name");
			return(NO);
		}
	}
	if(nmatch<1) {
		error("Invalid option");
		return(NO);
	}
	if(val==NULL && o->o_flags&HASARG) {
		error("Option requires value");
		return(NO);
	}
	if(val!=NULL && !(o->o_flags&(HASARG|OPTARG))) {
		error("Option does not take a value");
		return(NO);
	}
	if((filmode==COMMAND && !(o->o_flags&ONCOMM)) ||
	   (filmode==INPUT && !(o->o_flags&ONINPUT)) ||
	   (filmode==OUTPUT && !(o->o_flags&ONOUTPUT))) {
		error("Option in wrong position");
		return(NO);
	}
	optflags = o->o_flags;
	switch(o->o_type) {
	case O_IGNORE:
		return(YES);
	case O_SWITCH:
		if(o->o_csiopt != NULL)
			optappend(o->o_csiopt, val);
		return(YES);
	case O_FSPEC1:
	case O_FSPEC2:
	case O_FSPEC3:
		loc = o->o_type-O_FSPEC1;
		genfiles =& ~(GENMASK<<(loc*GENSHIFT));
		if(optflags&DELFSPEC)
			delfiles =| 1<<loc;
		else if((optflags&PRINTER) && val==NULL && (filmode==COMMAND || !(flags&GENOUT))) {
			/* if PRINTER flag is set use LP: if no file is provided.
			 * if the option is not on the command, then do not use
			 * LP:, since we will want a generated file name.  If there
			 * cannot be a generated file name, then also use LP:
			 */
			val = "LP:";
		}
		else if((optflags&TERMINAL) && val==NULL)
			val = "TT:";

		else if(val==NULL || isdev(val)) {
			/* we have an option like /LIST or /OBJECT, but
			 * no file specified.  make a record of which
			 * file name is to be generated.
			 */
			if(filmode==INPUT && ninargs!=0)
				i = ninargs-1;
			else
				i = 0;
			genfiles =| (DOGEN|i) << (loc*GENSHIFT);
		}
		return(insertfile(val, loc));
	case O_SUPPRESS:
		delopt(o->o_csiopt, comopts);
		return(YES);
	default:
		fatal("inval otype");
	}
}

/*
 * Tack a CSI switch option onto appropriate string.
 */
optappend(opt,val)
	char opt;
	char *val;
{
	char work[30];
	char *target;

	work[0]='/';
	work[1]=opt;
	work[2]='\0';

	if(filmode==COMMAND)
		target = comopts;
	else
		target = selectinout();
	append(work, target);
	if(val!=NULL && !(optflags&SKIPARG)) {
		while((val=getarg(val,work))!=NULL) {
			append(":", target);
			append(work, target);
		}
	}
}

/*
 * delete CSI switch from str.  OPT assumed to have no value.
 */
delopt(opt, str)
	char opt;
	char *str;
{
	register char *s;

	s = str;
	while(*s) {
		if(s[0]=='/' && s[1]!='\0') {
			if(s[1]==opt)
				copy(&s[2], s);
			else
				s =+ 2;
		}
		else
			s++;
	}
}

/*
 * Get an option "argument" (value).
 */
getarg(v, t)
	char *v, *t;
{
	register char *val, *target;
	register int i;
	target = t;
	val = v;
	if(val==NULL || *val=='\0')
		return(NULL);
	if(*val==':') {
		if(val[1]=='\0') {
			error("Invalid option value");
			return(NULL);
		}
		val++;
	}
	while(*val!='\0' && *val!=':') {
		if(digit(*val)) {
			do
				*target++ = *val++;
			while(digit(*val));
			if(*val=='\0' || *val==':') {
				*target++ = '.';	/* convert all nums to decimal */
				*target = '\0';
				return(val);
			}
			else {
				error("Invalid option value");
				return(NULL);
			}
		}
		else {
			for(i=0; i<3; i++) {
				if(*val!='\0' && *val!=':')
					*target++ = *val++;
			}
			while(*val!='\0' && *val!=':')
				val++;
			*target = '\0';
			return(val);
		}
	}
}

/*
 * extract the Nth filename (including device but not including
 * attached switches) from a CSI input or output list.
 */
getfile(which, fromstr, filename)
	char *fromstr;
	char *filename;
{
	register char *fi;
	register char *in;
	register int n;

	fi = filename;
	in = fromstr;
	n = which;
	while(n!=0) {
		while(*in!='\0' && *in++ != ',')
			;
		n--;
	}
	while(*in!='\0' && *in!=',' && *in!='/')
		*fi++ = *in++;
	*fi = '\0';
}

/*
 * replace an output file name.
 */
insertfile(name, loc)
	char *name;
{
	register int i;
	register char *p, *w;
	char work[STRLEN+1];

	w = work;
	p = outstring;
	if(name==NULL)
		name = "";
	for(i=0; i<loc; ) {
		if(*p=='\0') {
			*w++ = ',';
			i++;
		}
		else {
			if(*p==',')
				i++;
			*w++ = *p++;
		}
	}
	*w = '\0';
	if(!match(':', name) && name[0]!='\0')
		append("DK:", work);
	append(name, work);
	while(*p!='\0' && *p!=',')
		p++;
	append(p, work);
	copy(work, outstring);
	return(YES);
}

/*
 * gfile - generate an output file name from a supplied input name
 */
gfile(name, loc)
	char *name;
{
	register char *nm;
	register int len;
	char outname[30];

	replace('.', '\0', name);	/* remove any extension */
	/*
	 * if the output file consists of a device name, then save the
	 * device name, otherwise delete the entire existing output name.
	 */
	getfile(loc, outstring, outname);
	len = length(outname);
	if(len>0) {
		if(outname[len-1]!=':')
			outname[0] = '\0';
	}

	/*
	 * ignore any device which is part of an input file name.
	 * output is always to DK, unless specified by the option
	 * which causes name generation. (e.g. /LIST:DK2: )
	 */
	nm = name;
	while(*nm!='\0') {
		if(*nm++ == ':') {
			name = nm;
			break;
		}
	}
	append(name, outname);
	insertfile(outname, loc);
}

/*
 * delete an output file name.
 */
dfile(loc) {
	insertfile("", loc);
}

/*
 * execute - process a correctly analyzed command.
 */
execute() {
	register char **av;
	register char *in;
	register int i;
	char *arglist[20];
	char *tim;
	int tvec[2];
	int eloc1, eloc2;
	int corefile;
	int num;
	char work[300];
	int eflags;
	struct {
		char ispeed, ospeed;
		char erase, kill;
		int mode;
	} sgttybuf;

	switch(command[currcomm].c_type) {
	case EXTERNAL:
		/*
		 * The "normal" case.  Call "ru" to execute
		 * the desired program.
		 */
		if(!nextfile())
			break;
		if(debug)
			printf("R %s\n", commstr);
		else {
			av = arglist;
			*av++ = "ru";
		}
		/*
		 * save away CSI input lines
		 */
		for(i=0; i<ncsi; i++) {
			if(debug)
				printf("*%s\n", csistr[i]);
			else {
				*av++ = "-i";
				*av++ = csistr[i];
			}
		}
		/* and run it ... */
		if(!debug) {
			run(arglist, av, commstr);
		}
		break;


		/*
		 * The remaining commands must each be handled separately:
		 */


	case ASSIGN:
		assign(instring, outstring);
		break;
	case BASE:
		for(in=instring; *in != NULL; in++) {
			if(*in>'7' || *in<'0') {
				error("Invalid base");
				return(NO);
			}
		}
		exambase = atoo(instring)&~1;
		break;

	case BYE:
		terminate();
		break;

	case CLOSE:
		temps(CLOSETEMPS);
		break;


	case DATE:
		time(tvec);
		tim = ctime(tvec);
		tim[24] = '\0';
		tim[10] = '\0';
		tim[7] = '\0';
		printf("%s-%s-%s\n", &tim[8], &tim[4], &tim[22]);
		break;

	case DEASSIGN:
		assign("", instring);
		break;

	case EXAMINE:
		for(in=instring; *in!=NULL && *in!='-'; in++) {
			if(*in<'0' || *in>'7') {
				error("Bad examine");
				return(NO);
			}
		}
		if(*in=='-')
			*in++ = '\0';
		eloc1 = atoo(instring)&~1;
		if(digit(*in))
			eloc2=atoo(in)&~1;
		else
			eloc2=eloc1;
		if((corefile = open("core", 0)) == ERROR) {
			error("No core image to examine");
			return(NO);
		}
		i = 1;
		while(eloc1<=eloc2) {
			seek(corefile, 1024+exambase+eloc1, 0);
			num = ERROR;
			read(corefile, &num, 2);
			printo(num);
			if(i%8 == 0)
				putchar('\n');
			eloc1 =+ 2;
			i++;
		}
		putchar('\n');
		close(corefile);
		break;

	case INITIALIZE:
		dirinit();
		break;

	case SET:
		/*
		 * the SET command only allows EDIT, CRLF, and MSG to
		 * be set.  if the command is ever extended, it should
		 * be table driven.
		 *
		 * MSG is an extension, and controls the printing of
		 * emulator warning messages, and programmed request
		 * error return messages.
		 */
		replace(':', '\0', instring);
		if(equal(instring, "EDIT")) {
			if(equal(outstring, "EDIT"))
				defeditor = "edit";
			else if(equal(outstring, "TECO"))
				defeditor = "teco";
			else if(equal(outstring, "UNIX"))
				defeditor = "rted";
			else
				error("invalid SET");
		}
		else if(equal(instring, "TT") &&
		   (equal(outstring,"CRLF") || equal(outstring,"NOCRLF"))) {
			/*
			 * If CRLF is set, it means that the system should
			 * generate newlines when a line exceeds 80 chars.
			 * In the DCIEM system, the BS delay bit is used
			 * for this purpose.  This is very system dependant,
			 * and must be changed for other systems.
			 *
			 * This SET command is implemented so that the
			 * TECO VT-52 screen editor can be made to work.
			 */
#define CRLF 0100000
			gtty(1, &sgttybuf);
			if(outstring[0]=='N')
				sgttybuf.mode =& ~CRLF;
			else
				sgttybuf.mode =| CRLF;
			stty(1, &sgttybuf);
		}
		else if(equal(instring, "MSG")) {
			if(equal(outstring, "ERR"))
				emsg++;
			else if(equal(outstring, "NOERR"))
				emsg = 0;
			else if(equal(outstring, "WARN"))
				wmsg++;
			else if(equal(outstring, "NOWARN"))
				wmsg = 0;
			else
				error("Invalid SET");
		}
		else
			error("Invalid SET");
		break;

	case SHOW:
		show();
		break;

	case TIME:
		time(tvec);
		tim = ctime(tvec);
		tim[19] = '\0';
		printf("%s\n", &tim[11]);
		break;

	case EDIT:
		/*
		 * EDIT is called by chaining to the program with certain
		 * bits turned on in the chain area, and file names at
		 * certain pre-defined locations.
		 */
		if(match('I',comopts))		/* /INSPECT */
			eflags = 0200;
		else if(match('C',comopts))	/* /CREATE */
			eflags = 0100000;
		else if(match('X', comopts))	/* /EXEC */
			eflags = 0100200;
		else if(outstring[0]!='\0')	/* implies /OUTPUT, and none of above were found */
			eflags = 0200;
		else
			eflags = 0;		/* no options, so EB command implied */

		/*
		 * check whether /TECO, /EDIT, or /UNIX overrides default command.
		 */
		if(match('T', comopts))
			copy("teco", commstr);
		else if(match('E', comopts))
			copy("edit", commstr);
		else if(match('U', comopts))
			copy("rted", commstr);
		else if(defeditor!=NULL)
			copy(defeditor, commstr);
		/*
		 * now generate the chain area.  null bytes are handled
		 * specially since they cannot be passed as part of an
		 * argument string to exec.
		 */
		in = work;
		if(eflags.lobyte)
			*in++ = eflags.lobyte;
		else {
			*in++ = '\\';
			*in++ = '0';
		}
		if(eflags.hibyte)
			*in++ = eflags.hibyte;
		else {
			*in++ = '\\';
			*in++ = '0';
		}
		copy(instring, in);
		i = length(in);
		in =+ i;
		for(; i<030-2; i++) {
			*in++ = '\\';
			*in++ = '0';
		}
		/* this has brought us up to location 0540 in
		 * the chain info area.  if there is an output file,
		 * it goes here.
		 * If we are /EXECing a TECO macro, we must move up to
		 * location 0600, where we will generate a command string.
		 */
		if(match('X', comopts)) {
			for(i=0; i<040; i++) {
				*in++ = '\\';
				*in++ = '0';
			}
			copy("ER", in);
			append(instring, in);
			if(!match('.', instring))
				append(".TEC", in);
			append("\33YXZMZ\33\33", in);
		}
		else
			copy(outstring, in);

		if(debug) {
			printf("EDIT/%s\n", commstr);
		}
		else {
			/* execute the editor.  we have made an addition to
			 * rt-11 here.  SET EDIT UNIX or EDIT/UNIX will cause
			 * /bin/rted to be invoked.  this is a version of "ed"
			 * which knows about carriage returns on the ends of
			 * lines, and which translates input to upper case.
			 */
			av = arglist;
			if(equal(commstr, "rted")) {
				*av++ = commstr;
				copy(instring, work);
				if(!match(':', instring))
					prepend("DK:", work);
				doassign(work);
				replace(':', '/', work);
				*av++ = work;
				*av++ = NULL;
				ex("/bin/rted", arglist);
			}
			else {
				*av++ = "ru";
				*av++ = "-C";
				*av++ = work;
				run(arglist, av, commstr);
			}
		}
		break;

	case HELP:
		/*
		 * to execute the HELP command, chain to HELP.SAV with
		 * location 0510 equal to 0100200, and location 0600
		 * (070 bytes later) equal to a teco command string which
		 * will get things started.
		 */
		in = work;
		*in++ = 0200;
		*in++ = 0200;
		for(i=2; i<070; i++) {
			*in++ = '\\';
			*in++ = '0';
		}
		copy("@^UA!", in);
		if(match('P', comopts))
			append("/PRINTER ", in);
		else
			append("/TERMINAL ", in);
		if(instring[0]!='\0')
			append(instring, in);
		else
			append("HELP", in);
		if(outstring[0]!='\0') {
			append(" ", in);
			append(outstring, in);
		}
		in =+ length(in);	/* so  append size limit check doesn't complain */
		copy("!@:ER#SY:HELP.TEC#\"F@^A#?HELP-F-File not found HELP.TEC\r\n#HKEX'YHXZHKGAMZ", in);
		av = arglist;
		*av++ = "ru";
		*av++ = "-C";
		*av++ = work;
		run(arglist, av, commstr);
		break;

	case RUN:
		for(in=instring; *in!='\0'; in++)
			*in = lcase(*in);
		if(debug) {
			printf("RUN %s\n", instring);
			break;
		}
		av = arglist;
		*av++ = "ru";
		run(arglist, av, instring);
		break;

	case SHELL:
		av = arglist;
		*av++ = "-";
		*av = NULL;
		ex("/bin/sh", arglist);
		break;

	default:
		fatal("inval comm type");
	}
	return(YES);
}

/*
 * run - call the RT-11 emulator ("ru").
 *
 * tack on arguments which apply to any invocation of ru.
 */
#define COREDUMP	0200
#define ERRSTAT		04
#define SEVERE		010
run(arglist, argend, comm)
	char **arglist;
	char **argend;
	char *comm;
{
	register char *cmd;
	struct status {
		char	system;
		char	user;
	};
	int status;
	register char **av;
	register struct atab *a;

	cmd = comm;
	av = argend;
	if(inpsp!=0)
		*av++ = "-f";
	*av++ = "-c";
	if(emsg)
		*av++ = "-e";
	if(wmsg)
		*av++ = "-w";
	for(a=atab; a!= &atab[SIZEATAB]; a++) {
		if(a->aname[0]!=NULL) {
			*av++ = "-a";
			*av++ = a->iname;
			*av++ = a->aname;
		}
	}
	if(match(':', cmd)) {
		doassign(cmd);
		replace(':', '/', cmd);
	}
	*av++ = cmd;
	*av++ = NULL;
	status = ex("/bin/ru", arglist);
	if(status.system&COREDUMP) {
		error("Program terminates with core dump");
	}
	else if(status.system!=0 || status.user>=ERRSTAT)
		errors++;	/* which will force termination of indir file */
}

/*
 * fork and exec a program+arglist
 */
ex(prog, arglist)
	char *prog;
	char **arglist;
{
	int status;
	register int pid;
	register int i;

	switch(pid=fork()) {
	case ERROR:
		error("System overloaded, try later");
		return(0);
	case 0:
		signal(INTR, savintr);
		signal(QUIT, savquit);
		if(inpsp) {
			close(0);
			dup(input[inpsp]);
		}
		for(i=3; i<14; i++)
			close(i);
		execv(prog, arglist);
		error("Can't execute %s", prog);
		exit(SEVERE);
	default:
		status = 0;
		while(wait(&status)!=pid)
			;
		return(status);
	}
}


/*
 * assign a "physical" device name to a logical name
 * currently, logical name should be 2 or 3 characters.
 * one char logical names are extended to 2 with a blank.
 */
assign(physical, logical)
	char *physical, *logical;
{
	register char *ext, *in;
	register struct atab *a;
	struct atab *newtab;

	for(ext=logical; *ext!='\0'; ext++)
		*ext = lcase(*ext);
	ext = logical;	/* external name is logical name */
	if(length(ext)==1)
		append(" ", ext);
	for(in=physical; *in!='\0'; in++)
		*in = lcase(*in);
	in = physical;	/* internal name is physical name */
	replace(':', '\0', in);
	replace(':', '\0', ext);
	if(length(in)>3 || length(ext)>3 || (in[0]!='\0' && !legaldev(in))) {
		error("Invalid assignment");
		return;
	}
	newtab = NULL;
	for(a=atab; a!= &atab[SIZEATAB]; a++) {
		if(a->aname[0]==NULL)
			newtab = a;
		else if(ext[0]=='\0')
			a->aname[0] = '\0';
		else if(ext[0]==a->aname[0] && ext[1]==a->aname[1] && ext[2]==a->aname[2]) {
			newtab = a;
			break;
		}
	}
	if(ext[0]=='\0')
		return;
	a = newtab;
	if(in[0]==NULL && (a==NULL || a->aname[0]==NULL)) {
		error("Logical name not found");
		return;
	}
	if(a==NULL) {
		error("Assign table full");
		return;
	}
	if(in[0]=='\0')
		a->aname[0] = '\0';
	else
		copy(ext, a->aname);
	copy(in, a->iname);
}

/*
 * convert filename from logical to "physical" device name
 */
doassign(filename)
	char *filename;
{
	register struct atab *a;
	register char *f;

	for(f=filename; *f!='\0'; f++)
		*f = lcase(*f);
	f = filename;
	/* we think of "device" names as 2 or 3 char names.  extend one
	 * char name with a blank
	 */
	if(f[1]==':')
		prepend(" ", &f[1]);
	for(a=atab; a!= &atab[SIZEATAB]; a++) {
		if(a->aname[0]=='\0')
			continue;
		if(a->aname[0]==f[0] && a->aname[1]==f[1] &&
		   (a->aname[2]==f[2] || (a->aname[2]=='\0' && (f[2]==':' || digit(f[2]))))) {
			f[0] = a->iname[0];
			f[1] = a->iname[1];
                        if(!digit(f[2])) {
				if(f[2]==':')
					prepend("0", f+2);
				if(a->iname[2])
					f[2] = a->iname[2];
			}
			break;
		}
	}
	if(f[2]==':')
		prepend("0", f+2);
}

/*
 * implement the SHOW command - print assign table.
 */
show() {
	register struct atab *a;
        register char *p;
	printf("TT  (Resident)\nNL  (Resident)\nLP  (Resident)\nDK  (Resident)\n");
	for(a=atab; a< &atab[SIZEATAB]; a++) {
		if(a->aname[0]!=NULL) {
			for(p=a->iname; *p!='\0'; )
				putchar(ucase(*p++));
			printf("\t = ");
			for(p=a->aname; *p!='\0'; )
				putchar(ucase(*p++));
			putchar('\n');
		}
	}
	printf("\nUSR No Swap\n");
}

/* determine if legal physical device name.
 */
legaldev(d)
	char *d;
{
	register int len;
	register char *dev, *disk;

	dev = d;
	len = length(dev);
	if(len<2 || len>3)
		return(NO);
	if(equal(dev, "lp") || equal(dev, "tt") || equal(dev, "dk") || equal(dev, "nl"))
		return(YES);
	disk = "dk0";
	for(disk[2]='0'; disk[2]<='7'; disk[2]++)
		if(equal(dev, disk))
			return(YES);
	return(NO);
}


/*
 * The INIT command.  Verify that it should be done via prompt, and then
 * init appropriate directory.
 */
dirinit()
{
	register c, reply;
	if(comopts[0]!='/' || comopts[1]!='Y') {
		prompt(NULL, "Initialize %s, are you sure\7? ", instring);
		reply = getch();
		c = reply;
		while(c!='\n' && c!=EOF)
			c = getch();
		if(lcase(reply)!='y') {
			printf("Init not done.");
			return;
		}
	}
	if(debug)
		return;
	append(":", instring);	/* in case there wasn't one */
	doassign(instring);
	replace(':', '\0', instring);

	if(instring[0]!='d' || instring[1]!='k' || !digit(instring[2])
	  || instring[3] != '\0') {
		error("Invalid device");
		return;
	}
	newdir(instring);
}



/*
 * The following is a set of kludges for the LIBR command,
 * which is highly non-uniform.
 */
librkludge() {
	register nopts;

	nopts = 0;
	outmax--;	/* allow for generated arg */
	/* flags, etc., have already been set for /INSERT */
	if(match('I', comopts))		/* /INSERT */
		nopts++;
	if(match('C', comopts)) {	/* /CREATE */
		flags =& ~GENINPUT;
		outmax++;	/* prime count to allow one more, since no
				 * generated name.
				 */
		nopts++;
	}
	if(match('D', comopts)) {	/* /DELETE */
		outmax = 0;	/* do not allow any more args */
		flags =& ~NEEDOUTPUT;
		nopts++;
	}
	if(match('E', comopts)) {	/* /EXTRACT */
		outmax = 1;	/* allow only one more */
		flags =& ~(GENINPUT|BACKWARDS);
		nopts++;
	}
	if(match('M', comopts)) {	/* /MACRO */
		outmax = 6;
		flags =& ~GENINPUT;
		nopts++;
	}
	if(match('G', comopts)) {	/* /REMOVE */
		if(nopts==0) {
			outmax = 0;	/* no args, LIBR will prompt */
			flags =& ~NEEDOUTPUT;
		}
	}
        if(nopts==0 && ((genfiles&(DOGEN<<GENSHIFT)) || match(',',outstring))) { /* /LIST
		/* we have /LIST, and no other options specified,
		 * so we want only a library name as input.
		 */
		flags =& ~(GENINPUT|BACKWARDS|NEEDOUTPUT);
		outmax = 0;
	}
	/* delete phony switch options, no longer needed */
	delopt('C', comopts);
	delopt('I', comopts);
}
