#
/*
 *
 *
 * The  information  in  this  document  is  subject  to  change
 * without  notice  and  should not be construed as a commitment
 * by Digital Equipment Corporation or by DECUS.
 * 
 * Neither Digital Equipment Corporation, DECUS, nor the authors
 * assume any responsibility for the use or reliability of  this
 * document or the described software.
 * 
 * 	Copyright (C) 1980, DECUS
 * 
 * 
 * General permission to copy or modify, but not for profit,  is
 * hereby  granted,  provided that the above copyright notice is
 * included and reference made to  the  fact  that  reproduction
 * privileges were granted by DECUS.
 *
 */

/*
 * mkcook
 *
 * Make cookie.fil
 *
 *	mkcook [options] cookie.txt
 *
 * If no argument is given, cookie.txt will be looked for on
 * the current account.
 *
 * Options:
 *
 *	-d	Debug
 *	-s	Small -- output cookie.h for SMALL cookie.c
 *	-v	Verify (print each cookie line as it is read)
 *
 * Cookie text files have the following format:
 *
 *	A new cookie always begins in column 1.
 *	 It may be continued over several lines
 *	 by starting the continued text with one or
 *	 more spaces or tabs.
 *
 *	 A blank line may be included and will yield
 *	 a blank line within or following the cookie.
 *	 preceeding the first cookie will be ignored.
 *	 A line consisting of a formfeed only will also
 *	 be ignored.
 *
 *	 All the above constitutes one single cookie.
 *
 *	A cookie may be explicitly formatted
 *	/ into individual lines by
 *	/ preceeding each continuation line
 *	/ with a slash followed by a space or a tab.
 *
 *	To attribute a cookie. terminate it by terminating
 *	the sentence with a '.', '?', or '!' then append
 *	the author's name as shown.  -- Author's name.
 *
 *	Although not necessary, it is recommended that
 *	 cookies be stored in ascending Ascii order.
 *
 *	No cookie may contain more than 2047 bytes.
 *
 * Cookies are stored on a disk file in a format compatible
 * with the limited amount of random access capabilities available
 * on "standard" C libraries.  The format is, of course, called
 * "Cookie access method."  The output file is written by fput()
 * and must be read by fget().  Note also that a file written by
 * an RT11 library must be read by an RT11 library program.
 *
 * The cookie file has the following format:
 *
 *	Record 1:
 *		int		rsxflag;	-- 0 for RT11, 1 for RSX
 *		long		ncookie;	-- number of cookies
 *		int		nindex;		-- Dimension of index table
 *		int		subindex;	-- Number of subindex entries
 *		int		sindex;		-- sizeof index table
 *		char		date[26];	-- When file created.
 *
 *	Record 2:
 *		long	main_index[tabdim];
 *					-- Record number of each
 *						subindex entry.
 *
 *	Record 3 .. tabdim+2
 *		long	sub_index[tabdim];
 *					-- Record number of each
 *						cookie.
 *
 *	Cookie records have just plain text, one space between words.
 *	If an explicit return "/ text" is found, a newline will be
 *	included.  If the program thinks it has found an author,
 *	it will output "text.<tab>-- Author's name..."
 *
 * Thus, to find cookie N, proceed as follows:
 *
 *	Read record 1.
 *	Allocate space for index[].
 *	Read record 2 into index.
 *
 *	Read index[N / ncookies] into index.
 *	Read index[N % ncookies] -- it has the cookie.
 *
 * Note: if the maximum cookie contains 2047 bytes, the index table
 * may be dimensioned (2047 / sizeof (long)) (== 256) and the maximum
 * number of cookies equals 256 * 256 == 65536.  If this is done, and
 * only one cookie is to be read, only one buffer is needed, defined as:
 *
 *	union {
 *		char	text[TEXTSIZE];
 *		long	index[TEXTSIZE / sizeof (long)];
 *	} cookie;
 *
 */

#include <stdio.h>

#define	EOS	0

extern long	ftell();

FILE	*textfp;		/* Cookie text stored here	*/
FILE	*indexfp;		/* Indexes stored here		*/
FILE	*dummyfp;		/* Dummy output file		*/
FILE	*outfp;			/* Output file (cookie.fil)	*/

char	*input_file	= "cookie.txt";

/*
 * The following may need work on RT11 to insert file sizes
 */

char	*tiny_file	= "cookie.h";
char	*text_file	= "ctext.tmp";
char	*dummy_file	= "cdummy.tmp";
char	*index_file	= "cindex.tmp";
char	*cookie_file	= "cookie.fil";

struct header {
	int	rsxflag;	/* 1 if RSX mode		*/
	long	ncookie;	/* Number of cookies		*/
	int	nindex;		/* Dimension of top_index[]	*/
	int	subindex;	/* Number of subindex entries	*/
	int	sindex;		/* Sizeof index for alloc	*/
	char	date[28];	/* Date cookie file built	*/
} header;

#define	TEXTSIZE	2048
#define	INDEXMAX	(TEXTSIZE / sizeof (long))

char	text[TEXTSIZE];		/* Cookies stored here		*/

char	line[257];		/* Input text line		*/

long	*sub_index; 		/* Indexes stored here		*/
long	*top_index;		/* Top level indexes go here	*/
long	firstindex;		/* -> top index in indexfp	*/
long	firstcookie;		/* -> first cookie in dummyfp	*/

int	small		= 0;
int	verbose		= 0;
int	debug		= 0;


main(argc, argv)
int 	argc;
char	*argv[];
{
	register char	*ap;
	FILE		*file_open();

	while (argc > 1 && argv[1][0] == '-') {
		for (ap = &argv[1][1]; *ap; ap++) {
			switch (tolower(*ap)) {

			case 'd':
				debug++;
				break;

			case 's':
				small++;
				break;

			case 'v':
				verbose++;
				break;

			default:
				fprintf(stderr, "?Unknown option '%c'\n",
						*ap);
			}
		}
		argc--;
		argv++;
	}

	ap = (argc > 1 && argv[1][0] != ' ') ? argv[1] : input_file;
	if (*ap != '-') {
		if ((freopen(ap, "r", stdin)) == NULL)
			error("Can't open \"%s\"\n", ap);
	}
	textfp = file_open(text_file, "wnu");
	if (small) {
		outfp = file_open(tiny_file, "w");
		makesmall();
		fclose(stdin);
		fclose(outfp);
	}
	else {
		/*
		 * Copy raw cookies to a temp file.  Collect how many and
	 	 * dimension of indexes.
		 */
		maketext();
		fclose(stdin);
		fclose(textfp);
		/*
		 * Build dummy cookie file
		 */
		textfp = file_open(text_file, "rnu");
		dummyfp = file_open(dummy_file, "wnu");
		indexfp = file_open(index_file, "wnu");
		makedummy();
		delete(textfp);
		fclose(dummyfp);
		fclose(indexfp);
		/*
		 * Build real cookie file
		 */
		outfp = file_open(cookie_file, "wnu");
		dummyfp = file_open(dummy_file, "rnu");
		indexfp = file_open(index_file, "rnu");
		makecookie();
		fclose(outfp);
		delete(dummyfp);
		delete(indexfp);
	}
}


delete(fd)
FILE		*fd;
/*
 * Close or delete the file
 */
{
	if (debug)
		fclose(fd);
	else	fmkdl(fd);
}


FILE *
file_open(filename, mode)
char		*filename;
char		*mode;
/*
 * Open the file, die if failure
 */
{
	register FILE	*fd;

	if ((fd = fopen(filename, mode)) == NULL)
		error("?MKCOOK-F-Can't %s \"%s\"\n",
				(*mode == 'w') ? "create" : "open",
				filename);
	return(fd);
}


makesmall()
/*
 * Make the small cookie file
 */
{
	int		len;
	register char	*lp;
	register char	*start;
	register char	*end;

	fprintf(outfp, "char *ctext[] {\n");
	while (!feof(stdin)) {
		if ((len = gettext()) == 0)
			continue;
		end = &text[len];

		fprintf(outfp, "\"");

		for (start = text; start < end;) {
			if ((end - start) < 72) {	/* Short or end	*/
				outsmall(start, "\",\n");
				break;
			}
			for (lp = start+71; lp > start && *lp != ' '; lp--);
							/* lp -> blank	*/
			if (lp <= start) {		/* Long!! word	*/
				outsmall(start, "\",\n");
				break;
			}
			*lp = 0;			/* Mid piece	*/
			outsmall(start, "\\\n ");
			start = lp + 1;
		}
	}
	fprintf(outfp, "};\n");
}

outsmall(string, after)
char		*string;
char		*after;
{
	register char	*sp;
	register char	c;

	for (sp = string; (c = *sp++);) {
		switch (c) {
		case '\t':
			fprintf(outfp, "\\t");
			break;

		case '\n':
			fprintf(outfp, "\\n");
			break;

		case '\"':
		case '\\':
			fprintf(outfp, "\\");

		default:
			fprintf(outfp, "%c", c);
		}
	}
	fprintf(outfp, after);
}

maketext()
/*
 * Read stdin, building the temp. cookie file
 *
 *	Input:		stdin
 *	Output:		textfp;
 *
 *	Header is set as follows:
 *
 *		header.rsxflag		Operating system
 *		header.ncookie		Number of cookies (long)
 *		header.nindex		Index dimension
 *		header.subindex		Number of subindex entries
 *		header.sindex		Sizeof index[]
 *		header.date		ctime(0)
 */
{

	register int		len;
	long			subsquare;
	char			*cpystr();
	int			bigbytes;	/* Longest record bytes	*/


	bigbytes = 0;
	cpystr(&header.date, ctime(NULL));
	header.ncookie = 0;
	header.nindex = 0;
	subsquare = 0;
	line[0] = EOS;
	while (!feof(stdin)) {
		if ((len = gettext()) == 0)
			continue;
		if (len > bigbytes)
			bigbytes = len;
		header.ncookie++;
		while (subsquare < header.ncookie) {
			header.nindex++;
			subsquare = header.nindex;
			subsquare *= subsquare;
		}
		put(text, len + 1, textfp, "primary text");
	}

#ifdef	rsx
	header.rsxflag = 1;
#else
	header.rsxflag = 0;
#endif
	header.subindex = (header.ncookie + header.nindex) / header.nindex;
	header.sindex = header.nindex * sizeof (long);
	printf("%ld cookies read, the longest has %d bytes\n",
			header.ncookie, bigbytes);
	printf("top index = %d, sub index = %d, index area size = %d\n",
			header.nindex, header.subindex, header.sindex);
}

gettext()
/*
 * Read the next (complete) text line to text[].  Return the length
 * Initialize by setting line[0] to EOS.
 */
{
	register char		*tp;
	register char		*lp;
	register char		*ep;
	int			len;
	char			*append();

	lp = line;
	if (*lp == '\f')
		lp++;
	tp = append(text, lp, EOS);
	for (;;) {
		lp = line;
		if (gets(lp) == NULL) {
			*lp = EOS;
			break;
		}
		/*
		 * Form feeds are for teco, not for cookie
		 */
		if (*lp == '\f')
			lp++;
		ep = &lp[strlen(lp) - 1];
		while (ep >= lp && (*ep == ' ' || *ep == '\t'))
			ep--;
		ep[1] == EOS;			
		if (debug > 1)
			puts(lp);
		if (ep <= lp) {
			/*
			 * Empty line, ignore if between cookies
			 */
			if (tp = &text)
				continue;
			else {
				lp = line;
				cpystr(lp, "/ ");
			}
		}
		if (*lp == '/')
			tp = append(tp, lp + 1, '\n');
		else if (*lp == ' ' || *lp == '\t')
			tp = append(tp, lp, ' ');
		else 
			break;
	}
	return (tp - text);
}

char *
append(out, in, insert)
register char	*in;
register char	*out;
char		insert;
/*
 * Append in to out, inserting "insert" between them after skipping
 * over leading blanks/tabs.  Return a pointer to the trailing null.
 * Out must be in text.
 */
{
	register int		c;

	if (insert != EOS)
		*out++ = insert;
	while ((c = *in) == ' ' || c == '\t')
		in++;
	if ((out + strlen(in)) >= &text[TEXTSIZE]) {
		fprintf(stderr, "Long text at \"%s\"\n", in);
		fprintf(stderr, "current buffer is \"%s\"\n\n", text);
		return(out);
	}
	while (c != EOS) {
		if (c == ' ' || c == '\t')
			*out++ = ' ';
		while ((c = *in) == ' ' || c == '\t')
			in++;
		if (c == '-'	&& in[1] == '-'
				&& (in[2] == ' ' || in[2] == '\t')
				&& (out - text) > 3
				&& out[-1] == ' ') {
			switch (out[-2]) {
			case '.':
			case '?':
			case '!':
				out[-1] = '\t';
			}
		}
		while ((c = *in) != EOS && c != ' ' && c != '\t') {
			*out++ = c;
			in++;
		}
	}
	*out = EOS;
	return(out);
}

makedummy()
/*
 * Build a dummy cookie file in two separate files:
 *	textfp		Has the cookie data
 *	dummyfp		Gets the cookie data
 *	indexfp		Gets the indices.
 *
 * This way, we don't have to reposition the file, nor do we
 * have to read and write the same file.
 */
{
	register int	subi;		/* Index into sub_index[]	*/
	register int	topi;		/* Index into top_index[]	*/
	register int	len;		/* Input record length		*/
	long		count;

	count = 0;
	if ((top_index = calloc(header.sindex, 1)) == NULL
			|| (sub_index = calloc(header.sindex, 1)) == NULL)
		error("Can't allocate index buffers -- %d bytes\n",
				header.sindex);
	put(&header, sizeof header, dummyfp, "dummy header");
	put(&header, sizeof header, indexfp, "index header");
	put(sub_index, header.sindex, dummyfp, "dummy top index");
	put(sub_index, header.sindex, indexfp, "index top index");
	firstindex = ftell(indexfp);
	for (subi = header.subindex; --subi >= 0;) {
		put(sub_index, header.sindex, dummyfp, "dummy sub index");
	}
	firstcookie = ftell(dummyfp);
	subi = 0;
	topi = 0;
	for (;;) {
		len = get(text, sizeof text, textfp, 0, "first saved text");
		if (feof(textfp))
			break;
#ifdef	rt11
		if (len == 1 && text[0] == ' ')
			break;
#endif
		count++;
		if (subi >= header.nindex) {
			if (topi >= header.nindex) {
				error("Too many cookies, max is %d ** 2\n",
						header.nindex);
			}
			top_index[topi] = ftell(indexfp);
			topi++;
			put(sub_index, header.sindex, indexfp, "sub index");
			subi = 0;
		}
		sub_index[subi] = ftell(dummyfp);
		subi++;
		put(text, len, dummyfp, "secondary text");
	}
	/*
	 * Put the last record
	 */
	while (subi < header.nindex) {
		sub_index[subi] = -1;
		subi++;
	}
	top_index[topi] = ftell(indexfp);
	topi++;
	put(sub_index, header.sindex, indexfp, "last sub index");
	while (topi < header.nindex) {
		top_index[topi] = -1;
		topi++;
	}
	printf("Work files built, %ld cookies, %d index levels\n",
			count, header.nindex);
	if (count != header.ncookie)
		error("Expected %ld cookies, read %ld\n",
			header.ncookie, count);
}

makecookie()
/*
 * Write outfp with cookie file, using
 *
 *	indexfp		Index file (has sub-indexes)
 *	dummyfp		Cookie work file
 *
 */
{
	register int	i;
	register int	bytect;
	long		itemct;

	put(&header, sizeof header, outfp, "cookie header");
	put(top_index, header.sindex, outfp, "cookie top index");
	itemct = 0;
	if (fseek(indexfp, firstindex, 0) != 0)
		error("Can't seek to %ld on index file\n", firstindex);
	if (fseek(dummyfp, firstcookie, 0) != 0)
		error("Can't seek to %ld on cookie file\n", firstcookie);
	for (i = 0; i < header.subindex; i++) {
		get(sub_index, header.sindex, indexfp, 1, "sub index");
		put(sub_index, header.sindex, outfp, "cookie sub index");
	}
	printf("%d index records written\n", header.nindex + 1);
	for (;;) {
		i = get(text, sizeof text, dummyfp, 0, "second saved text");
		if (feof(dummyfp))
			break;
#ifdef	rt11
		if (i == 1 && text[0] == ' ')
			break;
#endif
		itemct++;
		put(text, i, outfp, "cookie text");
	}
	printf("%ld cookies written\n", itemct);
	if (itemct != header.ncookie)
		error("Wrong number of cookies written, expected %ld\n",
				header.ncookie);
}


/*
 * Raw I/O routines
 */

get(whereto, maxsize, fd, exact, why)
char		*whereto;		/* Where to read to		*/
int		maxsize;		/* Buffer (maximum) size	*/
FILE		*fd;			/* Input file descriptor	*/
int		exact;			/* = 1 if must read maxsize	*/
char		*why;			/* Who is reading for error	*/
/*
 * Read into the buffer.  Return the number of bytes read.
 * All errors are fatal.  EOF is fatal if exact == 1.
 */
{
	register int	bytect;

	bytect = fget(whereto, maxsize, fd);
	if (ferr(fd) || (exact && (bytect != maxsize || feof(fd)))) {
		error("%s reading %s, expected %s %d bytes,\
 read %d, code = %06o\n",
			(ferr(fd)) ? "Error"
			  : (feof(fd)) ? "End of file"
			  : "Byte count error",
			why, (exact) ? "exactly" : "up to",
			maxsize, bytect, $$ferr);
	}
	return(bytect);
}


put(wherefrom, size, fd, why)
char		*wherefrom;		/* Where to write from		*/
int		size;			/* Number of bytes to write	*/
FILE		*fd;			/* Output file descriptor	*/
char		*why;			/* Who is writeing for error	*/
/*
 * Write from the buffer. All errors are fatal.
 */
{
	fput(wherefrom, size, fd);
	if (ferr(fd)) {
		error("Error writing %d bytes to %s, code = %06o\n",
			size, why, $$ferr);
	}
}

/*
 * For debugging only
 */

dump(indextable, why)
long	indextable[];
char	*why;
{
	register int i;

	printf("\n%s\n", why);
	for (i = 0; i < header.nindex; i++)
		printf("%3d %ld\n", i, indextable[i]);
}


                                                                                                                                                                                                                                                                                                                                                                                   