#
/*
 *
 *
 * The  information  in  this  document  is  subject  to  change
 * without  notice  and  should not be construed as a commitment
 * by Digital Equipment Corporation or by DECUS.
 * 
 * Neither Digital Equipment Corporation, DECUS, nor the authors
 * assume any responsibility for the use or reliability of  this
 * document or the described software.
 * 
 * 	Copyright (C) 1980, DECUS
 * 
 * 
 * General permission to copy or modify, but not for profit,  is
 * hereby  granted,  provided that the above copyright notice is
 * included and reference made to  the  fact  that  reproduction
 * privileges were granted by DECUS.
 *
 */

#
/*
 *	getrno -- Convert macro comments to .rno for documentation
 *
 * Usage
 *
 *		getrno (-d) (-u) (-w) (-r) -o output -h header input_files
 *
 *	getrno reads a list of files compiling comments to runoff source.
 *	Switches are:
 *
 *		-d	Debugging
 *
 *		-r	RSTS/E title hack flag.  See below
 *
 *		-u	Usage -- output an abbreviated documentation
 *			containing only the information between
 *			"Usage" and "Description".
 *
 *		-w	Wizard:  output package internal calls, too.
 *
 *		-o file	Write output to "file", rather than stdout.
 *
 *		-h file	Prepend the contents of "file" to the output.
 *
 *
 *	There is an error in the RSTS/E V7.0 runoff (RNO.TSK) such
 *	that title lines do not get correct page numbers.  If the -r
 *	flag is set, .t lines in the header program will be marked for
 *	the fixdoc.c program.  Note, for a title to be recognized, it
 *	must have the exact form:
 *
 *		.t Anything  (or .t ########Anything)
 *
 *	Getrno will change it to:
 *
 *		.t [[Anything]]
 *
 *	And fixdoc.c will look for same for post processing.
 *
 *	Getrno runs only under the Decus compiler as it uses fwild/fnext.
 *
 *	Make sure you taskbuild with enough LUN's.  The worst case
 *	is 5 (stdin, stdout, stderr, -o, and a work file).
 *	Also, a larger than normal stack is needed.
 *
 *
 *	The macro input MUST have the following syntax:
 *
 *		.title	name	Title line
 *	;
 *	;+				(Start of commentary)
 *	;
 *	; Index		Short description for kwik index
 *	; Index		(Another short description)
 *	;
 *	; Usage				(Start of usage information)
 *	;
 *	;	Usage information, output, starting in column 1
 *	;	exactly as it is written (rno .nf mode).  The leading
 *	;	semicolon is not output, but the tab (or spaces) is.
 *	;
 *	; Description
 *	;
 *	;	Description, output, starting in column 8
 *	;	in rno .fill mode, except that any line starting
 *	;	with ";<TAB><TAB or SPACE>" will be output
 *	;	in .nf mode.  The leading ";<TAB>" will not be output.
 *	;
 *	;	A line consisting of just ";" will generate a rno .s
 *	;	While a line consisting of ";<SPACE>text" will
 *	;	generate an ".indent -8 ;text".
 *	;
 *	;-				(End of commentary)
 *
 * If the wizard flag (-w) is not given, a Macro source line of the format:
 *
 *	; Internal
 *
 * may be used to signal the start of package-internal information.
 * If "Internal" preceeds "Usage", no information will be output for
 * this file.  If it follows "Usage" or "Description", text following
 * (up to the next section initiator) will not be output.
 *
 * The header file is assumed to be in .rno format.  It should start
 * as follows:
 *
 *	.comment Set top-of-form at 4 lines below perforation.
 *	.ps 58,80		! 58 lines per page, 80 columns.
 *	.lm 8			! left margin at 8
 *	.rm 72			! right margin at 72
 *	.hd Mixed		! Mixed case headings
 *
 * Because the left margin is set to 8, titles and subtitles should
 * be written as:
 * 
 *	.t  ########Title
 *	.st ########Subtitle
 *
 * The wizard flag may be used to select parts of the header file:
 *
 *	.comment wizard
 *		*** For wizards only ***
 *	.comment else
 *		*** For non-wizards only ***
 *	.comment mundane
 *		*** Exit wizard section ***
 *
 * The format of the comments is EXACTLY ".comment<space>argument"
 * One space only.
 *
 * Workfile format:
 *
 *	Each file builds one or more records (lines) in the workfile.
 *	The first has the title line, (information following .title).
 *	To allow building an index, this should be in the format:
 *
 *		name<TAB>Title information
 *
 *	Following this are lines containing .rno text.  The last
 *	line is followed by a record containing "!!" in the first
 *	two bytes.
 *
 *	Records with "!" in the first byte may be used to communicate
 *	information between passes:  they are not written to the output.
 *	This allows writing Usage information as a seperate file.
 *
 * Note that there is a bug in the RSTS RNO (as distributed with Version
 * 7.0).  Consequently, if your runoff source has chapter headings,
 * You should not have explicit titles, or have page numbers.)
 */

#include	<stdio.h>

#define	NENTRIES	200
#define	NAMESIZE	8
#define	WORKFILE	"getrno.tmp"
#define	LINESIZE	257

struct entry {
	char	e_name[NAMESIZE];	/* Entry name		*/
	long	e_place;		/* File position	*/
};

struct entry	entries[NENTRIES];
struct entry	*free_entry = entries;
#define	ELAST	&entries[NENTRIES]

extern	long	ftell();

int		rstshack = 0;		/* If -r flag set	*/
unsigned	linect = 0;		/* Needed for debugging	*/
int		debug = 0;		/* Set for debugging	*/
int		uflag = 0;		/* Set if -u flagged	*/
int		wizard = 0;		/* Set if -w flagged	*/

char		line[LINESIZE];
char		macfile[81];
char		*section;		/* For bug()		*/
char		header[LINESIZE];	/* Header file name	*/

FILE		*commandfd;		/* Command input file	*/
FILE		*infd;			/* Input file		*/
FILE		*outfd;			/* Output file		*/
FILE		*workfd;		/* Temporary file	*/

main(argc, argv)
int		argc;
char		*argv[];

{
	register int	i;
	register char	*ap;
	register char	c;	

	commandfd = stdin;
	outfd = stdout;

	for (i = 1; i < argc; i++) {
		ap = argv[i];
		if (*ap == '-') {
			switch (c = tolower(ap[1])) {
			case 'd':	debug++;
					break;;

			case 'r':	rstshack++;
					break;

			case 'u':	uflag++;
					break;;

			case 'w':	wizard++;
					break;;

			default:
				argv[i] = 0;
				if (++i >= argc)
					error("?No file after \"%s\"\n", ap);
				switch (c) {

				case 'h':
					strcpy(header, argv[i]);
					break;

				case 'o':
					if ((outfd = fopen(argv[i], "w"))
							== NULL)
						cant("output", argv[i]);
					break;

				default:
					error("?Illegal switch '%c'\n", *ap);
				}
			}
			argv[i] = 0;		/* Erase this argument	*/
		}
	}
	/*
	 * Now open the work file and process all files
	 */
	if ((workfd = fopen(WORKFILE, "w")) == NULL)
		cant("work", WORKFILE);

	for (i = 1; i < argc; i++) {
		if ((ap = argv[i]) == 0)
			continue;		/* 'Twas a switch	*/
		if ((infd = fwild(ap, "r")) == NULL)
			cant("wild card input", ap);
		else {
			while (fnext(infd) != NULL) {
				filename(infd, macfile);
				if (debug)
					fprintf(stderr, "* %s\n", macfile);
				linect = 0;
				process();
			}
		}
	}				
	if (debug)
		fprintf(stderr, "* EOF\n");
	/*
	 * All file information read, now write it out
	 */
	fclose(workfd);
	doheader();
	if (debug)
		fprintf(stderr, "* Header processed\n");
	if ((workfd = fopen(WORKFILE, "r")) == NULL)
		cant("work (reopening)", WORKFILE);
	fprintf(outfd, ".lm 8.rm 72.nhy.hd mixed\n");
	output();
	fclose(outfd);
	fmkdl(workfd);			/* Delete work file		*/
}

filename(fd, outbuf)
FILE		*fd;
char		*outbuf;
/*
 * Get the file name, account number, and extension.
 * Remove the device name and the version number.
 */
{
	register char	*tp;
	register char	*op;
	register int	c;
	char		text[81];

	iovtoa(fd, text);
	/*
	 * Skip over the device name
	 */
	for (tp = text; (c = *tp++) && c != ':';);
	if (c == 0)
		tp = text;
	/*
	 * Don't bother outputting the version number
	 */
	for (op = tp; (c = *op) && c != ';'; op++);
	*op = 0;
	/*
	 * Copy the file spec, forcing lowercase.
	 */
	for (op = outbuf; (c = *tp++) ;) {
		*op++ = tolower(c);
	}
	*op = 0;
}

doheader()
/*
 * Process the header file
 */
{
	register int	inwizard;
	register int	skipit;
	register char	*lp;

	if (*header == '\0')
		return;
	if ((infd = fopen(header, "r")) == NULL) {
		fprintf(stderr, "?Can't open \"%s\" -- %06o\n",
				header, $$ferr);
		return;
	}
	strcpy(macfile, header);
	inwizard = 0;
	skipit = 0;
	linect = 0;
	while (getline(infd)) {
		if (match(line, ".comment wizard")) {
			if (inwizard)
				bug("headfile: embedded .comment wizard");
			inwizard = 1;
			skipit = !wizard;
		}
		else if (match(line, ".comment else")) {
			if (!inwizard)
				bug("headfile: else, no .comment wizard");
			else
				skipit = !skipit;
		}
		else if (match(line, ".comment mundane")) {
			if (!inwizard)
				bug("headfile: mundane, no .comment wizard");
			inwizard = 0;
			skipit = 0;
		}
		if (!skipit) {
			if (rstshack	&& line[0] == '.'
					&& tolower(line[1]) == 't'
					&& (line[2] == ' '
						|| line[2] == '\t'
						|| line[2] == ';')) {
				lp = &line[2];
				while (*lp && (*lp == ' ' || *lp == '\t'))
					lp++;
				if (*lp == ';')
					lp++;
				fprintf(outfd, ".t [[%s]]\n", lp);
			}
			else
				fprintf(outfd, "%s\n", line);
		}
	}
	fclose(infd);
	macfile[0] = 0;
}

static	int	fill_flag = 1;

process()
/*
 * Process input text, saving it in the workfile.
 */
{
	register int	flag;

	section = "Title scan";
	title();
	section = "Usage scan";
	flag = usage();
	if (flag == 2)
		return;
	else if (flag && !uflag) {
		section = "Document scan";
		rest();
	}
	fill_flag = 1;
	save("!!", 0);			/* Terminate entry	*/
}

long		place;
char		name_text[NAMESIZE];
char		title_text[LINESIZE];

title()
/*
 * First, skip to the title
 */
{
	register char		*lp;
	register char		*np;
	extern char		*skipbl();

	while (getline(infd)) {
		lp = skipbl(line);
		if (!match(lp, ".title"))
			continue;
		lp = skipbl(lp + 6);
		place = ftell(workfd);
		np = name_text;
		while (*lp > ' ' && np < &name_text[sizeof name_text - 1])
			*np++ = *lp++;
		while (np < &name_text[sizeof name_text])
			*np++ = 0;
		strcpy(title_text, lp);
		break;
	}
}

savename()
/*
 * Save info. in name/place which were setup by title()
 */
{
	register struct	entry	*ep;
	register struct entry	*lastep;

	/*
	 * Save in sorted order
	 */
	if ((lastep = free_entry++) >= ELAST)
		error("?Too many files, %d maximum", NENTRIES);
	for (ep = entries; ep < lastep &&
			strcmp(ep->e_name, name_text) <= 0; ep++);
	for (; lastep > ep; lastep--) {
		lastep->e_place = (lastep-1)->e_place;
		copy(lastep->e_name, (lastep-1)->e_name, NAMESIZE);
	}
	ep->e_place = place;
	copy(ep->e_name, name_text, NAMESIZE);
	/*
	 * Save the title line
	 */
	save((title_text[0]) ? title_text : "!", 1);	/* Title line	*/
	fill_flag = 1;
}


usage()
/*
 * Skip to ";+" (ignored what preceeds), then to "; Usage"
 * and put out usage section.  Return 1 if ok, 0 if trouble.
 */
{
	register char	*lp;
	register int	usage_seen;
	register int	skipit;

	skipit = 0;
	for (;;) {				/* Skip to ;+		*/
		if (!getline(infd))
			return(0);		/* Ignore if none	*/
		if (line[0] == ';' && line[1] == '+')
			break;
	}
	usage_seen = 0;
	while (getline(infd)) {
		if (line[0] != ';') {
			bug("Expecting ';', got \"%s\"\n", line);
			return(0);
		}
		else if (line[1] == 0) {
			if (usage_seen && !skipit)
				save(&line[1], 1);
			continue;
		}
		else if (line[1] == '-') {
			bug("No Description, etc.\n");
			return(0);
		}
		else if (line[1] == ' ') {
			if (match(&line[2], "internal")) {
				if (wizard) {		/* Do internal	*/
					if (usage_seen)
						unjust();
					continue;
				}
				else if (usage_seen) {
					skipit++;
					continue;
				}
				else	return(2);
			}
			else if (usage_seen == 0) {
				if (match(&line[2], "index"))
					continue;
				if (match(&line[2], "usage")) {
					savename();
					usage_seen++;
					save(".lm +8.nf", 0);
					if (wizard) {
						concat(line,
						    ".s.i -8;File name:\t",
						    macfile, 0);
						save(line, 0);
						save(".s 2", 0);
					}
					save(".i -8;Usage", 0);
					save("!b", 0);
					fill_flag = 0;
					continue;
				}
			}
			if (line[2] >= 'A') {
				if (usage_seen)
					save("!e", 0);
				return(1);
			}
			else if (usage_seen && line[2] == ' ') {
				if (!skipit) {
					unjust();
				}
				continue;
			}
		}
		else if (usage_seen) {
			if (line[1] == '\t') {
				if (!skipit)
					save(&line[2], 1);
				continue;
			}
		}
		bug("Ununderstandable line \"%s\"\n", line);
		return(0);
	}
	bug("No ;- at end of file\n");
	return(0);
}

rest()
/*
 * Output the remainder of the commentary.
 * The line buffer contains "; Description"
 */
{
	register int	skipit;

	skipit = 0;
	indent(&line[2]);
	while (getline(infd)) {
		if (line[0] != ';') {
			bug("Line doesn't start with a ';': \"%s\n",
				line);
			continue;
		}
		else if (line[1] == '-') {
			save(".lm -8.fill", 0);
			fill_flag = 1;
			return;
		}
		else if (line[1] == 0) {
			if (!skipit)
				save(&line[1], 1);
		}
		else if (line[1] == ' ') {
			if (line[2] < 'A') {
				if (!skipit) {
					unjust();
				}
			}
			else {
				if (!wizard) {
					if (match(&line[2], "internal"))
						skipit++;
					else	skipit = 0;
				}
				if (!skipit)
					indent(&line[2]);
			}
			continue;
		}
		else if (line[1] == '\t') {
			if (skipit)
				continue;
			else if (line[2] <= ' ') {
				nofill();
				save(&line[2], 1);
			}
			else {
				fill();
				save(&line[2], 1);
			}
			continue;
		}
		else
			bug("Ununderstandable line \"%s\"\n", line);
	}
	bug("Unexpected end of file\n");
	save(".lm -8.fill", 0);
}

unjust()
/*
 * Unjustify the line
 */
{
	register char	*lp;

	line[0] = '#' | 0200;
	for (lp = line + 1; *lp == ' ';)
		*lp++ = '#' | 0200;
	indent(line);
}

save(text, dotflag)
char		*text;
int		dotflag;
/*
 * Write the text to the work file.  If dotflag is set, initial '.'
 * is quoted.
 */
{
	if (dotflag && *text == '.')
		putc('_', workfd);
	fprintf(workfd, "%s\n", (*text != 0) ? text : ".s");
}

indent(text)
char		*text;
/*
 * Save an indented text item
 */
{
	char	buffer[LINESIZE];

	sprintf(buffer, ".i -8;%s", text);
	save(buffer, 0);
}

fill()
/*
 * Turn on fill mode
 */
{
	if (!fill_flag) {
		save(".fill", 0);
		fill_flag = 1;
	}
}

nofill()
/*
 * Turn off fill mode
 */
{
	if (fill_flag) {
		save(".nf", 0);
		fill_flag = 0;
	}
}

match(text, lookfor)
char		*text;
register char	*lookfor;
/*
 * If the beginning of text matches lookfor (ignoring case),
 * return true.  lookfor must always be in lowercase.
 */
{
	register int	c;
	register int	l;

	while ((l = *lookfor++) != 0) {
		if ((c = *text++) != l && (c + 040) != l)
			return(0);
	}					
	return(1);
}

output()
/*
 * Write out all records
 */
{
	int namlen;
	int stars;
	register char		*lp;
	register int		c;
	register struct	entry	*ep;

	for (ep = entries; ep < free_entry; ep++) {
		fseek(workfd, ep->e_place, 0);
		if (!getline(workfd)) {
			bug("Can't read record at %08lx\n", ep->e_place);
			continue;
		}
		if (uflag) {
			fprintf(outfd, ".tp 10.s 4.lm +8.nf\n.i -8 ;%s",
				ep->e_name);
			if (line[0] != '!')
				fprintf(outfd, "\t%s", line);
		}
		else {
			fprintf(outfd, ".st ########%s", ep->e_name);
			if (line[0] != '!')
				fprintf(outfd, "\t%s", line);
			fprintf(outfd, "\n.pg\n.hl 1 ");
			if (line[0] != '!')
				fprintf(outfd, "^&%s\\&\n", line+1);
			else
				fprintf(outfd, "#\n");
			fprintf(outfd, ".s 4\n.c ;");
			namlen = strlen(ep -> e_name);
			for (stars=0; stars < namlen+4; stars++)
				fprintf(outfd,"*");
			fprintf(outfd, "\n.c ;* %s *\n.c ;", ep -> e_name);
			for (stars=0; stars < namlen+4; stars++)
				fprintf(outfd,"*");
		}
		fprintf(outfd, "\n.s 2\n");
		if (uflag) {
			while (getline(workfd) &&
					line[0] != '!' && line[1] != 'b');
			while (getline(workfd)) {
				if (line[0] == '!' && (line[1] == 'e'
						|| line[1] == '!'))
					break;
				writeout(line);
			}
			writeout(".lm -8.fill");
			continue;
		}			
		while (getline(workfd)) {
			if (line[0] == '!') {
				if (line[1] == '!') {
					break;
				}
			}
			else writeout(line);		
		}
	}
}

writeout(text)
char	*text;
/*
 * Write this line to the .rno output, watching out for weird .rno
 * characters.  Note: to put a wierd .rno character out, flag it with
 * the parity bit.
 */
{
	register char	c;
	register char	*tp;

	tp = text;
	while ((c = *tp++) != 0) {
		switch (c) {
		case	'_':
				if (tp - text == 1 && *tp == '.')
					goto ignore;
		case	'%':
		case	'&':
		case	'\\':
		case	'^':
		case	'#':
				putc('_', outfd);
		default:
ignore:				putc(c & 0177, outfd);
		}
	}
	putc('\n', outfd);
}

getline(fd)
FILE	*fd;
/*
 * Read a line from fd into line[].  Return 0 at end of file.
 */
{
	register char	*lp;

	if (fgets(line, sizeof line, fd) == NULL) {
		if (debug)
			fprintf(stderr, "* end of \"%s\" after %u lines\n",
				macfile, linect);
		return(0);
	}	
	linect++;
	/*
	 * Erase trailing <Newline>, spaces, and tabs.  Note that
	 * line[strlen(line)] is the newline.
	 */
	for (lp = &line[strlen(line)]; lp > line && lp[-1] <= ' '; lp--);
	*lp = 0;
	if (debug > 1)
		fprintf(stderr, "\"%s\"\n", line);
	return(1);
}

char *skipbl(text)
register char	*text;
/*
 * Skip blanks, return -> first non blank (or -> trailing null)
 */
{
	register char	c;

	while ((c = *text) && c <= ' ')
		text++;
	return(text);
}

bug(arg)
{
	fprintf(stderr, "?Confused at %s", section);
	if (macfile[0])
		fprintf(stderr, " at line %u in file \"%s\"\n",
			linect, macfile);
	fprintf(stderr, "... %r", &arg);
}

cant(what, who)
char		*what;
char		*who;
/*
 * Can't open the file, die
 */
{
	error("?Can't open %s file \"%s\", fatal.\n", what, who);
}

                                                                                                                                                                                                                                                                                                        