/******************** MODULE INFO  ****************************/
/*
**  File name   :  chan.c
*/
/*  AUTHOR      :  Jan Erik Nilsen (jen@nilsenelektronikk.no) */
/*  VERSION     :  3.0                                        */
/*  DATE        :  Thu Aug 16 11:17:44 2001                   */
/*
*   Compiler    :  ANSI C
*   Project     :  proc  Real-Time Kernel
*   Functions   :  channel functions
*   Constants   :  see chan.h
*
*                  if you want to include memory allocation,
*                  define USE_MALLOC, and make your procMalloc
*
*
* Copyright (c) nilsen elektronikk as, Norway. (www.nilsenelektronikk.no)
* This software is the property of nilsen elektronikk as, Norway.
*
* The proc RTOS is free software; you can use it, redistribute it
* and/or modify it under the following terms:
* 1. You are not allowed to remove or modify this copyright notice
*    and License paragraphs, even if parts of the software is used.
* 2. The improvements and/or extentions you make SHALL be available
*    for the community under THIS license, source code included.
*    Improvements or extentions, including adaptions to new architectures,
*    SHALL be reported and transmitted to Nilsen Elektronikk AS.
* 3. You must cause the modified files to carry prominent notices stating
*    that you changed the files, what you did and the date of changes.
* 4. You may NOT distribute this software under another license without
*    explisit permission from Nilsen Elektronikk AS, Norway.
* 5. This software is free, and distributed in the hope that it will be
*    useful, but WITHOUT ANY WARRANTY; without even the implied warranty
*    of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*    You SHALL NOT use this software unless you accept to carry all
*    risk and cost of defects or limitations.
*
*    ------------  CHANGE RECORD  ----------------
*    Jan Erik Nilsen (jen@nilsenelektronikk.no) July 26. 1999:
*        First free version of this software published.
*
*    Andrey Melnikov, Russia. (melnikov@sietlab.com) Aug 15. 2001
*        To ensure atomic FIFO operations, disable_interrupt() and
*        enable interrupt() are used in chanInChar() and chanOutChar()
*
*    Naohiko Shimizu (nshimizu@keyaki.cc.u-tokai.ac.jp) Oct. 15.2002:
*        Add conditional compile statements for smaller object on GCC
*/
#define  NULL (void *)0
#include "proc.h"
#include "chan.h"






/*===========  chanInit  ============================
*
*   Purpose:      Initialize an character channel.
*                 Must be done before a channel is used.
*
*   Input:        ptr to channel
*                 size of channel [bytes]
*   Output:       none
*   Return:       If OK: 1, if malloc problem: 0.
*
*/
#ifdef L_chanInit
int chanInit(CHAN *cp, int size)
{
  procSemInit(&(cp->used), 0);
  procSemInit(&(cp->left), size);
  cp->ii   = cp->oi = 0;
  cp->size = size;
#ifdef USE_MALLOC
  if (cp->fifo == NULL) cp->fifo = (unsigned char *)procMalloc(size);
#endif
  return((cp->fifo)? 1 : 0);
}

#endif

/*===========  chanOutChar  ================================
*
*   Purpose:      Put a byte into a channel.
*                 If the channel is full, the function
*                 will block until space is available.
*
*   Input:        ptr to chan
*                 byte
*   Output:       none
*   Return:       none
*
*/
#ifdef L_chanOutChar
void chanOutChar(CHAN *cp, int byte)
{
  int i;
#ifdef REINT
  int sr;
#endif

  procSemWait(&(cp->left));
  disable_interrupt();
  if ((i = (cp->ii + 1)) >= cp->size) i = 0;
  cp->fifo[i] = (unsigned char)byte;
  cp->ii = i;
  enable_interrupt();
  procSemSignal(&(cp->used));
}

#endif

/*===========  chanOutNB  ===================================
*
*   Purpose:      Put a byte buffer of a given size
*                 into a channel.
*                 If the channel is full, the function
*                 will block until space is available.
*
*   Input:        ptr to chan
*                 ptr to byte buffer
*                 number of bytes to transmit
*   Output:       none
*   Return:       none
*
*/
#ifdef L_chanOutNB
void chanOutNB(CHAN *cp, char *msg, int num)
{
  while (num--) chanOutChar(cp, *msg++);
}

#endif

/*===========  chanInChar  ===============================
*
*   Purpose:      Block until a byte can be
*                 received from a channel.
*
*   Input:        ptr to chan
*   Output:       none
*   Return:       byte
*
*/
#ifdef L_chanInChar
int chanInChar(CHAN *cp)
{
  int  i, byte;
#ifdef REINT
  int sr;
#endif

  procSemWait(&(cp->used));
  disable_interrupt();
  if ((i = (cp->oi + 1)) >= cp->size) i = 0;
  byte = cp->fifo[i];
  cp->oi = i;
  enable_interrupt();
  procSemSignal(&(cp->left));
  return(byte);
}

#endif

/*===========  chanInNB  ===============================
*
*   Purpose:      Block until a string of bytes can be
*                 received from a channel.
*
*   Input:        ptr to chan
*                 ptr to byte buffer
*                 number of bytes to receive
*   Output:       byte buffer
*   Return:       none
*
*/
#ifdef L_chanInNB
void chanInNB(CHAN *cp, char *msg, int num)
{
  while (num--) *msg++ = (char)chanInChar(cp);
}
#endif
