/* $PDP11: ethtimed.c,v 1.2 2014/06/09 13:57:51 form Exp $ */

/*
 * Copyright (c) 2014 Oleg Safiullin <form@pdp-11.org.ru>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/utsname.h>

#if defined(__FreeBSD__) || defined(__NetBSD__)
#include <sys/endian.h>
#include <arpa/inet.h>
#endif	/* __FreeBSD__ || __NetBSD__ */

#ifdef __linux__
#include <sys/socket.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <endian.h>
#endif	/* __linux__ */

#include <err.h>
#include <pcap.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include "ethtimed.h"


#ifdef __FreeBSD__
#define __dead			__dead2
#endif

#ifdef __linux__
#define __dead			__attribute__((__noreturn__))
#endif

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__linux__)
#if BYTE_ORDER == LITTLE_ENDIAN
#define letoh16(x)	(x)
#endif

#if BYTE_ORDER == BIG_ENDIAN
#define letoh16(x)	ntohs(x)
#endif
#endif	/* __FreeBSD__ || __linux__ */


static char *interface;
static int do_debug;
static int do_log;
static int print_version;


static __dead void
usage(void)
{
	extern char *__progname;

	(void)fprintf(stderr, "usage: %s [-dlV] [-i interface]\n", __progname);
	_exit(1);
}

static const char *
ether_addr(const u_int8_t *ea)
{
	static char addr[18];

	(void)snprintf(addr, sizeof(addr), "%02x:%02x:%02x:%02x:%02x:%02x",
	    ea[0], ea[1], ea[2], ea[3], ea[4], ea[5]);

	return (addr);
}

static void
send_time(pcap_t *p, const struct ethtim *etp, u_int16_t hz, struct ethtim *et)
{
	struct tm *tm;
	time_t t;
	int i;

	if (do_debug || do_log) {
		if (etp->et_ident[0] == '\0') {
			if (do_debug)
				(void)fprintf(stderr,
				    "Request from %s, %u Hz\n",
				    ether_addr(etp->et_src),
				    letoh16(etp->et_hz));
			if (do_log)
				syslog(LOG_NOTICE, "request from %s, %u Hz",
				    ether_addr(etp->et_src),
				    letoh16(etp->et_hz));
		} else {
			if (do_debug)
				(void)fprintf(stderr, "Request from %s, %u Hz, %s\n",
				    ether_addr(etp->et_src),
				    letoh16(etp->et_hz), etp->et_ident);
			if (do_log)
				syslog(LOG_NOTICE, "request from %s, %u Hz, %s",
				    ether_addr(etp->et_src),
				    letoh16(etp->et_hz), etp->et_ident);
		}
	}
	bcopy(etp->et_src, et->et_dst, sizeof(et->et_dst));

	for (i = 0; i < ETHTIM_SEND_COUNT; i++) {
		t = time(NULL);
		tm = localtime(&t);

		et->et_rsx_tkps = et->et_hz = etp->et_hz;
		et->et_rsx_year = htole16(tm->tm_year);
		et->et_rsx_month = htole16(tm->tm_mon + 1);
		et->et_rsx_day = htole16(tm->tm_mday);
		et->et_rsx_hour = htole16(tm->tm_hour);
		et->et_rsx_minute = htole16(tm->tm_min);
		et->et_rsx_second = htole16(tm->tm_sec);

		if ((hz != 50 && hz != 60) || (tm->tm_year < 72 ||
		    tm->tm_year > 2099)) {
			et->et_rt_time[0] = et->et_rt_time[1] = 0177777;
			et->et_rt_date = 0177777;
		} else {
			u_int32_t tval;
			u_int16_t y, m, d;

			tval = (tm->tm_hour * 3600 + tm->tm_min * 60 +
			    tm->tm_sec) * hz;
			et->et_rt_time[0] = htole16(tval >> 16);
			et->et_rt_time[1] = htole16(tval & 0177777);

			/* XXX endian */
			if (tm->tm_year >= 72 && tm->tm_year <= 199) {
				y = tm->tm_year - 72;
				m = tm->tm_mon + 1;
				d = tm->tm_mday;
				et->et_rt_date = htole16((y & 037) |
				    ((y & 0140) << 9) | (m << 10) | (d << 5));
			} else
				et->et_rt_date = 0177777;
		}

		(void)pcap_inject(p, et, sizeof(*et));
		(void)usleep(300000);
	}
}

int
main(int argc, char *const argv[])
{
	static struct bpf_insn insn[] = {
		BPF_STMT(BPF_LD + BPF_H + BPF_ABS, 12),
		BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, ETHTIM_TYPE, 1, 0),
		BPF_STMT(BPF_RET + BPF_K, 0),
		BPF_STMT(BPF_RET + BPF_K, ETHTIM_SNAPLEN)
	};

	static struct bpf_program program = {
		sizeof(insn) / sizeof(insn[0]),
		insn
	};

#ifdef __linux__
	struct ifreq ifr;
#endif

	char errbuf[PCAP_ERRBUF_SIZE];
	struct ethtim et;
	struct utsname un;
	pcap_t *p;
	int ch;

	while ((ch = getopt(argc, argv, "di:lV")) != -1)
		switch (ch) {
		case 'd':
			do_debug++;
			break;
		case 'i':
			interface = optarg;
			break;
		case 'l':
			do_log++;
			break;
		case 'V':
			print_version++;
			break;
		default:
			usage();
			/*NOTREACHED*/
		}
	argc -= optind;
	argv += optind;

	if (print_version) {
		(void)fprintf(stderr, "ethtimed v%u.%u, %s\n",
		    ETHTIM_VERSION_MAJOR, ETHTIM_VERSION_MINOR,
		    pcap_lib_version());
		return (0);
	}

	if (uname(&un) < 0)
		err(1, "uname");

	bzero(&et, sizeof(et));
	et.et_type = htons(ETHTIM_TYPE);
	et.et_code = htons(ETHTIM_CODE_ANSWER);
	(void)snprintf(et.et_ident, sizeof(et.et_ident), "%s %s %s",
	    un.sysname, un.release, un.machine);

	if (interface == NULL && (interface = pcap_lookupdev(errbuf)) == NULL)
		errx(1, "%s", errbuf);

	if ((p = pcap_open_live(interface, ETHTIM_SNAPLEN, 0, 0,
	    errbuf)) == NULL)
		errx(1, "%s", errbuf);

	switch ((ch = pcap_datalink(p))) {
	case DLT_EN10MB:
	case DLT_EN3MB:
	case DLT_IEEE802:
		break;
	default:
		errx(1, "Unsupported datalink type: %d", ch);
	}

#ifdef __linux__
	bzero(&ifr, sizeof(ifr));
	(void)snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), "%s", interface);

	if ((ch = socket(PF_INET, SOCK_DGRAM, 0)) < 0)
		err(1, "socket");

	if (ioctl(ch, SIOCGIFHWADDR, &ifr) , 0)
		err(1, "SIOCGIFHWADDR");

	(void)close(ch);
	bcopy(&ifr.ifr_hwaddr.sa_data, et.et_src, sizeof(et.et_src));
#endif	/* __linux__ */

	if (pcap_setfilter(p, &program) < 0)
		errx(1, "%s", pcap_geterr(p));

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
	ch = 1;
	if (ioctl(pcap_fileno(p), BIOCIMMEDIATE, &ch) < 0)
		err(1, "BIOCIMMEDIATE");

	ch = 0;
	if (ioctl(pcap_fileno(p), BIOCSHDRCMPLT, &ch) < 0)
		err(1, "BIOCSHDRCMPLT");
#endif	/* __FreeBSD__ || __NetBSD__ || __OpenBSD__ */

#ifdef __OpenBSD__
	ch = BPF_DIRECTION_OUT;
	if (ioctl(pcap_fileno(p), BIOCSDIRFILT, &ch) < 0)
		err(1, "BIOCSDIRFILT");
#endif	/* __OpenBSD__ */

#ifdef __FreeBSD__
	ch = BPF_D_IN;
	if (ioctl(pcap_fileno(p), BIOCSDIRECTION, &ch) < 0)
		err(1, "BIOCSDIRECTION");
#endif	/* __FreeBSD__ */

	if (!do_debug && daemon(0, 0) < 0)
		err(1, "daemon");

	if (do_log)
		openlog("ethtimed", LOG_PID, LOG_DAEMON);

	if (do_debug)
		(void)fprintf(stderr, "Interface: %s, Ident: %s\n", interface,
		    et.et_ident);

	for (;;) {
		const struct ethtim *etp;
		struct pcap_pkthdr h;
		u_int16_t hz;

		if ((etp = (const struct ethtim *)pcap_next(p, &h)) == NULL ||
		    h.caplen != sizeof(struct ethtim) ||
		    ntohs(etp->et_code) != ETHTIM_CODE_REQUEST ||
		    (hz = letoh16(etp->et_hz)) > 1000)
			continue;
		send_time(p, etp, hz, &et);
	}

	return (0);
}
