/*
 * Copyright (c) 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)hk.c	2.1 (2.11BSD GTE) 1995/04/13
 */

/*
 * RK611/RK0[67] disk driver
 *
 * This driver mimics the 4.1bsd rk driver.
 * It does overlapped seeks, ECC, and bad block handling.
 * 	salkind@nyu
 *
 * dkunit() takes a 'dev_t' now instead of 'buf *'.  1995/04/13 - sms
 *
 * Modified to correctly handle 22 bit addressing available on DILOG
 * DQ615 controller. 05/31/90 -- tymann@oswego.edu
 *
 * Removed ifdefs on both Q22 and UNIBUS_MAP, substituting a runtime
 * test for presence of a Unibus Map.  Reworked the partition logic,
 * the 'e' partiton no longer overlaps the 'a'+'b' partitions - a separate
 * 'b' partition is now present.  Old root filesystems can still be used
 * because the size is the same, but user data will have to be saved and
 * then reloaded.  12/28/92 -- sms@wlv.iipo.gtegsc.com
 */

#include "hk.h"
#if	NHK > 0
#include "param.h"
#include "systm.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "map.h"
#include "uba.h"
#include "hkreg.h"
#include "dkbad.h"
#include "dk.h"
#include "disklabel.h"
#include "disk.h"
#include "syslog.h"

#define	NHK7CYL	815
#define	NHK6CYL	411
#define	HK_NSECT	22
#define	HK_NTRAC	3
#define	HK_NSPC		(HK_NTRAC*HK_NSECT)

struct	hkdevice *HKADDR;
struct size {
	daddr_t	nblocks;
	int	cyloff;
} hk6_sizes[8] =
	{
	8316,	0,	/* a: cyl    0 - 125 */
	8316,	90,	/* b: cyl  126 - 251 */
	27126,	0,	/* c: cyl    0 - 410, whole RK06 */
	0,	0,	/* d: Not Defined */
	0,	0,	/* e: Not Defined */
	0,	0,	/* f: Not Defined */
	10428,	252,	/* g: cyl  252 - 409 */
	27126,	0,	/* h: cyl   0 - 409, whole RK06 less 1 track */
	},
hk7_sizes[8] =
	{
	8316,	0,	/* a: cyl   0 -  125 */
	8316,	126,	/* b: cyl  126 - 251 */
	53790,	0,	/* c: cyl   0 - 814, whole RK07 */
	0,	0,	/* d: Not Defined */
	0,	0,	/* e: Not Defined */
	0,	0,	/* f: Not Defined */
	37092,	0,	/* g: cyl   252 - 813 */
	53724,	0	/* h: cyl   0 - 813, whole RK07 less 1 track */
};

/* Can be u_char because all are less than 0377 */
u_char	hk_offset[] =
{
	HKAS_P400,	HKAS_M400,	HKAS_P400,	HKAS_M400,
	HKAS_P800,	HKAS_M800,	HKAS_P800,	HKAS_M800,
	HKAS_P1200,	HKAS_M1200,	HKAS_P1200,	HKAS_M1200,
	0,		0,		0,		0,
};

int	hk_type[NHK];
int	hk_cyl[NHK];
struct	size *hk_sizes[NHK];
char	hk_pack[NHK];

struct hk_softc {
	int	sc_softas;
	int	sc_recal;
} hk;

struct	buf	hktab;
struct	buf	hkutab[NHK];
#ifdef BADSECT
struct	dkbad	hkbad[NHK];
struct	buf	bhkbuf[NHK];
#endif

#ifdef UCB_METER
static	int		hk_dkn = -1;	/* number for iostat */
#endif

#define	hkwait(hkaddr)		while ((hkaddr->hkcs1 & HK_CRDY) == 0)
#define	hkncyl(unit)		(hk_type[unit] ? NHK7CYL : NHK6CYL)
#define	hkunit(dev)		(((dev) >> 3) & 07)

void
hkroot()
{
	hkattach((struct hkdevice *)0177440, 0);
}

hkattach(addr, unit)
struct hkdevice *addr;
{
#ifdef UCB_METER
	if (hk_dkn < 0) {
		dk_alloc(&hk_dkn, NHK+1, "hk", 60L * (long)HK_NSECT * 256L);
		if (hk_dkn >= 0)
			dk_wps[hk_dkn+NHK] = 0L;
	}
#endif
	if (unit != 0)
		return(0);
	HKADDR = addr;
	return(1);
}

hkopen(dev, flag)
	dev_t dev;
	int flag;
{
	register int unit = hkunit(dev);
	register struct hkdevice *hkaddr = HKADDR;

	if (unit >= NHK || !HKADDR)
		return (ENXIO);

	hk_type[unit] = 0;
	hkaddr->hkcs1 = HK_CCLR;
	hkaddr->hkcs2 = unit;
	hkaddr->hkcs1 = HK_DCLR | HK_GO;
	hkwait(hkaddr);
	if((hkaddr->hkcs2&HKCS2_NED) || (hkaddr->hkds&HKDS_SVAL) == 0) {
		hkaddr->hkcs1 = HK_CCLR;
		hkwait(hkaddr);
		return(ENXIO);
	}
	if((hkaddr->hkcs1&HK_CERR) && (hkaddr->hker&HKER_DTYE)) {
		hk_type[unit] = HK_CDT;
		hkaddr->hkcs1 = HK_CCLR;
		hkwait(hkaddr);
		hk_sizes[unit] = hk7_sizes;
	}
	else
		hk_sizes[unit] = hk6_sizes;
	hk_cyl[unit] = -1;
	return(0);
}

hkstrategy(bp)
register struct buf *bp;
{
	register struct buf *dp;
	register unit;
	int s, part;
	long bn;
	long sz;
	struct size *szp;

	unit = dkunit(bp->b_dev);
	part = bp->b_dev & 7;
	if (unit >= NHK || !HKADDR  || !(szp = hk_sizes[unit])) {
		bp->b_error = ENXIO;
		goto bad;
	}
	sz = (bp->b_bcount + (NBPG-1)) >> PGSHIFT;
	if (bp->b_blkno < 0 || (bn = bp->b_blkno)+sz > szp[part].nblocks) {
		bp->b_error = EINVAL;
		goto bad;
	}
	bp->b_cylin = bn / HK_NSPC + szp[part].cyloff;
	mapalloc(bp);
	dp = &hkutab[unit];
	s = splbio();
	disksort(dp, bp);
	if (dp->b_active == 0) {
		hkustart(unit);
		if (hktab.b_active == 0)
			hkstart();
	}
	splx(s);
	return;
bad:
	bp->b_flags |= B_ERROR;
	iodone(bp);
}

hkustart(unit)
	int unit;
{
	register struct hkdevice *hkaddr = HKADDR;
	register struct buf *bp, *dp;
	int didie = 0;

#ifdef UCB_METER
	if (hk_dkn >= 0)
		dk_busy &= ~(1 << (hk_dkn + unit));
#endif
	if (hktab.b_active) {
		hk.sc_softas |= (1 << unit);
		return(0);
	}

	hkaddr->hkcs1 = HK_CCLR;
	hkaddr->hkcs2 = unit;
	hkaddr->hkcs1 = hk_type[unit] | HK_DCLR | HK_GO;
	hkwait(hkaddr);

	dp = &hkutab[unit];
	if ((bp = dp->b_actf) == NULL)
		return(0);
	if (dp->b_active)
		goto done;
	dp->b_active = 1;
	if ((hkaddr->hkds & HKDS_VV) == 0 || hk_pack[unit] == 0) {
		/* SHOULD WARN SYSTEM THAT THIS HAPPENED */
#ifdef BADSECT
		struct buf *bbp = &bhkbuf[unit];
#endif

		hkaddr->hkcs1 = hk_type[unit]|HK_PACK|HK_GO;
		hk_pack[unit]++;
#ifdef BADSECT
		bbp->b_flags = B_READ|B_BUSY|B_PHYS;
		bbp->b_dev = bp->b_dev;
		bbp->b_bcount = sizeof(struct dkbad);
		bbp->b_un.b_addr = (caddr_t)&hkbad[unit];
		bbp->b_blkno = (long)hkncyl(unit)*HK_NSPC - HK_NSECT;
		bbp->b_cylin = hkncyl(unit) - 1;
		mapalloc(bbp);
		dp->b_actf = bbp;
		bbp->av_forw = bp;
		bp = bbp;
#endif
		hkwait(hkaddr);
	}
	if ((hkaddr->hkds & HKDS_DREADY) != HKDS_DREADY)
		goto done;
#ifdef NHK > 1
	if (bp->b_cylin == hk_cyl[unit])
		goto done;
	hkaddr->hkcyl = bp->b_cylin;
	hk_cyl[unit] = bp->b_cylin;
	hkaddr->hkcs1 = hk_type[unit] | HK_IE | HK_SEEK | HK_GO;
	didie = 1;
#ifdef UCB_METER
	if (hk_dkn >= 0) {
		int dkn = hk_dkn + unit;

		dk_busy |= 1<<dkn;
		dk_seek[dkn]++;
	}
#endif
	return (didie);
#endif NHK > 1

done:
	if (dp->b_active != 2) {
		dp->b_forw = NULL;
		if (hktab.b_actf == NULL)
			hktab.b_actf = dp;
		else
			hktab.b_actl->b_forw = dp;
		hktab.b_actl = dp;
		dp->b_active = 2;
	}
	return (didie);
}

hkstart()
{
	register struct buf *bp, *dp;
	register struct hkdevice *hkaddr = HKADDR;
	daddr_t bn;
	int sn, tn, cmd, unit;

loop:
	if ((dp = hktab.b_actf) == NULL)
		return(0);
	if ((bp = dp->b_actf) == NULL) {
		hktab.b_actf = dp->b_forw;
		goto loop;
	}
	hktab.b_active++;
	unit = dkunit(bp->b_dev);
	bn = bp->b_blkno;

	sn = bn % HK_NSPC;
	tn = sn / HK_NSECT;
	sn %= HK_NSECT;
retry:
	hkaddr->hkcs1 = HK_CCLR;
	hkaddr->hkcs2 = unit;
	hkaddr->hkcs1 = hk_type[unit] | HK_DCLR | HK_GO;
	hkwait(hkaddr);

	if ((hkaddr->hkds & HKDS_SVAL) == 0)
		goto nosval;
	if (hkaddr->hkds & HKDS_PIP)
		goto retry;
	if ((hkaddr->hkds&HKDS_DREADY) != HKDS_DREADY) {
		log(LOG_WARNING, "hk%d: not ready\n", unit);
		if ((hkaddr->hkds&HKDS_DREADY) != HKDS_DREADY) {
			printf("\n");
			hkaddr->hkcs1 = hk_type[unit] | HK_DCLR | HK_GO;
			hkwait(hkaddr);
			hkaddr->hkcs1 = HK_CCLR;
			hkwait(hkaddr);
			hktab.b_active = 0;
			hktab.b_errcnt = 0;
			dp->b_actf = bp->av_forw;
			dp->b_active = 0;
			bp->b_flags |= B_ERROR;
			iodone(bp);
			goto loop;
		}
	}
nosval:
	hkaddr->hkcyl = bp->b_cylin;
	hk_cyl[unit] = bp->b_cylin;
	hkaddr->hkda = (tn << 8) + sn;
	hkaddr->hkwc = -(bp->b_bcount >> 1);
	hkaddr->hkba = bp->b_un.b_addr;
	if	(!ubmap)
		hkaddr->hkxmem=bp->b_xmem;

	cmd = hk_type[unit] | ((bp->b_xmem & 3) << 8) | HK_IE | HK_GO;
	if (bp->b_flags & B_READ)
		cmd |= HK_READ;
	else
		cmd |= HK_WRITE;
	hkaddr->hkcs1 = cmd;
#ifdef UCB_METER
	if (hk_dkn >= 0) {
		int dkn = hk_dkn + NHK;

		dk_busy |= 1<<dkn;
		dk_xfer[dkn]++;
		dk_wds[dkn] += bp->b_bcount>>6;
	}
#endif
	return(1);
}

hkintr()
{
	register struct hkdevice *hkaddr = HKADDR;
	register struct buf *bp, *dp;
	int unit;
	int as = (hkaddr->hkatt >> 8) | hk.sc_softas;
	int needie = 1;

	hk.sc_softas = 0;
	if (hktab.b_active) {
		dp = hktab.b_actf;
		bp = dp->b_actf;
		unit = dkunit(bp->b_dev);
#ifdef UCB_METER
		if (hk_dkn >= 0)
			dk_busy &= ~(1 << (hk_dkn + NHK));
#endif
#ifdef BADSECT
		if (bp->b_flags&B_BAD)
			if (hkecc(bp, CONT))
				return;
#endif
		if (hkaddr->hkcs1 & HK_CERR) {
			int recal;
			u_short ds = hkaddr->hkds;
			u_short cs2 = hkaddr->hkcs2;
			u_short er = hkaddr->hker;

			if (er & HKER_WLE) {
				log(LOG_WARNING, "hk%d: write locked\n", unit);
				bp->b_flags |= B_ERROR;
			} else if (++hktab.b_errcnt > 28 ||
			    ds&HKDS_HARD || er&HKER_HARD || cs2&HKCS2_HARD) {
hard:
				harderr(bp, "hk");
				log(LOG_WARNING, "cs2=%b ds=%b er=%b\n",
				    cs2, HKCS2_BITS, ds, 
				    HKDS_BITS, er, HKER_BITS);
				bp->b_flags |= B_ERROR;
				hk.sc_recal = 0;
			} else if (er & HKER_BSE) {
#ifdef BADSECT
				if (hkecc(bp, BSE))
					return;
				else
#endif
					goto hard;
			} else
				hktab.b_active = 0;
			if (cs2&HKCS2_MDS) {
				hkaddr->hkcs2 = HKCS2_SCLR;
				goto retry;
			}
			recal = 0;
			if (ds&HKDS_DROT || er&(HKER_OPI|HKER_SKI|HKER_UNS) ||
			    (hktab.b_errcnt&07) == 4)
				recal = 1;
			if ((er & (HKER_DCK|HKER_ECH)) == HKER_DCK)
				if (hkecc(bp, ECC))
					return;
			hkaddr->hkcs1 = HK_CCLR;
			hkaddr->hkcs2 = unit;
			hkaddr->hkcs1 = hk_type[unit]|HK_DCLR|HK_GO;
			hkwait(hkaddr);
			if (recal && hktab.b_active == 0) {
				hkaddr->hkcs1 = hk_type[unit]|HK_IE|HK_RECAL|HK_GO;
				hk_cyl[unit] = -1;
				hk.sc_recal = 0;
				goto nextrecal;
			}
		}
retry:
		switch (hk.sc_recal) {

		case 1:
			hkaddr->hkcyl = bp->b_cylin;
			hk_cyl[unit] = bp->b_cylin;
			hkaddr->hkcs1 = hk_type[unit]|HK_IE|HK_SEEK|HK_GO;
			goto nextrecal;
		case 2:
			if (hktab.b_errcnt < 16 ||
			    (bp->b_flags&B_READ) == 0)
				goto donerecal;
			hkaddr->hkatt = hk_offset[hktab.b_errcnt & 017];
			hkaddr->hkcs1 = hk_type[unit]|HK_IE|HK_OFFSET|HK_GO;
			/* fall into ... */
		nextrecal:
			hk.sc_recal++;
			hkwait(hkaddr);
			hktab.b_active = 1;
			return;
		donerecal:
		case 3:
			hk.sc_recal = 0;
			hktab.b_active = 0;
			break;
		}
		if (hktab.b_active) {
			hktab.b_active = 0;
			hktab.b_errcnt = 0;
			hktab.b_actf = dp->b_forw;
			dp->b_active = 0;
			dp->b_errcnt = 0;
			dp->b_actf = bp->av_forw;
			bp->b_resid = -(hkaddr->hkwc << 1);
			iodone(bp);
			if (dp->b_actf)
				if (hkustart(unit))
					needie = 0;
		}
		as &= ~(1<<unit);
	}
	for (unit = 0; as; as >>= 1, unit++)
		if (as & 1) {
			if (unit < NHK && hk_sizes[unit]) {
				if (hkustart(unit))
					needie = 0;
			} else {
				hkaddr->hkcs1 = HK_CCLR;
				hkaddr->hkcs2 = unit;
				hkaddr->hkcs1 = HK_DCLR | HK_GO;
				hkwait(hkaddr);
				hkaddr->hkcs1 = HK_CCLR;
			}
		}
	if (hktab.b_actf && hktab.b_active == 0)
		if (hkstart())
			needie = 0;
	if (needie)
		hkaddr->hkcs1 = HK_IE;
}

#ifdef HK_DUMP
/*
 *  Dump routine for RK06/07
 *  Dumps from dumplo to end of memory/end of disk section for minor(dev).
 *  It uses the UNIBUS map to dump all of memory if there is a UNIBUS map.
 */
#define	DBSIZE	(UBPAGE/NBPG)		/* unit of transfer, one UBPAGE */

hkdump(dev)
	dev_t dev;
{
	register struct hkdevice *hkaddr = HKADDR;
	daddr_t	bn, dumpsize;
	long paddr;
	register count;
	register struct ubmap *ubp;
	int com, cn, tn, sn, unit;
	struct size *szp;

	unit = hkunit(dev);
	szp = hk_sizes[unit];
	if (unit >= NHK || !szp)
		return(EINVAL);
	dumpsize = szp[dev & 7]->nblocks;
	if ((dumplo < 0) || (dumplo >= dumpsize))
		return(EINVAL);
	dumpsize -= dumplo;

	hkaddr->hkcs1 = HK_CCLR;
	hkwait(hkaddr);
	hkaddr->hkcs2 = unit;
	hkaddr->hkcs1 = hk_type[unit] | HK_DCLR | HK_GO;
	hkwait(hkaddr);
	if ((hkaddr->hkds & HKDS_VV) == 0) {
		hkaddr->hkcs1 = hk_type[unit]|HK_IE|HK_PACK|HK_GO;
		hkwait(hkaddr);
	}
	ubp = &UBMAP[0];
	for (paddr = 0L; dumpsize > 0; dumpsize -= count) {
		count = dumpsize>DBSIZE? DBSIZE: dumpsize;
		bn = dumplo + (paddr >> PGSHIFT);
		cn = (bn/HK_NSPC) + szp[dev & 7]->cyloff;
		sn = bn%HK_NSPC;
		tn = sn/HK_NSECT;
		sn = sn%HK_NSECT;
		hkaddr->hkcyl = cn;
		hkaddr->hkda = (tn << 8) | sn;
		hkaddr->hkwc = -(count << (PGSHIFT-1));
		com = hk_type[unit]|HK_GO|HK_WRITE;
		if (ubmap) {
			ubp->ub_lo = loint(paddr);
			ubp->ub_hi = hiint(paddr);
			hkaddr->hkba = 0;
		} else {
			/* non UNIBUS map */
			hkaddr->hkba = loint(paddr);
			hkaddr->hkxmem = hiint(paddr);
			com |= ((paddr >> 8) & (03 << 8));
		}
		hkaddr->hkcs2 = unit;
		hkaddr->hkcs1 = com;
		hkwait(hkaddr);
		if (hkaddr->hkcs1 & HK_CERR) {
			if (hkaddr->hkcs2 & HKCS2_NEM)
				return(0);	/* made it to end of memory */
			return(EIO);
		}
		paddr += (DBSIZE << PGSHIFT);
	}
	return(0);		/* filled disk minor dev */
}
#endif HK_DUMP

#define	exadr(x,y)	(((long)(x) << 16) | (unsigned)(y))

/*
 * Correct an ECC error and restart the i/o to complete
 * the transfer if necessary.  This is quite complicated because
 * the transfer may be going to an odd memory address base
 * and/or across a page boundary.
 */
hkecc(bp, flag)
register struct	buf *bp;
{
	register struct	hkdevice *hkaddr = HKADDR;
	ubadr_t	addr;
	int npx, wc;
	int cn, tn, sn;
	daddr_t	bn;
	unsigned ndone;
	int cmd;
	int unit;

#ifdef BADSECT
	if (flag == CONT) {
		npx = bp->b_error;
		ndone = npx * NBPG;
		wc = ((int)(ndone - bp->b_bcount)) / NBPW;
	} else
#endif
		{
		wc = hkaddr->hkwc;
		ndone = (wc * NBPW) + bp->b_bcount;
		npx = ndone / NBPG;
		}
	unit = dkunit(bp->b_dev);
	bn = bp->b_blkno;
	cn = bp->b_cylin - bn / HK_NSPC;
	bn += npx;
	cn += bn / HK_NSPC;
	sn = bn % HK_NSPC;
	tn = sn / HK_NSECT;
	sn %= HK_NSECT;
	hktab.b_active++;

	switch (flag) {
	case ECC:
		{
		register byte;
		int bit;
		long mask;
		ubadr_t bb;
		unsigned o;
		struct ubmap *ubp;

		log(LOG_WARNING, "hk%d%c:  soft ecc sn %D\n",
			unit, 'a' + (bp->b_dev & 07), bp->b_blkno + npx - 1);
		mask = hkaddr->hkecpt;
		byte = hkaddr->hkecps - 1;
		bit = byte & 07;
		byte >>= 3;
		mask <<= bit;
		o = (ndone - NBPG) + byte;
		bb = exadr(bp->b_xmem, bp->b_un.b_addr);
		bb += o;
		if (ubmap && (bp->b_flags & (B_MAP|B_UBAREMAP))) {
			ubp = UBMAP + ((bb >> 13) & 037);
			bb = exadr(ubp->ub_hi, ubp->ub_lo) + (bb & 017777);
		}
		/*
		 * Correct until mask is zero or until end of
		 * sector or transfer, whichever comes first.
		 */
		while (byte < NBPG && o < bp->b_bcount && mask != 0) {
			putmemc(bb, getmemc(bb) ^ (int)mask);
			byte++;
			o++;
			bb++;
			mask >>= 8;
		}
		if (wc == 0)
			return(0);
		break;
	}

#ifdef BADSECT
	case BSE:
		if ((bn = isbad(&hkbad[unit], cn, tn, sn)) < 0)
			return(0);
		bp->b_flags |= B_BAD;
		bp->b_error = npx + 1;
		bn = (long)hkncyl(unit)*HK_NSPC - HK_NSECT - 1 - bn;
		cn = bn/HK_NSPC;
		sn = bn%HK_NSPC;
		tn = sn/HK_NSECT;
		sn %= HK_NSECT;
		wc = -(NBPG / NBPW);
		break;

	case CONT:
		bp->b_flags &= ~B_BAD;
		if (wc == 0)
			return(0);
		break;
#endif BADSECT
	}
	/*
	 * Have to continue the transfer.  Clear the drive
	 * and compute the position where the transfer is to continue.
	 * We have completed npx sectors of the transfer already.
	 */
	hkaddr->hkcs1 = HK_CCLR;
	hkwait(hkaddr);
	hkaddr->hkcs2 = unit;
	hkaddr->hkcs1 = hk_type[unit] | HK_DCLR | HK_GO;
	hkwait(hkaddr);

	addr = exadr(bp->b_xmem, bp->b_un.b_addr);
	addr += ndone;
	hkaddr->hkcyl = cn;
	hkaddr->hkda = (tn << 8) + sn;
	hkaddr->hkwc = wc;
	hkaddr->hkba = (caddr_t)addr;

	if	(!ubmap)
		hkaddr->hkxmem=hiint(addr);
	cmd = hk_type[unit] | ((hiint(addr) & 3) << 8) | HK_IE | HK_GO;
	if (bp->b_flags & B_READ)
		cmd |= HK_READ;
	else
		cmd |= HK_WRITE;
	hkaddr->hkcs1 = cmd;
	hktab.b_errcnt = 0;	/* error has been corrected */
	return (1);
}

/*
 * Assumes the 'open' entry point has already been called to validate
 * the unit number.
*/
daddr_t
hksize(dev)
	register dev_t dev;
	{
	register struct size *szp = hk_sizes[hkunit(dev)];

	if	(!szp)
		return(-1);
	return(szp[dev & 7].nblocks);
	}
#endif NHK > 0
