/* en.c: streams MVME147 Am7990 Ethernet Driver */
#include "param.h"
#include "types.h"
#include "buf.h"
#include "errno.h"
#include "proc.h"
#include "inode.h"
#include "dir.h"
#include "file.h"
#include "ipm.h"
#include "sid.h"
#include "seg.h"
#include "signal.h"
#include "user.h"
#include "stream.h"
#include "map.h"
#include "enio.h"
#include "ethernet.h"

/* configuration */

#define	NTYPES		8	/* max number of ethernet packet types */
#define	NRRB		64	/* number of recieve buffers */
#define	LOGNRRB		6	/* log of NRRB*/
#define	NTRB		32	/* number of transmit buffers */
#define	LOGNTRB		5	/* log of NTRB */
#define	ENPRI		28	/* sleeping like a tty */
#define	SPLIMP		6	/* interrupt levell */
#define	LANCEAP		((u_short *)0xfffe1802)
#define	LANCEDP		((u_short *)0xfffe1800)
#define	NRBLOCKS		(NTRB+NRRB+100)


#define	RSUCC(x)	(((x)+1)%NRRB)
#define	TSUCC(x)	(((x)+1)%NTRB)

/* macro to map an address into the no cache segment */

#define	NCADDR(t, x)	((t *)(0x81000000|(u_long)(x)))

/*   mode bits in the lance initialization block */

#define	PROM	0x8000
#define	INTL	0x40
#define	DRTY	0x20
#define	COLL	0x10
#define	DTCR	0x8
#define	LOOP	0x4
#define	DTX	0x2
#define	DRX	0x1

/* CSR0.  l.rap points to to it most of the time */

#define	ERR0	0x8000
#define	BABL	0x4000
#define	CERR	0x2000
#define	MISS	0x1000
#define	MERR	0x800
#define	RINT	0x400
#define	TINT	0x200
#define	IDON	0x100
#define	INTR	0x80
#define	INEA	0x40
#define	RXON	0x20
#define	TXON	0x10
#define	TDMD	0x8
#define	STOP	0x4
#define	STRT	0x2
#define	INIT	0x1

/*  lance csr3   */

#define	BSWP	0x4
#define	ACON	0x2
#define	BCON	0x1


/*  flag bits from a buffer descriptor in the rcv/xmt rings */

#define	OWN	0x8000	/* 1 means that the buffer can be used by the chip */
#define ERR	0x4000	/* erro summary, the OR of all error bits */
#define	FRAM	0x2000	/* CRC error and incoming packet not a multiple of 8 bits */
#define	OFLO	0x1000	/* (recv) lost some of the packet */
#define	MORE	0x1000	/* (trans) more than 1 retry to send the packet */
#define	CRC	0x800	/* (recv) crc error reading packet */
#define	ONE	0x800	/* (trans) one retry to transmit the packet */
#define	BUF	0x400	/* (recv) out of buffers while reading a packet */
#define	DEF	0x400	/* (trans) deffered while transmitting packet */
#define	STP	0x200	/* start of packet */
#define	ENP	0x100	/* end of packet */

/* cntflags bits from a buffer descriptor in the rcv/xmt rings */

#define	BUFF	0x8000	/* buffer error (host screwed up?) */
#define	UFLO	0x4000	/* underflow from memory */
#define	LCOL	0x1000	/* late collision (ether too long?) */
#define	LCAR	0x800	/* loss of carrier (ether broken?) */
#define	RTRY	0x400	/* couldn't transmit (bad station on ether?) */
#define	TTDR	0x3FF	/* time domain reflectometer */

/* 
 *   Some macros for dealing with lance memory addresses.  The lance splits
 *   its 24 vbit addresses across two 16 bit registers.
 */

#define	HADDR(a) (((u_long)(a)) >> 16 & 0xff)
#define	LADDR(a) (((u_long)a) & 0xffff)

/*
 *   Initialization block, this must be in RAM accessible by the lance.
 *   It is used to configure the lance on startup and is ignored after
 *   that.
 */

typedef struct initblock {
	u_short	mode;		/* chip control (see below) */
	u_short	etheraddr[3];	/* the ethernet physical address */
	u_short multi[4];	/* multicast addresses, 1 bit for each of 64 */
	u_short	rdralow;	/* receive buffer ring */
	u_short	rdrahigh;	/* (top three bits define size of ring) */
	u_short	tdralow;	/* transmit buffer ring */
	u_short	tdrahigh;	/* (top three bits define size of ring) */
} Initblock;

/*
 *   Communication with the lance is via a transmit and receive ring of
 *   message descriptors.  The Initblock contains pointers to and sizes of
 *   these rings.  The rings must be in RAM addressible by the lance.
 */

typedef	struct {
	u_short	laddr;		/* low order piece of address */
	u_short	flags;		/* flags and high order piece of address */
	short	size;		/* size of buffer */
	u_short	cntflags;	/* (rcv)count of bytes in buffer; (xmt) more flags */
} Msg;


/*
 *   Ethernet packet buffers.  These must also be in alnce addressible RAM.
 */

typedef	struct {
	u_char	d[6];
	u_char	s[6];
	u_short	type;
	u_char	data[1500];
	u_char	crc[4];
} Etherp;

typedef	struct {
	u_char	d[6];
	u_char	s[6];
	u_char	type[2];
	u_char	data[60];
} MinEtherp;



/*	Stuff queues are made of.  Looks a lot like dmr streams */

typedef	struct {
	struct	queue	*q;
	u_short		flag;	/* 0=empty, 1=open, 2=ready to use */
	u_short		type;	/* ethernet type */
	struct	block	*outb;	/* buffer for output */
} Ethertype;

/*
 * Lance state
 */

typedef	struct {
	Initblock	*li;		/* init block */
	u_char		ea[6];		/* our ether addr */
	u_short		*rap;		/* lance address register */
	u_short		*rdp;		/* lance data register */
	Msg		*rring;		/* start of receive ring */
	u_short		rl;		/* first rcv message belonging to lance */
	u_short		rc;		/* first rcv messgae belonging to cpu */
	struct	block	*rbp[NRRB];	/* receive buffers */
	Msg		*tring;		/* start of xmt ring */
	u_short		tl;		/* first xmt message belonging to lance */
	u_short		tc;		/* first xmt message belonging to cpu */
	struct	block	*tbp[NTRB];	/* transmitt buffers */
	struct	block	blocks[NRBLOCKS];/* all ether blocks */
	struct block	*free;		/* free list of lance blocks */
	Ethertype	e[NTYPES];
} Lance;


/*
 *  lanceram must be addressable by the lance chip.
 *  on the 147 this is just local ram.  Other interfaces
 *  could require this to be mapped on to a board.
 */

struct	lanceram	{
	Initblock	initblock;
	Msg		rring[NRRB];
	Msg		tring[NTRB];
	Etherp		e[NRBLOCKS];
};


static	Lance	l;
static	int	inpackets, outpackets;
static	int endnsrv(), enqopen();
extern	int srv();
extern	int nulldev();
extern	int putq();

struct	qinit	enqinit[] = {
/* up */  { putq, srv, enqopen, nulldev, 1520, 20 },
/* dn */  { putq, endnsrv, nulldev, nulldev, 1520, 20 }
};
static	struct	block	*eallocb();

#define	DBREST	0001
#define	DBBLOCK	0002
#define DBSEND	0004

static	endebug = 00;


static
enqopen(dev, q)		/* link to a queue provided by streams */
	dev_t dev;
	struct queue *q;
{
	q->ptr = (char *)&l.e[minor(dev)];
	OTHERQ(q)->ptr = (char *)&l.e[minor(dev)];
	l.e[minor(dev)].q = q;
}

enopen(dev, flag)
	dev_t dev;
{
	register Ethertype *e;
	static inited = 0;

	if (minor(dev) < 0 || minor(dev) > NTYPES) {
		u.u_error = ENXIO;
		return;
	}
	if (!inited) {
		eninit();
		inited = 1;
	}
	if ((e = &l.e[minor(dev)])->flag) {
/*		u.u_error = ENXIO;[20~*/
		return;
	}
	e->outb = NULL;
	e->flag = 1;
	e->type = 0;
}

enclose(dev, flag)
	dev_t dev;
{
	register Ethertype *e;

	if (endebug & DBREST)
		printf("enclose: top\n");
	e = &l.e[minor(dev)];
	e->flag = 0;	/* closed */
	if (e->outb)
		efreeb(e->outb);
	e->q = NULL;
	e->type = 0;
	if (endebug & DBREST)
		printf("enclose: bot\n");
}

static
endnsrv(q)		/* down stream service procedure */
	register struct queue *q;
{
	register n;
	register struct block  *bp;
	register struct block *outb;
	register Ethertype *e;

	if (endebug & DBREST)
		printf("endnsrv: top\n");
	e = (Ethertype *)q->ptr;
	while (tring_full() == 0 && (bp = getq(q))) {
		if (endebug & DBSEND) 
			enprbp("endnsrv: from queue", bp);
		if (bp->type == M_IOCTL) {
			enioctl(q, bp);
			continue;
		}
		if (bp->type != M_DATA) {
			freeb(bp);
			continue;
		}
		if ((outb = e->outb) == NULL)
			outb = e->outb = eallocb();
		n = min(outb->lim - outb->wptr, bp->wptr - bp->rptr);
		if (n > 0) {
			bcopy(bp->rptr, outb->wptr, n);
			bp->rptr += n;
			outb->wptr += n;
		} else
			bp->rptr = bp->wptr; 	/* toss rest */
		if (endebug & DBSEND)
			enprbp("endnsrv: outb", outb);
		if (bp->class & S_DELIM) {
			if (endebug & DBREST)
				printpacket(outb->base);
			if (endebug & DBSEND)
				printf("calling put_tring\n");
			put_tring(outb);
			e->outb = NULL;
		}
		freeb(bp);
	}
	if (endebug & DBREST)
		printf("endnsrv: bot\n");
}

static
tring_full()	/* ...  Well?! */
{
	return TSUCC(l.tc) == l.tl;
}


static
put_tring(bp)	/* stick a lance space block into transmit ring */
	register struct block *bp;
{
	Etherp *e;
	register n;

	e = (Etherp *) bp->base;
	n = bp->wptr - bp->rptr;
	bcopy(l.ea, e->s, sizeof l.ea);
	
	/* set up the ring descriptor */

	l.tbp[l.tc] = bp;
	l.tring[l.tc].size = -(n > 60 ? n : 60);
	l.tring[l.tc].cntflags = 0;
	l.tring[l.tc].laddr = LADDR(bp->base);
	l.tring[l.tc].flags = OWN | STP | ENP | HADDR(bp->base);
	if (endebug & DBREST)
		printf("put_tring:bp 0x%x, l.tc %d\n", bp, l.tc);
	l.tc = TSUCC(l.tc);
}




enioctl(q, bp)	/* handle io control message */
	register struct queue *q;
	register struct block *bp;
{
	register Ethertype *e;

	e = (Ethertype *)q->ptr;
	switch (stiocom(bp)) {
	case ENIOTYPE:		/* set the protocol type */
		bcopy(stiodata(bp), (caddr_t)&e->type, sizeof e->type);
		e->flag = 2;		/* okay to receive messages */
		bp->type = M_IOCACK;
		bp->rptr = bp->wptr = bp->base;
		qreply(q, bp);
		break;
	case ENIOADDR:
		bcopy((caddr_t)l.ea, bp->rptr, sizeof l.ea);
		bp->type = M_IOCACK;
		bp->wptr = bp->rptr + sizeof l.ea;
		qreply(q, bp);
		break;
	default:
		bp->type = M_IOCNAK;
		bp->rptr = bp->wptr = bp->base;
		qreply(q, bp);
	}
}

enintr()
{
	u_short csr;
	int xrfreed = 0;
	Etherp *p;
	Ethertype *e;
	register t, size;
	struct block  *bp, *bp2;

	*l.rap = 0;
	csr = *l.rdp;
	if (endebug & DBREST)
		printf("enintr top: csr %x\n", csr);

	/* turn off interrupt and any error indicators */
	
	*l.rdp = IDON|INEA|TINT|RINT|BABL|CERR|MISS|MERR;

	/* see if an error occurred */

	if (*l.rdp & (BABL|MISS|MERR|CERR)) {
		*l.rdp = BABL|CERR|MISS|MERR;
		printf("en: err 0x%x\n", csr & (BABL|CERR|MISS|MERR));
	}
	if (csr & IDON)
		*l.rdp = INEA|STRT;

	/* look for rcv'd packets, put in queue for type */

	for (; l.rl!=l.rc && (l.rring[l.rl].flags & OWN)==0; l.rl=RSUCC(l.rl)) {
		inpackets++;
		bp = l.rbp[l.rl];
		if (endebug & DBREST)
			printf("enintr: bp 0x%x, l.rl %d\n", bp, l.rl);
		if (l.rring[l.rl].flags & ERR) {
#ifdef notdef
			printf("rcv error 0x%x\n",
				l.rring[l.rl].flags & (FRAM|OFLO|CRC|BUFF));
#endif
			efreeb(bp);
			continue;
		}

		/* see if a queue exists for this packet type */
	
		p = (Etherp *)bp->rptr;
		if (endebug & DBREST)
			printpacket(p);
		for (e = &l.e[0]; e < &l.e[NTYPES]; e++)
			if (e->flag == 2 && p->type == e->type)
				break;

 		/* 
		 * If no match, see if any stream has type -1.
		 * It matches all packtes.
		 */

		if (e == &l.e[NTYPES]) {
			for (e = &l.e[0]; e < &l.e[NTYPES]; e++)
				if (e->flag == 2 && e->type == 0xffff)
					break;
			if (e == &l.e[NTYPES]) {
				efreeb(bp);
				continue;
			}
		}
		bp->wptr += l.rring[l.rl].cntflags - 4;
		if (e->q->flag & QFULL) {
			efreeb(bp);
			continue;
		}
		do {
			bp2 = allocb(bp->wptr - bp->rptr);
			size = min(bp->wptr - bp->rptr, bp2->lim - bp2->base);
			bcopy(bp->rptr, bp2->wptr, size);
			bp->rptr += size;
			bp2->wptr += size;
			if (bp->rptr == bp->wptr)
				bp2->class |= S_DELIM;
			if (bp2->class == 077)
				printf("en:upsrv: enblock escaped 2\n");
			(*e->q->qinfo->putp)(e->q, bp2);
		} while (bp->rptr < bp->wptr);
		efreeb(bp);
	}

	/*
	 *  look for a xmitt'd packets, wake any process waiting for a
	 *  transmit buffer.
	 */

	xrfreed = 0;
	while (l.tl != l.tc && (l.tring[l.tl].flags & OWN) == 0) {
		if (l.tring[l.tl].flags & ERR)
			printf("en xmit error 0x%x 0x%x\n",
				l.tring[l.tl].flags,
				l.tring[l.tl].cntflags);
		efreeb(l.tbp[l.tl]);	/* put block back */
		l.tl = TSUCC(l.tl);
		xrfreed++;
	}
	if (xrfreed)
		wakeq();	/* see if any queue wants to write */
	stagerbuf();
	if (endebug & DBREST)
		printf("enintr bot: csr 0x%x\n", *l.rdp);
}

static
eninit()
{
	struct block  *bp;
	int i;
	Initblock *li;
	register a;
	register struct lanceram *lrp;
	Msg *msg;
	int x;

	splimp();
	if (endebug & DBREST)
		printf("eninit: top \n");

	/* Since we use local ram, get space & init pointers */

	if ((a = malloc(coremap, btoc(sizeof(struct lanceram)))) == 0)
		panic("eninit: no ram!");
	lrp = NCADDR(struct lanceram, ctob(a));
	bzero(lrp, sizeof (struct lanceram));
	li = NCADDR(Initblock, &lrp->initblock);

	/* init PCC for lance */

	pcc_lan_init(SPLIMP);

	/* initialize the free list */

	for (i = 0; i < NRBLOCKS; i++) {
		bp = l.blocks + i;
		bp->base = NCADDR(u_char, &lrp->e[i]);
		bp->lim = bp->base + sizeof(Etherp);
		bp->class = 077;	/* to keep them separate */
		bp->next = l.free;
		l.free = bp;
	}

	/* so that we don't have to indirect through constants */

	l.rring = NCADDR(Msg, lrp->rring);
	l.tring = NCADDR(Msg, lrp->tring);
	l.rap = LANCEAP;
	l.rdp = LANCEDP;
	*l.rap = 0;
	*l.rdp = STOP;		/* in case it was running */

	/* point lance to the initialization block */

	*l.rap = 1;
	*l.rdp = LADDR(li);
	*l.rap = 2;
	*l.rdp = HADDR(li);

	/* The lance byte swaps until we tell it not to */

	*l.rap = 3;
	*l.rdp = BSWP | BCON;

	/* creat the initialization block */

	li->mode = 0;

	/* set ether address from battery backed up ram */

	li->etheraddr[0] = 0x0800;
	li->etheraddr[1] = 0x3e00 | *((u_char *) 0xfffe0778);
	li->etheraddr[2] = *((u_short *) 0xfffe0779);
	bcopy((caddr_t)li->etheraddr, l.ea, 6);
	swab((caddr_t)li->etheraddr, (caddr_t)li->etheraddr, 6);

	/* no multi cast by default */

	for (i = 0; i < 4; i++)
		li->multi[i] = 0;

	/* set up rcv message ring */

	*l.rap = 0;
	stagerbuf();
	li->rdralow = LADDR(l.rring);
	li->rdrahigh = (LOGNRRB<<13) | HADDR(l.rring);

	/* set up xmit message ring */

	for (i = 0; i < NTRB; i++) {
		l.tring[i].cntflags = 0;
		l.tring[i].laddr = 0;
		l.tring[i].flags = 0;
		l.tring[i].size = 0;
	}

	li->tdralow = LADDR(l.tring);
	li->tdrahigh = (LOGNTRB<<13)|HADDR(l.tring);
	
	/* init lance, turn on interrupts, turn on transmit and recv */

	*l.rap = 0;
	*l.rdp = INEA|INIT;
	if (endebug & DBREST)
		printf("eninit: bot csr 0x%x\n", *l.rdp);
	spl0();
}

/*  stage as many receive buffers as possible */

static
stagerbuf()
{
	int next;
	struct block  *bp;
	int i = 0;
	int csr;

	for (next = RSUCC(l.rc); next != l.rl; next = RSUCC(next)) {
		l.rbp[l.rc] = bp = eallocb();
		if (bp == 0) {
			printf("en: no lance buffers!\n");
			break;
		}
		l.rring[l.rc].size = -(bp->lim - bp->base);
		l.rring[l.rc].cntflags = 0;
		l.rring[l.rc].laddr = LADDR(bp->base);
		l.rring[l.rc].flags = OWN|HADDR(bp->base);
		l.rc = next;
		i++;
	}
	if (endebug & DBREST)
		printf("staged %d buffers, csr 0x%x\n", i, *l.rdp);
}

static
struct block  *
eallocb()
{
	register struct block  *bp;
	register s;

	s = splimp();
	if ((bp = l.free) == NULL)
		panic("eallocb: out of blocks");
	l.free = bp->next;
	bp->type = M_DATA;
	bp->next = NULL;
	bp->rptr = bp->wptr = bp->base;
out:
	if (endebug & DBBLOCK)
		printf("eallocb: 0x%x\n", bp);
	splx(s);
	return bp;
}


static
efreeb(bp)		/* put a block back on the free list */
	register struct block  *bp;
{
	register s;

	s = splimp();
	if (bp->class != 077)
		printf("efreeb: this is not my block\n");
	bp->next = l.free;
	l.free = bp;
	if (endebug & DBREST)
		printf("efreeb: 0x%x\n", bp);
	splx(s);
}

static
printpacket(p)
	Etherp *p;
{
	printf("%d 0x%x\n", inpackets, *l.rdp);
	printf("d(%x.%x.%x.%x.%x.%x)\n",
		p->d[0], p->d[1], p->d[2], p->d[3], p->d[4], p->d[5]);
	printf("s(%x.%x.%x.%x.%x.%x)\n",
		p->s[0],  p->s[1],  p->s[2],  p->s[3],  p->s[4],  p->s[5]);
	printf("t(%x)\n", p->type);
}

wakeq()		/* check for queue waiting for space in transmit ring */
{
	register Ethertype *e;

	for (e = l.e; e < &l.e[NTYPES]; e++)
		if (e->flag)
		if (e->q->count)
			qenable(e->q);
}

static
struct block *
alloc_tring()		/* get a block from the transmit ring */
{
	return l.tbp[l.tc];
}


static
enprbp(s, bp)		/* print things about blocks */
	char *s;
	register struct block *bp;
{
	printf("%s: bp 0x%x, class 0%o, type 0%o\n", s, bp, bp->class, bp->type);
	printf("%s:base 0x%x, lim 0x%x\n", s, bp->base, bp->lim);
	printf("%s:rptr 0x%x, wptr 0x%x, count %d\n", s, bp->rptr, bp->wptr,
		bp->wptr - bp->rptr);
	printf("%s: next 0x%x\n", s, bp->next);
}
