/* msgsend.c: Sunos simulator for adaptive IPC functions */
/* the following shell variables are used to fully simulate the
 * gun envrionment:
 *	MY_SCOPE	scope for this processor 
 *	MY_NODE		the node number for this node
 *	NETWORK		network number
 *	SNM_SID		sid number for snm (inter-nodal)
 *	PEP_SID		sid number for pep (intra-nodal)
 */

#include <sys/types.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include "ipm.h"

#define	NPTEMPL	"/tmp/NP.%d.%d.%d"
#define	NULL	0

extern	int errno;
static	int sid_list[128];

extern	char *getenv();

msgbind(sid)		/* bind a service id to a fd */
	int sid;
{
	int fd;
	char b[64];
	char *name, *strdup();
	int clean_fifos();
	int cnt = 1000;

	if (sid == 0)
		sid = find_sid();
	mknp(b, getnode(), getscope(), sid);
	name = strdup(b);
	on_exit(clean_fifos, name);

	do
		mknod(b, S_IFIFO);
	while (chmod(b, 0666) < 0 && --cnt);
	if (!cnt)
		printf("msgbind:cnt = 0\n");
	fd =  open(b, O_RDWR);		/* RDWR to keep from blocking */
	sid_list[fd] = sid;
	return fd;
}

fdsid(fd)		/* given a fd, return the sid */
{
	if (sid_list[fd] == 0)
		return -1;
	return sid_list[fd];
}

char *
msgalloc(size)		/* allocate message memory */
	register size;
{
	register char *p;
	char *malloc();

	if ((p = malloc(size)) == NULL) {
		errno = ENOMEM;
		return (char *)-1;
	}
	memset(p, 0, size);
	return p;

}


msgsend(mhp, p)		/* send a message */
	register struct mesghdr *mhp;
	register char *p;
{
	char block[1024];
	register fd;
	
	if (mhp->length > MAXMESG) {
		errno = EINVAL;
		return -1;
	}
	if ((fd = getfd(mhp)) == -1)
		return -1;
	memcpy(block, (char *)mhp, sizeof (struct mesghdr));
	memcpy(block+sizeof(struct mesghdr), p, mhp->length);
	write(fd, block, sizeof block);
	free(p);
	close(fd);
	return 0;
}

char *
msgrecv(fd, mhp)		/* receive a message */
	int fd;
	struct mesghdr *mhp;
{
	register char *p;
	register struct mesghdr *mp;
	char block[1024];
	char *malloc();

	if (read(fd, block, sizeof block) != sizeof block)
		return 0;
	if (mhp != NULL)
		memcpy((char *)mhp, block, sizeof (struct mesghdr));
	mp = (struct mesghdr *)block;
	p = malloc(mp->length);
	memcpy(p, block + sizeof (struct mesghdr), mp->length);
	return p;
}

msgfree(p)			/* return message memory to the system */
	char *p;
{
	free(p);
	return 0;
}


setscope(node, scope)
{
}

unsigned short
getnode()
{
	register char *p;

	if ((p = getenv("MY_NODE")) == NULL)
		return 0;
	return atoi(p);
}

unsigned short
getscope()
{
	register char *p;

	if ((p = getenv("MY_SCOPE")) == NULL)
		return 0;
	return atoi(p);
}

static
this_scope()
{
	return getscope();
}


static
this_node()
{
	return getnode();
}


static
okpipe(b)		/* does `b' exist? */
	char *b;
{
	if (access(b, R_OK) == 0)
		return 1;
	return 0;
}

static
snm_sid()
{
	char *p;

	if ((p = getenv("SNM_SID")) == NULL)
		return 9998;
	return atoi(p);
}

static
pep_sid()
{
	char *p;

	if ((p = getenv("PEP_SID")) == NULL)
		return 9999;
	return atoi(p);
}

unsigned short
getnetwork()
{
	register char *p;

	if ((p = getenv("NETWORK")) == NULL)
		return -1;
	return atoi(p);
}
unsigned short
getnet()
{
	register char *p;

	if ((p = getenv("NETWORK")) == NULL)
		return -1;
	return atoi(p);
}
static
getfd(mhp)	/* get the fd for the service id */
	register struct mesghdr *mhp;
{
	register scope, node, fd;
	char b[80];

	node = mhp->dest.node ? mhp->dest.node : this_node();
	scope = mhp->dest.scope;
	if (mhp->dest.node == 0 || mhp->dest.node == this_node())
		scope = mhp->dest.scope ? mhp->dest.scope : this_scope();
	if ((fd = sidopen(node, scope, mhp->dest.sid)) != -1)
		return fd;
	if (node != this_node())
		if ((fd = sidopen(this_node(), this_scope(), snm_sid())) != -1)
			return fd;
	if (scope != this_scope())
		if ((fd = sidopen(this_node(), this_scope(), pep_sid())) != -1)
			return fd;
	errno = ENXIO;
	return -1;
}

static
sidopen(n, s, sid)
{
	char b[80];
	register fd;

	mknp(b, n, s, sid);
	if (okpipe(b) && (fd = open(b, 2)) != -1)
		return fd;
	return -1;
}

clean_fifos(status, name)
	char *name;
{
	unlink(name);
}

static
mknp(b, node, scope, sid)
	char *b;
{
	sprintf(b, NPTEMPL, node, scope, sid);
}

find_sid()		/* try various sids until one works */
{
	int sid;
	char b[128];

	for (sid = 10001; sid < 35000; sid++) {
		mknp(b, this_node(), this_scope(), sid);
		if (access(b, F_OK) == -1)
			break;
	}
	if (sid >= 35000)
		return -1;
	return sid;
}
