/*
 *	spname:	return correctly spelled filename
 *
 *	spname(oldname, newname) char *oldname, *newname;
 *	return -1 if no reasonable match to oldname,
 *		0 if exact match,
 *		1 if corrected.
 *	stores correcte name in newname
 */

# include <sys/types.h>
#include <limits.h>
#include <dirent.h>

#ifndef NAME_MAX
#define	NAME_MAX _POSIX_NAME_MAX
#endif

spname(oldname, newname)
	char *oldname, *newname;
{
	char *p, guess[NAME_MAX+1], best[NAME_MAX+1];
	char *new = newname, *old = oldname;

	for(;;) {
		while (*old == '/') /* skip slashes */
			*new++ = *old++;
		*new = '\0';
		if (*old == '\0')	/* exact or corrected */
			return strcmp(oldname, newname) != 0;
		p = guess;	/* copy next component into guess */
		for ( ; *old != '/' && *old != '\0' ; old++)
			if (p < guess+NAME_MAX)
				*p++ = *old;
		*p = '\0';
		if (mindist(newname, guess, best) >= 3)
			return -1;		/* hopeless */
		for (p = best; *new = *p++; )	/* add to end */
			new++;			/* of newname */
	}
}

mindist(dir, guess, best)	/* search dir for guess */
	char *dir, *guess, *best;
{
	/* set best, return distance 0..3 */
	int d, nd;
	DIR *fd;
	struct dirent *nbuf;

	if (dir[0] == '\0')		/* current directory */
		dir = ".";
	d = 3;	/* minimum distance */
	if ((fd = opendir(dir)) == NULL)
		return d;
	while (nbuf = readdir(fd)) {
		nd = spdist(nbuf->d_name, guess);
		if (nd <= d && nd != 3) {
			strcpy(best, nbuf->d_name);
			d = nd;
			if (d == 0)	/*exact match */
				break;
		}
	}
	closedir(fd);
	return d;
}

/*  spdist: return distance between two names */
/*
 *	very rough spelling metric:
 *	0 if the strings are identical
 *	1 if two chars are transposed
 *	2 if one char wrong, added or deleted
 *	3 otherwise
 */

# define EQ(s, t) (strcmp(s,t) == 0)

spdist(s, t)
	char *s, *t;
{
	while (*s++ == *t)
		if (*t++ == '\0')
			return 0;		/* exact match */
	if (*--s) {
		if (*t) {
			if (s[1] && t[1] && *s == t[1]
			  && *t == s[1] && EQ(s+2, t+2))
				return 1; /* transposition */
			if (EQ(s+1, t+1))
				return 2; /* 1 char mismatch */
		}
		if (EQ(s+1, t))
			return 2;	/* extra character */
	}
	if (*t && EQ(s, t+1))
		return 2;		/* missing character */
	return 3;
}
