/*  pokerd - a classical draw poker server
 *  Copyright (C) 2001,2003 Hans P. Rosenfeld <rosenfeld@grumpf.hope-2000.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "cards.h"

char *colors[]={ "Spades",   /* 0 */
		 "Clubs",    /* 1 */
		 "Diamonds", /* 2 */
		 "Hearts"    /* 3 */
};

char *numbers[]={ "Two",    /*  0 */
		  "Three",  /*  1 */
		  "Four",   /*  2 */
		  "Five",   /*  3 */
		  "Six",    /*  4 */
		  "Seven",  /*  5 */
		  "Eight",  /*  6 */
		  "Nine",   /*  7 */
		  "Ten",    /*  8 */
		  "Jack",   /*  9 */
		  "Queen",  /* 10 */
		  "King",   /* 11 */
		  "Ace"     /* 12 */
};

char *get_color(struct card_t *card)
{
     return(colors[card->color]);
}

char *get_number(struct card_t *card)
{
     return(numbers[card->number]);
}

struct card_t *get_first_card(struct card_deck_t *card_deck)
{
     return(card_deck->first);
}

struct card_t *get_last_card(struct card_deck_t *card_deck)
{
     return(card_deck->last);
}

struct card_t *pop_first_card(struct card_deck_t *card_deck)
{
     struct card_t *card=card_deck->first;

     if(card_deck->count!=1) {
	  card_deck->first=card->next;
	  card_deck->first->prev=NULL;
     } else card_deck->first=card_deck->last=NULL;
     card->next=NULL;
     card_deck->count--;
     return(card);
}

struct card_t *pop_last_card(struct card_deck_t *card_deck)
{
     struct card_t *card=card_deck->last;

     if(card_deck->count!=1) {
	  card_deck->last=card->prev;
	  card_deck->last->next=NULL;
     } else card_deck->first=card_deck->last=NULL;
     card->prev=NULL;
     card_deck->count--;
     return(card);
}

struct card_deck_t *push_first_card(struct card_t *card, struct card_deck_t *card_deck)
{
     if(card_deck->first==NULL)
	  card_deck->last=card_deck->first=card;
     else {
	  card->next=card_deck->first;
	  card->next->prev=card;
	  card_deck->first=card;
     }
     card_deck->count++;
     return(card_deck);
}

struct card_deck_t *push_last_card(struct card_t *card, struct card_deck_t *card_deck)
{
     if(card_deck->first==NULL)
	  card_deck->last=card_deck->first=card;
     else {
	  card->prev=card_deck->last;
	  card->prev->next=card;
	  card_deck->last=card;
     }
     card_deck->count++;
     return(card_deck);
}

struct card_deck_t *initialize_card_deck(void)
{
     int i, j;
     struct card_deck_t *card_deck=initialize_empty_card_deck();

     for(i=0; i!=4; i++)
	  for(j=0; j!=13; j++) {
	       struct card_t *card=calloc(1, sizeof(struct card_t));
	       card->color=i;
	       card->number=j;
	       card->next=card->prev=NULL;
	       push_first_card(card, card_deck);
	  }
     return(card_deck);
}

struct card_deck_t *initialize_empty_card_deck(void)
{
     struct card_deck_t *card_deck=calloc(1, sizeof(struct card_deck_t));

     card_deck->count=0;
     card_deck->first=card_deck->last=NULL;
     return(card_deck);
}

struct card_deck_t *shuffle_card_deck(struct card_deck_t *card_deck)
{
     int i, r;
     struct card_deck_t *new_deck=initialize_empty_card_deck();
     struct card_deck_t *tmp_deck=initialize_empty_card_deck();
     srand((unsigned int) time((time_t*) 0));

     while(card_deck->count>6) {
	  r=1+(int)(3.0*rand()/(RAND_MAX+1.0));
	  for(i=0; i!=r; i++)
	       push_first_card(pop_last_card(card_deck), new_deck);
	  r=1+(int)(3.0*rand()/(RAND_MAX+1.0));
	  for(i=0; i!=r; i++)                                        /* This is necessary to   */
	       push_first_card(pop_first_card(card_deck), tmp_deck); /* ensure that the cards  */
	  for(i=0; i!=r; i++)                                        /* are still in realistic */
	       push_first_card(pop_first_card(tmp_deck), new_deck);  /* order after shuffling. */
     }
     while(card_deck->count)
	  new_deck=push_first_card(pop_last_card(card_deck), new_deck);
     free(card_deck);
     free(tmp_deck);
     return(new_deck);
}

struct card_deck_t *lift_card_deck(struct card_deck_t *card_deck)
{ 
     int i, r;
     srand((unsigned int) time((time_t*) 0));

     r=1+(int)(52.0*rand()/(RAND_MAX+1.0));
     for(i=0; i!=r; i++)
	  push_last_card(pop_first_card(card_deck), card_deck);
     return(card_deck);
}

void print_card(FILE *stream, struct card_t *card)
{
     fprintf(stream, "%s %s\n", colors[card->color], numbers[card->number]);
}

int snprint_card(int i, char *buf, int size, struct card_t *card)
{
     return(snprintf(buf, size, "%d %s %s", i, colors[card->color], numbers[card->number]));
}

void print_card_deck(FILE *stream, struct card_deck_t *card_deck)
{
     struct card_t *card;

     for(card=card_deck->first; card!=NULL; card=card->next)
	  print_card(stream,card);
}

int snprint_card_deck(char *buf, int size, struct card_deck_t *card_deck)
{
     struct card_t *card;
     int i, j=size, k=1;
     
     for(card=card_deck->first; card!=NULL; card=card->next) {
	  i=snprint_card(k++, buf, size, card);
	  buf+=i;
	  size-=i+1;
	  *buf++='\n';
     }
     *buf=0;
     return(j-size);
}

int cmp_cards(const void *c1, const void *c2)
{
     return(((struct card_t*) c1)->number-((struct card_t*) c2)->number);
}

struct card_deck_t *sort_card_deck_by_number(struct card_deck_t *card_deck)
{
     int i;
     struct card_deck_t *c_deck[4], *n_deck[13];

     for(i=0; i!=4; i++)
	  c_deck[i]=calloc(1, sizeof(struct card_deck_t));
     for(i=0; i!=13; i++)
	  n_deck[i]=calloc(1, sizeof(struct card_deck_t));

     while(card_deck->count) {
	  struct card_t *card=pop_first_card(card_deck);
	  push_first_card(card, c_deck[card->color]);
     }
     for(i=0; i!=4; i++) {
	  while(c_deck[i]->count) {
	       struct card_t *card=pop_first_card(c_deck[i]);
	       push_first_card(card, n_deck[card->number]);
	  }
	  free(c_deck[i]);
     }
     for(i=0; i!=13; i++) {
	  while(n_deck[i]->count)
	       push_last_card(pop_first_card(n_deck[i]), card_deck);
	  free(n_deck[i]);
     }
     return(card_deck);
}

struct card_deck_t *sort_card_deck_by_color(struct card_deck_t *card_deck)
{
     int i;
     struct card_deck_t *c_deck[4], *n_deck[13];

     for(i=0; i!=4; i++)
	  c_deck[i]=calloc(1, sizeof(struct card_deck_t));
     for(i=0; i!=13; i++)
	  n_deck[i]=calloc(1, sizeof(struct card_deck_t));
     while(card_deck->count) {
	  struct card_t *card=pop_first_card(card_deck);
	  push_first_card(card, n_deck[card->number]);
     }
     for(i=0; i!=13; i++) {
	  while(n_deck[i]->count) {
	       struct card_t *card=pop_first_card(n_deck[i]);
	       push_first_card(card, c_deck[card->color]);
	  }
	  free(n_deck[i]);
     }
     for(i=0; i!=4; i++) {
	  while(c_deck[i]->count)
	       push_last_card(pop_first_card(c_deck[i]), card_deck);
	  free(c_deck[i]);
     }
     return(card_deck);
}
  
void dis_card_deck(struct card_deck_t *card_deck)
{
     if(card_deck==NULL)
	  return;
     while(card_deck->count)
	  free(pop_first_card(card_deck));
     free(card_deck);
}


#ifdef CARDSDEBUG

void print_deck_map(FILE *stream, struct card_deck_t *card_deck)
{
     struct card_t *card;

     fprintf(stream, "first: 0x%8x\n", (unsigned int) card_deck->first);
     for(card=card_deck->first; card!=NULL; card=card->next)
	  fprintf(stream, "card: 0x%8x\tprev: 0x%8x\tnext: 0x%8x\n",
		  (unsigned int) card, (unsigned int) card->prev, (unsigned int) card->next);
     fprintf(stream,"last: 0x%8x\n", (unsigned int) card_deck->last);
}

int main(int argc, char **argv)
{
     int i=atoi(argv[1]);
     struct card_deck_t *card_deck;

     printf("initializing card deck...");
     card_deck=initialize_card_deck();
     printf(".done\n\n"); 
     print_card_deck(stdout, card_deck);
     print_deck_map(stdout, card_deck);

     printf("lifting card deck...");
     card_deck=lift_card_deck(card_deck);
     printf(".done\n\n");
     print_card_deck(stdout, card_deck);

     while(i--) {
	  printf("shuffling card deck...");
	  card_deck=shuffle_card_deck(card_deck);
	  printf(".done\n\n");
	  print_card_deck(stdout,card_deck);
     }

     printf("sorting card deck by number...");
     card_deck=sort_card_deck_by_number(card_deck);
     printf(".done\n\n");
     print_card_deck(stdout,card_deck);

     printf("sorting card deck by color...");
     card_deck=sort_card_deck_by_color(card_deck);
     printf(".done\n\n");
     print_card_deck(stdout,card_deck);

     printf("discarding card deck...");
     dis_card_deck(card_deck);
     printf(".done\n");
     exit(0);
}

#endif
