;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	@(#)ccp.s	1.2	07/12/16
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	.title	"console command processor (CCP), ver 2.0"
;	assembly language version of the CP/M console command processor
;
;	version 2.2 February, 1980
;
;	Copyright (c) 1976, 1977, 1978, 1979, 1980
;	Digital Research
;	Box 579, Pacific Grove,
;	California, 93950
;

	.area CCP (ABS)

	.include "memcfg.s"
;
;	"bias" is address offset from 3400H for memory systems
;	than 16K (referred to as "b" throughout the text).
;
bias	= (msize-20)*1024
ccpl	= moff+bias		;base of ccp
bdosl	= ccpl+0x800		;base of bdos
bios	= ccpl+0x1600		;base of bios

	.org ccpl		;origin of this program

false	= 0x0000
true	= ~false

di	= 0xf3
hlt	= 0x76

tran	= 0x100
tranm	= .
ccploc	= .

;	********************************************************
;	*	Base of CCP contains the following code/data   *
;	*	ccp:	jmp ccpstart	(start with command)   *
;	*		jmp ccpclear    (start, clear command) *
;	*	ccp+6	127		(max command length)   *
;	*	ccp+7	comlen		(command length = 00)  *
;	*	ccp+8	' ... '		(16 blanks)	       *
;	********************************************************
;	* Normal entry is at ccp, where the command line given *
;	* at ccp+8 is executed automatically (normally a null  *
;	* command with comlen = 00).  An initializing program  *
;	* can be automatically loaded by storing the command   *
;	* at ccp+8, with the command length at ccp+7.  In this *
;	* case, the ccp executes the command before prompting  *
;	* the console for input.  Note that the command is exe-*
;	* cuted on both warm and cold starts.  When the command*
;	* line is initialized, a jump to "jmp ccpclear" dis-   *
;	* ables the automatic command execution.               *
;	********************************************************

	jp ccpstart		;start ccp with possible initial command
	jp ccpclear		;clear the command buffer

maxlen:	.db 127			;max buffer length
comlen:	.db 0			;command length (filled in by dos)
;	(command executed initially if comlen non zero)
combuf:
	.ascii "        "	;8 character fill
	.ascii "        "	;8 character fill
	.ascii "COPYRIGHT (C) 1979, DIGITAL RESEARCH  "; 38
	.ds 128-(.-combuf)
;	total buffer length is 128 characters
comaddr:.dw combuf		;address of next to char to scan
staddr:	.ds 2			;starting address of current fillfcb request

diska	= 0x0004		;disk address for current disk
bdos	= 0x0005		;primary bdos entry point
buff	= 0x0080		;default buffer
fcb	= 0x005c		;default file control block

rcharf	= 1			;read character function
pcharf	= 2			;print character function
pbuff	= 9			;print buffer function
rbuff	= 10			;read buffer function
breakf	= 11			;break key function
liftf	= 12			;lift head function (no operation)
initf	= 13			;initialize bdos function
self	= 14			;select disk function
openf	= 15			;open file function
closef	= 16			;close file function
searf	= 17			;search for file function
searnf	= 18			;search for next file function
delf	= 19			;delete file function
dreadf	= 20			;disk read function
dwritf	= 21			;disk write function
makef	= 22			;file make function
renf	= 23			;rename file function
logf	= 24			;return login vector
cself	= 25			;return currently selected drive number
dmaf	= 26			;set dma address
userf	= 32			;set user number

;	special fcb flags
rofile	= 9			;read only file
sysfile	= 10			;system file flag

;	special characters
cr	= 13			;carriage return
lf	= 10			;line feed
la	= 0x5f			;left arrow
eofile	= 0x1a			;end of file
space	= 0x20

;	utility procedures


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Print character.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
printchar:
	ld e,a			; move character to E
	ld c,#pcharf		; get the print char function
	jp bdos			; jump to BDOS


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Print character, but save b,c registers.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
printbc:
	push bc			; save these
	call printchar		; print the character in A
	pop bc			; restore these
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Print CRLF.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
crlf:
	ld a,#cr		; get CR
	call printbc		; print it
	ld a,#lf		; get LF
	jp  printbc		; print it and leave


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Print a blank.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
blank:
	ld a,#space		; get space character
	jp printbc		; print it


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Print string starting at BC until
;	next 00 entry.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
print:	
	push bc			; save the string pointer
	call crlf		; send CRLF
	pop hl			; get the string pointer
prin0:	ld a,(hl)		; get a character from the string
	or a,a			; EOL?
	ret z			; yes, leave
	inc hl			; no, point to next character
	push hl			; save string pointer
	call printchar		; print the character in A
	pop hl			; restore the string pointer
	jp prin0		; do it again.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Initialize the disk system.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
initialize:
	ld c,#initf		; Reset disk system function
	jp bdos			; jump to BDOS


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Select a disk.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
select:
	ld e,a			; get selected disk
	ld c,#self		; select disk function
	jp bdos			; jump to BDOS


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Call BDOS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
bdos$inr:
	call bdos		; call BDOS to process function
	ld (dcnt),a		; save disk directory count/error code
	inc a			;
	ret			; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Open the file given by DE.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
open:
	ld c,#openf		; open file function
	jp bdos$inr		; jump to BDOS


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Open COMFCB.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
openc:
	xor a			; clear A
	ld (comrec),a		; clear next record to read
	ld de,#comfcb		; get location of the COMFCB
	jp open			; do open file function


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Close the file given by DE.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
close:
	ld c,#closef		; close file function
	jp bdos$inr		; jump to BDOS


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Search the file given by DE.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
search:
	ld c,#searf		; search for first
	jp bdos$inr		; jump to BDOS


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Search for the next occurrence of the
;	file given by DE.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
searchn:
	ld c,#searnf		; search for next
	jp bdos$inr		; call BDOS


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Search for COMFCB file.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
searchcom:
	ld de,#comfcb		; get COMFCB
	jp search		; do search


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Delete the file given by DE.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
delete:
	ld c,#delf		; delete function
	jp bdos			; jump to BDOS


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Call BDOS and set the Z flag if error
;	returned.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
bdos$cond:
	call bdos		; call BDOS
	or a,a			; set Z flag if error
	ret			; done.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Read the next record from the file
;	given by DE.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
diskread:
	ld c,#dreadf		; disk read function
	jp bdos$cond		; jump to BDOS

diskreadc:
	;read the comfcb file
	ld de,#comfcb
	jp diskread

diskwrite:
	;write the next record to the file given by d,e
	ld c,#dwritf
	jp bdos$cond

make:	;create the file given by d,e
	ld c,#makef
	jp bdos$inr

renam:	;rename the file given by d,e
	ld c,#renf
	jp bdos

getuser:
	;return current user code in a
	ld e,#0x0ff		;drop through to setuser


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Set user code
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
setuser:
        ld c,#userf		; get/set user code function
	jp bdos			; set the user number

saveuser:
	;save user#/disk# before possible ^c or transient
	call getuser		;code to a
	add a,a
	add a,a
	add a,a
	add a,a			;rot left
	ld hl,#cdisk
	or a,(hl)		;4b=user, 4b=disk
	ld (diska),a		;stored away in memory for later
	ret

setdiska:
	ld a,(cdisk)
	ld (diska),a		;user/disk
	ret

translate:
	;translate character in register A to upper case
	cp #0x61
	ret c			;return if below lower case a
	cp #0x7b
	ret nc			;return if above lower case z
	and #0x5f
	ret			;translated to upper case

readcom:
	;read the next command into the command buffer
	;check for submit file
	ld a,(submit)
	or a,a
	jp z,nosub
	;scanning a submit file
	;change drives to open and read the file
	ld a,(cdisk)
	or a
	ld a,#0
	call nz,select
	;have to open again in case xsub present
	ld de,#subfcb
	call open
	jp z,nosub		;skip if no sub
	ld a,(subrc)
	dec a			;read last record(s) first
	ld (subcr),a		;current record to read
	ld de,#subfcb
	call diskread		;end of file if last record
	jp nz,nosub
	;disk read is ok, transfer to combuf
	ld de,#comlen
	ld hl,#buff
	ld b,#128
	call move0
	;line is transferred, close the file with a
	;deleted record
	ld hl,#submod
	ld (hl),#0		;clear fwflag
	inc hl
	dec (hl)		;one less record
	ld de,#subfcb
	call close
	jp z,nosub
	;close went ok, return to original drive
	ld a,(cdisk)
	or a
	call nz,select
	;print to the 00
	ld hl,#combuf
	call prin0
	call break$key
	jp z,noread
	call del$sub
	jp ccp			;break key depressed
	;
nosub:	;no submit file
	call del$sub
	;translate to upper case, store zero at end
	call saveuser		;user # save in case control c
	ld c,#rbuff
	ld de,#maxlen
	call bdos
	call setdiska		;no control c, so restore diska
noread:	;enter here from submit file
	;set the last character to zero for later scans
	ld hl,#comlen
	ld b,(hl)		;length is in b
readcom0:
	inc hl
	ld a,b
	or a			;end of scan?
	jp z,readcom1
	ld a,(hl)		;get character and translate
	call translate
	ld (hl),a
	dec b
	jp readcom0
	
readcom1:
	;end of scan, h,l address end of command
	ld (hl),a		;store a zero
	ld hl,#combuf
	ld (comaddr),hl		;ready to scan to zero
	ret

break$key:
	;check for a character ready at the console
	ld c,#breakf
	call bdos
	or a
	ret z
	ld c,#rcharf
	call bdos		;character cleared
	or a
	ret

cselect:
	;get the currently selected drive number to reg-A
	ld c,#cself
	jp bdos

setdmabuff:
	;set default buffer dma address
	ld de,#buff ;(drop through)

setdma:
	;set dma address to d,e
	ld c,#dmaf
	jp bdos

del$sub:
	;delete the submit file, and set submit flag to false
	ld hl,#submit
	ld a,(hl)
	or a
	ret z			;return if no sub file
	ld (hl),#0		;submit flag is set to false
	xor a
	call select		;on drive a to erase file
	ld de,#subfcb
	call delete
	ld a,(cdisk)
	jp select		;back to original drive

serialize:
	;check serialization
	ld de,#serial
	ld hl,#bdosl
	ld b,#6			;check six bytes
ser0:	ld a,(de)
	cp a,(hl)
	jp nz,badserial
	inc de
	inc hl
	dec b
	jp nz,ser0
	ret			;serial number is ok

comerr:
	;error in command string starting at position
	;'staddr' and ending with first delimiter
	call crlf		;space to next line
	ld hl,(staddr)		;h,l address first to print
comerr0:
	;print characters until blank or zero
	ld a,(hl)
	cp a,#space
	jp z,comerr1		; not blank
	or a,a
	jp z,comerr1		; not zero, so print it
	push hl
	call printchar
	pop hl
	inc hl
	jp comerr0		; for another character
comerr1:
	;print question mark,and delete sub file
	ld a,#'?
	call printchar
	call crlf
	call del$sub
	jp ccp			;restart with next command

; fcb scan and fill subroutine (entry is at fillfcb below)
	;fill the comfcb, indexed by A (0 or 16)
	;subroutines
delim:	;look for a delimiter
	ld a,(de)
	or a
	ret z			;not the last element
	cp #space
	jp c,comerr		;non graphic
	ret z			;treat blank as delimiter
	cp #'=
	ret z
	cp #la
	ret z			;left arrow
	cp #'.
	ret z
	cp #':
	ret z
	cp #';
	ret z
	cp #'<
	ret z
	cp #'>
	ret z
	ret			;delimiter not found

deblank: ;deblank the input line
	ld a,(de)
	or a
	ret z			;treat end of line as blank
	cp #space
	ret nz
	inc de
	jp deblank

addh:	;add a to h,l
	add l
	ld l,a
	ret nc
	inc h
	ret

fillfcb0:
	;equivalent to fillfcb(0)
	ld a,#0

fillfcb:
	ld hl,#comfcb
	call addh
	push hl
	push hl			;fcb rescanned at end
	xor a
	ld (sdisk),a		;clear selected disk (in case A:...)
	ld hl,(comaddr)
	ex de,hl		;command address in d,e
	call deblank		;to first non-blank character
	ex de,hl
	ld (staddr),hl		;in case of errors
	ex de,hl
	pop hl			;d,e has command, h,l has fcb address
	;look for preceding file name A: B: ...
	ld a,(de)
	or a
	jp z,setcur0		;use current disk if empty command
	sbc #'A-1
	ld b,a			;disk name held in b if : follows
	inc de
	ld a,(de)
	cp #':
	jp z,setdsk		;set disk name if :

setcur: ;set current disk
	dec de			;back to first character of command
setcur0:
	ld a,(cdisk)
	ld (hl),a
	jp setname

setdsk: ;set disk to name in register b
	ld a,b
	ld (sdisk),a		;mark as disk selected
	ld (hl),b
	inc de			;past the :

setname:
	;set the file name field
	ld b,#8			;file name length (max)
setnam0:
	call delim
	jp z,padname		;not a delimiter
	inc hl
	cp #'*
	jp nz,setnam1		;must be ?'s
	ld (hl),#'?
	jp setnam2		;to dec count

setnam1:
	ld (hl),a		;store character to fcb
	inc de
setnam2:
	dec b			;count down length
	jp nz,setnam0
	;
	;end of name, truncate remainder
trname: call delim
	jp z,setty		;set type field if delimiter
	inc de
	jp trname

padname:
	inc hl
	ld (hl),#space
	dec b
	jp nz,padname

setty:	;set the type field
	ld b,#3
	cp #'.
	jp nz,padty		;skip the type field if no .
	inc de			;past the ., to the file type field
setty0: ;set the field from the command buffer
	call delim
	jp z,padty
	inc hl
	cp #'*
	jp nz,setty1
	ld (hl),#'?		;since * specified
	jp setty2

setty1: ;not a *, so copy to type field mov m,a
	ld (hl),a
	inc de
setty2: ;decrement count and go again
	dec b
	jp nz,setty0

	;end of type field, truncate
trtyp:	;truncate type field
	call delim
	jp z,efill
	inc de
	jp trtyp

padty:	;pad the type field with blanks
	inc hl
	ld (hl),#space
	dec b
	jp nz,padty

efill:	;end of the filename/filetype fill, save command address
	;fill the remaining fields for the fcb
	ld b,#3
efill0: inc hl
	ld (hl),#0
	dec b
	jp nz,efill0
	ex de,hl
	ld (comaddr),hl		;set new starting point
	
	;recover the start address of the fcb and count ?'s
	pop hl
	ld bc,#11		;b=0, c=8+3
scnq:	inc hl
	ld a,(hl)
	cp #'?
	jp nz,scnq0
	;? found, count it in b
	inc b
scnq0:	dec c
	jp nz,scnq
	;
	;number of ?'s in c, move to a and return with flags set
	ld a,b
	or a
	ret

intvec:
	;intrinsic function names (all are four characters)
	.ascii 'DIR '
	.ascii 'ERA '
	.ascii 'TYPE'
	.ascii 'SAVE'
	.ascii 'REN '
        .ascii 'USER'
intlen	= (.-intvec)/4		;intrinsic function length
;serial: .db 0,0,0,0,0,0
serial: .db 9,89,0,0,7,137

intrinsic:
	;look for intrinsic functions (comfcb has been filled)
	ld hl,#intvec
	ld c,#0			;c counts intrinsics as scanned
intrin0:
	ld a,c
	cp a,#intlen		;done with scan?
	ret nc
	;no, more to scan
	ld de,#comfcb+1		;beginning of name
	ld b,#4			;length of match is in b
intrin1:
	ld a,(de)
	cp a,(hl)		;match?
	jp nz,intrin2		;skip if no match
	inc de
	inc hl
	dec b
	jp nz,intrin1		;loop while matching
	;
	;complete match on name, check for blank in fcb
	ld a,(de)
	cp a,#space
	jp nz,intrin3		;otherwise matched
	ld a,c
	ret			;with intrinsic number in a
	;
intrin2:
	;mismatch, move to end of intrinsic
	inc hl
	dec b
	jp nz,intrin2

intrin3:
	;try next intrinsic
	inc c			;to next intrinsic number
	jp intrin0		;for another round

ccpclear:
	;clear the command buffer
	xor a
	ld (comlen),a
	;drop through to start ccp

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Enter here from boot loader.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
ccpstart:
	ld sp,#stack		; initialize the stack
	push bc			; save initial disk number
        ;(high order 4bits=user code, low 4bits=disk#)
	ld a,c			; get user code & disk #
	rra			; shift the user code down
	rra			;
	rra			;
	rra			;
	and #0x0f		; mask out only the user code
	ld e,a			; move to E
	call setuser		; call BDOS to select the user code
	;initialize for this user, get $ flag
        call initialize		;0ffh in accum if $ file present
        ld (submit),a		;submit flag set if $ file present
        pop bc			;recall user code and disk number
	ld a,c
	and #0x0f		;disk number in accumulator
        ld (cdisk),a		;clears user code nibble
	call select		;proper disk is selected, now check sub files
	;check for initial command
	ld a,(comlen)
	or a,a
	jp nz,ccp0		;assume typed already

ccp:
	;enter here on each command or error condition
	ld sp,#stack
	call crlf		;print d> prompt, where d is disk name
	call cselect		;get current disk number
	add #'A
	call printchar
	ld a,#'>
	call printchar
	call readcom		;command buffer filled
ccp0:	;(enter here from initialization with command full)
	ld de,#buff
	call setdma		;default dma address at buff
	call cselect
	ld (cdisk),a		;current disk number saved
	call fillfcb0		;command fcb filled
	call nz,comerr		;the name cannot be an ambiguous reference
	ld a,(sdisk)
	or a,a
	jp nz,userfunc
	;check for an intrinsic function
	call intrinsic
	ld hl,#jmptab		;index is in the accumulator
	ld e,a
	ld d,#0
	add hl,de
	add hl,de		;index in d,e
	ld a,(hl)
	inc hl
	ld h,(hl)
	ld l,a
	jp (hl)
	;pc changes to the proper intrinsic or user function
jmptab:
	.dw direct		;directory search
	.dw erase		;file erase
	.dw type		;type file
	.dw save		;save memory image
	.dw rename		;file rename
	.dw user		;user number
	.dw userfunc		;user-defined function


badserial:
	ld hl,#(di | (hlt << 8))
	ld (ccploc),hl
	ld hl,#ccploc
	jp (hl)

;utility subroutines for intrinsic handlers
readerr:
	;print the read error message
	ld bc,#rdmsg
	jp print
rdmsg:	.ascii "READ ERROR"
	.db 0

nofile:
	;print no file message
	ld bc,#nofmsg
	jp print
nofmsg: .ascii "NO FILE"
	.db 0

getnumber:
	;read a number from the command line
	call fillfcb0		;should be number
	ld a,(sdisk)
	or a
	jp nz,comerr		;cannot be prefixed
	;convert the byte value in comfcb to binary
	ld hl,#comfcb+1
	ld bc,#11		;(b=0, c=11)
	;value accumulated in b, c counts name length to zero
conv0:	ld a,(hl)
	cp a,#space
	jp z,conv1
	;more to scan, convert char to binary and add
	inc hl
	sub a,#'0
	cp a,#10
	jp nc,comerr		;valid?
	ld d,a			;save value
	ld a,b			;mult by 10
	and a,#0b11100000
	jp nz,comerr
	ld a,b			;recover value
	rlca
	rlca
	rlca			;*8
	add a,b
	jp c,comerr
	add a,b
	jp c,comerr		;*8+*2 = *10
	add a,d
	jp c,comerr		;+digit
	ld b,a
	dec c
	jp nz,conv0		;for another digit
	ret
conv1:	;end of digits, check for all blanks
	ld a,(hl)
	cp a,#space
	jp nz,comerr		;blanks?
	inc hl
	dec c
	jp nz,conv1
	ld a,b			;recover value
	ret

movename:
	;move 3 characters from h,l to d,e addresses
	ld b,#3
move0:	ld a,(hl)
	ld (de),a
	inc hl
	inc de
	dec b
	jp nz,move0
	ret

addhcf:	;buff + a + c to h,l followed by fetch
	ld hl,#buff
	add c
	call addh
	ld a,(hl)
	ret

setdisk:
	;change disks for this command, if requested
	xor a
	ld (comfcb),a		;clear disk name from fcb
	ld a,(sdisk)
	or a
	ret z			;no action if not specified
	dec a
	ld hl,#cdisk
	cp (hl)
	ret z			;already selected
	jp select

resetdisk:
	;return to original disk after command
	ld a,(sdisk)
	or a
	ret z			;no action if not selected
	dec a
	ld hl,#cdisk
	cp (hl)
	ret z			;same disk
	ld a,(cdisk)
	jp select

	;individual intrinsics follow
direct:
	;directory search
	call fillfcb0		;comfcb gets file name
	call setdisk		;change disk drives if requested
	ld hl,#comfcb+1
	ld a,(hl)		;may be empty request
	cp #space
	jp nz,dir1		;skip fill of ??? if not blank
	;set comfcb to all ??? for current disk
	ld b,#11		;length of fill ????????.???
dir0:	ld (hl),#'?
	inc hl
	dec b
	jp nz,dir0
	;not a blank request, must be in comfcb
dir1:	ld e,#0
	push de			;E counts directory entries
	call searchcom		;first one has been found
	call z,nofile		;not found message
dir2:	jp z,endir
	;found, but may be system file
	ld a,(dcnt)		;get the location of the element
	rrca
	rrca
	rrca
	and #0b1100000
	ld c,a
	;c contains base index into buff for dir entry
	ld a,#sysfile
	call addhcf		;value to A
	rla
	jp c,dir6		;skip if system file
	;c holds index into buffer
	;another fcb found, new line?
	pop de
	ld a,e
	inc e
	push de
	;e=0,1,2,3,...new line if mod 4 = 0
	and #0b11
	push af			;and save the test
	jp nz,dirhdr0		;header on current line
	call crlf
	push bc
	call cselect
	pop bc
	;current disk in A
	add #'A
	call printbc
	ld a,#':
	call printbc
	jp dirhdr1		;skip current line hdr
dirhdr0:
	call blank		;after last one
	ld a,#':
	call printbc
dirhdr1:
	call blank
	;compute position of name in buffer
	ld b,#1			;start with first character of name
dir3:	ld a,b
	call addhcf		;buff+a+c fetched
	and #0x7f		;mask flags
	;may delete trailing blanks
	cp #space
	jp nz,dir4		;check for blank type
	pop af
	push af			;may be 3rd item
	cp #3
	jp nz,dirb		;place blank at end if not
	ld a,#9
	call addhcf		;first char of type
	and #0x7f
	cp #space
	jp z,dir5
	;not a blank in the file type field
dirb:	ld a,#space		;restore trailing filename chr
dir4:
	call printbc		;char printed
	inc b
	ld a,b
	cp #12
	jp nc,dir5
	;check for break between names
	cp #9
	jp nz,dir3		;for another char
	;print a blank between names
	call blank
	jp dir3
	;
dir5:	;end of current entry
	pop af			;discard the directory counter (mod 4)
dir6:	call break$key		;check for interrupt at keyboard
	jp nz,endir		;abort directory search
	call searchn
	jp dir2			;for another entry
endir:	;end of directory scan
	pop de			;discard directory counter
	jp retcom


erase:
	call fillfcb0		;cannot be all ???'s
	cp #11
	jp nz,erasefile
	;erasing all of the disk
	ld bc,#ermsg
	call print
	call readcom
	ld hl,#comlen
	dec (hl)
	jp nz,ccp		;bad input
	inc hl
	ld a,(hl)
	cp #'Y
	jp nz,ccp
	;ok, erase the entire diskette
	inc hl
	ld (comaddr),hl		;otherwise error at retcom
erasefile:
	call setdisk
	ld de,#comfcb
	call delete
	inc a			;255 returned if not found
	call z,nofile		;no file message if so
	jp retcom

ermsg:	.ascii "ALL (Y/N)?"
	.db 0

type:
	call fillfcb0
	jp nz,comerr		;don't allow ?'s in file name
	call setdisk
	call openc		;open the file
	jp z,typerr		;zero flag indicates not found
	;file opened, read 'til eof
	call crlf
	ld hl,#bptr
	ld (hl),#255		;read first buffer
type0:	;loop on bptr
	ld hl,#bptr
	ld a,(hl)
	cp #128			;end buffer
	jp c,type1
	push hl			;carry if 0,1,...,127
	;read another buffer full
	call diskreadc
	pop hl			;recover address of bptr
	jp nz,typeof		;hard end of file
	xor a
	ld (hl),a		;bptr = 0
type1:	;read character at bptr and print
	inc (hl)		;bptr = bptr + 1
	ld hl,#buff
	call addh		;h,l addresses char
	ld a,(hl)
	cp #eofile
	jp z,retcom
	call printchar
	call break$key
	jp nz,retcom		;abort if break
	jp type0		;for another character

typeof:	;end of file, check for errors
	dec a
	jp z,retcom
	call readerr
typerr:	call resetdisk
	jp comerr

save:
	call getnumber		; value to register a
	push af			;save it for later
	;
	;should be followed by a file to save the memory image
	call fillfcb0
	jp nz,comerr		;cannot be ambiguous
	call setdisk		;may be a disk change
	ld de,#comfcb
	push de
	call delete		;existing file removed
	pop de
	call make		;create a new file on disk
	jp z,saverr		;no directory space
	xor a
	ld (comrec),a		; clear next record field
	pop af			;#pages to write is in a, change to #sectors
	ld l,a
	ld h,#0
	add hl,hl
	ld de,#tran		;h,l is sector count, d,e is load address
save0:	;check for sector count zero
	ld a,h
	or l
	jp z,save1		;may be completed
	dec hl			;sector count = sector count - 1
	push hl			;save it for next time around
	ld hl,#128
	add hl,de
	push hl			;next dma address saved
	call setdma		;current dma address set
	ld de,#comfcb
	call diskwrite
	pop de
	pop hl			;dma address, sector count
	jp nz,saverr		;may be disk full case
	jp save0		;for another sector

save1:	;end of dump, close the file
	ld de,#comfcb
	call close
	inc a			; 255 becomes 00 if error
	jp nz,retsave		;for another command
saverr:	;must be full or read only disk
	ld bc,#fullmsg
	call print
retsave:
	;reset dma buffer
	call setdmabuff
	jp retcom

fullmsg:
	.ascii "NO SPACE"
	.db 0

rename:
	;rename a file on a specific disk
	call fillfcb0
	jp nz,comerr		;must be unambiguous
	ld a,(sdisk)
	push af			;save for later compare
	call setdisk		;disk selected
	call searchcom		;is new name already there?
	jp nz,renerr3
	;file doesn't exist, move to second half of fcb
	ld hl,#comfcb
	ld de,#comfcb+16
	ld b,#16
	call move0
	;check for = or left arrow
	ld hl,(comaddr)
	ex de,hl
	call deblank
	cp #'=
	jp z,ren1		;ok if =
	cp #la
	jp nz,renerr2
ren1:	ex de,hl
	inc hl
	ld (comaddr),hl		;past delimiter
	;proper delimiter found
	call fillfcb0
	jp nz,renerr2
	;check for drive conflict
	pop af
	ld b,a			;previous drive number
	ld hl,#sdisk
	ld a,(hl)
	or a
	jp z,ren2
	;drive name was specified.  same one?
	cp b
	ld (hl),b
	jp nz,renerr2
ren2:	ld (hl),b		;store the name in case drives switched
	xor a
	ld (comfcb),a
	call searchcom		;is old file there?
	jp z,renerr1
	;
	;everything is ok, rename the file
	ld de,#comfcb
	call renam
	jp retcom
	;
renerr1:; no file on disk
	call nofile
	jp retcom
renerr2:; ambigous reference/name conflict
	call resetdisk
	jp comerr
renerr3:; file already exists
	ld bc,#renmsg
	call print
	jp retcom
renmsg: .ascii "FILE EXISTS"
	.db 0

user:
	;set user number
	call getnumber		; leaves the value in the accumulator
	cp a,#16
	jp nc,comerr		; must be between 0 and 15
	ld e,a			;save for setuser call
	ld a,(comfcb+1)
	cp #space
	jp z,comerr
	call setuser		;new user number set
	jp endcom

userfunc:
	call serialize		;check serialization
	;load user function and set up for execution
	ld a,(comfcb+1)
	cp a,#space
	jp nz,user0
	;no file name, but may be disk switch
	ld a,(sdisk)
	or a
	jp z,endcom		;no disk name if 0
	dec a
	ld (cdisk),a
	call setdiska		;set user/disk
	call select
	jp endcom
user0:	;file name is present
	ld de,#comfcb+9
	ld a,(de)
	cp #space
	jp nz,comerr		;type ' '
	push de
	call setdisk
	pop de
	ld hl,#comtype		;.com
	call movename		;file type is set to .com
	call openc
	jp z,userer
	;file opened properly, read it into memory
	ld hl,#tran		;transient program base
load0:	push hl			;save dma address
	ex de,hl
	call setdma
	ld de,#comfcb
	call diskread
	jp nz,load1
	;sector loaded, set new dma address and compare
	pop hl
	ld de,#128
	add hl,de
	ld de,#tranm		;has the load overflowed?
	ld a,l
	sub e
	ld a,h
	sbc d
	jp nc,loaderr
	jp load0		;for another sector

load1:	pop hl
	dec a
	jp nz,loaderr		;end file is 1
	call resetdisk		;back to original disk
	call fillfcb0
	ld hl,#sdisk
	push hl
	ld a,(hl)
	ld (comfcb),a		;drive number set
	ld a,#16
	call fillfcb		;move entire fcb to memory
	pop hl
	ld a,(hl)
	ld (comfcb+16),a
	xor a
	ld (comrec),a		;record number set to zero
	ld de,#fcb
	ld hl,#comfcb
	ld b,#33
	call move0
	;move command line to buff
	ld hl,#combuf
bmove0:	ld a,(hl)
	or a
	jp z,bmove1
	cp #space
	jp z,bmove1
	inc hl
	jp bmove0		;for another scan
	;first blank position found
bmove1:	ld b,#0
	ld de,#buff+1
	;ready for the move
bmove2:	ld a,(hl)
	ld (de),a
	or a
	jp z,bmove3
	;more to move
	inc b
	inc hl
	inc de
	jp bmove2
bmove3:	;b has character count
	ld a,b
	ld (buff),a
	call crlf
	;now go to the loaded program
	call setdmabuff		;default dma
	call saveuser		;user code saved
	;low memory diska contains user code
	call tran		;gone to the loaded program
	ld sp,#stack		;may come back here
	call setdiska
	call select
	jp ccp

userer:	;arrive here on command error
	call resetdisk
	jp comerr

loaderr:;cannot load the program
	ld bc,#loadmsg
	call print
	jp retcom
loadmsg:.ascii "BAD LOAD"
	.db 0
comtype:.ascii "COM"		;for com files

retcom:	;reset disk before end of command check
	call resetdisk

endcom:	;end of intrinsic command
	call fillfcb0		;to check for garbage at end of line
	ld a,(comfcb+1)
	sub a,#space
	ld hl,#sdisk
	or a,(hl)
	;0 in accumulator if no disk selected, and blank fcb
	jp nz,comerr
	jp ccp

;
;	data areas
	.ds 16			;8 level stack
stack:
;
;	'submit' file control block
submit:	.db 0			;00 if no submit file, ff if submitting
subfcb:	.db 0
	.ascii "$$$     "	;file name is $$$
	.ascii "SUB"
	.db 0,0			;file type is sub
submod:	.db 0			;module number
subrc:	.ds 1			;record count filed
	.ds 16			;disk map
subcr:	.ds 1			;current record to read
;
;	command file control block
comfcb:	.ds 32			;fields filled in later
comrec:	.ds 1			;current record to read/write
dcnt:	.ds 1			;disk directory count (used for error codes)
cdisk:	.ds 1			;current disk
sdisk:	.ds 1			;selected disk for current operation
	 			;none=0, a=1, b=2 ...
bptr:	.ds 1			;buffer pointer

