	PAGE	,132
	TITLE	Z-150 Console Input Routines

;**********************************************************************
;
;                      ------------------------
;---------------------- Console Input Routines ------------------------
;                      ------------------------
;
;	       Copyright (C) 1983, by Zenith Data Systems
;
;**********************************************************************

MONITOR_SEGMENT SEGMENT WORD PUBLIC

	ASSUME	CS:MONITOR_SEGMENT, DS:ROM_DATA, ES:NOTHING

	INCLUDE ../ROM/ROM.LIT
	INCLUDE ../ROM/INTR.LIT
	INCLUDE ../VIDEO/VIDEO.LIT
	INCLUDE ../VIDEO/VIDEO.EXT
	INCLUDE DEBUG.LIT
	INCLUDE DEBUG.EXT
 
	EXTRN	GET_KEY_BUFF:NEAR, GET_CURSOR_POSITION:NEAR, BEEP:NEAR
	EXTRN	CONSOLE_OUTPUT:NEAR, BACK_SPACE:NEAR, MAP_UPPER:NEAR
	EXTRN	CRLF:NEAR, GET_REGISTER:NEAR, GET_INTR_PTR:NEAR

MONITOR_SEGMENT ENDS


MONITOR_SEGMENT SEGMENT


;**********************************************************************
; GET_STRING: (TEXT_PTR, STRING_ADDRESS)
;
;	Get_String gets a string of ASCII and hexadecimal characters
; from the console and places them in a user string buffer.  Ascii
; strings are enclosed in double quotes, and hex numbers are entered
; as the corresponding (upper or lower case) Ascii characters.  Each
; string/number is separated by commas and/or spaces.
;
; Input:
;	SI: Pointer to start of string within command line.
;	DI: Pointer to start of buffer for holding the converted
;	    string.
;
; Output:
;	CY: Set if the string is invalid
;	BX: Contains length of converted string, in bytes.
;	SI: Points to characters in error, if an invalid command line
;	    is entered.
;**********************************************************************
GET_STRING PROC NEAR
	PUBLIC	GET_STRING
	PUSHREG	<AX>
	MOV	BX,0			;Set byte string count to 0
	CALL	CHECK_CR		;Is there anything on the line
	JC	GS5			;No, must be at least one item!
GS1:	CMP	COMMAND_LINE[SI],'"'	;Is this the start of a string?
	JNE	GS3			;No, it must be CR or a hex byte
	INC	SI			;Yes - point past the '"'
GS2:	MOV	AL,COMMAND_LINE[SI]	;Retrieve an ASCII character
	INC	SI			;Point to the next one
	CMP	AL,CR			;Is the char a carriage return?
	STC				;[Set error flag in case error]
	JE	GS5			;ERROR - unbalanced quotes
	CMP	AL,'"'			;Is this a closing quote?
	JE	GS4			;Yes - exit enter-ASCII mode
	MOV	BYTE PTR CS:[DI][BX],AL	;ASCII char - place it in buffer
	INC	BX			;Increment the fill character count
	JMP	SHORT GS2		;Continue getting ASCII characters
GS3:	CALL	GET_HEX_WORD		;Get a value from the cmd line
	JC	GS5			;ERROR - invalid hex byte entered
	MOV	BYTE PTR CS:[DI][BX],AL	;Write the hex byte in the fill string
	INC	BX			;Increment the fill count by 1
GS4:	CALL	NEXT_PARM		;Point to the next parameter
	CMP	COMMAND_LINE[SI],CR	;Reached end of line?
	JNE	GS1			;No, go get some more!
	CLC				;Yes - clear the error flag
GS5:	POPREG	<AX>
	RET
GET_STRING ENDP



;**********************************************************************
; GET_HEX_RANGE: (TEXT_PTR, DEFAULT_RANGE_FLAG, DEFAULT_RANGE,
;		  DEFAULT_SEGMENT)
;
;	Get_Hex_Range gets a RANGE from the console.  A range is either
; a pair of addresses, or a single address followed by a L, then a
; length specification.
;
; Input:
;	SI: Pointer into COMMAND_LINE to text to be converted.
;	AH: Flag specifying whether or not a default length is
;	    to be accepted.
;	CX: Contains default length.
;	ES: Default segment value for pointer portion of range.
;
; Output:
;	CY: Error flag - set if invalid range specified.
;	SI: Pointer following range text.
;    ES:DI: Pointer to memory block
;	CX: Size of memory block in bytes
;**********************************************************************
GET_HEX_RANGE PROC NEAR
	PUBLIC	GET_HEX_RANGE
	PUSHREG	<AX>			;Save register(s)
	PUSH	AX			;Save the default length flag
	CALL	GET_HEX_POINTER		;Get the block pointer
	POP	AX			;[Restore the default length flag]
	JC	GHR4			;ERROR - invalid pointer entered
	MOV	AL,COMMAND_LINE[SI]	;Retrieve the character following ptr
	CMP	AL,CR			;Is this the end of the line?
	JNE	GHR1			;No, get the remaining parameter
	TEST	AH,AH			;Yes - is a default length acceptable?
	JNE	GHR3			;Return with CX unchanged
	STC				;Default not OK - return error!
	JMP	SHORT GHR4
GHR1:	CALL	MAP_UPPER		;Convert the character to upper case
	CMP	AL,'L'			;Is this a length specification?
	JNE	GHR2			;No, get second address spec
	INC	SI			;Yes - point past the 'L'
	CALL	GET_HEX_WORD		;Now get the length from the console
	JC	GHR4			;ERROR - invalid block length
	MOV	CX,AX			;Place length in CX
	JMP	SHORT GHR3		;Return
GHR2:	CALL	GET_HEX_WORD		;Get the second address
	CMP	AX,DI			;Is the first address smaller?
	JC	GHR4			;ERROR - invalid address pair
	MOV	CX,AX			;Yes - place second address in CX
	SUB	CX,DI			;Subtract first addr from second
	INC	CX			;Include last byte of block in count
GHR3:	CLC				;Clear carry - valid range
	CALL	NEXT_PARM		;Point to the next parameter
GHR4:	POPREG	<AX>			;Restore register(s)
	RET
GET_HEX_RANGE ENDP



;**********************************************************************
; GET_HEX_POINTER: (TEXT_PTR, DEFAULT_SEGMENT)
;
;	Get_Hex_Pointer retrieves an ASCII-Hex pointer from a command
; line and returns the contents in binary form.
;
; Input:
;	SI: Pointer to string containing pointer.
;	ES: Default value for segment register, in case no segment
;	    is entered.
;
; Output:
;	CY: If the conversion process was valid, the carry flag will
;	    be returned reset.  Otherwise, the carry flag will be set.
;    ES:DI: If conversion valid, ES:DI will contain the converted
;	    pointer.
;	SI: Pointer to string in error (if CY set), or text following
;	    pointer (if CY reset).
;**********************************************************************
GET_HEX_POINTER PROC NEAR
	PUBLIC	GET_HEX_POINTER
	PUSHREG	<AX>
	CALL	GET_HEX_WORD		;Retrieve a word from the console
	JC	GHP2			;ERROR - invalid hex word
	MOV	DI,AX			;Place it in DI as the offset
	CMP	COMMAND_LINE[SI],':'	;Is this a segment/offset separator?
	CLC				;[Clear error flag if not...]
	JNE	GHP2			;No, return
	INC	SI			;Yes - advance past the character
	MOV	ES,DI			;Place segment value in ES
GHP1:	CALL	GET_HEX_WORD		;Retrieve the address offset
	JC	GHP3			;ERROR - invalid offset entered!
	MOV	DI,AX			;Place offset into DI
GHP2:	CALL	NEXT_PARM		;Skip past separators, etc.
GHP3:	POPREG	<AX>
	RET
GET_HEX_POINTER ENDP



;**********************************************************************
; GET_HEX_WORD: (TEXT_PTR)
;
;	Get_Hex_Word retrieves a word value from the command line.  The
; word value may be either the usual string of ASCII-Hex digits, or any
; register name may be entered, instead.
;
; Input:
;	SI: Offset into COMMAND_LINE to begin conversion.
;
; Output:
;	AX: Converted word value.
;	SI: Pointer to character following hex characters.
;**********************************************************************
GET_HEX_WORD PROC NEAR
	PUBLIC	GET_HEX_WORD
	PUSHREG	<BX,DX>
	CALL	CHECK_CR		;Is there at least one char here?
	JC	GHW4			;ERROR - cannot have 'blank' number
	CALL	GET_REGISTER		;Try to get a register name
	JC	GHW1			;Not a register - must be hex
	MOV	DX,REGISTERS[BX]	;Register name - get contents of reg
	JMP	SHORT GHW3		;Return
GHW1:	MOV	DX,0			;Set output value to 0
	CALL	GET_CHAR		;Get a character from the cmd line
GHW2:	CALL	MAP_UPPER		;Convert char to upper-case ASCII
	CALL	ASCII_TO_HEX		;Convert the character to hex
	JC	GHW4			;ERROR - invalid character
	SHL	DX,1			;Multiply
	SHL	DX,1			;     ...output
	SHL	DX,1			;	    ...by 16
	SHL	DX,1			;		 ...for next digit.
	OR	DL,AL			;Include the new digit in the output
	MOV	AL,COMMAND_LINE[SI]	;Get the next character
	CALL	MAP_UPPER		;Convert the char to upper case
	CMP	AL,' '			;Is this char a space?
	JE	GHW3			;Yes - finished w/conversion
	CMP	AL,','			;Is this char a comma?
	JE	GHW3			;Yes - valid separator entered
	CMP	AL,':'			;Segment:offset separator found?
	JE	GHW3			;Yes - consider word ended
	CMP	AL,'L'			;Is this the start of a length spec?
	JE	GHW3			;Yes - finished w/word input
	CMP	AL,CR			;End of line reached?
	JE	GHW3			;Yes - done with hex conversion
	INC	SI			;No, point to the next character
	JMP	SHORT GHW2		;Loop until non-hex char found
GHW3:	MOV	AX,DX			;Get the result into AX
	CLC				;Clear carry - all OK
	CALL	NEXT_PARM		;Point on to the next parameter
GHW4:	POPREG	<DX,BX>
	RET
GET_HEX_WORD ENDP



;**********************************************************************
; NEXT_PARM: (TEXT_PTR)
;
;	Next_Parms is called to advance the line pointer beyond
; inter-element separators, such as commas and spaces.
;
; Input:
;	SI: Pointing to spaces or other separators, within COMMAND_LINE.
;
; Output:
;	SI: Pointing to first char following terminator/separator.
;**********************************************************************
NEXT_PARM PROC NEAR
	PUBLIC	NEXT_PARM
	PUSHF				;Save flags on stack
	PUSHREG	<AX>			;Save other register(s)
	CALL	SKIP_SPACES		;Advance past leading spaces
	MOV	AL,COMMAND_LINE[SI]	;Pick up the char following spaces
	CMP	AL,','			;Is it a comma?
	JE	NP1			;Yes - advance beyond it
	CMP	AL,'-'			;Is the character a hyphen?
	JNE	NP2			;No, return with pointer set as is
NP1:	INC	SI			;Point beyond separator
NP2:	CALL	SKIP_SPACES		;Skip spaces after separator, as well
	POPREG	<AX>			;Restore register(s)
	POPF				;Restore flags
	RET
NEXT_PARM ENDP



;**********************************************************************
; CHECK_CR: (TEXT_PTR)
;
;	Check_CR is called to determine if the TEXT_PTR points to
; an empty line (ie, only spaces), or to the end of a line.
;
; Input:
;	SI: Pointer into COMMAND_LINE.
;
; Output:
;	CY: Set if remainder of line is blank.
;	SI: Pointing beyond any spaces initially pointed to.
;**********************************************************************
CHECK_CR PROC NEAR
	PUBLIC	CHECK_CR
	PUSHREG	<AX>
	CALL	SKIP_SPACES		;Advance past any spaces
	MOV	AL,COMMAND_LINE[SI]	;Get the current character
	SUB	AL,CR			;Set AL=0 if pointing to CR
	ADD	AL,0FFH			;Set carry if NOT CR
	CMC				;Set carry if CR, clear otherwise
	POPREG	<AX>
	RET
CHECK_CR ENDP


;**********************************************************************
; SKIP_SPACES: (TEXT_PTR)
;
;	Skip_Spaces is called to advance a text pointer beyond any
; sequence of spaces which it may point to.
;
; Input:
;	SI: Pointer to string possibly containing spaces.
;
; Output:
;	SI: Pointing to string following any spaces.
;**********************************************************************
SKIP_SPACES PROC NEAR
	PUBLIC	SKIP_SPACES
SS1:	CMP	COMMAND_LINE[SI],' '	;Is this character a space?
	JNZ	SS2			;No, return
	INC	SI			;Yes - point to the next character
	JMP	SHORT SS1		;Loop until non-space found
SS2:	RET
SKIP_SPACES ENDP



;**********************************************************************
; GET_CHAR: (TEXT_PTR)
;
;	Get_Char retrieves the next non-blank character from the
; command line pointed to by TEXT_PTR.  The pointer is also advanced
; to point to the next character.
;
; Input:
;	SI: Pointer into command line.
;
; Output:
;	AL: Contains contents of character from line.
;	SI: Pointing to next character
;**********************************************************************
GET_CHAR PROC NEAR
	PUBLIC	GET_CHAR
	CALL	SKIP_SPACES		;Skip leading spaces
	MOV	AL,COMMAND_LINE[SI]	;Get the char from the line
	INC	SI			;Point to the next
	RET				;Return character & pointer
GET_CHAR ENDP



;**********************************************************************
; GET_LINE:
;
;	Get_Line reads a command line into the global 'COMMAND_LINE'.
; The line is entered via the 'RETURN' or 'ENTER' keys, and an ASCII
; CR character is placed after the line's contents.  Note that Get_Line
; sets the global variable 'LINE_START' as the cursor's position at the
; first character in the line (this is used for error reporting).
;**********************************************************************
GET_LINE PROC NEAR
	PUBLIC	GET_LINE
	PUSHREG	<AX,CX,DX,DI>
	CALL	GET_CUR_POSN		;Read the current cursor position
	MOV	LINE_START,DL		;Save offset of cursor
	MOV	DI,0			;Clear offset into COMMAND_LINE
GL1:	CALL	GET_KEY_BUFF		;Get a keyboard key
	CMP	AL,BS			;Was it a backspace?
	JNE	GL2			;No, check for CR
	CALL	GET_CUR_POSN		;Yes - get the current cursor posn
	CMP	DL,LINE_START		;Is the cursor at the prompt?
	JBE	GL1			;Yes - ignore the backspace
	CALL	BACK_SPACE		;No, delete last char from display
	DEC	DI			;Remove the char from the cmd buffer
	JMP	SHORT GL1		;Go get another key
GL2:	CMP	AL,CR			;Is the user finished w/the line?
	JNE	GL3			;No, see if char can go in line
	MOV	COMMAND_LINE[DI],AL	;Yes - place the CR in the buffer
	CALL	CRLF			;Echo a newline on the console
	JMP	SHORT GL6		;Return contents of COMMAND_LINE
GL3:	CMP	AL,' '			;Ok, is this char printable?
	JAE	GL5			;Yes - place it in the buffer
GL4:	CALL	BEEP			;No, complain about it
	JMP	SHORT GL1		;Get another keystroke
GL5:	CMP	DI,MAX_COLUMNS		;Entered too many keys yet?
	JAE	GL4			;Yes - complain about it!
	MOV	COMMAND_LINE[DI],AL	;No, place the char in the buffer
	INC	DI			;Advance to the next buffer location
	CALL	CONSOLE_OUTPUT		;Echo the character on the screen
	JMP	SHORT GL1		;Go get another keystroke
GL6:	POPREG	<DI,DX,CX,AX>
	RET
GET_LINE ENDP



;**********************************************************************
; ASCII_TO_HEX: (CHAR)
;
;	Ascii_To_Hex is called to determine if a character is a valid
; hex character, and if so, to convert it into its binary equivalent.
;
; Input:
;	AL: Character to convert
;
; Output:
;	CY: Set if the input character was not a hex digit.
;	AL: If the CY flag is set, AL contains the original character.
;	    Otherwise, AL contains the binary equivalent of the char.
;**********************************************************************
ASCII_TO_HEX PROC NEAR
	PUBLIC	ASCII_TO_HEX
	PUSHREG	<BX>
	MOV	BL,AL			;Save the character in BL
	SUB	AL,'0'			;Is the char less than '0'?
	JC	ATH2			;Yes - return character w/CY set
	CMP	AL,9 + 1		;Is the character a digit?
	CMC				;[Clear carry if so]
	JNC	ATH3			;Yes - finished with conversion
	CMP	AL,'A'-'0'		;Is the char > '9', < 'A'?
	JC	ATH2			;Yes - return 'invalid'
	SUB	AL,'A'-'9'-1		;No, convert char to A-F
	CMP	AL,10H			;Is the result a valid hex digit?
	CMC				;Clear the carry flag if so
	JNC	ATH3			;No, return OK
ATH2:	MOV	AL,BL			;Restore the original character
ATH3:	POPREG	<BX>
	RET
ASCII_TO_HEX ENDP



;**********************************************************************
; GET_CUR_POSN:
;
;	Get_Cur_Posn gets the cursor position for the currently
; active page.
;
; Output:
;	DX: Cursor position (DH:row, and DL:column)
;**********************************************************************
GET_CUR_POSN PROC NEAR
	PUBLIC	GET_CUR_POSN
	PUSHREG	<AX,BX,SI,DS>
	MOV	BH,DISPLAYED_PAGE	;Get the currently displayed page
	CALL	GET_CURSOR_POSITION	;Retrieve the Z-150 cursor position
	POPREG	<DS,SI,BX,AX>
	RET
GET_CUR_POSN ENDP



MONITOR_SEGMENT ENDS

	END

