/******************** MODULE INFO  ****************************/
/*
**  File name   :  queue.c
*/
/*  AUTHOR      :  Jan Erik Nilsen (jen@nilsenelektronikk.no) */
/*  VERSION     :  3.0                                        */
/*  DATE        :  Thu Aug 16 11:17:44 2001                   */
/*
*   Compiler    :  ANSI C
*   Project     :  proc  Real-Time Kernel
*   Functions   :  queue functions
*   Constants   :  see queue.h
*
*                  if you want to include memory allocation,
*                  define USE_MALLOC, and make your procMalloc
*
*
* Copyright (c) nilsen elektronikk as, Norway. (www.nilsenelektronikk.no)
* This software is the property of nilsen elektronikk as, Norway.
*
* The proc RTOS is free software; you can use it, redistribute it
* and/or modify it under the following terms:
* 1. You are not allowed to remove or modify this copyright notice
*    and License paragraphs, even if parts of the software is used.
* 2. The improvements and/or extentions you make SHALL be available
*    for the community under THIS license, source code included.
*    Improvements or extentions, including adaptions to new architectures,
*    SHALL be reported and transmitted to Nilsen Elektronikk AS.
* 3. You must cause the modified files to carry prominent notices stating
*    that you changed the files, what you did and the date of changes.
* 4. You may NOT distribute this software under another license without
*    explisit permission from Nilsen Elektronikk AS, Norway.
* 5. This software is free, and distributed in the hope that it will be
*    useful, but WITHOUT ANY WARRANTY; without even the implied warranty
*    of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*    You SHALL NOT use this software unless you accept to carry all
*    risk and cost of defects or limitations.
*
*    ------------  CHANGE RECORD  ----------------
*    Jan Erik Nilsen (jen@nilsenelektronikk.no) July 26. 1999:
*        First free version of this software published.
*
*    Andrey Melnikov, Russia. (melnikov@sietlab.com) Aug 15. 2001
*        To ensure atomic FIFO operations, disable_interrupt() and
*        enable interrupt() are used in queueGet() and queuePut()
*
*/
#define  NULL (void *)0
#include "proc.h"
#include "queue.h"






/*===========  queueInit  ============================
*
*   Purpose:      Initialize an integer queue.
*                 Must be done before a queue is used.
*
*   Input:        ptr to queue, queue length
*   Output:       none
*   Return:       If OK: 1, if malloc problem: 0.
*
*/
int queueInit(QUEUE *qp, int size)
{
  procSemInit(&(qp->used), 0);
  procSemInit(&(qp->left), size);
  qp->ii   = qp->oi = 0;
  qp->size = size;
#ifdef USE_MALLOC
  if (qp->fifo == NULL) qp->fifo = (char **)procMalloc(sizeof(void*) * size);
#endif
  return((qp->fifo)? 1 : 0);
}



/*===========  queuePut  ============================
*
*   Purpose:      Put a value into a queue
*                 If the queue is full, the function
*                 will block until space is available.
*
*   Input:        ptr to queue, value x
*   Output:       none
*   Return:       none
*
*/
void queuePut(QUEUE *qp, char *x)
{
  int i;
#ifdef REINT
  int sr;
#endif

  procSemWait(&(qp->left));
  disable_interrupt();
  if ((i = (qp->ii + 1)) >= qp->size) i = 0;
  qp->fifo[i] = x;
  qp->ii = i;
  enable_interrupt();
  procSemSignal(&(qp->used));
}



/*===========  queueGet  ============================
*
*   Purpose:      Block until a value can be taken
*                 from a queue.
*
*   Input:        ptr to queue
*   Output:       none
*   Return:       value
*
*/
char *queueGet(QUEUE *qp)
{
  int  i;
  char *x;
#ifdef REINT
  int sr;
#endif

  procSemWait(&(qp->used));
  disable_interrupt();
  if ((i = (qp->oi + 1)) >= qp->size) i = 0;
  x = qp->fifo[i];
  qp->oi = i;
  enable_interrupt();
  procSemSignal(&(qp->left));
  return(x);
}
