/******************** MODULE INFO  ****************************/
/*
**  File name   :  proc.c
*/
/*  AUTHOR      :  Jan Erik Nilsen (jen@nilsenelektronikk.no) */
/*  VERSION     :  3.1                                        */
/*  DATE        :  Tue Feb 12 13:16:26 2002                   */
/*
*   Compiler    :  ANSI C
*   Project     :  proc  Real-Time Kernel
*   Functions   :  basic functions and semaphore functions
*
*
* Copyright (c) nilsen elektronikk as, Norway. (www.nilsenelektronikk.no)
* This software is the property of nilsen elektronikk as, Norway.
*
* The proc RTOS is free software; you can use it, redistribute it
* and/or modify it under the following terms:
* 1. You are not allowed to remove or modify this copyright notice
*    and License paragraphs, even if parts of the software is used.
* 2. The improvements and/or extentions you make SHALL be available
*    for the community under THIS license, source code included.
*    Improvements or extentions, including adaptions to new architectures,
*    SHALL be reported and transmitted to Nilsen Elektronikk AS.
* 3. You must cause the modified files to carry prominent notices stating
*    that you changed the files, what you did and the date of changes.
* 4. You may NOT distribute this software under another license without
*    explisit permission from Nilsen Elektronikk AS, Norway.
* 5. This software is free, and distributed in the hope that it will be
*    useful, but WITHOUT ANY WARRANTY; without even the implied warranty
*    of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*    You SHALL NOT use this software unless you accept to carry all
*    risk and cost of defects or limitations.
*
*    ------------  CHANGE RECORD  ----------------
*    Jan Erik Nilsen (jen@nilsenelektronikk.no) July 26. 1999:
*        First free version of this software published.
*
*    Dag Magne Vedaa (dagmagne@otrum.no) Jan 21. 2000:
*        procSemWait(). moved enable_interrupt(). Dangerous!!
*
*    Reinhard Doerner (r.doerner@mru.de) 19.3.2001
*        procSemSignal(). Aktiviert Prio x einen Task mit Prio > x
*        dann muss ein Rescheduling erfolgen
*
*    Jan Erik Nilsen (jen@nilsenelektronikk.no) Feb 12. 2002:
*        if (this->priority < proc_curpid->priority)
*        I had used '>' instead of '<' at line 451. Sorry.
*    Naohiko Shimizu (nshimizu@keyaki.cc.u-tokai.ac.jp) Oct. 15.2002:
*        Add conditional compile statements for smaller object on GCC
*/
/*  GLOBAL  */
#define THIS_IS_PROC_C
#include "proc.h"
#define  NULL (void *)0

/*  GLOBAL END  */

/*  FILE=procinit.c  */

#ifdef L_procGlobal
PID   *proc_curpid;               /* pointer to current process         */
#ifdef INT_ATOMIC
int  proc_nodisp;                 /* disallow tick dispatch if not zero */
#else
char  proc_nodisp;                /* disallow tick dispatch if not zero */
#endif
int   proc_runcnt;                /* number of running processes        */
int   proc_maxage;                /* max age before pre-emtive dispatch */
char  proc_pattern;               /* workspace filled with pattern      */
PID   *proc_pritab[PROC_PRISIZE]; /* priority table                     */
void (*proc_at_next)(void);       /* to be done after contex switching  */

#else
extern PID   *proc_curpid;        /* pointer to current process         */
#ifdef INT_ATOMIC
extern int  proc_nodisp;          /* disallow tick dispatch if not zero */
#else
extern char  proc_nodisp;         /* disallow tick dispatch if not zero */
#endif
extern int   proc_runcnt;         /* number of running processes        */
extern int   proc_maxage;         /* max age before pre-emtive dispatch */
extern char  proc_pattern;        /* workspace filled with pattern      */
extern PID   *proc_pritab[PROC_PRISIZE]; /* priority table              */
extern void (*proc_at_next)(void); /* to be done after contex switching  */

void procInsertProcess(PID *pid);
int procRemoveProcess(PID *pid);
void procNextProcess(void);
int procAnyNextProcess(void);
#endif

/*===========  procInsertProcess  ============================
*
*   Purpose:      PRIVATE FUNCTION - DONT USE THIS
*                 Insert the given pid into the link.
*
*   Input:        ptr to pid
*   Output:       none
*   Return:       none
*/
#ifdef L_procInsertProcess
void procInsertProcess(PID *pid)
{
  int pri = pid->priority;

  if (proc_pritab[pri] == NULL)
    proc_pritab[pri] = pid->next = pid;
  else {
    pid->next = proc_pritab[pri]->next;
    proc_pritab[pri]->next = pid;
  }
  if (!pid->susp) proc_runcnt++;
}

#endif


/*===========  procRemoveProcess  ============================
*
*   Purpose:      PRIVATE FUNCTION - DONT USE THIS
*                 Remove the given pid from the link.
*
*   Input:        ptr to pid
*   Output:       none
*   Return:       see proc.h
*/
#ifdef L_procRemoveProcess
int procRemoveProcess(PID *pid)
{
  int pri = pid->priority;
  PID *pp = proc_pritab[pri];
  PID *this;

  if (pp == NULL) return(PROC_BADPID); /* inconsistent data         */
  for (this = pp; this->next != pp; this = this->next) {
    if (this->next == pid) {           /* found pid somewhere       */
      this->next = this->next->next;   /* take off link             */
      if (!pid->susp) proc_runcnt--;
      return(PROC_YOU_OK);
    }
  }
  if (pp == pid) {                     /* terminate current process */
    proc_runcnt--;                     /* one less running          */
    if (pid->next == pid) {            /* if last proc              */
      proc_pritab[pri] = NULL;
      return(PROC_ME_OK);
    }
    proc_pritab[pri] = this->next = this->next->next;
    return(PROC_ME_OK);
  }
  return(PROC_BADPID);                 /* should not happend        */
}
#endif

/*  FILE END         */
/*  FILE=procnp.c    */
/*  INCLUDE=proc.var */



/*===========  procNextProcess  ============================
*
*   Purpose:      PRIVATE FUNCTION - DONT USE THIS
*                 Called from the assembly function procReschedule()
*                 Finds the next process to start.
*                 NB: This function does not return until one process
*                 is runable. Therefore, interrupt MUST be enabled.
*                 The global proc_curpid is changed to point to the
*                 next process to start and (*proc_at_next) is
*                 called.
*
*   Global:       proc_curpid
*   Input:        none
*   Return:       none
*/
#ifdef L_procNextProcess
void procNextProcess(void)
{
  PID *this;
  int i = 0;

  for (;;) {
    if (proc_pritab[i] != NULL) {
      for (this = proc_pritab[i];
           this->next != proc_pritab[i] && this->next->susp;
           this = this->next);
      if (this->next->susp == 0) break; /* runable */
    }
    if (++i >= PROC_PRISIZE) i = 0;
  }
  proc_pritab[i] = proc_curpid = this->next;
  if (proc_at_next) (*proc_at_next)();
}

#endif

/*  FILE END         */
/*  FILE=procanp.c   */
/*  INCLUDE=proc.var */



/*===========  procAnyNextProcess  ============================
*
*   Purpose:      PRIVATE FUNCTION - DONT USE THIS
*                 Called from the assembly level.
*                 This function scans through the processes.
*                 If a no runable process is found, 0 is returned.
*                 Otherwise, if a runable process is found, but the
*                 process is the original process, 1 is returned.
*                 If a new runable process is found, proc_curpid
*                 is changed to point to it, (*proc_at_next) is
*                 called and -1 is returned.
*
*   Global:       proc_curpid
*   Input:        none
*   Return:       0, 1 or -1, see above
*/
#ifdef L_procAnyNextProcess
int procAnyNextProcess(void)
{
  PID *this;
  int i;

  for (i = 0; i < PROC_PRISIZE; i++) {
    if (proc_pritab[i] != NULL) {
      for (this = proc_pritab[i];
           this->next != proc_pritab[i] && this->next->susp;
           this = this->next);
      if (this->next->susp == 0) { /* runable */
        if (proc_curpid == this->next) return(1);
        proc_pritab[i] = proc_curpid = this->next;
        if (proc_at_next) (*proc_at_next)();
        return(-1);
      }
    }
  }
  return(0);
}
#endif


/*===========  procInitialize  ============================
*
*   Purpose:      Initializes process system, must be called
*                 before anything else. Inserts the current ``main''
*                 process into the system.
*                 NB: If the priority value is out of range, the system
*                 will blow up.
*
*   Input:        ptr to pid
*                 priority of this process
*   Output:       none
*   Return:       none
*/
#ifdef L_procInitialize
void procInitialize(PID *pid, int priority)
{
  int  i;

  pid->age      = 0;            /* is fresh          */
  pid->susp     = 0;            /* not sleeping      */
  pid->priority = priority;
  pid->stack    = NULL;
  pid->ws       = NULL;
  pid->wssize   = 0;
  proc_nodisp   = 0;            /* dispatching is ok */
  proc_runcnt   = 0;            /* 0 running process */
  proc_maxage   = 10;           /* default max age   */

  for (i = 0; i < PROC_PRISIZE; i++) proc_pritab[i] = NULL;
  BEGIN_CRITICAL_REGION();
  proc_curpid = pid;
  procInsertProcess(pid);
  END_CRITICAL_REGION();
}

#endif

/*===========  procCreateProcess  ============================
*
*   Purpose:      Initializes the pid and the stack frame,
*                 Inserts the pid into the system.
*
*                 NB: If the priority value is out of range, the system
*                 will blow up.
*
*   Input:        see below
*   Output:       none
*   Return:       none
*/
#ifdef L_procCreateProcess
void procCreateProcess(
  PID  *pid,                 /* process descriptor         */
  void (*func)(),            /* process function           */
  int priority,              /* priority                   */
  char *ws,                  /* work space (stack etc.)    */
  int  wssize,               /* work space size [Bytes]    */
  int  nparam,               /* # of parameters to process */
  ...)                       /* parameters to process      */
{
  int i;
#ifdef PROC_USE_STDARG
  va_list ap;
#endif

  for (i = 0; i < wssize; i++) *(ws+i) = proc_pattern;

#ifndef PROC_USE_STDARG
  pid->stack    = stack_frame(ws, wssize, pid, func, &nparam);
#else
  va_start(ap, nparam);
  pid->stack    = stack_frame(ws, wssize, pid, func, nparam, ap);
  va_end(ap);
#endif

  pid->priority = priority;
  pid->ws       = ws;
  pid->wssize   = wssize;
  pid->age      = 0;
  pid->susp     = 0;
  BEGIN_CRITICAL_REGION();
  procInsertProcess(pid);
  END_CRITICAL_REGION();
}

#endif

/*  FILE END         */
/*  FILE=procpost.c  */
/*  INCLUDE=proc.var */



/*===========  procTerminate  ============================
*
*   Purpose:      Terminates i.e. destroies the given process.
*
*                 NB: NEVER ATTEMPT TO TERMINATE THE LAST PROCESS.
*
*   Input:        ptr to pid
*   Output:       none
*   Return:       see proc.h
*/
#ifdef L_procTerminate
int procTerminate(PID *pid)
{
  int r;

  BEGIN_CRITICAL_REGION();
  r = procRemoveProcess(pid);
  END_CRITICAL_REGION();
  if (r == PROC_ME_OK) procReschedule();    /* get out of dead process */
  return(r);
}

#endif


/*  FILE END         */
/*  FILE=procpri.c   */
/*  INCLUDE=proc.var */



/*===========  procChangePriority  ============================
*
*   Purpose:      Changes the priority of the given process.
*                 The change works after the first rescheduling.
*                 NB: If the priority value is out of range, the system
*                 will blow up.
*
*   Input:        ptr to pid
*                 new priority
*   Output:       none
*   Return:       none
*/
#ifdef L_procChangePriority
void procChangePriority(PID *pid, int priority)
{
  BEGIN_CRITICAL_REGION();
  if (procRemoveProcess(pid) != PROC_BADPID) {
    pid->priority = priority;
    procInsertProcess(pid);
  }
  END_CRITICAL_REGION();
}

#endif

/*  FILE END         */
/*  FILE=procwss.c   */
/*  INCLUDE=proc.var */



/*===========  procGetUsedWSSize  ============================
*
*   Purpose:      Return the size of the work space which is used.
*                 This is done by finding the largest continuous
*                 block with pattern equal to that of proc_pattern.
*                 The size of that block is subtracted from the
*                 work space size and returned.
*
*                 NB: The size of the first process, inserted by
*                 procInitialize() can not be found.
*
*   Input:        ptr to pid
*   Output:       none
*   Return:       see above
*/
#ifdef L_procGetUsedWSSize
int procGetUsedWSSize(PID *pid)
{
  char *cp = pid->ws;
  int  i;
  int  max = 0;
  int  num = 0;

  for (i = 0; i < pid->wssize; i++) {
    if (*cp++ == proc_pattern) {
      num++;
      if (max < num) max = num;
    }
    else num = 0;
  }
  return(pid->wssize - max);
}

#endif

/*  FILE END         */
/*  FILE=procsem.c   */
/*  INCLUDE=proc.var */



/*===========  procSemInit  ============================
*
*   Purpose:      Initialize semaphore. Semaphores must
*                 be initialized prior to use.
*
*   Input:        ptr to semaphore, initial value
*   Output:       none
*   Return:       none
*/
#ifdef L_procSemInit
void procSemInit(SEMAPHORE *sp, int value)
{
  sp->value = value;
  sp->ph = sp->pt = NULL;
}

#endif

/*===========  procSemSignal  ============================
*
*   Purpose:      Increment semaphore value. If negative
*                 or zero then take oldest process from
*                 head of semaphore list and activate it.
*
*   Input:        ptr to semaphore
*   Output:       none
*   Return:       none
*/
#ifdef L_procSemSignal
void procSemSignal(SEMAPHORE *sp)
{
  PID *this;
#ifdef REINT
  int sr;
#endif

  disable_interrupt();
  if (++sp->value <= 0) {
    this = sp->ph;
    if ((sp->ph = this->sema) == NULL) sp->pt = NULL;
    this->susp = 0;
    enable_interrupt();
    proc_runcnt++;
    if (this->priority < proc_curpid->priority) /* R.Doerner 19.3.2001 */
      procReschedule();
  }
  else enable_interrupt();
}

#endif

/*===========  procSemSignalCareful  =======================
*
*   Purpose:      Same as procSemSignal, but interrupt is
*                 NOT disabled during the processing.
*                 Intended for use in interrupt handles.
*
*   Input:        ptr to semaphore
*   Output:       none
*   Return:       Waiting process was made runable: 1
*                 Otherwise: 0
*/
#ifdef L_procSemSignalCareful
int procSemSignalCareful(SEMAPHORE *sp)
{
  PID *this;

  if (++sp->value <= 0) {
    this = sp->ph;
    if ((sp->ph = this->sema) == NULL) sp->pt = NULL;
    this->susp = 0;
    proc_runcnt++;
    return(1);
  }
  return(0);
}

#endif

/*===========  procSemWait  ============================
*
*   Purpose:      Decrement semaphore value. If negative then
*                 put current process on tail of semaphore list,
*                 suspend process and reschedule
*
*   Input:        ptr to semaphore
*   Output:       none
*   Return:       none
*/
#ifdef L_procSemWait
void procSemWait(SEMAPHORE *sp)
{
  PID *this;
#ifdef REINT
  int sr;
#endif

  disable_interrupt();
  proc_curpid->sema = NULL;              /* mark end of list    */
  if (--sp->value < 0)  {
    if ((this = sp->pt) == NULL)
      sp->ph = sp->pt = proc_curpid;     /* this is the only    */
    else
      this->sema = sp->pt = proc_curpid; /* insert at list tail */
    proc_curpid->susp = 1;
    proc_runcnt--;
    enable_interrupt();
    procReschedule();
  }
  else enable_interrupt();
}

#endif
/*  FILE END  */
