/******************** MODULE INFO  ****************************/
/*
**  File name   :  timer.c
*/
/*  AUTHOR      :  Jan Erik Nilsen (jen@nilsenelektronikk.no) */
/*  VERSION     :  3.2                                        */
/*  DATE        :  Sun Apr 02 23:40:08 2000                   */
/*
*   Compiler    :  ANSI C
*   Project     :  proc  Real-Time Kernel
*   Functions   :  timer functions
*   Global var  :  see below
*
*
* Copyright (c) nilsen elektronikk as, Norway. (www.nilsenelektronikk.no)
* This software is the property of nilsen elektronikk as, Norway.
*
* The proc RTOS is free software; you can use it, redistribute it
* and/or modify it under the following terms:
* 1. You are not allowed to remove or modify this copyright notice
*    and License paragraphs, even if parts of the software is used.
* 2. The improvements and/or extentions you make SHALL be available
*    for the community under THIS license, source code included.
*    Improvements or extentions, including adaptions to new architectures,
*    SHALL be reported and transmitted to Nilsen Elektronikk AS.
* 3. You must cause the modified files to carry prominent notices stating
*    that you changed the files, what you did and the date of changes.
* 4. You may NOT distribute this software under another license without
*    explisit permission from Nilsen Elektronikk AS, Norway.
* 5. This software is free, and distributed in the hope that it will be
*    useful, but WITHOUT ANY WARRANTY; without even the implied warranty
*    of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*    You SHALL NOT use this software unless you accept to carry all
*    risk and cost of defects or limitations.
*
*    ------------  CHANGE RECORD  ----------------
*    Jan Erik Nilsen (jen@nilsenelektronikk.no) July 26. 1999:
*        First free version of this software published.
*    Gerard Willemsen (gerard.w@faber-electronic.nl) Jan 13. 2000:
*        changed: timerStart  (See GW below)
*         reason: if we want to add a timer bigger then timer_start->timer
*                 we ran into a bug. The time became wrong.
*    Naohiko Shimizu (nshimizu@keyaki.cc.u-tokai.ac.jp) Oct. 15.2002:
*        Add conditional compile statements for smaller object on GCC

*/
#define  NULL (void *)0
#include "timer.h"
#include "proc.h"



#ifdef L_timerGlobal

TIMER         *timer_start;
char          timer_busy;
unsigned char timer_lost;
unsigned long timer_counter;
#else
extern TIMER         *timer_start;
extern char          timer_busy;
extern unsigned char timer_lost;
extern unsigned long timer_counter;
#endif

/*===========  timerInit  ============================
*
*   Purpose:      To be done first.
*
*   Input:        none
*   Output:       none
*   Return:       none
*
*/
#ifdef L_timerInit
void timerInit(void)
{
  timer_start = NULL;
  timer_busy  = timer_lost = 0;
}

#endif

/*===========  timerStart  ============================
*
*   Purpose:      A timer is started, or re-started if
*                 it already excists. The timer will
*                 excist until it is deleted by timerStop
*                 or until it has timed out. If it excists
*                 when the time elapses,  the function
*                 checkEventTimerList(); will delete the
*                 timer and send a value x to the target CHANnel.
*
*                 The timers are elements of a simple linked
*                 list. The first elements time is absolute,
*                 and the next element(s) time is relative
*                 to the current. At insertion, the time of
*                 the current, and the next, element may be
*                 adjusted. At deletion, the time of the
*                 following element must be adjusted.
*                 The function checkEventTimerList() will
*                 decrement the first timer, and then remove
*                 all the timers with zero time.
*
*   Input:        ptr to timer, time, x
*   Output:       none
*   Return:       none
*
*/
#ifdef L_timerStart
void timerStart(TIMER *tp, int time, int x)
{
  TIMER  *temp;
  int    acct;

  tp->x = x;

  BEGIN_CRITICAL_REGION();
  timer_busy++;
  if (!timer_start) {             /* if list is empty */
    tp->timer = time;
    tp->next  = NULL;
    timer_start = tp;
    timer_busy--;
    END_CRITICAL_REGION();
    return;
  }
  if (tp == timer_start) {        /* if This is the first    */
    if (!timer_start->next) {     /* if This is the only one */
      timer_start->timer = time;
      timer_busy--;
      END_CRITICAL_REGION();
      return;
    }
    timer_start->next->timer += timer_start->timer;
    timer_start = timer_start->next;
  }
  else {
    for (temp = timer_start; temp->next; temp = temp->next)
      if (temp->next == tp) {
        if (tp->next) tp->next->timer += tp->timer;
        temp->next = tp->next;
        break;
      }
  }

  if (time <= timer_start->timer) {    /* insert prior to first */
    tp->next = timer_start;
    tp->timer = time;
    timer_start = tp;
    tp->next->timer -= time;
    timer_busy--;
    END_CRITICAL_REGION();
    return;
  }


  acct = timer_start->timer;   /* GW */

  for (temp = timer_start; temp->next; temp = temp->next) {
    if ((acct + temp->next->timer) >= time) {
      temp->next->timer -= time - acct;
      break;
    }
    else acct += temp->next->timer;  /* GW */
  }
  tp->next = temp->next;
  temp->next = tp;
  tp->timer = time - acct;
  timer_busy--;
  END_CRITICAL_REGION();
}

#endif


/*===========  timerStop  ============================
*
*   Purpose:      A timer, started using timerStart(),
*                 is deleted.
*
*   Input:        ptr to timer
*   Output:       none
*   Return:       If the has not been removed by timeout
*                 return 0, otherwise 1
*
*/
#ifdef L_timerStop
int timerStop(TIMER *tp)
{
  TIMER  *temp;
  int    removed = 0;

  BEGIN_CRITICAL_REGION();
  timer_busy++;
  if (timer_start) {
    if (tp == timer_start) {
      if (timer_start->next)
        timer_start->next->timer += timer_start->timer;
      timer_start = timer_start->next;
      removed = 1;
    }
    else {
      for (temp = timer_start; temp->next; temp = temp->next)
        if (temp->next == tp) {
          if (tp->next) tp->next->timer += tp->timer;
          temp->next = tp->next;
          removed = 1;
          break;
        }
    }
  }
  timer_busy--;
  END_CRITICAL_REGION();
  return(removed);
}

#endif


/*===========  timerWait  ============================
*
*   Purpose:      Wait, i.e. sleep n ticks.
*                 Uses timerStart and chanInChar
*
*   Input:        n: number of ticks to wait
*   Output:       none
*   Return:       none
*
*/
#ifdef L_timerWait
void timerWait(int n)
{
  TIMER         tim;
  CHAN          c;
  unsigned char cfifo[2];

  c.fifo = cfifo;
  chanInit(&c, 2);
  tim.chan = &c;
  timerStart(&tim, n, 0);
  chanInChar(&c);
}

#endif

/*===========  timerCheckList  ============================
*
*   Purpose:      To be called frequently from a cyclic process.
*                 This function scans a list of timers.
*                 For each timer the following is done:
*                 If the timer == 0, it is removed, and a value x,
*                   given to timerStart(), is sent to the target.
*                 Otherwise the timer is decremented.
*
*                 If the event channel is full, the timer event will be lost.
*                 The return value can be used to suggest for
*                 rescheduling because some process is waked up.
*
*   Uses:         chanIfOutChar() in case of timeout
*
*   Input:        none
*   Output:       none
*   Return:       if a process is waked up, 1 is returned, else 0.
*
*/
#ifdef L_timerCheckList
int timerCheckList(void)
{
  int r = 0;

  timer_counter++;
  if (timer_busy) timer_lost++;
  else {
    for (;;) {
      if (timer_start) {
        (timer_start->timer)--;
        while (timer_start->timer <= 0) {
          r |= chanIfOutChar(timer_start->chan, timer_start->x);
          /* we are fucked if the queue is full */
          timer_start = timer_start->next;
          if (!timer_start) break;
        }
      }
      if (!timer_lost) break;
      timer_lost--;
    }
  }
  return(r > 1);
}
#endif
