/******************* FILE - HEADING **************************
*
*   File name	:  packet.c
*/
/*  DATE        :  Tue Jul 27 08:56:44 1999                   */
/*  AUTHOR      :  Jan Erik Nilsen (jen@nilsenelektronikk.no) */
/*  VERSION     :  3.0                                        */
/*
*   Compiler    :  ANSI C
*   Project     :  proc  Real-Time Kernel
*   Contents	:  packet pool and packet put and get
*
*
* Copyright (c) nilsen elektronikk as, Norway. (www.nilsenelektronikk.no)
* This software is the property of nilsen elektronikk as, Norway.
*
* The proc RTOS is free software; you can use it, redistribute it
* and/or modify it under the following terms:
* 1. You are not allowed to remove or modify this copyright notice
*    and License paragraphs, even if parts of the software is used.
* 2. The improvements and/or extentions you make SHALL be available
*    for the community under THIS license, source code included.
*    Improvements or extentions, including adaptions to new architectures,
*    SHALL be reported and transmitted to Nilsen Elektronikk AS.
* 3. You must cause the modified files to carry prominent notices stating
*    that you changed the files, what you did and the date of changes.
* 4. You may NOT distribute this software under another license without
*    explisit permission from Nilsen Elektronikk AS, Norway.
* 5. This software is free, and distributed in the hope that it will be
*    useful, but WITHOUT ANY WARRANTY; without even the implied warranty
*    of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*    You SHALL NOT use this software unless you accept to carry all
*    risk and cost of defects or limitations.
*
*    ------------  CHANGE RECORD  ----------------
*    Jan Erik Nilsen (jen@nilsenelektronikk.no) July 26. 1999:
*        First free version of this software published.
*
*    Jan Erik Nilsen May 18. 2000:
*      ALIGN introduced. Some re-writing. See also proc.h
*/
#define  NULL (void *)0
#include "proc.h"
#include "packet.h"


/*
    The packet pool is organized as a single linked list of packets.
    Packets are allocated from the front of the list.
    Free packets are inserted at the front of the list.

    The pool memory is divided into packets of given size.
    If demanded, memory references are aligned
    (ALIGN_VAL is defined in proc.h)
*/



/*===========  packetInitPool  =======================
*
*   Purpose:      To be done first.
*                 An area of memory and its size is given.
*                 The net size of a packet is given as well.
*                 The first part of the pool memory is used for
*                 a pool handle. The memory size (m) needed for a
*                 n number of packets is:
*                   m = sizeof(PACKETPOOL) + (alignment space) + 
*                     n * (sizeof(PACKET_HEAD) + msgsize + (alignment space))
*
*   Input:        ptr to pool buffer
*                 size of pool buffer     (bytes)
*                 net size of data packet (bytes)
*   Output:       none
*   Return:       pointer to packet pool struct
*/
PACKETPOOL *packetInitPool(char *ppool, int ppsize, int msgsize)
{
  int    num;
  PACKET *pp;
  PACKETPOOL *polp;

#ifdef ALIGN_VAL
  num = ALIGN_ADDON((int)ppool);
  ppool += num;
  ppsize -= num;
#endif

  polp = (PACKETPOOL *)ppool;
  ppsize -= (int)sizeof(PACKETPOOL);
  ppool += sizeof(PACKETPOOL);

#ifdef ALIGN_VAL
  num = ALIGNED_NUMBER((int)sizeof(PACKET_HEAD) + msgsize);
#else
  num = (int)sizeof(PACKET_HEAD) + msgsize;
#endif
  polp->packetsize    = num;
  polp->packetnum     = 0;
  polp->packetmsgsize = num - (int)sizeof(PACKET_HEAD);
  polp->packetstart   = NULL;

  while (ppsize >= num) {
    pp = (PACKET *)ppool;
    pp->ppool = polp;
    pp->next = polp->packetstart;
    polp->packetstart = pp;
    ppsize -= num;
    ppool += num;
    (polp->packetnum)++;
  }
  procSemInit(&polp->packetleft, polp->packetnum);
  return(polp);
}



/*===========  packetAlloc  ==========================
*
*   Purpose:      Alloc a Packet from the pool. If the
*                 pool is empty, then block until a packet
*                 is available.
*
*   Input:        pointer to pool
*   Output:       none
*   Return:       ptr to Packet
*
*/
PACKET *packetAlloc(PACKETPOOL *h)
{
  PACKET *pp;
  int    pkused;

  procSemWait(&h->packetleft);

  BEGIN_CRITICAL_REGION();
  pkused = h->packetnum - h->packetleft.value;
  if (pkused > h->packetusedmax) h->packetusedmax = pkused;
  pp = h->packetstart;
  h->packetstart = h->packetstart->next;
  END_CRITICAL_REGION();
  return(pp);
}



/*===========  packetFree  =======================
*
*   Purpose:      Free a Packet to the pool.
*
*   Input:        ptr to Packet
*   Output:       none
*   Return:       none
*
*/
void packetFree(PACKET *pp)
{
  PACKETPOOL *h;

  if (pp == NULL) return;
  h = pp->ppool;
  BEGIN_CRITICAL_REGION();
  pp->next = h->packetstart;
  h->packetstart = pp;
  END_CRITICAL_REGION();
  procSemSignal(&h->packetleft);
}



/*===========  packetPut  ================================
*
*   Purpose:      A packet, allocated and then written to,
*                 is put into a queue. If the queue is full,
*                 this function blocks.
*
*   Input:        queue ptr
*                 packet ptr
*   Output:       none
*   Return:       none
*
*/
void packetPut(QUEUE *qp, PACKET *pp)
{
  queuePut(qp, (char *)pp);
}



/*===========  packetGet  ===============================
*
*   Purpose:      Return a packet from the packet queue.
*                 If the queue is empty, this function blocks.
*
*   Input:        queue ptr
*   Output:       none
*   Return:       packet ptr
*
*/
PACKET *packetGet(QUEUE *qp)
{
  return((PACKET *)queueGet(qp));
}
