/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Firmware re-flash utility (present in the absence of working AlphaBIOS) */
/* Begun by Stig Telfer, Alpha Processor Inc, 8 June 1999 */

/* NOTE: the assumption made throughout this module is that firmware image
 * follows firmware image header in the ROM.  This may not always be the case,
 * for example with version 2 headers (which permit a 'table of contents
 * approach') */


#include "lib.h"
#include "uilib.h"
#include "platform.h"
#include "rom.h"
#include "floppy.h"
#include "nvram.h"


/*--------------------------------------------------------------------*/
/* Private data */

/* Because flash ROMs typically work in sectors, we can't just think of an 
 * image as starting at some address and ending at some other: we must also
 * take into account the sectors occupied or partially occupied.
 *
 * This code cannot work unless the ROM is organised so that no two ROM images
 * share the same sector.  Otherwise, erasing one image would corrupt the other.
 * 
 * When quoting space available between images, the outerbound may safely be
 * used if the above rule is followed.
 * 
 * When quoting space available in empty regions, the inner bound must be used.
 * this preserves the above no-sharing rule.
 */


/* Max practical size for a ROM file.  NB Linux kernels may be ROM files */
/* FIXME: Can we find a practical way of deducing file size and allocating
 * just that?
 */
#define MAX_ROMIMG_SIZE	(1536<<10)
static char *dld_buf = NULL;


static unsigned nblks, blksize;		/* map block management */

static unsigned nregions;		/* flashable regions, image or empty */
static int isize;
static FWID itype;


enum test_states { NONE, SCAN, LOAD, BURN };
static unsigned progress = NONE;

static const Rect r_fwbar  = {  1,  1, 63, 10 };
#define NROWS			5
#define MAX_ROWS_DISPLAYABLE	10
#define	COL1			3
#define COL2			33

static const Rect r_key    = { 64,  1, 17, 10 };
static const Rect r_dialog = {  1, 11, 80,  8 };

static const Point p_fwbar = {  3, 3 };		/* relates to the above */
static const Point p_scale = { 56, 3 };		/* relates to the above */
static const Point p_fwimg = {  3, 5 };		/* relates to the above */

#define MAXBLKS	52				/* blocks in firmware bar */


/* commands available in this module */

static DBM_STATUS scan( int argc, char *argv[] );
static DBM_STATUS load( int argc, char *argv[] );
static DBM_STATUS erase( int argc, char *argv[] );
static DBM_STATUS burn( int argc, char *argv[] );
static DBM_STATUS help( int argc, char *argv[] );

static const Cmd_t fw_cmds[] =
{
    { "scan - analyse firmware for images, check integrity",
			"scan", scan, SYS_NONE },
    { "load - get new image for firmware update", 
			"load", load, SYS_NONE },
    { "burn - flash new image and verify",
			"burn", burn, SYS_NONE },
    { "help - in case of confusion, start here",
			"help", help, SYS_NONE },
    { "quit - return to main diagnostics",
			"quit", NULL, SYS_NONE },

    /* commands after this line are non-essential, and don't get printed */
    /* as menu options - they can still be used from the keyboard though */

    { "erase - remove an image from the firmware",
			"erase", erase, SYS_NONE },
};

#define NUMFWMENUS 5


/*--------------------------------------------------------------------*/
/* if the user was diverted due to buggered firmware, dont scare them */

static const String dont_panic_msg = 
"If you did not expect to see this message, then it is likely that\r"
"your BIOS firmware image is corrupted.  This program will attempt to help\r"
"restore your machine to working order.  The process involves:\r\r"
"   * Analysis of your ROM for defects ('scan')\r"
"   * Loading of new firmware ('load')\r"
"   * Re-flashing of the ROM ('burn')\r"
"   * Verification of new ROM integrity\r\r"
"Press any key to begin.";

static void user_dont_panic( void )
{
    mobo_box( r_lrgapp, "Don't Panic!" );
    printf_dbm( dont_panic_msg );
    (void) mobo_key( 0 );
}



/*--------------------------------------------------------------------*/

/* This extremely simple progress bar callback depends on nobody moving
 * the cursor during its operation and not putting it back in the right
 * place */

static void blkCallBack( void )
{ printf_dbm( FG_LGREY BLK_HALF /* CU_NORMAL */ ); }

/* Firmware key, drawn next to the map */

static void drawkey( void )
{
    unsigned i;
    String C,N;

    static const FWID keymember[] = {		/* firmware favourites */
	FW_CBOX, FW_SR, FW_WNT, FW_SRM,
	FW_DIAG, FW_OSFPAL, FW_LINUX, FW_UNKNOWN };

    mobo_box( r_key, "Firmware Key" );		/* moves to point inside box */

    for ( i=0; i<ARRAYLEN( keymember ); i++ )
    {
	C = rom_I2colour( keymember[i] );
	N = rom_I2name( keymember[i] );
	printf_dbm("%s" BLK_FULL CU_NORMAL " %s\r", C, N );
    }
}



/*--------------------------------------------------------------------*/
/* Firmware map, drawn for the user in block form */

static void initmapparams( void )
{
    /* calculate the block sizes required, assume minimum ROM size of 1K/blk */
    blksize = 1<<10;			/* start low, work up */
    do {
	blksize <<= 1;			/* double blocksize, half blocks */
	nblks = plat_romsize / blksize;
    } while ( nblks > MAXBLKS );
	

    /* note: assuming the ROM size is a multiple of the block size */
}

/* takes an offset and calculates the coords of the block in the bar */
static Point off2blk( unsigned offset )
{
    Point P;
    P.x = p_fwbar.x + offset / blksize, P.y = p_fwbar.y;
    return P;
}


/* takes an image number and returns the data record for it */
static rom_t *imgno2img( unsigned imgno )
{
    rom_t *R;
    unsigned i;

    /* find the firmware record for the image referred to */
    for ( i=1, R=rom_phys_map; i < imgno; i++, R=R->next )
    {
        if ( R==NULL )			break;	/* off the end of the list */
    }
    return R;
}

/* takes a FWID type and returns a region number in the ROM, or -1 */

static int fwid2imgno( FWID ID )
{
    rom_t *R;
    unsigned i;

    if ( ID == FW_UNKNOWN )		return -1;	/* reject immediately */

    for ( i=1, R=rom_phys_map; ; i++, R=R->next )
    {
        if ( R==NULL )                          return -1;	/* no match */
	if ( R->hdr->romh.V1.fw_id == ID )	return i;	/* match */
    }
    return -1;
}



static void displayimg( const rom_t *R, const char leg_char )
{
    unsigned imgblks;
    unsigned i;

    if ( R->flags & ROM_EMPTYRGN )	return;	/* empties don't get drawn */

    /* calculate the number of blocks for this image */
    imgblks = R->rsize / blksize;
    if ( R->rsize % blksize )	imgblks++;	/* round up */
    
    /* draw the new image onto the map */
    mobo_goto( off2blk( R->rstart ) );
    printf_dbm( R->defcol );

    if ( leg_char == -1 )		/* Do we have anything to print? */
    {
	for ( i = 0; i < imgblks; i++ )
	    printf_dbm( BLK_HALF );
    } else {
	for ( i = 0; i < imgblks; i++ )
	    printf_dbm( "%c", leg_char );
    }

    printf_dbm( CU_NORMAL );
}


/* draws the ROM firmware setup, including any details currently known */
static void drawmap( void ) 
{
    unsigned i;
    rom_t *R;
    Point drawpos;
    char valid;

    mobo_box( r_fwbar, "System Firmware Map" );

    /* draw the empty ROM in block form */
    mobo_goto( p_fwbar );
    printf_dbm( FG_DGREY );
    for ( i = 0; i < nblks; i++ )	printf_dbm( BLK_HALF );
    mobo_goto( p_scale );
    printf_dbm( BLK_HALF CU_NORMAL "=%dKB", blksize >> 10 );


    /* traverse the list of images found, putting each on the map */
    mobo_goto( p_fwimg );
    if ( progress < SCAN ) {				/* scan not done yet */
	printf_dbm("(System firmware map has not been scanned yet)");
	return;
    } 

    for ( R=rom_phys_map, nregions=0; R != NULL; R=R->next, nregions++ )
    {
	if ( (R->flags & ROM_EMPTYRGN) == 0 ) 
	    displayimg( R, nregions + 'A' );
	else
	    displayimg( R, -1 );


	/* we still draw on the map but have no space for the name */
	if ( nregions >= MAX_ROWS_DISPLAYABLE )		continue;


	/* Do we have a checksummable image that failed checksum? */
	if ( (R->flags & (ROM_IMGCSUM | ROM_IMGVALID)) == ROM_IMGCSUM )
	    valid = '!';
	else
	    valid = ' ';


	/* Compute the cursor position and put out the message */
	if ( nregions < NROWS )
	    drawpos.x = COL1, drawpos.y = p_fwimg.y + nregions;
	else
	    drawpos.x = COL2, drawpos.y = p_fwimg.y + (nregions - NROWS);
	mobo_goto( drawpos );
	printf_dbm( "%c)%c%s (%dK/%dK)", nregions + 'A', valid, 
			R->name, R->rsize >> 10, R->asize >> 10 );
    }
}



/*--------------------------------------------------------------------*/
/* drawdialog - a central box that handles user interaction */

static void rundialog( void )
{
    const Cmd_t *C;
    unsigned i;

    do {
	mobo_cls();
        drawmap();
        drawkey();

	mobo_box( r_dialog, "Flash Firmware Management Options" );
	
	/* list the commands available */
	for ( C = fw_cmds, i = 1; i <= NUMFWMENUS; C++, i++)
	    printf_dbm("%d) %s\r", i, C->desc);

	/* now run menus - note same options in cmd vocab and on menu */

    } while ( mobo_menu( fw_cmds, NUMFWMENUS, fw_cmds, ARRAYLEN( fw_cmds ) ) == 0);
}



/*--------------------------------------------------------------------*/
/* Command implemenations */


/*--------------------------------------------------------------------*/
/* Scan firmware map for recognisable images.  Since we may have entered
 * because the AlphaBIOS image is corrupt, we must be paranoid 
 */

static DBM_STATUS scan( int argc, char *argv[] )
{
    DBM_STATUS sval;
    rom_t *R;
    BOOLEAN rval;

    /* Build a new list of all headers present in the ROM */
    sval = rom_index();
    switch ( sval )
    {
	case STATUS_FAILURE:			/* Meaning ROM unreadable */
	    mobo_alertf( "ROM Access Error",
		"The ROM is unreadable by diags!\r"
		"Unless you expected this, there may be a flash chip defect" );
	    return STATUS_FAILURE;
	    

	case STATUS_WARNING:			/* Meaning no ROM headers */
	    mobo_alertf( "ROM Access Warning",
		"The ROM appears to be empty!  Diags found no images.\r"
		"Unless you expected this, there may be a flash chip defect" );
	    break;				/* Press on with reflash */

	default: case STATUS_SUCCESS:		/* Everything as expected */
	    break;
    }



    /* for each ROM image found, verify the image integrity, if applicable */
    for ( R = rom_phys_map; R != NULL; R=R->next )
    {
	displayimg( R, -1 );

	if ( R->next != NULL && (R->next->flags & ROM_EMPTYRGN) == 0 )
	{
	    if( R->next->astart < (R->astart + R->asize) )
		mobo_alertf( "ROM firmware image overlap",
		    "%s (at %dKB) and %s (at %dKB) overlap in the ROM!\r"
		    "You should be cautious using either firmware image",
		    R->name, R->astart>>10, R->next->name, R->next->astart>>10);
	}


	/* Are we able to checksum this object? */
	if ( (R->flags & ROM_IMGCSUM) == 0 )
	    continue;

	rval = rom_body_valid( R->hdr, R->body );
	rom_dump( R );
	if ( rval )	
	{
	    R->flags |= ROM_IMGVALID;
	}
	else
	{
	    mobo_logf( LOG_CRIT "FLASH: Image (%s) did not verify\n", R->name );
            mobo_alertf( "ROM corruption found during scan",
                         "This image (%s, %dK at %dKb) appears to be corrupt",
                         R->name, R->rsize >> 10, R->rstart >> 10 );
	    continue;
	}
    }

    progress = SCAN;		/* we have scanned */
    return STATUS_SUCCESS;	/* the firmware map will be redrawn now */
}



/*--------------------------------------------------------------------*/
/* load a ROM image in one of the standard formats, through any of a
 * number of mechanisms.
 */




static DBM_STATUS ldimg_floppy( int a, char **b )
{
    unsigned ipl = swpipl( 7 );			/* disable ints during floppy */


    mobo_box( r_dialog, "Load from floppy disk" );
    if ( FileDirectory("\\", 'w') == STATUS_FAILURE )
    {
	mobo_alertf( "Floppy disk error", 
		     "Read of floppy failed, is there a DOS disk?");
	return STATUS_FAILURE;
    }

    /* setup for user interaction */
    mobo_goto( p_help );
    printf_dbm( "%-64s", "Enter filename [Or press Enter to cancel]:" );
    mobo_goto(p_prompt);
    printf_dbm( Prompt );               /* platform prompt string */
    mobo_input(cbuf, sizeof(cbuf));
    if ( strlen(cbuf) == 0 )
    {
	return STATUS_FAILURE;
    }

    printf_dbm( "..." );		/* respond to user */
    isize = LoadAFile( cbuf, dld_buf );	/* load file to download buffer */

    swpipl( ipl );					/* re-enable any ints */
    if (isize == -1)
    {
	mobo_alertf("Floppy load error", "Couldn't load file %s", cbuf );
	return STATUS_FAILURE;
    }

    return STATUS_SUCCESS;
}


/* used by COM1, COM2 and SROM for generic Xmodem image transfer */

static DBM_STATUS do_ser_dld( io_dev D )
{
    isize = XReceive( dld_buf, D );	/* download from requested port */

    if ( isize <= 0 ) 	return STATUS_FAILURE;
    else		return STATUS_SUCCESS;
}

static DBM_STATUS ldimg_com1( int a, char **b )
{ return do_ser_dld( DEV_COM1 ); }
static DBM_STATUS ldimg_com2( int a, char **b )
{ return do_ser_dld( DEV_COM2 ); }
static DBM_STATUS ldimg_srom( int a, char **b )
{ return do_ser_dld( DEV_SROM ); }


static const Cmd_t ld_cmds[] =
{
    { "floppy - MS-DOS format floppy",
                        "floppy", ldimg_floppy, SYS_NONE },
    { "com1 - Xmodem transfer of image",
                        "com1", ldimg_com1, SYS_NONE },
    { "com2 - Xmodem transfer of image",
                        "com1", ldimg_com2, SYS_NONE },
    { "srom - Xmodem transfer of image",
                        "srom", ldimg_srom, SYS_NONE },
    { "quit - return to main menu",
			"quit", NULL, SYS_NONE },
    { "help - in case of confusion, start here",
                        "help", help, SYS_NONE },
};


static DBM_STATUS load( int argc, char *argv[] )
{
    const Cmd_t *C;
    unsigned i;
    romheader_t *H;
    rom_t *R;
    int rval;
    char buf[32];
    DBM_STATUS sval;

    if ( progress < SCAN ) {
	help( 0, NULL );
	return STATUS_FAILURE;
    }


    dld_buf = malloc( MAX_ROMIMG_SIZE );
    if ( dld_buf == NULL )
    {
	mobo_alertf( "Memory allocation failure",
		"Not enough heap space to process file load.\r"
		"Sorry." );
	return STATUS_FAILURE;
    }

    /* Don't want to be confused by preloaded stuff */
    rom_free( rom_ram_map );
    rom_ram_map = NULL;
    memset( dld_buf, 0, MAX_ROMIMG_SIZE );
    isize = 0;


    /* list the commands available */
    mobo_box( r_dialog, "Load new firmware image" );
    for ( C = ld_cmds, i = 1; i <= ARRAYLEN( ld_cmds ); C++, i++)
	printf_dbm("%d) %s\r", i, C->desc);


    /* now run menus - note same options in cmd vocab and on menu */
    rval = mobo_menu( ld_cmds, ARRAYLEN(ld_cmds), ld_cmds, ARRAYLEN(ld_cmds) );
    if ( rval == -1 )	
    {
	free( dld_buf );
	dld_buf = NULL;				/* defensiveness */
	return STATUS_FAILURE;			/* quit cmd */
    }


    /* a download operation has been attempted */
    /* we can tell if the download was successful by seeing if isize > 0 */

    if ( isize <= 0 ) {
	mobo_alertf( "ROM Image Load Failure",
		     "The download operation was not successful" );
	progress = SCAN;			/* step back one from LOAD */
	free( dld_buf );
	dld_buf = NULL;				/* defensiveness */
	return STATUS_FAILURE;
    }

    /* now check to see if the image we loaded is a recognisable ROM image */
    /* This is done by scanning it as ROM-in-RAM */

    progress = LOAD;

    sval = rom_ram_scan( dld_buf, MAX_ROMIMG_SIZE );
    R = rom_ram_map;

    /* What should be done if more than one image was found? */
    switch( sval )
    {
	case STATUS_SUCCESS:		/* A recognised image was found */
	    itype = R->fwid;
	    H = R->hdr;

	    mobo_alertf( "Download Successful",
			 "Image (%dKb) is type %s (%d) Rev %s\r"
			 "Choose 'burn' to write this image into the ROM.",
			 isize >> 10, rom_I2name( itype ), itype,
			 rom_parseRevision( H, buf ) );
	    break;
	    

	case STATUS_WARNING:		/* No recognised image was found */
	    itype = FW_UNKNOWN;
	    H = NULL;

	    mobo_alertf( "Unstructured Image Warning",
		 "This image (%dKb) has no header structure to validate.\r"
		 "Unless you expected this, you should not attempt reflash!",
		 isize >> 10 );
	    break;


	case STATUS_FAILURE:		/* Recognised image with bad checksum */
	default:			/* (Defensive coding) */
	    itype = R->fwid;
	    H = R->hdr;

	    mobo_alertf( "Downloaded Image Corrupted",
		"Image (%dKB) is recognised as '%s' but has bad checksum\r"
		"Unless you expected this, you should not attempt to reflash!",
		isize >> 10, rom_I2name( itype ) );
	    break;
    }

    return sval;
}


/*--------------------------------------------------------------------*/
/* remove an image from the rom, by letter */

static DBM_STATUS erase( int argc, char *argv[] )
{
    char imgchar;
    int imgno;
    DBM_STATUS sval = STATUS_SUCCESS;
    rom_t *R;

    if ( progress < SCAN )
    {
	mobo_alertf( "Firmware scan required",
		"You must scan the firmware for images before erasure!");
	return STATUS_FAILURE;
    }

    if ( argc != 2 )
    {
	mobo_alertf( "Argument required",
		"Please specifiy the image number you wish to erase" );
	return STATUS_FAILURE;
    }
    sscanf( argv[1], "%c", &imgchar );

    /* find the firmware record for the image referred to */
    imgno = tolower(imgchar) - 'a' + 1;		/* Image 'A' equates to 1 */
    R = imgno2img( imgno );

    /* Validation: Is this an image that we can safely erase? */
    if ( (R->fwid == FW_NVLOG) && (nvlog_state != NVLOG_CLOSED) )
    {
	mobo_alertf( "Operation Cannot Proceed",
		"The region '%s' cannot be erased because it is currently\r"
		"Being used (log state is %d, need %d to proceed)",
		R->name, nvlog_state, NVLOG_CLOSED );
	return STATUS_FAILURE;
    }


    mobo_goto( off2blk( R->astart ) );

    /* clear all available space - overkill, or playing safe? */
    sval = plat_romerase( R->astart, R->astart + R->asize - 1,
				blkCallBack, blksize );

    if ( sval != STATUS_SUCCESS )
    {
	mobo_alertf("Operation failed", "Erasure failed!" );
	sval = STATUS_FAILURE;
    }

    /* perform a re-scan of the ROM */
    scan( 0, NULL );

    return sval;
}


/*--------------------------------------------------------------------*/
/* perform a reflashing and verification */

static DBM_STATUS burn( int argc, char *argv[] )
{
    char imgchar;
    unsigned imgno=0;
    int address_selected = FALSE;
    unsigned rstart=0, astart=0;
    int rval;
    DBM_STATUS sval;
    rom_t *R;

    /*--------------------------------------------------------------------*/
    /* Validation sanity checks and image region selection */

    if ( progress < LOAD )
    {
        mobo_alertf( "Firmware load required",
                "You must scan the firmware and load the new image first!");
        return STATUS_FAILURE;
    }

    if ( itype == FW_UNKNOWN )
    {
	rval = mobo_alertf( "Message from the programmer",
	    "This unknown image may break your system.  Press 'q' to quit.\r"
	    "Press any other key to carry on (if you are certain)." );

	if ( tolower( rval ) == 'q')
		return STATUS_FAILURE;		/* user bottled it */
    }


    /* Find the region to be reflashed.  The policy here is:
     * 1) V2 may and V3 must specify an address in the ROM header
     * 2) Undocumented fixed-address mode - user specifies @hex-addr
     * 3) if the user specified a number on the command line, it goes 
     * 4) if there is a matching image in the ROM, that's second
     * 5) prompt the user for a region to overwrite as last choice
     */

    R = rom_ram_map;		/* pick up whatever was downloaded from here */
    if( ROMH_VERSION( R->hdr ) == 3 )
    {
	do {
	    rval = mobo_alertf( "Flashing of fixed-location image",
		"Please confirm (y/n) writing this image at its fixed address\r"
		"The image is programmed for flashing at %d KB", R->rstart );

	    if ( tolower( rval ) == 'n' )
		return STATUS_FAILURE;

	} while ( tolower( rval ) != 'y' );

	astart = R->astart;
	rstart = R->rstart;
	address_selected = TRUE;
    }

    /* secret option */
    if ( !address_selected && argv[1][0] == '@' )
    {
	address_selected = TRUE;
	if ( sscanf( argv[1], "@%x", &rstart ) != 1 )
	{
	    mobo_alertf( "Error in secret usage mode",
		"Specify args (in hex) such as 'burn @40000'" );
	    return STATUS_FAILURE;
	}
	astart = rstart;
    }

    if ( !address_selected ) 		/* normal arg validation */
    {
	if ( argc != 2 )
	{
	    imgno = fwid2imgno( itype );		/* attempt to infer */
	    if ( imgno == -1 )
	    {
		rval = mobo_alertf( "Argument required",
			"Please select a region to reflash, or 'q' to quit");
		if ( tolower( rval ) == 'q' )		return STATUS_FAILURE;
		imgno = tolower(rval) - 'a' + 1;
	    }

	} else {
	    sscanf( argv[1], "%c", &imgchar );
	    while ( !isalpha( imgchar ) )
	    {
		imgchar = mobo_alertf( "Argument required",
		    "Please select a region to reflash now (by letter), "
			"or 'q' to quit");
		if ( tolower( imgchar ) == 'q' )	return STATUS_FAILURE;
	    }
	    imgno = tolower(imgchar) - 'a' + 1;
	}


	/* validate the selected region, if it is a valid region at all */

	R = imgno2img( imgno );
	if ( R == NULL ) {
	    mobo_alertf("Bad region selection",
			"The region selected (%c) doesn't match the firmware!",
			imgchar );
	    return STATUS_FAILURE;
	}

	if ( R->asize < isize ) {
	    mobo_alertf( "Selected region too small",
		"Insufficient space for the loaded image (%dK req, %dK avail)\r"
		"Try another firmware region, or erasing something...",
		isize >> 10, R->asize >> 10 );
	    return STATUS_FAILURE;
	}

	astart = R->astart;
	rstart = R->rstart;
	address_selected = TRUE;
    }

    /*--------------------------------------------------------------------*/
    /* At this point, we could perhaps consider the possibility of having
     * disjoint header and image.  Currently, we only support scenarios where
     * header and image are contiguous (as they would have been at download)
     * This seems like a reasonable restriction to make at this point.
     */


    mobo_goto( off2blk( rstart ) );
    sval = plat_romerase( astart, astart + isize - 1, blkCallBack, blksize);

    if ( sval != STATUS_SUCCESS )			/* failure case */
    {
	mobo_alertf("Erase failed", "Erasure failed!" );
	scan( 0, NULL );
	return STATUS_FAILURE;
    }


    mobo_goto( off2blk( rstart ) );
    sval = plat_romwrite( astart, (const unsigned char *)dld_buf, isize,
                        blkCallBack, blksize);

    if ( sval != STATUS_SUCCESS )			/* failure case */
    {
	mobo_alertf("Reflash failed", "Writing of new image failed!" );
	scan( 0, NULL );
	return STATUS_FAILURE;
    }

    mobo_alertf("Firmware update complete",
		"The new image has been written to firmware\r"
		"The ROM will now be verified" );

    scan( 0, NULL );				/* re-scan with new image */

    progress = BURN;
    return STATUS_SUCCESS;
}



static DBM_STATUS help( int argc, char *argv[] )
{
    String hmsg;
    switch ( progress ) {
	case NONE:
	    hmsg = "The first step is to analyse your firmware\r"
		   "Choose 'scan' to check ROM integrity";
	    break;
	
	case SCAN:
	    hmsg = "The next stage is to load a firmware upgrade\r"
		   "Choose 'load' and follow the directions from there";
	    break;

	case LOAD:
	    hmsg = "The next stage is to reflash your ROM with the upgrade\r"
		   "Choose 'burn' to start this process";
	    break;

	case BURN:
 	default:
	    hmsg = "Sorry, even I'm confused, there is no help at this stage\r"
		   "try 'quit' if you want to transfer to system firmware";
	    break;
    }
    mobo_alertf( "Don't Panic!", hmsg );
    return STATUS_SUCCESS;
}




/*--------------------------------------------------------------------*/
/* MAIN: Take a user through the process of reflashing a firmware image,
 * from start to finish */

DBM_STATUS reflash( int argc, char *argv[] )
{
    int ipl = swpipl( 7 );

    mobo_cls();

    /* was the user diverted here due to a corrupt firmware image? */
    if ( StartupMode == BIOSRECOVER || StartupMode == ADERECOVER ) 
		user_dont_panic();

    /* setup ROM data */
    initmapparams();

    /* draw the interface */
    rundialog();

    /* quit and clean-up code */
    progress = NONE;
    if ( dld_buf != NULL )
	free( dld_buf );

    swpipl( ipl );
    return STATUS_SUCCESS;
}


