/* file:	gray.c
 *
 * Copyright (C) 1993 by
 * Digital Equipment Corporation, Maynard, Massachusetts.
 * All rights reserved.
 *
 * This software is furnished under a license and may be used and copied
 * only  in  accordance  of  the  terms  of  such  license  and with the
 * inclusion of the above copyright notice. This software or  any  other
 * copies thereof may not be provided or otherwise made available to any
 * other person.  No title to and  ownership of the  software is  hereby
 * transferred.
 *
 * The information in this software is  subject to change without notice
 * and  should  not  be  construed  as a commitment by digital equipment
 * corporation.
 *
 * Digital assumes no responsibility for the use  or  reliability of its
 * software on equipment which is not supplied by digital.
 */

/*
 *++
 *  FACILITY:
 *
 *      Common console Firmware
 *
 *  MODULE DESCRIPTION:
 *
 *      GRAY, Gray code generator and checker.
 *
 *  AUTHORS:
 *
 *      AJ Beaverson
 *
 *  CREATION DATE:
 *  
 *	ajb	26-May-1993
 *
 *  MODIFICATION HISTORY:
 *
 *--
 */

#include "cp$src:platform.h"
#include	"cp$src:kernel_def.h"
#include	"cp$src:parse_def.h"
#include	"cp$src:stddef.h"
#include	"cp$src:common.h"
#include "cp$inc:prototypes.h"
#include	"cp$src:msg_def.h"

#define stdin  (struct FILE *) (((struct PCB *) (getpcb()))->pcb$a_stdin)
#define stdout (struct FILE *) (((struct PCB *) (getpcb()))->pcb$a_stdout)
#define stderr (struct FILE *) (((struct PCB *) (getpcb()))->pcb$a_stderr)


#define	QSTART	0	/* start position		*/
#define	QSTOP	1	/* end position			*/
#define	QLENGTH	2	/* length			*/
#define QCHECK	3	/* check mode (default)		*/
#define QGEN	4	/* generate data		*/
#define QINVERT	5	/* report indecies		*/
#define	QBYTE	6
#define QWORD	7
#define QLONG   8	/* longword (default)		*/
#define	QMAX	9
#define QSTRING	"%xstart %xstop %xlength check gen invert byte word long"


/*
 * Generate a graycode given an index.  Invert every other entry so that
 * the patterns generate have all bits changing except one.
 */
unsigned int gen_gray (unsigned int gix) {
	if (gix & 1) {
	    return ~((gix >> 1) ^ gix);
	} else {
	    return (gix >> 1) ^ gix;
	}
}

/*
 * Given a graycode, calculate the index of that graycode, assuming the
 * graycode was generated by:
 *
 *    graycode (x) = (x >> 1) ^ x
 */
unsigned int invert_gray (unsigned int gray) {
  int i;
  unsigned int index;
  unsigned int lastbit;
  unsigned int gcbit;

  lastbit = 0;
  index = 0;
  for (i=sizeof (int) * 8 - 1; i>=0; i--) {
    gcbit = (gray >> i) & 1;
    index |= (gcbit ^ lastbit) << i;
    lastbit ^= gcbit;
  }
  return index;
}


/*+
 * ============================================================================
 * = gray - Gray code generator/checker.				      =
 * ============================================================================
 *
 * OVERVIEW:
 *  
 *	GRAY is a program that handles gray code patterns.  It has three
 *	operating modes, check, generation or pattern inversion.
 *	
 *	By default, it works in check mode.  It reads all files on the
 *	command line (or stdin if no arguments are present), and verifies that
 *	the streams conform to the gray code pattern.  It reports all locations
 *	that do not conform.  Furthermore, locations that don't conform are
 *	run through the pattern inversion routines so see where they could
 *	have come from.
 *
 *	When generating data, GRAY generates gray code data to all files on
 *	the command line (or to stdout if no files are present).  Typically,
 *	a length qualifier would be specified to keep the program from
 *	generating data indefinately.
 *
 *	In pattern inversion mode, GRAY treats all arguments as hex numbers
 *	and reports which indecies those patterns could have been generated
 *	by. This is useful when the check mode reports discrepencies and you
 *	want to know from where the data could have come from.
 *	
 *   COMMAND FMT:	gray 0 0 0 gray
 *
 *   COMMAND FORM:
 *  
 *	gray ( [-{start|stop|length|check|gen|invert|byte|word|long}] [<file>...] )
 *  
 *   COMMAND TAG:	gray 0 RXBZ gray
 *
 *   COMMAND ARGUMENT(S):
 *
 * 	<file>... - Specifies the file(s) on which to perform the checks.
 *
 *   COMMAND OPTION(S):
 *
 *	-start - Specifies the starting offset within the file.
 *	-stop - Specifies the ending offset within the file.
 *	-length - Specifies the number of bytes to dump.
 *	-check	- Check data
 *	-gen    - Generate data
 *	-invert	- treat all arguments as patterns
 *	-byte	
 *	-word
 *	-long
 *
 *   COMMAND EXAMPLE(S):
 *~
 *	>>> gray mopdl:gray.dat/eza0
 *	>>> gray -l 1000 -o  >foo.dat
 *	>>> gray -l 1000 -o | gray 
 *	>>>echo -n 123 |gray -byte
 *	00000000: expected 00000000 got 00000031 (from 00000021 or AAAAAA8B )
 *	00000001: expected 000000FE got 00000032 (from 00000023 or AAAAAA89 )
 *	00000002: expected 00000003 got 00000033 (from 00000022 or AAAAAA88 )
 *	>>>gray -byte -start AAAAAA8B -len 1 -gen | hd
 *	00000000  31 1
 *	>>>>
 *~
 *
 * FORM OF CALL:
 *  
 *	gray (int argc, char *argv [])
 *  
 * RETURN CODES:
 *
 *	msg_success - normal completion
 *       
 * ARGUMENTS:
 *
 * 	argc	- number of words present on command line
 *	argv	- array of pointers to words on command line
 *
 * SIDE EFFECTS:
 *
 *	None
 *
-*/
gray (int argc, char *argv []) {
	struct FILE	*fin;
	int		status,argix;
	unsigned int	length, offset;
	struct QSTRUCT	qual [QMAX];
	int		data_len;
	int		mode; /* one of QCHECK, QGEN or QINVERT */
	unsigned int	gcd, gix;

	/*
	 * Decode the flags (argc,argv are adjusted and flags culled out).
	 */
	status = qscan (&argc, argv, "-", QSTRING, qual);
	if (status != msg_success) {
	    err_printf (status);
	    return status;
	}

	/* decode the boundaries */
	offset = 0;
	length = -1;
	if (qual [QSTART].present) {
	    offset = qual [QSTART].value.integer;
	}
	if (qual [QSTOP].present) {
	    length = qual [QSTOP].value.integer - offset;
	}

	/* length over rides end qualifier */
	if (qual [QLENGTH].present) {
	    length = qual [QLENGTH].value.integer;
	}

	/* data length */
	data_len = 4;
	if (qual [QBYTE].present) data_len = 1;
	if (qual [QWORD].present) data_len = 2;
	if (qual [QLONG].present) data_len = 4;

	/* determine which mode we're in */
	mode = QCHECK;
	if (qual [QCHECK].present) mode = QCHECK;
	if (qual [QGEN].present) mode = QGEN;
	if (qual [QINVERT].present) mode = QINVERT;

	/*
	 * No argument on the command line, so either read from stdin
	 * (check mode), or write to stdout (generate mode).
	 */	
	if (argc == 1) {
	    if (mode == QCHECK) {
		gray_checker (stdin, data_len, offset, length);
	    } else if (mode == QGEN) {
		gray_generator (stdout, data_len, offset, length);
	    }


	/*
	 * Visit each argument on the command line
	 */
	} else {
	    for (argix=1; argix<argc; argix++) {
		if (killpending()) break;
		switch (mode) {
		case QCHECK:
		    fin = (struct FILE *) fopen (argv [argix], "r");
		    if (!fin) {
			err_printf ("%s: can't open %s\n", argv [0], argv [argix]);
			continue;
		    }
		    gray_checker (fin, data_len, offset, length);
		    fclose (fin);
		    break;

		case QGEN:
		    fin = (struct FILE *) fopen (argv [argix], "w");
		    if (!fin) {
			err_printf ("%s: can't open %s\n", argv [0], argv [argix]);
			continue;
		    }
		    gray_generator (fin, data_len, offset, length);
		    fclose (fin);
		    break;

		case QINVERT:
		    gcd = xtoi (argv [argix]);
		    printf ("%08X can be derived from %08X or %08X\n",
			gcd,
			invert_gray (gcd),
			invert_gray (~gcd)
		    );
		    break;
		}
	    }
	}

	return msg_success;
}

/*----------*/
gray_checker (struct FILE *fp, unsigned int data_len, unsigned int offset, unsigned int length) {
	unsigned int datum;
	unsigned int gix, gcd;
	unsigned int status;
	int	i;
	char *buf;
	int read_size;
	int len;
	int x;
	unsigned int mask;

	switch (data_len) {
	case 1: mask = 0x000000ff; break;
	case 2: mask = 0x0000ffff; break;
	case 4: mask = 0xffffffff; break;
	}


	/* skip initial data */
	buf = (char *) malloc (2048);
	x = offset;
	while (x) {
	    read_size = min (2048, x);
	    len = fread (buf, 1, read_size, fp);
	    if (len == 0) return;
	    x -= len;
	}

	gix = offset;
	while (length -= data_len) {
	    gcd = gen_gray (gix);
	    status = fread (&datum, 1, data_len, fp);
	    if (status == 0) break;
	    if (status != data_len) {
	        err_printf ("Premature end of file\n");
		break;
	    }
	    if ((datum & mask) != (gcd & mask)) {
		printf ("%08X: expected %08X got %08X (from %08X or %08X ) \n",
		    gix*data_len,
		    (gcd & mask),
		    (datum & mask),
		    data_len * invert_gray (datum & mask),
		    data_len * invert_gray (~(datum & mask))
		);
	    }
	    gix++;
	}

	free (buf);
}

/*----------*/
gray_generator (struct FILE *fp, unsigned int data_len, unsigned int offset, unsigned int length) {
	unsigned int gix;
	unsigned int gcd;

	if (!length) return;

	gix = offset;
	do {
	    gcd = gen_gray (gix);
	    fwrite (&gcd, 1, data_len, fp);
	    gix++;
	    if (((gix & 0x1ff) == 0) && killpending ()) break;
	} while (length -= data_len);
}
