/* Copyright (c)1994-2000 Begemot Computer Associates. All rights reserved.
 * See the file COPYRIGHT for details of redistribution and use. */

/*
 * toy clock
 *
 * Arguments:
 *	ctrl toy csr_base
 */

# include "proc.h"

RCSID("$Id: dev_toy.c,v 1.1 2000/03/04 08:17:55 hbb Exp $")

typedef struct TOY TOY;

/*
 * Keep it simple: in start we keep the hosts number of seconds, when the
 * toy was last set (or when it was started). clock keeps the value it was
 * set to. If we read the clock, we look at the current host number of seconds,
 * compute how many have elapsed and add this to the 'clock'.
 */
struct TOY {
	unsigned csr_base;

	ushort	csr;
	UQuad	code;
	int	state;
	UQuad	value;

	UQuad	clock;		/* the toy, as set by the user */
	UQuad	start;		/* the clock at start time */
	int	cnt;
};

enum {
	TOY_WAITI	= 0,
	TOY_INIT	= 1,
	TOY_READ	= 2,
	TOY_WRITE	= 3,
};

void	toy_ctrl_create(IODev *, int, char **);
void	toy_ctrl_complete(IODev *);
ushort	toy_fetch(IODev *, unsigned);
void	toy_store(IODev *, unsigned, int, ushort);
void	toy_info(IODev *);
void	toy_command(IODev *, int, char **);

static int	toyc_info(IODev *, int argc, char **);


IODevCmd toy_cmds[] = {
	{ "?",		"[command ...]",	dev_help },
	{ "help",	"[command ...]",	dev_help },
	{ "info",	"",			toyc_info },
	{ NULL }
};

IOOps	toy_ops = {
	toy_ctrl_create,	/* ctrl_create */
	0,			/* dev_create */
	toy_ctrl_complete,	/* ctrl_complete */
	0,			/* reset */
	toy_fetch,		/* fetch */
	toy_store,		/* store */
	0,			/* vector */
	0,			/* dma */
	0,			/* onsig */
	toy_info,		/* info */
	0,			/* flush */
	toy_cmds,		/* cmds */
};

static UQuad conv2bcd(UQuad);
static UQuad toyset(UQuad);

/*
 * create controller for parallel interface
 * - nothing to do
 */
void
toy_ctrl_create(IODev *dev, int argc, char **argv)
{
	TOY *toy;

	if(argc != 1)
		conf_panic("toy: need 1 arg in controller configuration");
	toy = dev->data = xalloc(sizeof(TOY));
	(void)memset(toy, 0, sizeof(TOY));

	toy->csr_base = parse_csr(argv[0], "toy");
	toy->code = 0;
	toy->state = TOY_WAITI;

	proc.iopage[IOP(toy->csr_base+0)] = dev;
}

void	
toy_ctrl_complete(IODev *dev)
{
	TOY *toy = (TOY *)dev->data;
	struct timeval tp;

	gettimeofday(&tp, NULL);
	toy->clock = tp.tv_sec;
	toy->start = tp.tv_sec;
}

/*
 * fetch from csr/buf
 */
ushort	
toy_fetch(IODev *dev, unsigned a)
{
	TOY *toy = dev->data;
	struct timeval tp;
	UQuad diff;

	if(a == toy->csr_base) {
		switch(toy->state) {

		  case TOY_WAITI:	/* waiting for init sequence */
		  case TOY_WRITE:
			return toy->csr;

		  case TOY_INIT:	/* first read after init -> read clock */
			gettimeofday(&tp, NULL);
			diff = tp.tv_sec - toy->start;
			toy->value = conv2bcd(toy->clock + diff);
			toy->state = TOY_READ;

		  case TOY_READ:	/* next read */
			toy->csr = (toy->csr & 0xfeff) | ((toy->value & 1) << 8);
			toy->value >>= 1;
			return toy->csr;
		}
	}
	printf("toy_fetch(%o)\n", a);
	Trap4(0100);
}

void	
toy_store(IODev *dev UNUSED, unsigned a UNUSED, int mode UNUSED, ushort v UNUSED)
{
	TOY *toy = dev->data;
	struct timeval tp;

	if(a == toy->csr_base) {
		if(toy->state == TOY_READ) {
			toy->state = TOY_WAITI;
			toy->code = 0;
		}

		if(!(mode & M_High))
			SLB(toy->csr, v);
		if(!(mode & M_Low))
			SHB(toy->csr, v);


		switch(toy->state) {

		  case TOY_WAITI:	/* waiting for recognition code */
			toy->code = (toy->code >> 1)
				  | ((UQuad)(toy->csr & 0x100) << 55);
			if(toy->code == ((056243ULL << 48) | (035305ULL << 32)
				       | (056243ULL << 16) | (035305ULL << 0)))
				toy->state = TOY_INIT;
			break;

		  case TOY_INIT:	/* code recognized - first write */
			toy->cnt = 0;
			toy->state = TOY_WRITE;
			toy->value = 0;

		  case TOY_WRITE:
			toy->cnt++;
			toy->value = (toy->value >> 1)
				  | ((UQuad)(toy->csr & 0x100) << 55);
			if(toy->cnt == 64) {
				gettimeofday(&tp, NULL);
				toy->start = tp.tv_sec;
				toy->clock = toyset(toy->value);
				toy->state = TOY_WAITI;
				toy->code = 0;
			}
			break;

		  case TOY_READ:
			toy->state = TOY_WAITI;
			toy->code = 0;
			break;
		}

		return;
	}
	printf("toy_store(%o)\n", a);
	Trap4(0100);
}

/*
 * print info about device
 */
void	
toy_info(IODev *dev)
{
	TOY *toy = dev->data;

	printf("Time Of Day clock\n");
	printf("CSR at %08o\n", toy->csr_base);
}

static int
toyc_info(IODev *dev, int argc, char **argv UNUSED)
{
	if(argc != 0)
		return 1;
	toy_info(dev);
	return 0;
}

static uint
bcd(uint v)
{
	return (((v / 10) % 10) << 4) | (v % 10);
}

static uint
bcd2bin(uint v)
{
	return ((v >> 4) & 0xf) * 10 + (v & 0xf);
}

static UQuad
conv2bcd(UQuad clock)
{
	struct tm *tm;
	UQuad ret;
	clock_t c = clock;

	tm = gmtime(&c);

	ret = (UQuad)bcd(tm->tm_sec) << 8
	    | (UQuad)bcd(tm->tm_min) << 16
	    | (UQuad)bcd(tm->tm_hour) << 24
	    | (UQuad)bcd(tm->tm_wday) << 32
	    | (UQuad)bcd(tm->tm_mday) << 40
	    | (UQuad)bcd(tm->tm_mon + 1) << 48
	    | (UQuad)bcd(tm->tm_year % 100) << 56;
# if 0
	printf("%d/%d/%d %d:%d:%d\n",
		tm->tm_mday, tm->tm_mon, tm->tm_year,
		tm->tm_hour, tm->tm_min, tm->tm_sec);
# endif

	return ret;
}

static UQuad
toyset(UQuad val)
{
	struct tm tm;

# if 0
	printf("VAL: %qx\n", val);
# endif
	tm.tm_sec = bcd2bin(val >> 8);
	tm.tm_min = bcd2bin(val >> 16);
	tm.tm_hour = bcd2bin(val >> 24);
	tm.tm_mday = bcd2bin(val >> 40);
	tm.tm_mon = bcd2bin(val >> 48) - 1;
	tm.tm_year = bcd2bin(val >> 56);
	if(tm.tm_year < 70)
		tm.tm_year += 100;

# if 0
	printf("%d/%d/%d %d:%d:%d\n",
		tm.tm_mday, tm.tm_mon, tm.tm_year,
		tm.tm_hour, tm.tm_min, tm.tm_sec);
# endif

	return timegm(&tm);
}
