/* ODT.  We assume the cons driver is installed and has (a) put stdin/stdout
   into async, non-blocking, CBREAK, ~ECHO, and has set up a SIGIO handler. */

#include <stdio.h>
#include <setjmp.h>
#include <signal.h>

#include <signaltype.h>

#include "pdp11.h"
#include "driverint.h" /* for drivermask[] */
#define NEED_NAME
#define NEED_FUNCTIONS
#define NEED_EXTERN
#include "instrtbl.h"

extern int notraps;
extern jmp_buf notraps_jmp;

extern short int deftbl[65536];

#define CONS_I_STAT 0177560
#define CONS_I_DATA 0177562
#define CONS_O_STAT 0177564
#define CONS_O_DATA 0177566

enum odt_state_{
  ODT_PROMPT = 1,
  ODT_R,
  ODT_OPENLOC,
  ODT_OPENREG,
  ODT_LOC,
  ODT_REG,
  ODT_NEWVAL_LOC,
  ODT_NEWVAL_REG,
  ODT_PROMPT_PREV_REG,
  ODT_PROMPT_PREV_LOC,
  } ;
typedef enum odt_state_ ODT_STATE;

static jmp_buf odtdone;
static struct sigvec oldsigio;

static ODT_STATE odtstate;
static unsigned long int xaddr;
static word xregno;
static word newvalue;
static int singlestep;

static Gcmd(addr)
word addr;
{
 mmr0 = 0;
 mmr3 = 0;
 pirq = 0;
 fps = 0;
 cpu_error = 0;
 /* set ccr<8> (flush), clear mem_sys_err */
 psw = 0;
 pc = addr;
 halted = 0;
 longjmp(odtdone,1);
}

static Pcmd()
{
 halted = 0;
 longjmp(odtdone,1);
}

static Scmd()
{
 halted = 1;
 singlestep = 1;
 longjmp(odtdone,1);
}

static signaltype nullh()
{
}

static setsigio()
{
 struct sigvec sv;

 sv.sv_handler = nullh;
 sv.sv_mask = 0;
 sv.sv_flags = 0;
 sigvec(SIGIO,&sv,&oldsigio);
}

static resetsigio()
{
 sigvec(SIGIO,&oldsigio,(struct sigvec *)0);
}

static flushin()
{
 char ch;

 while (read(0,&ch,1) == 1) ;
}

static int get()
{
 char ch;

 while (1)
  { if (read(0,&ch,1) == 1) return(ch);
    sigpause(0);
  }
}

static put(ch)
int ch;
{
 char chc;

 chc = ch;
 write(1,&chc,1);
}

static puts(s)
#define CONST const
CONST char *s;				/* To keep damn gcc & <stdio.h> happy */
{
 for (;*s;s++) put(*s);
}

static putq()
{
 puts("?\r\n@");
}

static put8(val)
unsigned long int val;
{
 int i;

 for (i=21;i>=0;i-=3) put('0'+((val>>i)&7));
}

static put6(val)
word val;
{
 int i;

 for (i=15;i>=0;i-=3) put('0'+((val>>i)&7));
}

void odt()
{
 setsigio();
 if (setjmp(odtdone))
  { fflush(stdout);
    resetsigio();
    notraps = 0;
    return;
  }
 flushin();
 notraps = 1;
 while (setjmp(notraps_jmp))
  { puts("(trap)");
  }
 puts("\r\n");
 if (singlestep)
  { singlestep = 0;
    disas(pc);
  }
 else
  { put6(pc);
  }
 puts("\r\n@");
 fflush(stdout);
 odtstate = ODT_PROMPT;
 while (1)
  { int ch;
    ch = 0x7f & get();
    if (ch != 012) put(ch);
    switch (odtstate)
     { case ODT_PROMPT:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		odtstate = ODT_LOC;
		xaddr = ch - '0';
		break;
	     case 'r': case 'R': case '$':
		odtstate = ODT_R;
		xregno = 0;
		break;
	     case 'g': case 'G':
		Gcmd((word)0);
		break;
	     case 'p': case 'P':
		Pcmd();
		break;
	     case 's': case 'S':
		Scmd();
		break;
	     case 'Q':
		cons__reset();
		exit(0);
		break;
	     case 'X':
		xodt();
		break;
	     case 'Z':
		kill(getpid(),SIGTSTP);
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_R:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		odtstate = ODT_REG;
		xregno = (xregno << 3) | (ch - '0');
		break;
	     case 's': case 'S':
		odtstate = ODT_REG;
		xregno = 077;
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_OPENLOC:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		odtstate = ODT_NEWVAL_LOC;
		newvalue = ch - '0';
		break;
	     case '\r':
		odtstate = ODT_PROMPT_PREV_LOC;
		puts("\n@");
		break;
	     case '\n':
		xaddr = (xaddr & ~0xffff) | ((xaddr + 2) & 0xfffe);
		if (goodaddr(xaddr))
		 { odtstate = ODT_OPENLOC;
		   puts("\r\n@");
		   put8(xaddr,8);
		   put('/');
		   put6(fetchphys(xaddr));
		   put(' ');
		 }
		else
		 { odtstate = ODT_PROMPT_PREV_LOC;
		   puts("\r\n?\r\n@");
		 }
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_OPENREG:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		odtstate = ODT_NEWVAL_REG;
		newvalue = ch - '0';
		break;
	     case '\r':
		odtstate = ODT_PROMPT_PREV_REG;
		puts("\n@");
		break;
	     case '\n':
		if (xregno > 7)
		 { odtstate = ODT_PROMPT_PREV_REG;
		   puts("\r\n@");
		 }
		else
		 { xregno = (xregno + 1) & 7;
		   odtstate = ODT_OPENREG;
		   puts("\r\n@R");
		   put('0'+xregno);
		   put('/');
		   put6(*regs[xregno]);
		   put(' ');
		 }
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_LOC:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		xaddr = (xaddr << 3) | (ch - '0');
		break;
	     case '/':
		xaddr &= 017777776;
		if (goodaddr(xaddr))
		 { odtstate = ODT_OPENLOC;
		   put6(fetchphys(xaddr));
		   put(' ');
		 }
		else
		 { odtstate = ODT_PROMPT_PREV_LOC;
		   puts("?\r\n@");
		 }
		break;
	     case 'g': case 'G':
		Gcmd((word)(xaddr&0xffff));
		break;
	     case 's': case 'S':
		pc = xaddr & 0xffff;
		Scmd();
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_REG:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		xregno = (xregno << 3) | (ch - '0');
		break;
	     case 's': case 'S':
		xregno = 077;
		break;
	     case '/':
		xregno &= 0377;
		if (xregno != 077) xregno &= 7;
		odtstate = ODT_OPENREG;
		put6((xregno>7)?psw:*regs[xregno]);
		put(' ');
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_NEWVAL_LOC:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		newvalue = (newvalue << 3) | (ch - '0');
		break;
	     case '\r':
		storephys(xaddr,newvalue);
		odtstate = ODT_PROMPT_PREV_LOC;
		puts("\n@");
		break;
	     case '\n':
		storephys(xaddr,newvalue);
		xaddr = (xaddr & ~0xffff) | ((xaddr + 2) & 0xfffe);
		if (goodaddr(xaddr))
		 { odtstate = ODT_OPENLOC;
		   puts("\r\n@");
		   put8(xaddr,8);
		   put('/');
		   put6(fetchphys(xaddr));
		   put(' ');
		 }
		else
		 { odtstate = ODT_PROMPT_PREV_LOC;
		   puts("\r\n?\r\n@");
		 }
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_NEWVAL_REG:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		newvalue = (newvalue << 3) | (ch - '0');
		break;
	     case '\r':
		if (xregno > 7)
		 { psw = (psw & PSW_T) | (newvalue & ~PSW_T);
		 }
		else
		 { *regs[xregno] = newvalue;
		 }
		odtstate = ODT_PROMPT_PREV_REG;
		puts("\n@");
		break;
	     case '\n':
		if (xregno > 7)
		 { psw = (psw & PSW_T) | (newvalue & ~PSW_T);
		   odtstate = ODT_PROMPT_PREV_REG;
		   puts("\r\n@");
		 }
		else
		 { *regs[xregno] = newvalue;
		   xregno = (xregno + 1) & 7;
		   odtstate = ODT_OPENREG;
		   puts("\r\n@R");
		   put('0'+xregno);
		   put('/');
		   put6(*regs[xregno]);
		   put(' ');
		 }
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_PROMPT_PREV_REG:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		odtstate = ODT_LOC;
		xaddr = ch - '0';
		break;
	     case 'r': case 'R': case '$':
		odtstate = ODT_R;
		xregno = 0;
		break;
	     case 'g': case 'G':
		Gcmd((word)0);
		break;
	     case 'p': case 'P':
		Pcmd();
		break;
	     case 's': case 'S':
		Scmd();
		break;
	     case 'Q':
		cons__reset();
		exit(0);
		break;
	     case 'X':
		xodt();
		break;
	     case 'Z':
		kill(getpid(),SIGTSTP);
		break;
	     case '/':
		xregno &= 0377;
		if (xregno != 077) xregno &= 7;
		odtstate = ODT_OPENREG;
		put6((xregno>7)?psw:*regs[xregno]);
		put(' ');
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_PROMPT_PREV_LOC:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		odtstate = ODT_LOC;
		xaddr = ch - '0';
		break;
	     case 'r': case 'R': case '$':
		odtstate = ODT_R;
		xregno = 0;
		break;
	     case 'g': case 'G':
		Gcmd((word)0);
		break;
	     case 'p': case 'P':
		Pcmd();
		break;
	     case 's': case 'S':
		Scmd();
		break;
	     case 'Q':
		cons__reset();
		exit(0);
		break;
	     case 'X':
		xodt();
		break;
	     case 'Z':
		kill(getpid(),SIGTSTP);
		break;
	     case '/':
		xaddr &= 017777776;
		if (goodaddr(xaddr))
		 { odtstate = ODT_OPENLOC;
		   put6(fetchphys(xaddr));
		   put(' ');
		 }
		else
		 { odtstate = ODT_PROMPT_PREV_LOC;
		   puts("?\r\n@");
		 }
		break;
	     default:
		odtstate = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
     }
  }
}

void xodt()
{
 /* nothing here yet */
 puts("\r\n@");
}

void disas(addr)
word addr;
{
 word inst;
 int dtx;

 put6(addr);
 puts(": ");
 opcode = fetchword(addr,MMAN_ISPACE);
 dtx = get_deftbl(inst);
 if (deftbl[dtx] == DEF_UNUSED)
int get_deftbl(inst)
word inst;
{
 int l;
 int m;
 int h;

 if (deftbl[inst] != DEF_UNKNOWN) return(deftbl[inst]);
 l = 0;
 h = N_INSTR_DEF - 1;
 while (h-l > 1)
  { m = (h + l) / 2;
    if (inst >= instr_defs[m].value)
     { l = m;
     }
    else
     { h = m;
     }
  }
 if ((inst & instr_defs[l].mask) != instr_defs[l].value) l = DEF_UNUSED;
 deftbl[inst] = l;
 return(l);
}

}
