/* Copyright (c)1994-1999 Begemot Computer Associates. All rights reserved.
 * See the file COPYRIGHT for details of redistribution and use. */

/*
 * This program makes a copy of a file and substitutes zero blocks
 * by holes. This can drastically reduce the storage requirements
 * for disc images.
 */
# include <stdio.h>
# include <stdlib.h>
# include <fcntl.h>
# include <unistd.h>
# include <string.h>
# include <errno.h>

# include <begemot.h>
# include "cdefs.h"
# include "util.h"

static char usgtxt[] =
"Usage: zcp [-v] [-b n] infile ofile\n"
"Options:\n"
"	-v	be verbose (more v's may give more verbosity)\n"
"	-b n	set blocksize to n 512 byte blocks\n";

int
main(int argc, char *argv[])
{
	int opt, ifd, ofd, ret, oret, verbose = 0;
	unsigned char *buf, *p;
	unsigned long bufl = 0;
	off_t pos, opos;

	set_argv0(argv[0]);
	while((opt = getopt(argc, argv, "b:v")) != EOF)
		switch(opt) {

		  case 'b':
			bufl = strtoul(optarg, NULL, 0);
			break;

		  case 'v':
			verbose++;
			break;
		}

	argc -= optind;
	argv += optind;

	if(argc != 2) {
		fprintf(stderr, usgtxt);
		return 1;
	}

	if((ifd = open(argv[0], O_RDONLY, 0)) < 0)
		panic("%s: %s", argv[0], strerror(errno));
	if((ofd = open(argv[1], O_WRONLY|O_CREAT|O_TRUNC, 0666)) < 0)
		panic("%s: %s", argv[1], strerror(errno));

	if(bufl == 0)
		bufl = 1;
	bufl *= 512;

	if(verbose) {
		fprintf(stderr, "copy from %s to %s; ", argv[0], argv[1]);
		fprintf(stderr, "blocksize %lu bytes\n", bufl);
	}
	buf = xalloc(bufl);

	opos = pos = 0;
	while((ret = read(ifd, buf, bufl)) > 0) {
		for(p = buf; p < &buf[ret]; p++)
			if(*p != 0)
				break;
		if(p == &buf[ret]) {
			pos += ret;
			if(verbose > 1)
				fprintf(stderr, "skipping %d bytes\n", ret);
			continue;
		}
		if(opos != pos) {
			if(verbose > 1)
				fprintf(stderr, "seeking to %qd\n", (long long)pos);
			if(lseek(ofd, pos, SEEK_SET) != pos)
				panic("seek: %s", strerror(errno));
			opos = pos;
		}
		if(verbose > 1)
			fprintf(stderr, "writing %d bytes\n", ret);
		if((oret = write(ofd, buf, ret)) != ret) {
			if(oret < 0)
				panic("write: %s", strerror(errno));
			panic("short write: %d instead of %d", oret, ret);
		}
		pos += ret;
		opos += ret;
	}
	if(ret < 0)
		panic("read: %s", strerror(errno));

	if(opos != pos) {
		if(verbose)
			fprintf(stderr, "extending to %qd\n", (long long)pos);
		if(ftruncate(ofd, pos))
			panic("ftruncate: %s", strerror(errno));
		opos = pos;
	}

	if(verbose)
		fprintf(stderr, "done.\n");

	return 0;
}
