/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

# include <sys/types.h>
# include <sys/param.h>
# include <sys/socket.h>
# include <sys/time.h>
# include <sys/mman.h>
# include <sys/stat.h>
# include <sys/wait.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <signal.h>
# ifdef USE_VARARGS
#  include <varargs.h>
# else
#  include <stdarg.h>
# endif
# ifdef HAS_ASYNC_STREAMS
#  include <stropts.h>
# endif
# include <stdlib.h>
# include <string.h>
# include <setjmp.h>
# include <termios.h>
# include <errno.h>

# include "system.h"
# include "machine.h"
# include "compat.h"
# include "pathnames.h"

/*
 * tunable parameters
 */
# define PHYSMEM 	(1024*1024)
# define FD_DEVICES	64
# define P11CONF	"p11conf"
# define MAXTIMEOUT	13

# define _STRING(C)#C
# define STRING(S)_STRING(S)

# ifndef HAS_UCHAR
typedef unsigned char	uchar;
# endif
typedef signed char	schar;
typedef signed short	sshort;
typedef long long 	Quad;
typedef unsigned long long	UQuad;
# ifndef HAS_ULONG
typedef unsigned long	ulong;
# endif
typedef void		(*const VPF)(void);

typedef struct Proc 	Proc;
typedef struct MMG 	MMG;
typedef struct IOConf 	IOConf;
typedef struct IOOps  	IOOps;
typedef struct IODev  	IODev;
typedef struct Memops	Memops;
typedef struct FPRegs 	FPRegs;
typedef struct Timeout	Timeout;
typedef struct PDP_Float PDP_Float;

typedef void	(*PFstore)(unsigned, int, int, int, ushort);
typedef unsigned(*PFfetch)(unsigned, int, int);

/*
 * entry points to non-generic memory part
 */
struct Memops {
	void	(*reset)(MMG *);		/* bus reset		*/
	void	(*mmr0)(MMG *);			/* mmr0 written		*/
	void	(*mmr3)(MMG *);			/* mmr3 written		*/
	void	(*par)(MMG *, int, int, int);	/* par written		*/
	void	(*pdr)(MMG *, int, int, int);	/* pdr written		*/
	void	(*ccr)(MMG *);			/* ccr written		*/
	void	(*info)(MMG *);			/* odt info		*/
	PFstore	store16, store18, store22;
	PFfetch	fetch16, fetch18, fetch22;
	ushort  *(*addrphys)(MMG *, unsigned);	/* for dma		*/
};

/*
 * floating status bit definitions
 */
enum {
	FER	= 0100000,	/* floating error */
	FID	=  040000,	/* interrupt disable */
	FIUV	=   04000,	/* interrupt on undefined variable */
	FIU	=   02000,	/* interrupt on underflow */
	FIV	=   01000,	/* interrupt on overflow */
	FIC	=    0400,	/* interrupt on integer conversion error */
	FD	=    0200,	/* floating double precision mode */
	FL	=    0100,	/* floating long-integer mode */
	FT	=     040,	/* floating chop mode */
	FN	=     010,	/* neg */
	FZ	=      04,	/* zero */
	FV	=      02,	/* overflow */
	FC	=      01,	/* carry */
};

/*
 * floating point errors
 */
enum {
	FE_ILL	= 2,		/* illegal opcode */
	FE_DIV0	= 4,		/* divide by zero */
	FE_INT	= 6,		/* integer conversion error */
	FE_OFL	= 8,		/* overflow */
	FE_UFL	= 10,		/* underflow */
	FE_UND	= 12,		/* undefined variable */
};

/*
 * pdp float/double format
 * this structure MUST be 4 ushorts (64 bits) long or some code will break
 */
# ifdef USE_BIG_ENDIAN
struct PDP_Float {
	unsigned	s : 1;		/* sign */
	unsigned	e : 8;		/* expo offset 0200 */
	unsigned	m0 : 7;		/* 7 bits in 1st word */
	unsigned	m1 : 16;	/* 1nd word */
	unsigned	m2 : 16;	/* 2nd word */
	unsigned	m3 : 16;	/* 3nd word */
};
# else
struct PDP_Float {
	unsigned	m0 : 7;		/* 7 bits in 1st word */
	unsigned	e : 8;		/* expo offset 0200 */
	unsigned	s : 1;		/* sign */
	unsigned	m1 : 16;	/* 1nd word */
	unsigned	m2 : 16;	/* 2nd word */
	unsigned	m3 : 16;	/* 3nd word */
};
# endif
# define PDP_OFF	0200		/* exponent excess */

/*
 * floating point processor
 */
struct FPRegs {
	PDP_Float	ac[6];		/* AC0-6 */
	ushort		st;		/* state */
	ushort		fec;		/* exception code */
	ushort		fea;		/* exception address */
};


/*
 * processor
 */
struct Proc {
	ushort		reg[8];		/* registers for cur. mode	*/
	ushort		ureg[6];	/* R0-R5			*/
	ushort		kreg[6];	/* R0'-R5'			*/
	ushort		sp[4];		/* SP				*/

	ushort		cpuerr;		/* CPU error register		*/
	ushort		pirq;		/* programmed interrupt request	*/
	ushort		ltc;		/* line time clock		*/
	ushort		maint;		/* maintenance			*/
	ushort		ccr;		/* cache control :-)		*/

	int		fpd;		/* disable FPP			*/
	FPRegs		fpp;		/* floating point registers	*/

	/* PSW */
	uchar		c, n, v, z;
	int		t;
	int		pri;
	int		regs;
	int		curmode;	/* mode from psw 		*/
	int		prevmode;	/* previous mode  		*/

	MMG		*mmg;		/* memory manager		*/
	Memops		*memops;	/* operations			*/
	ushort		mmr0;
	unsigned	mmr2;		/* trapping pc			*/
	unsigned	mmr3;		/* 				*/
	int		mmfrozen;
	ushort		fmmr0, fmmr1, fmmr2;
	int		mmr1reg[2];	/* register part		*/
	int		mmr1chg[2];	/* change part			*/
	PFfetch 	fetch;
	PFstore		store;
	ushort		par[4][2][8];
	ushort		pdr[4][2][8];

	unsigned 	physmem;	/* amount of phys. mem (bytes)	*/
	IODev		*iopage[4096];

	int		reqpri;		/* highest pending bus request	*/
	IODev		*devices;

	unsigned	cmd;		/* last ifetch			*/
	ushort		bp;		/* breakpoint 			*/
	int		tinhibit;	/* tbit trap pending		*/
	int		halt;		/* stop processor		*/
};

/*
 * "Bootstrap" structure.
 */
struct IOConf {
	char	name[4];		/* name ala "kl", "rl", ...	*/
	IOOps	*ioops;			/* class specific functions	*/
};

/*
 * Per-Controller-Class structure
 */
struct IOOps {
	void		(*ctrl_create)(IODev *,int, char **);
	void		(*dev_create)(IODev *,int, char **);
	void		(*ctrl_complete)(IODev *);
	void		(*reset)(IODev *);
	ushort		(*fetch)(IODev *, unsigned);
	void		(*store)(IODev *, unsigned, int, ushort);
	ushort		(*vector)(IODev *);
	void		(*dma)(IODev *);
	void		(*async)(IODev *, void *);
	void		(*info)(IODev *);
	void		(*command)(IODev *, int, char **);
};

/*
 * Per-Controller-structure
 */
struct IODev {
	int		reqpri;
	IODev		*next;
	void		*data;
	char		*name;
	IOOps		*ioops;
	int		unit;
};

/*
 * timeout-structure
 */
struct Timeout {
	void		(*func)(void *);
	void		*arg;
	unsigned	time;
	unsigned	curr;
};


extern Proc 	proc;
extern ushort	startloc;
extern int	ill_trace;

extern int	tinterval;
extern char	verbose[128];

extern IOOps	rl_ops;
extern IOOps	kl_ops;
extern IOOps	mr_ops;

extern IODev	internals;
extern IODev	memsys;

extern jmp_buf	trap;

void	push(ushort);
ushort	pop(void);
void	switchmode(int, int);
void	interp(void);
void	Reset(void);
void	set_fd_async(int);
void	catch_io_sigs(void);

/*
 * generic memory interface
 */
ushort 	iofetch(unsigned);
void 	iostore(unsigned, int, ushort);
# define mmfetch(A,D,M)		proc.fetch((A),(D),(M))
# define mmstore(A,T,D,M,V)	proc.store((A),(T),(D),(M),(V))
# define mfetch(A,D) 		mmfetch((A),(D),proc.curmode)
# define mstore(A,T,D,V)	mmstore((A),(T),(D),proc.curmode,(V))
void	mem_init(void);		/* 1st initialisation	*/


volatile void	Trap(ushort) DEAD_FUNC;
volatile void	Trap10(void) DEAD_FUNC;
volatile void	Trap4(ushort) DEAD_FUNC;
volatile void 	MemTrap(int, int, int, ushort) DEAD_FUNC;
void		SetupTrap(ushort);
void		proc_pwrfail();
void		pwrfail_exit(Proc *);

void		mon_fpp(void);
void		store_long(int);
void		store_float(ushort *, int, int);
void 		load_float(ushort *, int, int);
void		load_long(int);

unsigned	parse_csr(char *, char *);
ushort		parse_vec(char *, char *);
ushort		parse_irq(char *, char *);
int		getmfields(char *, char **, int);
char		*setfields(char *);

int		register_handler(char *, char *[], IODev *, void *);
int		register_timer(unsigned, void (*)(void *), void *);
void		monitor(void);
void		request(void);
void		Warning(char *, ...);
volatile void	panic(char *, ...) DEAD_FUNC;
void		verb(char, char *, ...);
void		conf_warning(char *, ...);
volatile void	conf_panic(char *, ...) DEAD_FUNC;
void		*xalloc(size_t, char *);

void		start_timer(void);
int		stop_timer(void);

enum {
	M_Mode		= 03,	/* mode mask			*/
	M_Low		= 010,	/* low byte			*/
	M_High		= 020,	/* high byte			*/
	M_Word		= 040,	/* word				*/
};

# define Max(A,B)	(((A) > (B)) ? (A) : (B))
# define Off(A) ((A) >> 1)
# define IOP(A) (((A) & 017777) >> 1)

/*
 * host byte ordering dependend stuff
 */
# ifdef USE_BIG_ENDIAN
#  define SETLB(W, V)    ((W) = ((W) & 0377) | ((V) << 8 ))
#  define SETHB(W, V)    ((W) = ((W) & 0177400) | ((V) >> 8))
#  define SETW(W,V)      ((W) = ((V)<<8) | (((V)>>8)&0xff))
#  define GETW(V)        ((((V)&0xff)<<8)|(((V)>>8)&0xff))
# else
#  define SETLB(W, V)    ((W) = ((W) & 0177400) | ((V) & 0377))
#  define SETHB(W, V)    ((W) = ((W) & 0377) | ((V) & 0177400))
#  define SETW(W,V)      ((W) = (V))
#  define GETW(V)        (V)
# endif
# define SLB(W, V)    ((W) = ((W) & 0177400) | ((V) & 0377))
# define SHB(W, V)    ((W) = ((W) & 0377) | ((V) & 0177400))

# define ArraySize(a)	(sizeof(a)/sizeof((a)[0]))

/*
 * Must be higher, than anything else ...
 */
# define NMIPRI 	10
# define DMAPRI		11
# define IRQ(D, L)	((D)->reqpri = (L), proc.reqpri = Max(proc.reqpri,(L)))
