/*
 * dir.c
 *
 * Modules supporting operations which require manipulation of the
 * directory information stored in the "dirname" file.
 */

#include <stdio.h>
#include <errno.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include "rstr.h"

extern int	errno;
extern char	*sys_errlist[];

/*
 * GLOBALS (see rstr.c for description)
 */

extern int	(*do_inode)();

extern ushort	dumpmap[MSIZ];

/*
 * command-line argument modes
 */

extern char	*progname,
		*file_list[];

extern int	file_count,	/* number of file arguments on command line */
		verbose,	/* be wordy (-v) */
		ext_count,	/* number of files to extract */
		ext_ena,	/* extraction of files enabled (-x) */
		ena_all,	/* extract or list everything */
		listing,	/* table of contents (-t) */
		interact;	/* interactive mode (-i) */

/*
 * data global to this module; not visible to the outside world
 */

static struct dino	*dirs;		/* this often is a BIG array */
static int		max_ino;	/* maximum inode dumped to tape */

static char	*dirfile = "/tmp/dirXXXXXX",	/* directory entries */
		*namfile = "/tmp/namXXXXXX";	/* list of files to restore */

static FILE	*dp,
		*np;

/*
 * clean_up
 *
 * Called when we'd like to give up the ghost (often
 * is a signal handler).  We don't bother closing file descriptors
 * or checking return values of unlink; the files may not be
 * open in all cases, and exit() will close them if needed.
 */
clean_up(arg)
{
	(void) unlink(dirfile);
	(void) unlink(namfile);

	exit(arg);
}

/*
 * clear
 *
 * Take inode off extraction list.  If inode describes a directory,
 * recursively clear all inodes marked in that directory.
 */
clear(inode)
ushort	inode;
{
	struct dir	*dr, *d, *get_contents();
	struct dinode	di;
	int		i, nitems;

	dirs[inode].dn_type &= ~MARKED;

	if (ino_type(inode) == DIRECT) {
		dr = d = get_contents(inode, &di, &nitems);
		d += 2;
		nitems -= 2;
		for (i = 0; i < nitems; i++, d++)
			if (marked(d->d_inode))
				clear(d->d_inode);
		free((char *)dr);
	}
}

/*
 * complete_dirs
 *
 * Called once when we've seen the first TS_INODE record describing
 * a regular file.  We close dirfile for writing and reopoen it for
 * reading; create namfile and also leave it open for reading.
 *
 * Finish by invoking mechanisms to mark files on extraction list,
 * and invoking routine to create all needed directories.
 */
complete_dirs()
{
	int	i;
	ushort	inode, nami();

	(void) fclose(dp);
	if ((dp = fopen(dirfile, "r")) == NULL) {
		perror(dirfile);
		clean_up(1);
	}

	if (interact) {
		shell();
	} else if ((listing || ext_ena) && file_count) {
		for (i = 0; i < file_count; i++) {
			if ((inode = nami(ROOTDIR, file_list[i], 1)) == 0) {
				fprintf(stderr, "%s: %s not on tape\n",
					progname, file_list[i]);
				continue;
			}
			if (!BIT(inode, dumpmap)) {
				fprintf(stderr, "%s: inode %d not on tape\n",
					progname, inode);
				continue;
			}
			mark(inode);
		}
	}

	ext_count = marked_files();

	if (verbose)
		extraction_info();

	if (verbose)
		printf("Generating pathname%s.\n", listing ?
			"s" : " file");
	make_names(ROOTDIR, "./");

	/*
	 * Make_names will actually do the table of contents if
	 * -t option is in effect, so we can go now.
	 */
	if (listing)
		clean_up(0);

	(void) fclose(np);
	if ((np = fopen(namfile, "r")) == NULL) {
		perror(namfile);
		clean_up(1);
	}

	if (verbose)
		printf("Making directories:\n");
	make_dirs(ROOTDIR, "./");
	if (!ext_count) {
		printf("No files to extract.\n");
		clean_up(0);
	}
}

/*
 * count_ents
 *
 * Given a pointer to a core image of a directory file (d)
 * and the number of entries in that file, return the number
 * of objects in that directory actually on the tape.
 */
count_ents(d, ents)
struct dir	*d;
{
	int	i, count = 0;

	for (i = 0; i < ents; i++, d++)
		if (d->d_inode != 0 && BIT(d->d_inode, dumpmap))
			count++;
	return count;
}

/*
 * dir_compar
 *
 * Routine usable by qsort(3) to sort the entries in an in-core
 * directory file.
 */
dir_compar(d1, d2)
struct dir	*d1, *d2;
{
	return strncmp(d1->d_name, d2->d_name, 14);
}

/*
 * dirname
 *
 * Return a pointer to a (static buffer containing a) character
 * string that's the pathname of the directory associated with
 * inode.  We write the name into dirbuf on the return side
 * of the recursive calls.
 */
char *
dirname(inode)
ushort	inode;
{
	static char	dirbuf[MAXNAME+1];
	struct dir	*d, *dp, *get_contents();
	struct dinode	di;
	int		nitems, i;
	ushort		parent;

	if (ino_type(inode) != DIRECT) {
		fprintf(stderr, "dirname: inode %d not directory\n", inode);
		return NULL;
	}

	/*
	 * This is a bit expensive just to get the parent inode,
	 * but can't see any other way.
	 */
	d = get_contents(inode, &di, &nitems);
	parent = (d+1)->d_inode;
	free((char *)d);

	if (inode == parent) {		/* we've reached ROOTDIR... */
		(void) strcpy(dirbuf, "/");	/*  start writing pathname */
		return dirbuf;
	} else {
		if (!dirname(parent))
			return NULL;
		/*
		 * Find inode's name in parent's directory and
		 * append it to path so far.
		 */
		d = dp = get_contents(parent, &di, &nitems);
		for (i = 0; i < nitems; i++,dp++)
			if (inode == dp->d_inode) {
				(void) strncat(dirbuf, dp->d_name, 14);
				(void) strcat(dirbuf, "/");
				free((char *)d);
				return dirbuf;
			}
		/*
		 * Oops.  Couldn't find our name.
		 */
		free((char *)d);
		fprintf(stderr, "dirname: bogus directory\n");
		return NULL;
	}
}

/*
 * extraction_info
 *
 * Print operator-helpful stuff.  Not much so far.
 */

extraction_info()
{
	printf("%s: %d file%s to %s.\n", progname, ext_count,
		ext_count == 1 ? "" : "s",
		listing ? "list" : "extract");
}

/*
 * get_contents
 *
 * Central routine to get contents of dirfile associated with
 * a particular inode.  This is kind of a kitchen-sink routine:
 * we return a pointer to a newly-malloc'd buffer containing a
 * copy of the directory file, we place a copy of the disk inode
 * into the area referenced by di, and the number of items
 * in the directory in the location referenced by num.
 *
 * Callers of get_contents need to free the buffer returned when
 * done with it.
 *
 * Errors within this routine cause program exit; doesn't seem
 * to be anything to do if our central data base is corrupt.
 */
struct dir *
get_contents(inode, di, num)
ushort		inode;
struct dinode	*di;
int		*num;
{
	int	nitems;
	char	*d, *malloc();

	if (fseek(dp, dirs[inode].dn_seek, 0) == -1) {
		perror("get_contents: fseek");
		clean_up(1);
	}
	if (fread((char *)di, sizeof(struct dinode), 1, dp) != 1) {
		perror("get_contents: fread1");
		clean_up(1);
	}

	nitems = di->di_size / sizeof(struct dir);

	if ((d = malloc((unsigned) di->di_size)) == NULL) {
		perror("get_contents malloc");
		clean_up(1);
	}
	if (fread(d, sizeof(struct dir), nitems, dp) != nitems) {
		perror("get_contents: fread2");
		clean_up(1);
	}
	*num = nitems;
	return (struct dir *) d;
}

/*
 * inam
 *
 * Given an inode, return a pointer to a string (dug out of namfile)
 * that's the pathname of the regular file associated with that
 * inode.
 */
char *
inam(inode)
ushort	inode;
{
	static char	path[MAXNAME+1];
	static int	maxerrs = 10;

	if (ino_type(inode) != REGFILE) {
		fprintf(stderr, "inam: inode %d not a regular file\n",
			inode);
		if (maxerrs--)
			return NULL;
		else
			clean_up(1);
	}
	if (fseek(np, dirs[inode].dn_seek, 0) == -1) {
		perror("fseek in inam");
		if (maxerrs--)
			return NULL;
		else
			clean_up(1);
	}
	if (fgets(path, MAXNAME+1, np) == NULL) {
		perror("fgets in inam");
		if (maxerrs--)
			return NULL;
		else
			clean_up(1);
	}
	path[strlen(path)-1] = '\0';	/* don't want fgets' newline */
	return path;
}

/*
 * init_dir
 *
 * Open our two data files for writing.
 */
init_dir()
{
	char	*mktemp();

	(void) mktemp(dirfile);
	if ((dp = fopen(dirfile, "w")) == NULL) {
		perror(dirfile);
		exit(1);
	}
	(void) mktemp(namfile);
	if ((np = fopen(namfile, "w")) == NULL) {
		perror(namfile);
		(void) fclose(dp);
		(void) unlink(dirfile);
		exit(1);
	}
}

/*
 * init_dump
 *
 * Called after the dumpmap bitmap is read.
 * Check the bitmap to find the maximum inode dumped
 * and malloc space with enough headroom for that inode.
 *
 * Global var max_ino is set to this maximum inode when
 * we leave.
 */
init_dump()
{
	register	i, j;
	char		*s, *calloc();

	for (i = MSIZ-1; i >= 0; i--)
		if (dumpmap[i])
			break;
	if (i < 0) {
		fprintf(stderr, "init_dump: no inodes dumped!\n");
		clean_up(1);
	}
	for (j = MLEN; j; j--)
		if (MBIT(j) & dumpmap[i])
			break;
	if (j == 0) {
		fprintf(stderr, "init_dump: this error \"can't happen\"\n");
		clean_up(1);
	}

	max_ino = i * MLEN + j;

	s = calloc((unsigned) (max_ino + 1), sizeof(struct dino));

	if (s == NULL) {
		perror("init_dump: calloc");
		clean_up(0);
	}
	dirs = (struct dino *) s;
}

/*
 * ino_type
 *
 * Return the type of the passed inode.  Comparisons like
 *	"if (ino_type(i) == REGFILE)" are more convenient.
 */
ino_type(inode)
ushort	inode;
{
	return dirs[inode].dn_type & D_TYPE;
}

/*
 * make_dirs
 *
 * Used to make a breadth-first search of the directory tree
 * and and actually create the directories maked for extraction.
 *
 * We use breadth-first to guarantee that parent directories are
 * made before children are created.
 */
make_dirs(dir, path_so_far)
ushort	dir;
char	*path_so_far;
{
	struct dir	*dr, *d, *get_contents();
	struct dinode	di;
	int		nitems, i;
	char		newpath[MAXNAME];

	dr = d = get_contents(dir, &di, &nitems);

	/*
	 * We're asking for trouble if we start walking
	 * the cycles implied by . and ..
	 */
	d += 2;
	nitems -= 2;

	/*
	 * first pass; make directories at this level
	 */
	for (i = 0; i < nitems; i++, d++) {
		(void) strcpy(newpath, path_so_far);
		if (ino_type(d->d_inode) == DIRECT) {
			(void) strncat(newpath, d->d_name, 14);
			if (marked(d->d_inode))
				/*
				 * prep_dir does the mkdir(2)
				 */
				prep_dir(newpath, d->d_inode);
		}
	}
	/*
	 * second; start cursing
	 */
	d = dr + 2;
	for (i = 0; i < nitems; i++, d++) {
		(void) strcpy(newpath, path_so_far);
		if (ino_type(d->d_inode) == DIRECT && marked(d->d_inode)) {
			(void) strncat(newpath, d->d_name, 14);
			(void) strcat(newpath, "/");
			make_dirs(d->d_inode, newpath);
		}
	}
	free((char *)dr);
}

/*
 * make_names
 *
 * Make a full tree walk and generate the namfile.  When we begin
 * to extract files we will not need to consult dirfile any more;
 * we just make one namfile access in inam to get the pathname
 * associated with the inode.
 *
 * If we're just giving a table of contents, write the generated
 * pathnames to stdout rather than to a file; the program will
 * exit after we return.
 *
 * This is a depth-first walk.
 */
make_names(dir, path_so_far)
ushort	dir;
char	*path_so_far;
{
	struct dir	*dr, *d, *get_contents();
	struct dinode	di;
	int		nitems, i;
	char		newpath[MAXNAME];

	bzero(newpath, MAXNAME);

	dr = d = get_contents(dir, &di, &nitems);

	/*
	 * Again, . and .. will cause nasty problems.
	 */
	d += 2;
	nitems -= 2;

	for (i = 0; i < nitems; i++, d++) {
		(void) strcpy(newpath, path_so_far);
		if (marked(d->d_inode)) {
			if (ino_type(d->d_inode) == DIRECT) {
				(void) strncat(newpath, d->d_name, 14);
				(void) strcat(newpath, "/");
				make_names(d->d_inode, newpath);
			} else {
				(void) strncat(newpath, d->d_name, 14);
				if (listing) {
					if (on_list(newpath))
						printf("%s\n", newpath);
				} else {
					dirs[d->d_inode].dn_type |= REGFILE;
					dirs[d->d_inode].dn_seek = ftell(np);
					fprintf(np, "%s\n", newpath);
				}
			}
		}
	}
	free((char *)dr);
}

/*
 * mark
 *
 * Put inode on extraction list.  If inode describes a directory,
 * recursively mark all inodes in that directory.  We assume that
 * non-directory inodes describe regular files, and set the dn_type
 * field as such.
 */
mark(inode)
ushort	inode;
{
	struct dir	*dr, *d, *get_contents();
	struct dinode	di;
	int		i, nitems;

	dirs[inode].dn_type |= MARKED;

	if (ino_type(inode) == DIRECT) {
		dr = d = get_contents(inode, &di, &nitems);
		d += 2;
		nitems -= 2;
		for (i = 0; i < nitems; i++, d++)
			mark(d->d_inode);
		free((char *)dr);
	} else {
		dirs[inode].dn_type |= REGFILE;
	}
}

/*
 * mark_to_root
 *
 * Starting at inode, mark directories up to the root directory.
 */
mark_to_root(inode)
ushort	inode;
{
	struct dir	*d, *get_contents();
	struct dinode	di;
	int		nitems;
	ushort		parent;

	if (inode == ROOTDIR || marked(inode))
		return;

	dirs[inode].dn_type |= MARKED;

	d = get_contents(inode, &di, &nitems);
	parent = (d+1)->d_inode;
	free((char *)d);

	mark_to_root(parent);
}

/*
 * marked
 *
 * Return non-zero if the file associated with inode
 * is marked (on extraction list).  BLOCKDIR (inode 1)
 * is a special case we look for (is never dumped, but
 * is in bitmap).
 */
marked(inode)
ushort	inode;
{
	if (ena_all && inode != BLOCKDIR)
		return BIT(inode, dumpmap);
	else
		return dirs[inode].dn_type & MARKED;
}

/*
 * marked_files
 *
 * Return the number files in our inode
 * array that are marked for extraction.
 */
marked_files()
{
	ushort	i, count = 0;

	if (ena_all)
		return MAXINO;

	for (i = ROOTDIR + 1; i <= max_ino; i++) {
		if (marked(i)) {
			if (ino_type(i) == REGFILE)
				count++;
		}
	}

	return count;
}

/*
 * nami
 *
 * Return the inode of pathname s, relative the the inode
 * (directory) dir.
 *
 * Vital side-effect: if mark_flag is set we mark the directories
 * we're descending on our way to s -- this makes sure that
 * directories are marked for creation when we extract files.
 * (Mark_flag is set when the invocation of nami will be immediately
 * be followed by a call to mark().)
 */
ushort
nami(dir, s, mark_flag)
ushort	dir;
char	*s;
{
	struct dir	*d, *dr, *get_contents();
	struct dinode	di;
	int		i,
			nitems;
	char		name[15], *n, *index();

	while (*s == ' ')
		s++;
	
	if (*s == '/') {
		dir = ROOTDIR;
		s++;
	} else if (!strncmp(s, "./", 2)) {
		dir = ROOTDIR;
		s += 2;
	}

	if (mark_flag)
		mark_to_root(dir);

	while (*s) {
		n = name;
		while (*s && *s != '/')
			*n++ = *s++;
		*n = '\0';
		if (*s)
			s++;
		d = dr = get_contents(dir, &di, &nitems);
		for (i = 0; i < nitems; i++, d++) {
			if (!strncmp(name, d->d_name, 14)) {
				dir = d->d_inode;
				break;
			}
		}
		free((char *)dr);
		if (i == nitems) {
			fprintf(stderr, "nami: %s not found\n", name);
			return 0;
		}
		if (*s && ino_type(dir) != DIRECT) {
			fprintf(stderr, "nami: %s not a directory\n", name);
			return 0;
		}
		if (mark_flag && dir != ROOTDIR)
			dirs[dir].dn_type |= MARKED;
	}
	return dir;
}

/*
 * on_list
 *
 * Check the particular pathname "name" against the names
 * in the global file_list.  This is used to generate the
 * extraction list for the -x option (or lists for -t).
 *
 * Eg., we're just checking to find pathnames that match
 * the leading pathnames in file_list.
 */
on_list(name)
char	*name;
{
	int	i;

	if (ena_all)
		return 1;
	if (*name == '.')
		name++;
	
	for (i = 0; i < file_count; i++)
		if (!strncmp(name, file_list[i], strlen(file_list[i])))
			return 1;

	return 0;
}

/*
 * prep_dir
 *
 * Make the directory "name".  Use inode to dig original
 * ownership and permissions out of dirfile, and call rest_mods
 * to reset modes.
 */
prep_dir(name, inode)
char	*name;
ushort	inode;
{
	struct dinode	di;
	struct dir	*d;
	int		dummy;

	if (verbose)
		printf("x %s\n", name);
	if (mkdir(name, 0700) == -1) {
		if (errno == EEXIST) {
			fprintf(stderr, "%s warning: %s already exists.\n",
				progname, name);
			return;
		} else {
			perror("prep_dir: mkdir");
			clean_up(1);
		}
	}
	d = get_contents(inode, &di, &dummy);
	rest_mods(name, &di);
	free((char *)d);
}

/*
 * read_dir
 *
 * Read in the contents of a directory file and save 
 * it to the dirfile.  The directory file contains, in
 * order, a copy of the dinode structure and then directory
 * contents usable by us.  The seek address of this info
 * is placed in the proper dirs item.
 *
 * We process the directory file before writing it;
 * don't write directory entries for files not identified
 * (in the bitmaps) as written to the archive, and adjust the
 * copy of the disk inode before writing it.
 */
read_dir(fd, hp)
struct spcl	*hp;
{
	extern int	begin_extract(),
			read_regfile();
	struct dinode	*di = &hp->c_dinode;
	struct dir	*d;
	char		*alloc_mem();
	int		i, d_ents, d_valid,
			copy_buf(), flush_buf();


	if ((di->di_mode & S_IFMT) != S_IFDIR) {
		/*
		 * Here's where we switch over from reading
		 * directory information on the tape to regular
		 * file info.
		 *
		 * Complete_dirs consolidates directory info we've
		 * seen into "dirfile", and writes full pathnames of
		 * all files seen into "namfile"; it may not return
		 * if all we're doing is getting a listing (-t).
		 */
		complete_dirs();

		/*
		 * read_dir will not be called again
		 */
		do_inode = read_regfile;

		return begin_extract(fd, hp);
	}

	d_ents = di->di_size/sizeof(struct dir);
	/*
	 * (the buffer's declared in getfile.c)
	 */
	d = (struct dir *) alloc_mem((unsigned) di->di_size);

	if (getfile(fd, hp, copy_buf, flush_buf) == -1) {
		fprintf(stderr, "read_dir: getfile problem\n");
		return -1;
	}

	d_valid = count_ents(d, d_ents);

	/*
	 * Adjust the size entry in the copy of inode; we won't
	 * be writing directory entries if the file described isn't
	 * on tape.
	 */
	di->di_size = (long) (d_valid * sizeof(struct dir));

	dirs[hp->c_inumber].dn_type |= DIRECT;
	dirs[hp->c_inumber].dn_seek = ftell(dp);

	if (fwrite((char *)di, sizeof(struct dinode), 1, dp) != 1) {
		perror("read_dir: fwrite1:");
		return -1;
	}

	for (i = 0; i < d_ents; i++, d++) {

		if (d->d_inode == 0 || !BIT(d->d_inode, dumpmap))
			continue;

		if (fwrite((char *)d, sizeof(struct dir), 1, dp) != 1) {
			perror("read_dir: fwrite2:");
			return -1;
		}
	}

	return 0;
}
