/* Copyright (c)1994-2000 Begemot Computer Associates. All rights reserved.
 * See the file COPYRIGHT for details of redistribution and use. */

# include "proc.h"
# include <sys/ioctl.h>

RCSID("$Id: mon.c,v 1.7 2000/03/04 08:03:30 hbb Exp $")

# ifndef TCSASOFT
#  define TCSASOFT 0
# endif

/*
 * magic addresses for register access
 */
enum {
	MAXADDR	= 017777777,
	R0, R1, R2, R3, R4, R5,
	R0s, R1s, R2s, R3s, R4s, R5s,
	KSP, USP, SSP, PC,
	MAXREG
};
	
# define LINES	24

static void	raw(void);
static void	noraw(void);
static int	gch(void);
static void	crlf(void);
static void	save_regs(void);
static void	restore_regs(void);
static void	out_registers(void);
static void	out_mmg(void);
static void	out_word(unsigned);
static void	out_byte(unsigned);
static void	out_space(void);
static void	open_loc(void);
static int	mon_fetch(u_int, int);
static int	mon_store(u_int, int, ushort);
static int	get8(void);
static int	getreg(void);
static void	doinfo(void);
static void	find(int);
static int	pins(FILE *, unsigned, int);
static void	preqs(void);
static void	device_command(void);
static void	escape();
static void	dump_ins(void);
static void	open_monfile();
static int	read_monfile();
static int	getstring(char *, int);
static void	save_tty();



static int	loc;	/* current location, -1 means none */
static char	c;
static int	monstate;
static FILE	*monfile;

static int	f_utrace = 0;
static int	first_call = 1;

void
monitor(void)
{
	int	t = 0;		/* timer flag */

	if(first_call) {
		save_tty();
		first_call = 0;
	}
	if(!(proc.maint & 1))
		pwrfail_exit(&proc);
	if(nomon) {
		/*
		 * we are running in the background so we have no
		 * monitor. monstate counts how often we were in the
		 * monitor.
		 */
		if(monstate == 0) {
			/*
			 * simulate 173000g
			 */
			Reset();
			proc.reg[7] = 0173000;
			monstate = 1;
		} else if(monstate == 1) {
			/*
			 * back in monitor - exit with powerfail
			 */
			proc_pwrfail(1);
			monstate = 2;
		} else
			exit(0);
		proc.halt = 0;
		return;
	}

	Flush();

	t = stop_timer();
	proc.halt = 0;
	save_regs();

	pins(stdout, proc.reg[7], proc.curmode);
	preqs();

	raw();
	loc = -1;
again:
	putchar('@');
	c = gch();
sw:
	switch(c) {

	case '$':
		if((loc = getreg()) < 0)
			goto err;
		goto sw;

	case '0': case '1': case '2': case '3':
	case '4': case '5': case '6': case '7':
		loc = get8();
		goto sw;

	case 'r':
		crlf();
		out_registers();
		goto again;

# ifdef FPP
	case 'R':
		crlf();
		noraw();
		mon_fpp();
		raw();
		goto again;
# endif

	case 'm':
		crlf();
		out_mmg();
		goto again;

	case 'x':
		proc_pwrfail(1);
		break;

	case 'X':
		crlf();
		noraw();
		exit(0);

	case 'g':
		Reset();
		proc.reg[7] = (loc >= 0 && loc <= MAXADDR) ? (loc & ~1) : 0;
	case 'p':
		break;
	case 's':
		proc.halt = 1;
		break;

	case '/': case '\\':
		if(loc < 0)
			goto err;
		open_loc();
		goto again;

	case 'b':
		if(loc >= 0200000 || loc < 0)
			goto err;
		crlf();
		proc.bp = loc;
		goto again;

	case 'i':
		doinfo();
		goto again;

	case 'I':
		device_command();
		goto again;

	case 'z':
		fflush(stdout);
		noraw();
		kill(getpid(), SIGTSTP);
		raw();
		goto again;

	case 'Z':
		nomon = 1;
		monstate = 1;
		break;

	case 'f':
		if(loc > MAXADDR || loc < 0)
			goto err;
		crlf();
		find(loc);
		goto again;


	case 't':
	case 'T':
		switch(gch()) {
		case 't':
			if(c == 'T')
				f_utrace = !f_utrace;
			printf("utrace: %s", f_utrace ? "on" : "off");
			break;
# ifdef FPP
		case 'f':
			if(c == 'T')
				proc.fpd = !proc.fpd;
			printf("fpp: %sabled", proc.fpd ? "dis" : "en");
			break;
# endif

		case 'q':
			if(c == 'T')
				proc.bevent = !proc.bevent;
			printf("timer %s", proc.bevent ? "on" :  "off");
			break;

		case 'i':
			if(c == 'T')
				ill_trace = !ill_trace;
			printf("ill-trace %s", ill_trace ? "on" : "off");
			break;

		default:
			putchar('?');
		}
		crlf();
		goto again;
		
	case '!':
		crlf();
		noraw();
		escape();
		printf("!\n");
		raw();
		goto again;

	case 'd':
		if(loc > MAXADDR || loc < 0)
			goto err;
		dump_ins();
		goto again;
		
	case '?':
		crlf();
		noraw();
		type_help_file(_PATH_MON_HELP);
		raw();
		goto again;

	case 'Y':
		crlf();
		noraw();
		type_help_file(_PATH_COPYING);
		raw();
		goto again;

	case '<':
		open_monfile();
		goto again;

	default:
	err:
		loc = -1;
		putchar('?');
		crlf();
		goto again;
	}

	crlf();
	restore_regs();
	noraw();
	if(t)
		start_timer();
}

/*
 * open locations and let the user enter new contents
 */
static void
open_loc(void)
{
	int	val;	/* new value (-1 means none) */
	int	bmode;	/* byte mode if true */
	int	v;	/* current value */

	bmode = (c == '\\');

stage1:
	/*
	 * print contents and wait for input
	 */
	if(loc <= MAXADDR && (loc & 1))
		bmode = 1;
	if(loc > MAXADDR)
		bmode = 0;
	if((v = mon_fetch(loc, bmode)) == -1) {
		putchar('?');
		crlf();
		return;
	}
	if(bmode)
		out_byte(v);
	else
		out_word(v);
	out_space();

	val = -1;

/* stage2: */
	c = gch();	

/* stage3: */
	/*
	 * decode user response
	 */
	if(c >= '0' && c <= '7')
		val = get8();

	switch(c) {

	case '/':	
		if(val >= 0) loc = val; 
		bmode = 0; 
		break;

	case '\\':	
		if(val >= 0) loc = val; 
		if(loc > 0)
			bmode = 1;
		else
			bmode = 0;
		break;

	case '\r':	
		if(val >= 0)
			mon_store(loc, bmode, val);
		crlf();
		return;

	case '\n':
		if(val >= 0)
			mon_store(loc, bmode, val);
		if(loc >= MAXADDR) {
			/*
			 * wrap back to R0
			 */
			if(++loc >= MAXREG)
				loc = R0;
		} else {
			/*
			 * wrap to phys. 0
			 */
			if((loc += bmode ? 1 : 2) >= MAXADDR)
				loc = 0;
		}
		break;

	case '^':
		if(val >= 0)
			mon_store(loc, bmode, val);
		if(loc >= MAXADDR) {
			/*
			 * wrap to last register (PC)
			 */
			if(--loc == MAXADDR)
				loc = MAXREG-1;
		} else {
			/*
			 * wrap to highest address
			 */
			if((int)(loc -= bmode ? 1 : 2) < 0)
				loc = bmode ? MAXADDR : (MAXADDR & ~1);
		}
		break;

	default:
		putchar('?');
		return;
	}

	crlf();

	/*
	 * print new location
	 */
	if(loc >= R0 && loc <= R5)
		printf("R%d", loc - R0);
	else if(loc >= R0s && loc <= R5s)
		printf("R%d'", loc - R0s);
	else if(loc >= KSP && loc <= SSP)
		printf("%cSP", "KUS"[loc - KSP]);
	else if(loc == PC)
		printf("PC");
	else
		printf("%08o", loc);

	putchar("/\\"[bmode != 0]);

	goto stage1;
}

/*
 * fetch value from register or memory.
 * avoid traps.
 * return -1 on trap.
 */
static int
mon_fetch(u_int a, int bmode)
{
	ushort	v;

	if(a >= R0 && a <= R5)
		return proc.ureg[a - R0];
	if(a >= R0s && a <= R5s)
		return proc.kreg[a - R0s];
	if(a == KSP)
		return proc.sp[0];
	if(a == SSP)
		return proc.sp[1];
	if(a == USP)
		return proc.sp[3];
	if(a == PC)
		return proc.reg[7];

	if(a >= 017760000 && a <= 017777776) {
		IODev *dev = proc.iopage[IOP(a & ~1)];
		if(!dev || !dev->ioops->fetch)
			return -1;
		v = iofetch(a & ~1);
	} else if(a < proc.physmem) {
		v = *proc.memops->addrphys(proc.mmg, a & ~1);
		v = GETW(v);
	} else
		return -1;

	if(!bmode)
		return v;
	else if(a & 1)
		return (v >> 8) & 0377;
	else
		return v & 0377;
}

/*
 * store value into register or memory.
 * avoid traps
 */
static int
mon_store(u_int a, int bmode, ushort v)
{
	if(a >= R0 && a <= R5)
		proc.ureg[a - R0] = v;
	else if(a >= R0s && a <= R5s)
		proc.kreg[a - R0s] = v;
	else if(a == KSP)
		proc.sp[0] = v;
	else if(a == SSP)
		proc.sp[1] = v;
	else if(a == USP)
		proc.sp[3] = v;
	else if(a == PC)
		proc.reg[7] = v;
	else {
		if(a >= 017760000 && a <= 017777776) {
			IODev *dev = proc.iopage[IOP(a & ~1)];
			if(!dev || !dev->ioops->store)
				return -1;
			if(bmode) {
				if(a & 1)
					iostore(a & 1, M_High, v);
				else
					iostore(a & 1, M_Low, v);
			} else
				iostore(a & ~1, M_Word, v);
		} else if(a < proc.physmem) {
			ushort *ov = proc.memops->addrphys(proc.mmg, a & ~1);
			if(bmode) {
				if(a & 1)
					SETHB(*ov, v);
				else
					SETLB(*ov, v);
			} else
				SETW(*ov, v);
		} else {
			return -1;
		}
	}
	return 0;
}


/*
 * switch input to file.
 */
static void
open_monfile()
{
	char	buf[80];

	printf(" ");
	if(!getstring(buf, sizeof(buf)))
		return;
	if((monfile = fopen(buf, "r")) == NULL)
		fprintf(stderr, "p11: %s: %s\n", buf, strerror(errno));
}

/*
 * print processor registers
 */
static void
out_registers(void)
{
	int	i;

	for(i = 0; i < 6; i++)
		printf("R%d =%06ho  ", i, proc.ureg[i]);
	crlf();
	for(i = 0; i < 6; i++)
		printf("R%d'=%06ho  ", i, proc.kreg[i]);
	crlf();
	printf("PC =%06ho  ", proc.reg[7]);
	printf("USP=%06ho  ", proc.sp[3]);
	printf("KSP=%06ho  ", proc.sp[0]);
	printf("SSP=%06ho  ", proc.sp[1]);
	printf("PSW=%06ho  ", iofetch(017777776));
	if(proc.n) putchar('N');
	if(proc.z) putchar('Z');
	if(proc.v) putchar('V');
	if(proc.c) putchar('C');
	crlf();
}

/*
 * print memory management registers
 */
static void
out_mmg(void)
{
	int i;

	printf("MMR: %06ho, %06ho, %06ho, %06ho", 
		iofetch(017777572),
		iofetch(017777574),
		iofetch(017777576),
		iofetch(017772516));
	crlf();

	printf("KERNEL-I     KERNEL-D      ");
	printf("USER-I       USER-D        ");
	printf("SUPERV-I     SUPERV-D"); crlf();
	for(i = 0; i < 8; i++) {
		printf("%05ho %06ho %05ho %06ho  "
		       "%05ho %06ho %05ho %06ho  "
		       "%05ho %06ho %05ho %06ho",
			iofetch(017772300 + 2*i),
			iofetch(017772340 + 2*i),
			iofetch(017772320 + 2*i),
			iofetch(017772360 + 2*i),
			iofetch(017777600 + 2*i),
			iofetch(017777640 + 2*i),
			iofetch(017777620 + 2*i),
			iofetch(017777660 + 2*i),
			iofetch(017772200 + 2*i),
			iofetch(017772240 + 2*i),
			iofetch(017772220 + 2*i),
			iofetch(017772260 + 2*i));
		crlf();
	}
}

static void
out_word(unsigned v)
{
	printf("%06o", v);
}

static void
out_byte(unsigned v)
{
	printf("%03o", v & 0377);
}

static void
out_space(void)
{
	putchar(' ');
}

/*
 * save current registers
 */
static void
save_regs(void)
{
	if(proc.regs)
		memcpy(proc.kreg, proc.reg, sizeof(proc.kreg));
	else
		memcpy(proc.ureg, proc.reg, sizeof(proc.ureg));
	proc.sp[proc.curmode] = proc.reg[6];
}

static void
restore_regs(void)
{
	if(proc.regs)
		memcpy(proc.reg, proc.kreg, sizeof(proc.kreg));
	else
		memcpy(proc.reg, proc.ureg, sizeof(proc.ureg));
	proc.reg[6] = proc.sp[proc.curmode];
}


/*
 * I/O 
 */
static struct termios tty_saved;

static void
save_tty()
{
	tcgetattr(0, &tty_saved);
}

static void
raw(void)
{
	struct termios t;

	t = tty_saved;
	cfmakeraw(&t);
	t.c_iflag |= IXON;
	tcsetattr(0, TCSANOW | TCSASOFT, &t);
}

static void
noraw(void)
{
	tcsetattr(0, TCSANOW | TCSASOFT, &tty_saved);
}

static int
gch(void)
{
	char 	c;
	int	cc;
	int	ret;

	fflush(stdout);

	if(monfile == NULL || (cc = read_monfile()) == EOF) {
		/*
		 * read from standard input
		 */
		if((ret = read(0, &c, 1)) != 1) {
			noraw();
			panic("tty read error: %s\n", strerror(errno));
		}
		cc = c;
	}

	if(cc == 3) {
		printf("^C"); crlf();
		return -1;
	}
	cc &= 0177;
	if(cc == '\r')
		putchar(cc);
	else if(cc < ' ')
		printf("^%c", cc + 0100);
	else if(cc == 0177)
		printf("^?");
	else
		putchar(cc);
	return cc;
}

/*
 * read next token from input file. If EOF close file and return EOF.
 */
static int
read_monfile()
{
	int	backs = 0;
	int	c;

	for(;;) {
		if((c = getc(monfile)) == EOF) {
			if(ferror(monfile))
				panic("monitor input: %s", strerror(errno));
			fclose(monfile);
			monfile = NULL;
			return EOF;
		}
		if(c == '\\' && !backs)
			backs = 1;
		else if(c != '\n')
			break;
	}
	if(!backs)
		return c;

	switch(c) {

	case 'r':
		return '\r';

	case 'n':
		return '\n';

	case 'e':
		return '[' - 0100;

	case '\\':
		return '\\';
	}
	ungetc(c, monfile);
	return '\\';
}


static void
crlf(void)
{
	putchar('\r');
	putchar('\n');
}

/*
 * read an octal number. The first character is already in `c'.
 * leave terminating one in c.
 */
static int
get8(void)
{
	unsigned v = 0;

	do {
		v = 8 * v + (c - '0');
	} while((c = gch()) >= '0' && c <= '7');
	return v;
}

static int
getstring(char *buf, int len)
{
	char	*p = buf, *end = buf + len;

	while((c = gch()) != '\n' && c != '\r') {
		if(c == '\b' || c == 0177) {
			if(p > buf) {
				p--;
				printf("\b\b\b   \b\b\b");
			} else
				printf("\b\b  \b\b");
		} else if(p < end - 1) {
			*p++ = c;
		}
	}
	*p = 0;
	crlf();
	return p - buf;
}

/*
 * read register name
 * 1st character is already in c.
 * leave next character in c.
 * returns -1 on failure.
 * "$0" - "$5" means 1st register set
 * "$0'" - "$5'" means 2nd register set
 * "$6u", "$6s", "$6k", "$6" means user, supervisor, kernal stack pointer
 * "$7" means PC
 */
static int
getreg()
{
	int a;

	switch(c = gch()) {
	
	case '0': 
	case '1': 
	case '2': 
	case '3': 
	case '4': 
	case '5': 
		a = R0 + c - '0';
		if((c = gch()) == '\'')
			a += R0s - R0;
		else
			return a;
		break;

	case '6': 
		if((c = gch()) == 'u')
			a = USP;
		else if(c == 's')
			a = SSP;
		else if(c == 'k')
			a = KSP;
		else
			return KSP;
		break;

	case '7': 
		a = PC;
		break;

	default:
		return -1;
	}
	c = gch();
	return a;
}

static IODev *
get_device(char ch)
{
	char buf[10];
	char *p = buf, c;
	IODev *dev, *d;
	int len, cnt;

	putchar(' ');
	while((c = gch()) != '\r' && c != '\n' && c != ' ') {
		if(p == &buf[sizeof(buf)-1]) {
			putchar('?');
			crlf();
			return 0;
		}
		if(c == '\b' || c == '\177') {
			if(p > buf) {
				p--;
				printf("\b\b\b   \b\b\b");
			} else
				printf("\b\b  \b\b");
		} else if(c == '\t') {
			dev = 0;
			*p = 0;
			cnt = 0;
			len = strlen(buf);
			for(d = proc.devices; d; d = d->next) {
				if(!strncmp(buf, d->name, len)) {
					if(dev) {
						if(cnt == 1)
							crlf();
						printf("%s", dev->name);
						crlf();
					}
					dev = d;
					cnt++;
				}
			}
			if(dev) {
				printf("%s", dev->name);
				crlf();
			}
			if(cnt == 1)
				return dev;
			printf("@%c %s", ch, buf);
		} else
			*p++ = c;
	}
	*p = 0;
	crlf();

	for(dev = proc.devices; dev; dev = dev->next) {
		if(buf[0] == 0) {
			printf("%s", dev->name);
			crlf();
		} else if(!strcmp(dev->name, buf))
			return dev;
	}

	return 0;
}


static void
doinfo(void)
{
	IODev *dev = get_device('i');

	if(dev) {
		noraw();
		(*dev->ioops->info)(dev);
		raw();
	}
}

static void
device_command(void)
{
	IODev 	*dev = get_device('i');
	char	buf[70];
	char	*argv[20];
	int	argc;

	if(!dev)
		return;
	for(;;) {
		printf("%s >", dev->name);
		if(!getstring(buf, 70))
			return;
		if((argc = getmfields(buf, argv, ArraySize(argv))) == 0)
			return;
		noraw();
		dev_command(dev, argc, argv);
		raw();
	}
}

static void
find(int iv)
{
	unsigned a;
	ushort v = iv, vv;

	for(a = 0; a < PHYSMEM; a += 2) {
		vv = *proc.memops->addrphys(proc.mmg, a);
		vv = GETW(vv);
		if(vv == v) 
			printf("%08o/%06o\r\n", a, v);
	}
}

/*
 * Type a help file onto stdout.
 *
 * If 'libdir' is set the help file is searched in libdir else
 * in the current working directory.
 *
 * If stdout is not a tty (as indicated by isatty) the file is simply
 * copied to stdout. If stdout is a tty and enviroment variable PAGER
 * exists, the pager is called with the appropriate full filename. If PAGER
 * is not set, the window size is obtained trough an ioctl (or a default
 * of 24 used) and the file is typed one page at a time. The user may
 * quit by typing 'q' or ESC.
 */
void
type_help_file(char *help)
{
	struct winsize ws;
	int	lines;
	int	lno = 0;
	char	buf[200];
	FILE	*hf;
	char	*fname;
	char	*command;
	char	*pager;


	/*
	 * construct full filename. Don't forget to free later
 	 */
	fname = xalloc(strlen(help) + (libdir ? (strlen(libdir)+1) : 0) + 1);
	if(libdir) {
		strcpy(fname, libdir);
		strcat(fname, "/");
		strcat(fname, help);
	} else
		strcpy(fname, help);

	/*
	 * now look what we must do
	 */
	if(isatty(fileno(stdout))) {
		if((pager = getenv("PAGER")) != 0) {
			command = xalloc(strlen(pager) + strlen(fname) + 2);
			strcpy(command, pager);
			strcat(command, " ");
			strcat(command, fname);
			system(command);
			free(command);
			free(fname);
			return;
		} 
		if(ioctl(fileno(stdout), TIOCGWINSZ, &ws) || (lines = ws.ws_row) <= 0)
			lines = LINES;
	} else
		lines = 0;


	if((hf = fopen(fname, "r")) == NULL) {
		printf("p11: %s: %s\n", fname, strerror(errno));
		free(fname);
		return;
	}

	while(fgets(buf, sizeof(buf), hf) != NULL) {
		if(lines != 0 && lno % (lines - 1) == (lines - 2)) {
			printf("continue?");
			raw(); c = gch(); noraw();
			printf("\n");
			if(c == 'q' || c == '\033')
				break;
		}
		fputs(buf, stdout);
		lno++;
	}
	if(ferror(hf))
		printf("p11: read error: %s\n", strerror(errno));

	free(fname);
	fclose(hf);
}


# define B		0100000
# define TNONE		1
# define TONE		2
# define TREG		3
# define TSPL		4
# define TCC		5
# define TBRANCH	6
# define TRONE		7
# define TDBL		8
# define TSOB		9
# define TEMT		10
# define TFONE		11
# define TFDBL		12


static struct ins {
	int	type;
	int	val;
	int	mask;
	char	*ins;
} instab[] = {
	{ TNONE,	0000000, 0000000, "halt" },
	{ TNONE,	0000001, 0000000, "wait" },
	{ TNONE,	0000002, 0000000, "rti" },
	{ TNONE,	0000003, 0000000, "bpt" },
	{ TNONE,	0000004, 0000000, "iot" },
	{ TNONE,	0000005, 0000000, "reset" },
	{ TNONE,	0000006, 0000000, "rtt" },
	{ TNONE,	0000007, 0000000, "mfpt" },
	{ TONE,		0000100, 0000077, "jmp" },
	{ TREG,		0000200, 0000007, "rts" },
	{ TSPL,		0000230, 0000007, "spl" },
	{ TNONE,	0000240, 0000000, "nop" },
	{ TCC,		0000240, 0000017, "cflg" },
	{ TCC,		0000260, 0000017, "sflg" },
	{ TONE,		0000300, 0000077, "swab" },
	{ TBRANCH,	0000400, 0000377, "br" },
	{ TBRANCH,	0001000, 0000377, "bne" },
	{ TBRANCH,	0001400, 0000377, "beq" },
	{ TBRANCH,	0002000, 0000377, "bge" },
	{ TBRANCH,	0002400, 0000377, "blt" },
	{ TBRANCH,	0003000, 0000377, "bgt" },
	{ TBRANCH,	0003400, 0000377, "ble" },
	{ TBRANCH,	0100000, 0000377, "bpl" },
	{ TBRANCH,	0100400, 0000377, "bmi" },
	{ TBRANCH,	0101000, 0000377, "bhi" },
	{ TBRANCH,	0101400, 0000377, "blos" },
	{ TBRANCH,	0102000, 0000377, "bvc" },
	{ TBRANCH,	0102400, 0000377, "bvs" },
	{ TBRANCH,	0103000, 0000377, "bcc" },
	{ TBRANCH,	0103400, 0000377, "bcs" },
	{ TRONE,	0004000, 0000777, "jsr" },
	{ TONE | B,	0005000, 0100077, "clr" },
	{ TONE | B,	0005100, 0100077, "com" },
	{ TONE | B,	0005200, 0100077, "inc" },
	{ TONE | B,	0005300, 0100077, "dec" },
	{ TONE | B,	0005400, 0100077, "neg" },
	{ TONE | B,	0005500, 0100077, "adc" },
	{ TONE | B,	0005600, 0100077, "sbc" },
	{ TONE | B,	0005700, 0100077, "tst" },
	{ TONE | B,	0006000, 0100077, "ror" },
	{ TONE | B,	0006100, 0100077, "rol" },
	{ TONE | B,	0006200, 0100077, "asr" },
	{ TONE | B,	0006300, 0100077, "asl" },
	{ TONE,		0006500, 0000077, "mfpi" },
	{ TONE,		0006600, 0000077, "mtpi" },
	{ TONE,		0006700, 0000077, "sxt" },
	{ TDBL | B,	0010000, 0107777, "mov" },
	{ TDBL | B,	0020000, 0107777, "cmp" },
	{ TDBL | B,	0030000, 0107777, "bit" },
	{ TDBL | B,	0040000, 0107777, "bic" },
	{ TDBL | B,	0050000, 0107777, "bis" },
	{ TDBL,		0060000, 0007777, "add" },
	{ TRONE,	0070000, 0000777, "mul" },
	{ TRONE,	0071000, 0000777, "div" },
	{ TRONE,	0072000, 0000777, "ash" },
	{ TRONE,	0073000, 0000777, "ashc" },
	{ TRONE,	0074000, 0000777, "xor" },
	{ TSOB,		0077000, 0000777, "sob" },
	{ TEMT,		0104000, 0000377, "emt" },
	{ TEMT,		0104400, 0000377, "trap" },
	{ TONE,		0106400, 0000077, "mtps" },
	{ TONE,		0106500, 0000077, "mfpd" },
	{ TONE,		0106600, 0000077, "mtpd" },
	{ TONE,		0106700, 0000077, "mfps" },
	{ TDBL,		0160000, 0007777, "sub" },

	{ TNONE,	0170000, 0000000, "cfcc" },
	{ TNONE,	0170001, 0000000, "setf" },
	{ TNONE,	0170002, 0000000, "seti" },
	{ TNONE,	0170011, 0000000, "setd" },
	{ TNONE,	0170012, 0000000, "setl" },
	{ TONE,		0170100, 0000077, "ldfps" },
	{ TONE,		0170200, 0000077, "stfps" },
	{ TONE,		0170300, 0000077, "stst" },
	{ TFONE,	0170400, 0000077, "clrf" },
	{ TFONE,	0170500, 0000077, "tstf" },
	{ TFONE,	0170600, 0000077, "absf" },
	{ TFONE,	0170700, 0000077, "negf" },
	{ TFDBL,	0171000, 0000377, "mulf" },
	{ TFDBL,	0171400, 0000377, "modf" },
	{ TFDBL,	0172000, 0000377, "addf" },
	{ TFDBL,	0172400, 0000377, "ldf" },
	{ TFDBL,	0173000, 0000377, "subf" },
	{ TFDBL,	0173400, 0000377, "cmpf" },
	{ TFDBL,	0174000, 0000377, "std" },
	{ TFDBL,	0174400, 0000377, "divf" },
	{ TFDBL,	0175000, 0000377, "stexp" },
	{ TFDBL,	0175400, 0000377, "stci" },
	{ TFDBL,	0176000, 0000377, "stcf" },
	{ TFDBL,	0176400, 0000377, "ldexp" },
	{ TFDBL,	0177000, 0000377, "ldci" },
	{ TFDBL,	0177400, 0000377, "ldcf" },

	{ 0, 0, 0, 0 },
};

/*
 * dump an instruction
 */
static ushort
pins_fetch(unsigned loc, int mode, unsigned *pa, ushort *pv)
{
	if(mode >= 0) {
		ushort par = proc.par[mode][0][(loc >> 13) & 7];
		ushort off = loc & 017777;

		if(!(proc.mmr0 & 1)) {
			*pa = loc & 0177776;
			if(loc >= 0160000)
				*pa += 017600000;
		} else if(!(proc.mmr3 & 020)) {
			*pa = (((int)par << 6) + off) & 0777776;
			if(*pa >= 0760000)
				*pa += 017000000;
		} else {
			*pa = (((int)par << 6) + off) & 017777776;
		}
	} else {
		*pa = loc;
	}
	
	if(*pa >= 017760000 && *pa <= 017777776) {
		IODev *dev = proc.iopage[IOP(*pa)];
		if(!dev || !dev->ioops->fetch)
			return 1;
		*pv = iofetch(*pa);
	} else if(*pa >= proc.physmem)
		return 1;
	else {
		*pv = *proc.memops->addrphys(proc.mmg, *pa);
		*pv = GETW(*pv);
	}
	return 0;
}

static void pins_addr(FILE *, unsigned *, int, int);

static int
pins(FILE *fp, unsigned loc, int mode)
{
	unsigned a;
	ushort v;
	int i;
	unsigned old_loc = loc;

	i = pins_fetch(loc, mode, &a, &v);
	fprintf(fp, "%06ho %c -> %08o", loc, "PKS?U"[mode+1], a);
	loc += 2;
	if(i) {
		crlf();
		return -1;;
	}
	fprintf(fp, " = %06ho", v);

	for(i = 0; instab[i].ins; i++)
		if((v & ~instab[i].mask) == instab[i].val) {
			fprintf(fp, ":\t%s", instab[i].ins);
			if((instab[i].type & B) && (v & 0100000))
				putc('b', fp);
			putc('\t', fp);

			switch(instab[i].type & ~B) {

			case TNONE:
				break;

			case TREG:
				fprintf(fp, "r%d", v & 7);
				break;

			case TSPL:
				fprintf(fp, "%d", v & 7);
				break;

			case TCC:
				if(!(v & 017))
					putc('0', fp);
				else {
					int j = 4;
					while(--j >= 0)
						if(v & (1 << j))
							putc("cvzn"[j], fp);
				}
				break;

			case TBRANCH:
				if((v & 0377) >= 0200)
					fprintf(fp, "%o", loc - ((-v & 0377) << 1));
				else 
					fprintf(fp, "%o", loc + ((v & 0377) << 1));
				break;

			case TSOB:
				fprintf(fp, "%o", loc - ((v & 077) << 1));
				break;

			case TEMT:
				fprintf(fp, "%o", v & 0377);
				break;

			case TONE:
				pins_addr(fp, &loc, mode, (v & 077));
				break;

			case TRONE:
				fprintf(fp, "r%d,", (v >> 6) & 7);
				pins_addr(fp, &loc, mode, (v & 077));
				break;
				
			case TDBL:
				pins_addr(fp, &loc, mode, ((v >> 6) & 077));
				putc(',', fp);
				pins_addr(fp, &loc, mode, (v & 077));
				break;

			case TFDBL:
				fprintf(fp, "ac%c,", "0123"[(v >> 6) & 3]);

			case TFONE:
				if((v & 070) == 0)
					fprintf(fp, "ac%c", "012345??"[v & 7]);
				else
					pins_addr(fp, &loc, mode, (v & 077));
				break;

			default:
				fprintf(fp, "bad type %d", instab[i].type);
			}
			break;
		}
	putc('\n', fp);
	return loc - old_loc;
}

static void
pins_addr(FILE *fp, unsigned *loc, int mode, int t)
{
	unsigned a;
	ushort v;

	if(t == 027 || t == 037 || (t & 060) == 060) {
		if(pins_fetch(*loc, mode, &a, &v)) {
			*loc += 2;
			putc('?', fp);
			return;
		}
		*loc += 2;
	}
	if(t & 010) {
		putc('@', fp);
		t &= ~010;
	}
	if(t == 027) {
		fprintf(fp, "#%o", v);
		return;
	}
	if(t == 067) {
		fprintf(fp, "%o", (*loc + v) & 0177777);
		return;
	}

	switch(t & 060) {

	case 0:
		fprintf(fp, "r%d", t & 7);
		break;

	case 020:
		fprintf(fp, "(r%d)+", t & 7);
		break;

	case 040:
		fprintf(fp, "-(r%d)", t & 7);
		break;

	case 060:
		fprintf(fp, "%o(r%d)", v, t & 7);
		break;
	}
}

/*
 * user process tracing
 */
FILE	*fputrace;
static void
close_utrace(void)
{
	fclose(fputrace);
}

void
utrace(void)
{
	if(!f_utrace)
		return;
	if(!fputrace) {
		if(!(fputrace = fopen("utrace", "w"))) {
			printf("utrace: %s\n", strerror(errno));
			f_utrace = 0;
			return;
		}
		atexit(close_utrace);
	}
	pins(fputrace, proc.reg[7], proc.curmode);
}

/*
 * print outstanding interrupt requests
 */
static void
preqs(void)
{
	IODev *d;

	for(d = proc.devices; d; d = d->next) {
		if(d->reqpri == DMAPRI)
			printf("[DMA - %s]\n", d->name);
		else if(d->reqpri)
			printf("[%-3d - %s]\n", d->reqpri, d->name);
	}
}

static void
escape(void)
{
	char	*shell = getenv("SHELL");
	char	buf[1000];

	if(!shell)
		shell = _PATH_BSHELL;
	sprintf(buf, "exec %s", shell);
	system(buf);
}

/*
 * dump instructions
 */
static void
dump_ins(void)
{
	int inc, cnt = 1;

	crlf();
	for(;;) {
		inc = pins(stdout, loc, -1);
		putchar('\r');
		if(inc <= 0)
			break;
		loc += inc;
		if(--cnt <= 0) {
			switch(gch()) {

			case '\033':
				return;

			case ' ':
				cnt = 10;
				break;

			default:
				cnt = 1;
			}
		}
	}
}
