/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

/*
 * line printer
 *
 * Arguments:
 *	ctrl lp csr_base vector irq
 *	  | prog-name args ...
 * or
 *	  file-name
 */

# include "proc.h"

void 	pipeto(int, char *[], pid_t *, int *, char *);

typedef struct LP LP;

struct LP {
	unsigned csr_base;
	ushort vector;
	ushort ireq;
	int lpint;		/* interrupt requested */
	
	uchar	lpbuf;
	ushort	lpcsr;

	pid_t	pid;
	int	fd;
	LP	*next;
};

void	lp_ctrl_create(IODev *, int, char **);
void	lp_dev_create(IODev *, int, char **);
void	lp_ctrl_complete(IODev *);
void	lp_reset(IODev *);
ushort	lp_fetch(IODev *, unsigned);
void	lp_store(IODev *, unsigned, int, ushort);
ushort	lp_vector(IODev *);
void	lp_info(IODev *);
void	lp_kill(void);
void	lp_command(IODev *, int, char **);


IOOps	lp_ops = {
	lp_ctrl_create,		/* ctrl_create */
	lp_dev_create,		/* dev_create */
	lp_ctrl_complete,	/* ctrl_complete */
	lp_reset,		/* reset */
	lp_fetch,		/* fetch */
	lp_store,		/* store */
	lp_vector,		/* vector */
	0,			/* dma */
	0,			/* onsig */
	lp_info,		/* info */
	lp_command,		/* command interface */
};

static LP	*lps = 0;

/*
 * create controller for parallel interface
 * - nothing to do
 */
void
lp_ctrl_create(IODev *dev, int argc, char **argv)
{
	LP *lp;

	if(argc != 3)
		conf_panic("lp: need 3 args in controller configuration");
	lp = dev->data = xalloc(sizeof(LP), "lp_ctrl_create");
	lp->next = lps;
	lps = lp;

	lp->csr_base = parse_csr(argv[0], "lp");
	lp->vector = parse_vec(argv[1], "lp");
	lp->ireq = parse_irq(argv[2], "lp");

	proc.iopage[IOP(lp->csr_base+0)] = dev;
	proc.iopage[IOP(lp->csr_base+2)] = dev;
}

/*
 * create device
 */
void
lp_dev_create(IODev *dev, int argc, char **argv)
{
	LP *lp = dev->data;

	if(argc == 0)
		conf_panic("lp: missing args in device configuration");

	if(!strcmp(argv[0], "|")) {
		/*
		 * pipe into program
		 */
		pipeto(--argc, ++argv, &lp->pid, &lp->fd, "lp");
		ATEXIT(lp_kill, NULL);

	} else {
		/*
		 * write to file
		 */
		if(argc != 1)
			conf_warning("lp: too many args in dev config\n");

		if((lp->fd = open(argv[0], O_WRONLY)) < 0)
			conf_panic("lp: cannot open %s: %s\n", argv[0], strerror(errno));
		lp->pid = 0;
	}
}


void	
lp_ctrl_complete(IODev *dev)
{
}

/*
 * kill output processes
 */
void
lp_kill(void)
{
	LP *lp = lps;

	while(lp) {
		if(lp->pid)
			kill(lp->pid, SIGINT);
		lp = lp->next;
	}
}

/*
 * bus reset
 */
void	
lp_reset(IODev *dev)
{
	LP *lp = dev->data;

	lp->lpint = 0;
	lp->lpbuf = 0;
	lp->lpcsr = 0200;
}

/*
 * fetch from csr/buf
 */
ushort	
lp_fetch(IODev *dev, unsigned a)
{
	LP *lp = dev->data;

	if(a == lp->csr_base)
		return lp->lpcsr;
	else if(a == lp->csr_base + 2)
		return lp->lpbuf;
	printf("lp_fetch(%o)\n", a);
	Trap4(0100);
}

/*
 * store into csr/buffer 
 */
void	
lp_store(IODev *dev, unsigned a, int mode, ushort v)
{
	LP *lp = dev->data;
	char c;

	if(a == lp->csr_base) {
		if(mode & M_High)
			return;
		if(!(lp->lpcsr & 0100) && (v & 0100)) {
			IRQ(dev, lp->ireq);
			lp->lpint = 1;
		}
		lp->lpcsr = (v & 0100) | 0200;
		return;
	}

	if(a == lp->csr_base + 2) {
		if(mode & M_High)
			return;
		c = v;
		write(lp->fd, &c, 1);
		if(lp->lpcsr & 0100) {
			IRQ(dev, lp->ireq);
			lp->lpint = 1;
		}
		return;
	}

	printf("lp_fetch(%o)\n", a);
	Trap4(0100);
}

/*
 * return vector on interrupt
 */
ushort	
lp_vector(IODev *dev)
{
	LP *lp = dev->data;

	if(!lp->lpint)
		return 04;
	dev->reqpri = 0;
	return lp->vector;
}

/*
 * print info about device
 */
void	
lp_info(IODev *dev)
{
	LP *lp = dev->data;

	printf("LP parallel interface\n");
	printf("CSR at %08o: %06o\n", lp->csr_base, lp->lpcsr);
	printf("BUF at %08o: %06o\n", lp->csr_base+2, lp->lpbuf);
}

/*
 * lp command
 */
void
lp_command(IODev *dev, int argc, char **argv)
{

}

/*
 * create a pipe
 */
void
pipeto(int argc, char *argv[], pid_t *ppid, int *pfd, char *s)
{
	int p[2];
	char *arg;
	int len, i;

	if(argc == 0)
		conf_panic("%s: nothing to pipe", s);

	/*
	 * collect args
	 */
	for(i = len = 0; i < argc; i++)
		len += strlen(argv[i]);
	len += argc;
	arg = xalloc(len+1, "pipeto");		/* get one more */
	for(i = 0, arg[0] = 0; i < argc; i++) {
		strcat(arg, argv[i]);
		strcat(arg, " ");
	}

	if(pipe(p))
		conf_panic("%s: can't create pipe: %s", s, strerror(errno));

	if((*ppid = fork()) == 0) {
		/*
		 * child
		 */
		int fd;
		static char *av[4] = { _PATH_BSHELL, "-c", 0, 0 };

		close(0);
		dup(p[0]);
		for(fd = 3; fd < getdtablesize(); fd++)
			close(fd);
		av[2] = arg;
		signal(SIGINT, SIG_IGN);
		signal(SIGQUIT, SIG_IGN);
		execv(av[0], av);
		panic("%s: can't exec %s: %s", s, av[0], strerror(errno));
	}
	if(*ppid < 0)
		conf_panic("%s: cannot fork\n", s);
	close(p[0]);
	*pfd = p[1];
	free(arg);
}
