/* defines.h	- Definitions of things needed in all C files */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>

/* Defines for ifdef'd code -- define them in the Makefile */

/* #define DEBUG			adds in debugging code */
/* #define STREAM_BUFFERING		converts reads/writes to stdio calls */


/* Type definitions for PDP data types. */

#if __FreeBSD__ != 2
typedef char int8_t;
typedef short int16_t;
typedef long int32_t;
#endif
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned long uint32_t;


/* PDP processor defines. */

#define SP	6		/* stack pointer */
#define PC	7		/* program counter */


/* Instruction Table for Fast Decode. */

typedef void (*_itab) ();

#define PDP_MEM_SIZE	65536
#define MAX_ARGS	200

/* Global variables. */

extern uint16_t regs[8];	/* general registers */
extern uint16_t stacks[4];	/* stack pointers */
extern uint16_t psw;		/* processor status word */
extern uint16_t sr;		/* switch register */
extern uint16_t ir;		/* current instruction register */
extern uint16_t ea_addr;	/* stored address for dest modifying insts */
extern _itab itab[];
extern uint8_t *ispace, *dspace;

/* Function prototypes */
int load_a_out();


/* Inline defines. */

#define CC_N	010
#define CC_Z	04
#define CC_V	02
#define CC_C	01

#define CLR_CC_V()	psw &= ~CC_V
#define CLR_CC_C()	psw &= ~CC_C
#define CLR_CC_Z()	psw &= ~CC_Z
#define CLR_CC_N()	psw &= ~CC_N
#define CLR_CC_ALL()	psw &= ~(CC_V|CC_C|CC_Z|CC_N)

#define SET_CC_V()	psw |= CC_V
#define SET_CC_C()	psw |= CC_C
#define SET_CC_Z()	psw |= CC_Z
#define SET_CC_N()	psw |= CC_N

#define SRC_MODE	(( ir & 07000 ) >> 9 )
#define SRC_REG		(( ir & 0700 ) >> 6 )
#define DST_MODE	(( ir & 070 ) >> 3 )
#define DST_REG		( ir & 07 )

#define LSBIT	1		/* least significant bit */

#define	MPI	0077777		/* most positive integer */
#define MNI	0100000		/* most negative integer */
#define NEG_1	0177777		/* negative one */
#define SIGN	0100000		/* sign bit */
#define CARRY   0200000		/* set if carry out */

#define	MPI_B	0177		/* most positive integer (byte) */
#define MNI_B	0200		/* most negative integer (byte) */
#define NEG_1_B	0377		/* negative one (byte) */
#define SIGN_B	0200		/* sign bit (byte) */
#define CARRY_B	0400		/* set if carry out (byte) */

#define LOW16( data )	(( data ) & 0177777 )	/* mask the lower 16 bits */
#define LOW8( data )	(( data ) & 0377 )	/* mask the lower 8 bits */

#define CHG_CC_N( d )	if ((d) & SIGN ) \
					SET_CC_N(); \
				else \
					CLR_CC_N()

#define CHGB_CC_N( d )	if ((d) & SIGN_B ) \
				SET_CC_N(); \
			else \
				CLR_CC_N()

#define CHG_CC_Z( d )	if ( d ) \
					CLR_CC_Z(); \
				else \
					SET_CC_Z()

#define CHGB_CC_Z( d )	if ( LOW8( d )) \
				CLR_CC_Z(); \
			else \
				SET_CC_Z()

#define CHG_CC_C( d )	if ((d) & CARRY ) \
					SET_CC_C(); \
				else \
					CLR_CC_C()

#define CHG_CC_IC( d )	if ((d) & CARRY ) \
					CLR_CC_C(); \
				else \
					SET_CC_C()

#define CHGB_CC_IC( d )	if ((d) & CARRY_B ) \
				CLR_CC_C(); \
			else \
				SET_CC_C()

#define CHG_CC_V( d1, d2, d3 )	\
				if ((( d1 & SIGN ) == ( d2 & SIGN )) \
				&& (( d1 & SIGN ) != ( d3 & SIGN ))) \
					SET_CC_V(); \
				else \
					CLR_CC_V()

#define CHG_CC_VC( d1, d2, d3 )	\
				if ((( d1 & SIGN ) != ( d2 & SIGN )) \
				&& (( d2 & SIGN ) == ( d3 & SIGN ))) \
					SET_CC_V(); \
				else \
					CLR_CC_V()

#define CHG_CC_VS( d1, d2, d3 )	\
				if ((( d1 & SIGN ) != ( d2 & SIGN )) \
				&& (( d1 & SIGN ) == ( d3 & SIGN ))) \
					SET_CC_V(); \
				else \
					CLR_CC_V()

#define CHGB_CC_V( d1, d2, d3 )	\
				if ((( d1 & SIGN_B ) == ( d2 & SIGN_B )) \
				&& (( d1 & SIGN_B ) != ( d3 & SIGN_B ))) \
					SET_CC_V(); \
				else \
					CLR_CC_V()

#define CHGB_CC_VC(d1,d2,d3)	\
				if ((( d1 & SIGN_B ) != ( d2 & SIGN_B )) \
				&& (( d2 & SIGN_B ) == ( d3 & SIGN_B ))) \
					SET_CC_V(); \
				else \
					CLR_CC_V()

#define CHG_CC_V_XOR_C_N()	\
				if ((( psw & CC_C ) && \
				   ( psw & CC_N )) \
				|| ((!( psw & CC_C )) && \
				   ( ! ( psw & CC_N )))) \
					CLR_CC_V(); \
				else \
					SET_CC_V()



#ifdef __STDC__
#define	P(s) s
#else
#define P(s) ()
#endif


/* Macros to read and write loctions in
 * main memory.
 */

extern uint16_t *adptr;

/* lli_word() - Load a word from the given ispace logical address. */
#define lli_word(addr, word) \
	{ adptr= (uint16_t *)&(ispace[addr]); word= *adptr; }

/* ll_word() - Load a word from the given logical address. */
#define ll_word(addr, word) \
	{ adptr= (uint16_t *)&(dspace[addr]); word= *adptr; }

/* sl_word() - Store a word at the given logical address. */
#define sl_word(addr, word) \
	{ if (addr < dwrite_base) seg_fault(); \
	  adptr= (uint16_t *)&(dspace[addr]); *adptr= word; }

/* lli_byte() - Load a byte from the given logical ispace address. */
#define lli_byte(addr, byte) \
    byte = ispace[addr];

/* ll_byte() - Load a byte from the given logical address. */
#define ll_byte(addr, byte) \
    byte = dspace[addr];

/* sl_byte() - Store a byte at the given logical address. */
#define sl_byte(addr, byte) \
    	{ if (addr < dwrite_base) seg_fault(); \
	  dspace[addr]= byte; }

/* aout.c */
int load_a_out P((const char *file));

/* branch.c */
void brx P((unsigned clear, unsigned set));
void br P((void));
void blos P((void));
void bge P((void));
void blt P((void));
void ble P((void));
void bgt P((void));
void jmp P((void));
void jsr P((void));
void rts P((void));

/* double.c */
void mov P((void));
void cmp P((void));
void add P((void));
void sub P((void));
void bit P((void));
void bic P((void));
void bis P((void));
void movb P((void));
void cmpb P((void));
void bitb P((void));
void bicb P((void));
void bisb P((void));

/* ea.c */
void load_ea P((uint16_t * addr));
void pop P((uint16_t * data));
void push P((uint16_t data));
void loadb_dst P((uint8_t * data));
void loadb_src P((uint8_t * data));
void storeb_dst P((uint8_t data));
void storeb_dst_2 P((uint8_t data));
void loadp_dst P((uint16_t * data));
void storep_dst P((uint16_t data));
void load_src P((uint16_t * data));
void store_dst P((uint16_t data));
void load_dst P((uint16_t * data));
void store_dst_2 P((uint16_t data));

/* fp.c */
void fpset P((void));
void fp11 P((int IR ));
void illegal_fp P((void ));


/* itab.c */
void dositab0 P((void));
void dositab1 P((void));

/* main.c */
int main P((int argc, char **argv));
void sim_init P((void));
void set_arg_env P((int argc, char **argv, int envc, char **envp, int oldexec));
void bus_error P((void));
void seg_fault P((void));


/* run.c */
void run P((void));

/* single.c */
void adc P((void));
void asl P((void));
void asr P((void));
void clr P((void));
void com P((void));
void dec P((void));
void inc P((void));
void neg P((void));
void rol P((void));
void ror P((void));
void sbc P((void));
void swabi P((void));
void sxt P((void));
void tst P((void));
void tstb P((void));
void aslb P((void));
void asrb P((void));
void clrb P((void));
void comb P((void));
void decb P((void));
void incb P((void));
void negb P((void));
void rolb P((void));
void rorb P((void));
void adcb P((void));
void sbcb P((void));

/* trap.c */
void trap P((void));
char *xlate_filename P((char *name));
void trap_exec P((int oldexec));
int open_dir P((char *name));
int trap_gtty(uint16_t fd, uint16_t ucnt);
int trap_stty(uint16_t fd, uint16_t ucnt);

/* weird.c */
void waiti P((void));
void halt P((void));
void iot P((void));
void emt P((void));
void bpt P((void));
void fis P((void));
void illegal P((void));
void not_impl P((void));
void mark P((void));
void mfpd P((void));
void mtpd P((void));
void bne P((void));
void beq P((void));
void bpl P((void));
void bmi P((void));
void bhi P((void));
void bvc P((void));
void bvs P((void));
void bcc P((void));
void bcs P((void));
void scc P((void));
void ccc P((void));
void sob P((void));
void mfps P((void));
void mtps P((void));
void mfpi P((void));
void mtpi P((void));
void ash P((void));
void mul P((void));
void divide P((void));
void ashc P((void));
void xor P((void));

#undef P
