/*
 *	dlscan - scan all mounted file-systems for disk usage by users & groups
 *
 *		update passwd file entries appropriately,
 *		note any discrepancies on stdout
 *
 *				Piers Lauder
 *				Dept of Comp Sci
 *				Sydney University
 *	May '80.
 */


#include	<local-system>		/* global parameters */
#include	<param.h>		/* system parameters */
#include	<ino.h>		/* disk inode structure */
#include	<inode.h>		/* inode mode definitions */
#include	<filsys.h>		/* super-block structure */
#include	<mtab.h>		/* mounted file table structure */
#include	<passwd.h>		/* "passwd" entry structure */
#include	<stdio.h>		/* standard i/o streams */


#define	SPARSE_FILES	1	/* 1: understand sparse files - MUCH SLOWER */

#define	NBLOCKS		16	/* read NBLOCKS of inodes at a time */
#define	MINOR		50	/* complain if difference larger than this */

dusage_t dusages[NUSERS+1];	/* real usage per user */

#define	NINODES	(INOPB*NBLOCKS)
struct dinode	inbuf[NINODES];

char	rawname[]	= "/dev/r";
char	mtabf[]		= MTABF;
uid_t	uid		= -1;
long	lseek();


main(ac, av)
char **av;
{
	register char *fs;
	char s[16+(sizeof rawname)];
	char *getfs();

	if (ac != 1)
		if (av[1][0] == '-')
			uid = NUSERS;
		else
			uid = atoi(av[1]);
	while ((fs = getfs(s)) != NULL)
		chkfs(fs);
	if (uid == -1)
		updtpasswd();
	exit(0);
}



/*
 *	getfs - returns pointer to next file system name
 */

char *
getfs(s)
char *s;
{
	static init, fd;
	struct mtab mtabent;	/* entry from "mtabf" */
	extern char *strncat();
	extern char *strcpy();

	if (init == 0)
	{
		init++;
		if ((fd = open(mtabf, 0)) == SYSERROR)
		{
			perror(mtabf);
			exit(1);
		}
	}
	if (read(fd, (char *)&mtabent, sizeof mtabent) != sizeof mtabent)
	{
		close(fd);
		return(NULL);
	}
	return(strncat(strcpy(s, rawname), mtabent.m_spec, sizeof mtabent.m_spec));
}


/*
 *	chkfs - check file-system "fs" for usage per uid
 */

chkfs(fs)
char *fs;
{
	register fd;
	register long ninodes;
	unsigned inum;
	long offset;
	struct filsys super;

	if ((fd = open(fs, 0)) == SYSERROR)
	{
		perror(fs);
		return;
	}
	lseek(fd, (long)(SUPERB*BSIZE), 0);
	if (read(fd, (char *)&super, sizeof super) != sizeof super)
	{
		perror(fs);
		close(fd);
		return;
	}
	offset = (SUPERB+1) * BSIZE;
	if (uid != -1)
		printf("%s:\n", fs);
	for (ninodes = (super.s_isize-(SUPERB+1)) * INOPB ; ninodes > 0 ; ninodes -= NINODES)
	{
		register unsigned i;

		i = sizeof inbuf;
		if (ninodes < NINODES)
			i = ninodes * (sizeof (struct dinode));
		lseek(fd, offset, 0);
		if (read(fd, (char *)inbuf, (int)i) != (int)i)
		{
			perror(fs);
			close(fd);
			return;
		}
		offset += i;
		i /= sizeof (struct dinode);
		while (i--)
		{
			register dusage_t d;
			register uid_t u;
			extern dusage_t fsize();

			d = IWEIGHT;
			switch(inbuf[i].di_mode & IFMT)
			{
		    case 0:		/* Unallocated */
				break;

		    case IFDIR:
		    case IFREG:
		    case IFLOK:
		    case IFALK:
				d += fsize(fd, &inbuf[i]);
		    default:
				u = inbuf[i].di_uid;
				if (u > NUSERS)		/* illegal */
					u = NUSERS;
				if (uid && u == uid)
					printf("inum %u (%u units)\n", inum + i + 1, d);
				dusages[u] += d;
			}
		}
		inum += NINODES;
	}
	close(fd);
}


/*
 *	fsize - return block usage of file described by ip
 */

dusage_t
fsize(fd, ip)
int fd;
struct dinode *ip;
{

#if	SPARSE_FILES == 0
	register dusage_t b, ib, ib2;

	b = (ip->di_size + BMASK) >> BSHIFT;
	if (b > 10)
		if ((ib = (b-10+NMASK)>>NSHIFT) > 1)
			if ((ib2 = (ib-1+NMASK)>>NSHIFT) > 1)
				return(1+ib2+ib+b);
			else
				return(1+ib+b);
		else
			return(1+b);
	else
		return(b);
#else
	register i;
	register dusage_t b = 0;
	daddr_t bladdr[13];
	extern dusage_t ibcount();

	l3tol(bladdr, ip->di_addr, 13);
	for (i = 0 ; i < 13 ; i++)
	{
		if (bladdr[i])
		{
			b++;
			if (i >= 10)
				b += ibcount(fd, bladdr[i], i-10);
		}
	}
	return(b);
#endif
}


#if	SPARSE_FILES
dusage_t
ibcount(fd, ba, i)
int fd;
daddr_t ba;
register int i;
{
	register dusage_t b;
	daddr_t indb[BSIZE/sizeof(daddr_t)];
	register j;

	b = 0;
	lseek(fd, (long)ba * BSIZE, 0);
	if (read(fd, (char *)indb, sizeof indb) != sizeof indb)
		return(0);
	for (j = 0 ; j < (sizeof indb/sizeof indb[0]) ; j++)
		if (indb[j])
		{
			b++;
			if (i)
				b += ibcount(fd, indb[j], i-1);
		}
	return(b);
}
#endif


/*
 *	updtpasswd - updates passwd file entries with disk usage
 *		and checks consistency where limits exist.
 */
updtpasswd()
{
	uid_t i;
	short small;

	small = 0;
	for (i = 0 ; i < NUSERS ; i++)
	{
		register dusage_t u;

		if (u = dusages[i])
		{
			struct pwent pwe;

			pwe.pw_limits.l_uid = i;
			if (getpwlog(&pwe, (char *)0, 0) == PWERROR)
				printf("%u: %u units (DOES NOT EXIST)\n", i, u);
			else
			{
				register dusage_t d0, d1;
				register logged_in;

				logged_in = (limits(&pwe.pw_limits, L_OTHLIM) != SYSERROR);
				d0 = pwe.pw_limits.l_duse;
				if (d0 != u)
				{
					if (pwe.pw_limits.l_dlimit &&
					    logged_in == 0)
					{
						if (u > d0)
							d1 = u - d0;
						else
							d1 = d0 - u;
						if (d1 < MINOR)
							small++;
						else
							printf("%u: %u (pw) %u (real)\n", i, d0, u);
					}
					pwe.pw_limits.l_duse = u;
					updtpwent(&pwe);
				}
			}
		}
	}
	if (small)
		printf("%u minor infringements (diff < %u)\n", small, MINOR);
	if (dusages[NUSERS])
		printf("ILLEGAL uids possess %u units!\n", dusages[NUSERS]);
	pwclose();
}
