/*
**	Network protocol handler, routines to interface with netd.c
*/

#include	<local-system>
#ifdef	Aus2
#include	<types.h>
#include	<dir.h>
#endif	Aus2
#ifdef	Aus1
#include	<dir.h>
#endif	Aus1
#include	<sgtty.h>
#include	<setjmp.h>
#include	<stdio.h>
#include	<signal.h>
#include	<errno.h>
#include	"net.h"
#include	"netd.h"
#if	Aus2 & ~L7C
typedef	unsigned char	unsndch;
#endif	Aus2 & ~L7C
#ifdef	Aus1
#include	"lvl7.h"
#endif	Aus1
#if	Aus1 | L7C
typedef	char		unsndch;
#endif	Aus1 | L7C

/*
**	Regulate file transfer with a file protocol as follows:-
**
**	Blocks consist of a header followed by data followed by a trailer.
**	 The header contains a sequence number (0-MAXSEQ) and the size
**	  (transmitted twice, 2nd as complement of first).
**	 The trailer contains a longitudinal parity check of the data,
**	  followed by a sum check of the data.
**
**	Block format is
**	 SOH SEQ SIZ ~SIZ ...DATA... LPC SUM
**
**	Blocks are acknowledged by a 2 byte reply consisting of
**	 ACK or NAK followed by the relevant sequence number.
**
**	The protocol is multi-buffered for speed, with at most NBUFS blocks
**	 unacknowledged at any one time.
**
**	Note: that the DATAZ is really dependant on the accuracy
**	 of the crc check.
*/

#define	MAXSEQ		0177
#define	NBUFS		3
#define	HEADER		4
#define	BTYP		0
#define	BSEQ		1
#define	BSIZ1		2
#define	BSIZ2		3
#define	TRAILER		2
#define OVERHEAD	(HEADER+TRAILER)
#define	DATAZ		64		/* N.B. nice round number */
#define	BUFZ		(DATAZ+OVERHEAD)

unsndch		buf[BUFZ];
unsndch		c;

struct reply_s
{
	unsndch		r_typ;
	unsndch		r_seq;
}
		reply;

#define	R_EOF		-1
#define	R_TIMEOUT	-2
#define	R_RESET		-3

short		eof;
short		starting;

/*
**	Errors
*/

enum e_errors
{
	e_xretrys, e_routsize, e_rtimeouts, e_xtimeouts, e_crc,
	e_xwrite, e_lwrite, e_xnaks, e_sync, e_rsync, e_xsync,
	e_lread, e_rbadseq, e_rbadsiz,
	e_routseq, e_rskpseq, e_rignerr,
	e_nerrs
};

#define	E_XRETRYS	(int)e_xretrys
#define	E_ROUTSIZE	(int)e_routsize
#define	E_RTIMEOUTS	(int)e_rtimeouts
#define	E_XTIMEOUTS	(int)e_xtimeouts
#define	E_CRC		(int)e_crc
#define	E_XWRITE	(int)e_xwrite
#define	E_LWRITE	(int)e_lwrite
#define	E_XNAKS		(int)e_xnaks
#define	E_SYNC		(int)e_sync
#define	E_RSYNC		(int)e_rsync
#define	E_XSYNC		(int)e_xsync
#define	E_LREAD		(int)e_lread
#define	E_RBADSEQ	(int)e_rbadseq
#define	E_RBADSIZ	(int)e_rbadsiz
#define	E_ROUTSEQ	(int)e_routseq
#define	E_RSKPSEQ	(int)e_rskpseq
#define	E_RIGNERR	(int)e_rignerr
#define	NERRS		(int)e_nerrs

struct	err
{
	char *		e_mess;
	unsigned short	e_count;
}
		errs[NERRS] =
{
	 {"retrys"}
	,{"outsize"}
	,{"recv timeouts"}
	,{"xmit timeouts"}
	,{"crc"}
	,{"remote write"}
	,{"local write"}
	,{"naks"}
	,{"sync"}
	,{"recv sync"}
	,{"xmit sync"}
	,{"local read"}
	,{"bad sequence"}
	,{"bad size"}
	,{"out of sequence"}
	,{"sequence skipped"}
	,{"error ignored"}
};


/*
**	Protocol bytes and messages
*/

#define	SOH		0201	/* start of block */
#define	EOT		0204	/* remote reset */
#define	RCV		0205	/* remote recv request */
#define	ACK		0206	/* request/xfer ok */
#define	SND		0220	/* request to send */
#define	SAK		0221	/* sync acknowledge */
#define	SNA		0222	/* sync negative acknowledge */
#define	RST		0223	/* sync restart header */
#define	NAK		0225	/* xfer failed */
#define	SYN		0226	/* synchronisation */

enum m_typs
{
	m_reset, m_reset2, m_sync, m_mesgs
};

#define	MESGSIZ		2
#define	MRESET		(char *)&mesgs[(int)m_reset],(MESGSIZ*2)
#define	MSYNC		(char *)&mesgs[(int)m_sync],MESGSIZ

struct mesg
{
	unsndch		m_byts[MESGSIZ];
}
		mesgs[(int)m_mesgs] =
{
	 {EOT,EOT}
	,{EOT,EOT}
	,{SYN,SYN}
};

/*
**	active transmission data
*/

enum list_st
{
	null_st, ok_st, wait_st, err_st
};

struct list
{
	long		l_addr;
	enum list_st	l_state;
	unsigned short	l_age;
	unsndch		l_size;	/* WARNING! - used by crc.s */
	unsndch		l_seq;	/* WARNING! - used by crc.s */
}
	activelist[NBUFS];

typedef struct list *	listp;

extern listp		inlist();
extern listp		enter();

#define	FLUSHTIMEOUT	2	/* minimum recommended */
#define	XRETRYS		4
#define	SYNCRETRYS	5
#define	XTIMEOUT	15
#define	INRTIMEOUT	10
#define	RTIMEOUT	60

jmp_buf			le_jb, rto_jb, wto_jb;

#define	ack(s)		rep(ACK,s)
#define	nak(s)		rep(NAK,s)

unsndch			getreply();
unsndch			getsync();
extern long		time();
extern long		lseek();
extern long		atol();
extern char *		strncpy();
#ifdef	Aus1
extern long		tell();
#endif	Aus1

/*
**	Synchronisation data
*/

#define	RSTDATA		HEADER
#define	RSTDATAZ	14
#define	RSTBZ		(RSTDATAZ+OVERHEAD)

struct reply_s	reqst;

enum saval
{
	sa_fail, sa_ok, sa_cont, sa_reset, sa_restart, sa_write
}
		(*systat[6][7])();

extern enum saval	syreset();

enum systs
{
	recv1_syst, recv2_syst, recv3_syst, recv4_syst,
	send1_syst, send2_syst
}
		sync_state;

static long	rstaddr;
static char *	rstfn;



/*
**	Synchronise protocol direction
*/

sync(direction, fd, fn)
	enum sync_proto	direction;
	int		fd;
	char *		fn;
{
	register int	retrys;
	register int	rc;

#	ifdef	DEBUG
	fprintf(stderr, "sync %s\n", direction==send_pro?"send":"recv");
#	endif	DEBUG

	if ( setjmp(wto_jb) )
	{
		hostdown++;
		return 0;
	}

	hostdown = 0;
	starting = 1;	/* disable error counts */
	rstfn = fn;

resync:
	twrite(MSYNC);

	/** Set arbitration priority **/

	if ( lokk.lk_file[0] != '\0' )
	{
		reqst.r_seq = 0xff;	/* attempt restart */
		rstaddr = lseek(fd, (long)0, 2);
#		ifdef	Aus1
		rstaddr = tell(fd);
#		endif	Aus1
	}
	else
#		if	Aus2 & ~L7C
		if ( (reqst.r_seq = sync_time) == 0xff )
#		endif	Aus2 & ~L7C
#		if	Aus1 | L7C
		if ( ((reqst.r_seq = sync_time) & 0xff) == 0xff )
#		endif	Aus1 | L7C
			reqst.r_seq = 0xfe;

	switch ( direction )
	{
	 case send_pro:	reqst.r_typ = SND;
			sync_state = send1_syst;
			break;
	 case recv_pro:	reqst.r_typ = RCV;
			if ( lokk.lk_file[0] == '\0' && sndflg )
				reqst.r_seq = 0;	/* don't really want to send */
			sync_state = recv1_syst;
	}

	for ( retrys = 0 ; ; )
	{
#		if	Aus1 | L7C
		if ( rc = getsync() & 0xff )
#		endif	Aus1 | L7C
#		if	Aus2 & ~L7C
		if ( rc = getsync() )
#		endif	Aus2 & ~L7C
			retrys = 0;
		else
			if ( retrys++ > SYNCRETRYS )
			{
				hostdown = 1;
				return 0;
			}

		switch ( rc )
		{
		 case 0:		break;
		 case RCV:	rc = 1; break;
		 case SND:	rc = 2; break;
		 case SAK:	rc = 3; break;
		 case RST:	rc = 4; break;
		 case SNA:	rc = 5; break;
		 case SYN:	rc = 6;	break;

		 case EOT:	while(!(tread(&c, 1, 2)));	/* flush line */
				goto resync;
		 default:	syreset();
				continue;
		}

		switch ( (*systat[(int)sync_state][rc])() )
		{
		 case sa_fail:
				return 0;	/* change mode */
		 case sa_ok:
				rstaddr = 0;
		 case sa_restart:
				sync_time = time((long *)0);
				starting = 0;
				return 1;	/* success */
		 case sa_reset:
				/* goto resync; */
		 case sa_cont:
				continue;
		}

		twrite((char *)&reqst, sizeof reqst);
	}
}



/*
**	Read sync sequences from remote
*/

unsndch
getsync()
{
	register int	retrys;

#	ifdef	DEBUG
	fprintf(stderr, "getsync\n");
#	endif	DEBUG

	if ( tread((unsndch *)&reply, sizeof reply, RTIMEOUT) )
		return 0;

	for ( retrys = 0 ; ; )
	{
#		if	Aus1 | L7C
		switch ( reply.r_typ & 0xff )
#		endif	Aus1 | L7C
#		if	Aus2 & ~L7C
		switch ( reply.r_typ )
#		endif	Aus2 & ~L7C
		{
		 case RCV:
		 case SND:
		 case SAK:
		 case SNA:
			return reply.r_typ;
		 case ACK:
			if ( Oldnetd )
				return SAK;
			goto resync;
		 case SYN:
		 case EOT:
			if ( reply.r_seq == reply.r_typ )
				return reply.r_typ;
		 default:
resync:
			errs[E_SYNC].e_count++;
			if ( retrys++ > (2*RSTBZ) )
				return 0xff;
			reply.r_typ = reply.r_seq;
			if ( tread(&reply.r_seq, 1, INRTIMEOUT) )
				return 0;
			continue;
		 case RST:
			if ( tread(buf+(sizeof reply), RSTBZ-(sizeof reply), RTIMEOUT) )
				return 0;
			return RST;
		}
	}
}



/*
**	The following routines are part of the state table responses
*/

enum saval
synull()
{
	return sa_cont;
}



enum saval
syretry()
{
	return sa_write;
}



enum saval
syreset()
{
	errs[E_SYNC].e_count++;
	while ( !tread(&c, 1, 2) );	/* flush line */
	twrite(MRESET);
	return sa_reset;
}



enum saval
syr1()
{
	sync_state = recv1_syst;
	reply.r_typ = RCV;
	return sa_write;
}



enum saval
syr2()
{
	sync_state = recv2_syst;
	reqst.r_typ = ACK;
	reqst.r_seq = reply.r_seq;
	return sa_write;
}



enum saval
syr3()
{
	sync_state = recv3_syst;
	reqst.r_typ = RCV;
	reqst.r_seq = reply.r_seq;
	return sa_write;
}



enum saval
syr23()
{
	return Oldnetd?syr2():syr3();
}



enum saval
syrarb()
{
	if ( reply.r_seq && reqst.r_seq && !sndflg )
		return sa_cont;
#	if	Aus1 | L7C
	if ( (reply.r_seq&0xff) > (reqst.r_seq&0xff) )
#	endif	Aus1 | L7C
#	if	Aus2 & ~L7C
	if ( reply.r_seq > reqst.r_seq )
#	endif	Aus2 & ~L7C
		return sa_fail;	/* try sending */
	if ( reply.r_seq == reqst.r_seq )
		reqst.r_seq = time((long *)0);
	return syr1();
}



enum saval
syrrst()
{
	struct list	rl;

	Oldnetd = 0;
	reqst.r_seq = reply.r_seq;
	rl.l_seq = reply.r_seq;
	if ( (rl.l_size = buf[BSIZ1]) != RSTDATAZ
		|| ((~(buf[BSIZ2]))&0xff) != RSTDATAZ
		|| (crc(&rl) && ++errs[E_CRC].e_count)
	   )
		return syr3();
	sync_state = recv4_syst;
	strncpy(rstfn, (char *)&buf[RSTDATA], RSTDATAZ);
	if ( strncmp(lokk.lk_file, rstfn, RSTDATAZ) )
	{
		reqst.r_typ = SNA;
		return sa_write;
	}
	sprintf((char *)&buf[RSTDATA], "%-*ld", RSTDATAZ, rstaddr);
	syrstb(&rl);
	return sa_cont;
}



syrstb(lp)
	struct list *	lp;
{
	crc(lp);
	buf[BTYP] = RST;
	twrite((char *)buf, RSTBZ);
}



enum saval
syr2ok()
{
	if ( reply.r_seq == reqst.r_seq )
	{
		rstfn[0] = '\0';
		return sa_ok;
	}
	return syr1();
}



enum saval
syr4ok()
{
	if ( reply.r_seq == reqst.r_seq )
#		if	Aus2 & ~L7C
		return reply.r_typ==SAK?sa_restart:sa_ok;
#		endif	Aus2 & ~L7C
#		if	Aus1 | L7C
		return (reply.r_typ&0xff)==SAK?sa_restart:sa_ok;
#		endif	Aus1 | L7C
	return syr3();
}



enum saval
syx1()
{
	sync_state = send1_syst;
	return sa_write;
}



enum saval
syx2()
{
	struct list	rl;

	if ( reply.r_seq != reqst.r_seq )
		return sa_cont;
	reqst.r_seq++;
	sync_state = send2_syst;
	strncpy((char *)&buf[RSTDATA], rstfn, RSTDATAZ);
	rl.l_seq = reqst.r_seq;
	rl.l_size = RSTDATAZ;
	syrstb(&rl);
	return sa_cont;
}



enum saval
syx12()
{
	if ( Oldnetd )
		return sa_write;
	if ( reply.r_seq != reqst.r_seq )
		return sa_write;
	return syx2();
}



enum saval
syxarb()
{
#	if	Aus1 | L7C
	if ( (reply.r_seq&0xff) > (reqst.r_seq&0xff) )
#	endif	Aus1 | L7C
#	if	Aus2 & ~L7C
	if ( reply.r_seq > reqst.r_seq )
#	endif	Aus2 & ~L7C
		return sa_fail;	/* gazzumped */
	if ( reply.r_seq == reqst.r_seq )
		reqst.r_seq = time((long *)0);
	return syx1();
}



enum saval
syx1sak()
{
	if ( !Oldnetd )
		return sa_cont;
	if ( reply.r_seq != reqst.r_seq )
		return sa_write;
	twrite((char *)&reply, sizeof reply);
	return sa_ok;
}



enum saval
syxrst()
{
	struct list	rl;
	long		ra;

	rl.l_seq = reply.r_seq;
	if ( (rl.l_size = buf[BSIZ1]) != RSTDATAZ
		|| ((~(buf[BSIZ2]))&0xff) != RSTDATAZ
		|| (crc(&rl) && ++errs[E_CRC].e_count)
	   )
		return syx2();
	if ( (ra = atol((char *)&buf[RSTDATA])) <= rstaddr )
	{
		rstaddr = ra;
		reply.r_typ = SAK;
	}
	else
	{
		rstaddr = 0;
		reply.r_typ = SNA;
	}
	twrite((char *)&reply, sizeof reply);
#	if	Aus2 & ~L7C
	return reply.r_typ==SAK?sa_restart:sa_ok;
#	endif	Aus2 & ~L7C
#	if	Aus1 | L7C
	return (reply.r_typ&0xff)==SAK?sa_restart:sa_ok;
#	endif	Aus1 | L7C
}



enum saval
syx2sna()
{
	twrite((char *)&reply, sizeof reply);
	return sa_ok;
}



/*
**	Synchronisation State Table
*/

enum saval	(*systat[6][7])() =
{
     /* timeout	RCV	SND	SAK	RST	SNA	SYN */
	syretry,syrarb,	syr23,	synull,	syrrst,	synull,	syretry,/* recv1 */
	synull,	syrarb,	syr2,	syr2ok,	syrrst,	synull,	syretry,/* recv2 */
	syretry,syrarb,	syr3,	synull,	syrrst,	synull,	syr1,	/* recv3 */
	synull,	syrarb,	syr3,	syr4ok,	syrrst,	syr4ok,	syr1,	/* recv4 */
	syretry,syx12,	syxarb,	syx1sak,syreset,synull,	syretry,/* send1 */
	synull,	syx2,	syxarb,	synull,	syxrst,	syx2sna,syx1	/* send2 */
};


/*
**	Catch write timeouts
*/

wtimout()
{
	longjmp(wto_jb, 1);
}



/*
**	Write remote line with timeout
*/

twrite(m, n)
	char *	m;
	int	n;
{
	signal(SIGALRM, wtimout);
	alarm((unsigned)(SYNCRETRYS*RTIMEOUT));
	write(net_fd, m, n);
	alarm(0);
#	ifdef	TRACE
	trace("twrite", n, m);
#	endif	TRACE
}



/*
**	Transmit protocol handler
**
**	send ahead NBUFS and then start looking for acknowledgements
*/

enum r_states
xproto(local)
{
	register listp	lp;
	register int	nextseq = 0;
	register long	nextaddr = rstaddr;
	register int	retrys = 0;
	listp		alp;

#	ifdef	DEBUG
	fprintf(stderr, "xproto\n");
#	endif	DEBUG

	if ( setjmp(le_jb) )
		return r_localerr;

	eof = 0;

	for ( lp = activelist ; lp < &activelist[NBUFS] && !eof ; lp++ )
	{
		lp->l_seq = nextseq++;
		lp->l_addr = nextaddr;
		nextaddr += putblk(lp, local);
	}

	for(;;)
	{
#		if	Aus1 | L7C
		switch ( getreply(&alp) & 0xff )
#		endif	Aus1 | L7C
#		if	Aus2 & ~L7C
		switch ( getreply(&alp) )
#		endif	Aus2 & ~L7C
		{
		 case EOT:	break;
		 case ACK:	retrys = 0;
				lp = alp;
				if ( eof )
				{
					lp->l_state = ok_st;

					for ( lp = activelist ; lp < &activelist[NBUFS] ; lp++ )
						if ( lp->l_state == wait_st )
						{
							retrys++;
							break;
						}

					if ( !retrys )
					{
						proto_bytes += nextaddr - rstaddr;
						proto_time += time((long *)0) - sync_time;
						return r_ok;
					}

					retrys = 0;
					continue;
				}

				if ( (lp->l_seq = nextseq++) == MAXSEQ )
					nextseq = 0;
				lp->l_addr = nextaddr;
				nextaddr += putblk(lp, local);
				continue;
		 case NAK:	if ( retrys++ < XRETRYS )
				{
					errs[E_XRETRYS].e_count++;
					putblk(alp, local);
					continue;
				}
				break;
		 default:	if ( retrys++ < XRETRYS )
				{
					register int	minseq = MAXSEQ;

					for ( lp = activelist, alp = (listp)0 ; lp < &activelist[NBUFS] ; lp++ )
						if ( lp->l_state == wait_st && lp->l_seq <= minseq )
						{
							minseq = lp->l_seq;
							alp = lp;
						}

					if ( (lp = alp) == (listp)0 )
						continue;

					do
					{
						if ( lp->l_state == wait_st )
						{
							errs[E_XRETRYS].e_count++;
							putblk(lp, local);
						}

						if ( ++lp >= &activelist[NBUFS] )
							lp = activelist;
					}
					while
						( lp != alp );

					continue;
				}
				break;
		}
		break;
	}
	write(net_fd, MRESET);
	proto_bytes += nextaddr - rstaddr;
	proto_time += time((long *)0) - sync_time;
	return r_remoterr;
}




/*
**	Read local file and write data block + protocol
*/

putblk(lp, local)
	register listp	lp;
{
	register int	x;

	if ( lseek(local, lp->l_addr, 0) == SYSERROR
		|| (x = read(local, (char *)buf+HEADER, DATAZ)) == SYSERROR
	   )
	{
		errs[E_LREAD].e_count++;
		longjmp(le_jb, 1);
	}

	if ( (lp->l_size = x) == 0 )
		eof++;

	lp->l_state = wait_st;
	crc(lp);

	if ( write(net_fd, (char *)buf, (int)lp->l_size+OVERHEAD) != (lp->l_size+OVERHEAD) )
	{
		errs[E_XWRITE].e_count++;
		longjmp(le_jb, 2);
	}

#	ifdef	TRACE
	trace("putblk", lp->l_size+OVERHEAD, buf);
#	endif	TRACE
#	if	Aus1 | L7C
	return lp->l_size & 0xff;
#	endif	Aus1 | L7C
#	if	Aus2 & ~L7C
	return lp->l_size;
#	endif	Aus2 & ~L7C
}



/*
**	Read reply packet from remote
*/

unsndch
getreply(alp)
	listp *	alp;
{
	register int	retrys;

#	ifdef	DEBUG
	fprintf(stderr, "getreply\n");
#	endif	DEBUG

	do
	{
		if ( tread((unsndch *)&reply, sizeof reply, XTIMEOUT) )
		{
			errs[E_XTIMEOUTS].e_count++;
			return 0;
		}

		retrys = 0;

		for(;;)
		{
#			if	Aus1 | L7C
			switch ( reply.r_typ & 0xff )
#			endif	Aus1 | L7C
#			if	Aus2 & ~L7C
			switch ( reply.r_typ )
#			endif	Aus2 & ~L7C
			{
			 case NAK:
				errs[E_XNAKS].e_count++;
			 case ACK:
				break;
			 case SYN:
			 case EOT:
				if ( reply.r_seq == reply.r_typ )
					return reply.r_typ;
			 default:
				errs[E_XSYNC].e_count++;
				if ( retrys++ > (2*RSTBZ) )
					return EOT;
				reply.r_typ = reply.r_seq;
				if ( tread(&reply.r_seq, 1, XTIMEOUT) )
				{
					errs[E_XTIMEOUTS].e_count++;
					return 0;
				}
				continue;
			}
			break;
		}
	}
	while
		( (*alp = inlist(reply.r_seq)) == (listp)0 );

	return reply.r_typ;
}




/*
**	Receive protocol handler
*/

enum r_states
rproto(local)
{
	register listp		lp;
	register unsigned	seq;
	register unsigned	nextseq = 0;
	register listp		op;
	register int		retrys = 0;
	int			size;
	int			count = 0;

#	ifdef	DEBUG
	fprintf(stderr, "rproto\n");
#	endif	DEBUG

	eof = 0;
	strclr((char *)activelist, sizeof activelist);

	while ( (seq = (unsigned)getblk(&size)) <= MAXSEQ )
		if ( (lp = inlist((unsndch)seq)) == (listp)0 )
		{
			if ( seq != nextseq )
			{
#				ifdef	DEBUG
				fprintf(stderr, "seq=%d, expected=%d\n", seq, nextseq);
#				endif	DEBUG
				errs[E_ROUTSEQ].e_count++;
				if ( retrys++ >= (2*NBUFS) )
					break;
				write(net_fd, MSYNC);
			}
			else
			{
				if ( ++nextseq > MAXSEQ )
					nextseq = 0;
				if ( (lp = enter((unsndch)seq)) == (listp)0 )
				{
					errs[E_RIGNERR].e_count++;
					break;
				}
				if ( (op = inlist(seq==0?(unsndch)MAXSEQ:(unsndch)(seq-1))) == (listp)0 )
				{
					if ( count )
					{
						errs[E_RSKPSEQ].e_count++;
						break;
					}
					lp->l_addr = rstaddr;
				}
				else
					lp->l_addr = op->l_addr + op->l_size;
				lp->l_size = size;

				count++;
write_l:
				if ( (crc(lp) && ++errs[E_CRC].e_count)
					|| ((lseek(local, lp->l_addr, 0) == SYSERROR
						|| write(local, (char *)buf+HEADER, size) != size
					   ) && ++errs[E_LWRITE].e_count )
				   )
				{
					if ( retrys++ > (2*NBUFS) )
						break;
					lp->l_state = err_st;
					nak(seq);
				}
				else
				{
					retrys = 0;
					if ( size == 0 )
						eof++;
					else
						proto_bytes += size;
					lp->l_state = ok_st;
					ack(seq);
					if ( eof )
					{
						for ( lp = activelist ; lp < &activelist[NBUFS] ; lp++ )
							if ( lp->l_state == err_st )
								break;
						if ( lp = &activelist[NBUFS] )
							break;
					}
				}
			}
		}
		else
			if ( lp->l_state == ok_st )
				ack(seq);
			else
				goto write_l;

	for ( lp = activelist ; lp < &activelist[NBUFS] ; lp++ )
		if ( lp->l_state == err_st )
		{
			errs[E_RIGNERR].e_count++;
			eof = 0;
		}

	proto_time += time((long *)0) - sync_time;

	if ( eof )
		return r_ok;

	write(net_fd, MRESET);
	return r_remoterr;
}



/*
**	Catch alarm signals
*/

timeout()
{
	longjmp(rto_jb, 1);
}


/*
**	Read remote line with timeout
*/

tread(b, s, outtime)
	register unsndch *	b;
	register int		s;
	int			outtime;
{
	register int		x;
	register int		error = 0;
#	ifdef	TRACE
	register int		i = s;
	register unsndch *	b1 = b;
#	endif	TRACE
	extern int		errno;

#	ifdef	DEBUG
	fprintf(stderr, "tread\n");
#	endif	DEBUG

	if ( setjmp(rto_jb) )
	{
#		ifdef	TRACE
		trace("tread - timeout", 0);
#		endif	TRACE
		return 1;
	}
	signal(SIGALRM, timeout);
	alarm((unsigned)outtime);

	while ( (x = read(net_fd, (char *)b, s)) != s )
		if ( x != SYSERROR )
		{
			if ( x == 0 )
				sleep(outtime+2);
			else
			{
				b += x;
				s -= x;
			}
		}
		else
		{
			error++;
			break;
		}

	alarm(0);
#	ifdef	TRACE
	if ( !error )
		trace("tread", i, b1);
	else
		trace("tread - read error", 0);
#	endif	TRACE
	if ( error && errno != EINTR )
		sleep(60);
	return(error);
}



/*
**	Search list for sequence
*/

listp
inlist(seq)
	unsndch	seq;
{
	register listp	lp;

	for ( lp = activelist ; lp < &activelist[NBUFS] ; lp++ )
		if ( lp->l_seq == seq && lp->l_state != null_st )
			return lp;

	return (listp)0;
}



/*
**	Read a block from line
*/

enum get_st
{
	newbl_st, seq_st, siz1_st, siz2_st, reset_st
};


getblk(sizea)
	int *			sizea;
{
	register enum get_st	state;
	register unsndch	seq;
#	if	Aus1 | L7C
	register int		cc;
#	endif	Aus1 | L7C
#	if	Aus2 & ~L7C
#	define			cc	c
#	endif	Aus2 & ~L7C
	register int		retrys;

#	ifdef	DEBUG
	fprintf(stderr, "getblk\n");
#	endif	DEBUG

	state = newbl_st;

	for ( retrys = 0 ; retrys < (2*NBUFS*BUFZ) ; )
	{
		if ( tread(&c, 1, INRTIMEOUT) )
		{
			state = newbl_st;

			if ( tread(&c, 1, RTIMEOUT) )
			{
				errs[E_RTIMEOUTS].e_count++;
				return R_TIMEOUT;
			}
		}

#		if	Aus1 | L7C
		cc = c & 0xff;
#		endif	Aus1 | L7C

		switch ( state )
		{
		 case newbl_st:	if ( cc == SOH )
					state = seq_st;
				else
				if ( cc == EOT )
					state = reset_st;
				else
				{
					errs[E_RSYNC].e_count++;
					break;
				}
				continue;
		 case seq_st:	if ( cc <= MAXSEQ )
				{
					seq = cc;
					state = siz1_st;
					continue;
				}
				errs[E_RBADSEQ].e_count++;
				break;
		 case siz1_st:	if ( (*sizea = cc) <= DATAZ )
				{
					state = siz2_st;
					continue;
				}
				errs[E_ROUTSIZE].e_count++;
				break;
		 case siz2_st:	if ( cc == ((~(*sizea))&0xff) )
				{
					if ( tread(buf+HEADER, *sizea+TRAILER, RTIMEOUT) )
					{
						errs[E_RTIMEOUTS].e_count++;
						return R_TIMEOUT;
					}
					return (int)seq;
				}
				errs[E_RBADSIZ].e_count++;
				break;
		 case reset_st:	if ( cc == EOT )
					return R_RESET;
		}

		state = newbl_st;
		retrys++;
	}

	return R_RESET;
}
#if	Aus2 & ~L7C
#undef	cc
#endif	Aus2 & ~L7C



/*
**	Enter a sequence into active list
*/

listp
enter(seq)
	unsndch			seq;
{
	register listp		lp;
	register unsigned	oldest = 0;
	register listp		op = (listp)0;

	for ( lp = activelist ; lp < &activelist[NBUFS] ; lp++ )
	{
		lp->l_age++;

		switch ( lp->l_state )
		{
		 case null_st:
		 case ok_st:
			if ( oldest < lp->l_age )
			{
				oldest = lp->l_age;
				op = lp;
			}
		}
	}

	if ( op != (listp)0 )
	{
		op->l_seq = seq;
		op->l_age = 0;
		op->l_state = wait_st;
	}

	return op;
}



/*
**	Make reply to block
*/

rep(typ, seq)
	unsndch	typ, seq;
{
	reply.r_typ = typ;
	reply.r_seq = seq;
	write(net_fd, (char *)&reply, sizeof reply);
#	ifdef	TRACE
	trace("rep", sizeof reply, (unsndch *)&reply);
#	endif	TRACE
}


/*
**	Print errors
*/

errors()
{
	register struct err *	ep;
	register int		nerrs;

	nerrs = 0;

	for (ep = errs; ep < &errs[NERRS]; ep++)
	if (ep->e_count)
	{
		if (nerrs++ == 0)
			fprintf(stderr, "  file transfer protocol errors:-\n");
		fprintf(stderr, "  %s: %u", ep->e_mess, ep->e_count);
		ep->e_count = 0;
	}

	if (nerrs)
		fprintf(stderr, "\n");
	return(nerrs);
}
#ifdef	TRACE


/*VARARGS2*/
trace(d, s, p)
	char *			d;
	register int		s;
	register unsndch *	p;
{
	register int		i;
	register unsndch	c;
	unsndch			cc[2];
	long			ntime;
	static long		ltime;
	extern int		traceflag;

	if ( !traceflag )
		return;

	cc[1] = '\0';

	ntime = time((long *)0);
	fprintf(stderr, "%2ld: ", ntime - ltime);
	ltime = ntime;

	fprintf(stderr, "%s:", d);
	for ( i = 0 ; i < s ; i++ )
	{
		c = *p++;
		cc[0] = (c >= ' ' && c <= '~') ? c : '\0';
		fprintf(stderr, "<%o>%s", c&0xff, cc);
	}
	fprintf(stderr, "\n");
	fflush(stderr);
}
#endif	TRACE
