;;;	/ p o p / u s r / l i b / l o g i c  . p
;;; 	by Aaron Sloman		March 1978
;;;	for details see /pop/demos/logic
;;; also see the files /pop/usr/lib/rparse.explain and /pop/usr/lib/helplogic.p
;;;	The top level operation is dologic, which sets up a special environment
;;;	reading in commands, or logical formulas, and trapping errors.

;;; first some parsing routines

;;; A program to read in logical formulas, parse them, and print them
;;; out either in a symbolic or a verbal form.
Uses lineread;
Uses member;

vars istrue;	member(%[%true% 1 t T true TRUE]%) -> istrue;
vars isfalse;	member(%[%false% 0 f F false FALSE]%) ->isfalse;
vars istruthvalue;
member(%[%true,false% 0 1 t f T F true false TRUE FALSE]%) ->istruthvalue;

vars isyes isno;
member(%[[yes][y][YES][Y][ok][OK]]%)	->isyes;
member(%[[no] [n][NO] [n][nope]]%)	->isno;

function allof(List,Pred);
	;;; true if every element of List satisfies Pred, false otherwise.
	if	List==[]
	then true
	elseif	not(Pred(hd(List)))
	then	false
	else	allof(tl(List),Pred)
	close
end;


function rerror(l);
	ppr(l);
	interrupt();
end;
vars pppr;
ppr ->pppr;
	;;; but can be altered if required.
vars logout;	unless isfunc(logout) then popmess(%[1]%) -> logout close;
		;;; for debugging, assign setpop to logout.
vars getlinedepth; 	0 -> getlinedepth;	;;; Used in next function.

function getline(mess)=>ll;
	vars getlinedepth; getlinedepth + 1 -> getlinedepth;
	if getlinedepth > 3 then ppr('\nType H for help\n'); interrupt() close;
	pppr(mess);
	lineread() ->ll;
	if	ll = [bye] or ll == termin
	then	ppr('\nBYE\n');
		logout()
	elseif	ll = [stop]
	then	interrupt()
	elseif	ll= [setpop]
	then	setpop()	;;; for debugging
	elseif	atom(ll) then getline('') ->ll
	close
end;

vars input;
vars currformula;
[v [-p] q] ->currformula;		;;; a default initial formula.

vars iffwds thenwds exorwds orwds andwds notwds;
[iff <-> == <=>] ->iffwds;
[implies -> => 'if...then..']->thenwds;
[xor x ex ] ->exorwds;
[or v ] ->orwds;
[and & . ] ->andwds;
[not - ~ ] ->notwds;

vars rnoisy;		;;; Controls printing of explanatory message.
if rnoisy then
	compile('/user3/EECF-Maint/pop/usr/lib/rparse.explain')
close;

vars englishof;
newassoc([[ <-> iff ] [ -> implies ] [ x xor ][ v or ] [ & and ] [ - not ]]) ->englishof;
	;;; Used for translating operators back to English words.

;;; now a list assigning precedence
vars operatorseq;
[%iffwds,thenwds,orwds,exorwds,andwds,"not","atom"%] ->operatorseq;

;;; and a list used to check for unwanted operators;
vars operators;
iffwds<>(thenwds<>(orwds<>(exorwds<>(andwds<>notwds)))) ->operators;

;;; The next functions are concerned with parsing, on the assumption that
;;; a line of text has been read in and assigned to input.
;;; the top level function is parse, which calls rparse, which in turn calls itself
;;; recursively using less and less of the list operatorseq

function testinput;
	if input==[]
	then	rerror(['incomplete formula']);
	close;
end;

function try(wds);
	;;; used to check a hypothesis about the next word in input.
	;;; wds may be either a single word, or a list of words.
	if input==[]
	then	false;
	elseif	isword(wds)
	then	if	wds==hd(input)
		then	tl(input) ->input;
			true
		else	false
		close
	elseif	member(hd(input),wds)
	then	tl(input) ->input;
		true;
		testinput()
	else	false
	close;
end;

vars rparse toplevel;
true ->toplevel;

function parse;
	vars toplevel flag;
	toplevel ->flag; false ->toplevel;
	rparse(operatorseq);
	if	flag and input/==[]
	then	rerror(['surplus input: '%input%]);
	close
end;

function rparse(precedence) =>ss;
	vars thisop nextops;
	dest(precedence) -> nextops ->thisop;
	if	thisop == "not"
	then	hd(input) ->ss;
		if	ss == "--"
		then		[ - - ]<>tl(input) ->input
		elseif	ss=="---"
		then	[ - - - ]<>tl(input) ->input
		elseif	ss == "----"
		then	[ - - - - ]<>tl(input) ->input
		close;
		if	try(notwds)
		then	[ - % rparse(precedence)%] ->ss;
		else	rparse(tl(precedence)) ->ss
		close
	elseif	thisop == "atom"
	then
		;;; expect "(", "if", or some atom which is not in the list operators.
		if	hd(input)=="then"
		then	rerror('missing antecedent before'::input)
		elseif	hd(input) == ")"
		then	rerror('unexpected closing bracket: '::input)
		elseif	member(hd(input),operators)
		then	rerror('argument missing before'::input)
		elseif	try("if")
		;;; this bit copes with conditionals of the form IF P THEN Q.
		;;; P may be an arbitrary formula. Q may be any formula except a biconditional
		then	parse() ->ss;
			if	try("then")
			then	[-> %ss, rparse(tl(operatorseq))%] ->ss
				;;; use the tl to prevent bi-conditionals after "then"
			else	rerror('missing then clause'::
					if atom(input)
					then	[]
					else 'before'::input
					close)
			close
		elseif	try("(")
		then	parse() ->ss;
			if	input==[]	then rerror('Missing )')
			elseif	not(try(")"))	then rerror('Missing ")" before'::input)
			close
		else	hd(input) ->ss;		;;; found an atom, return it
			tl(input) ->input
		close
	else
		;;; thisop specifies a type of binary operator, e.g. andwds.
		;;; try to parse suitable arguments before and after an
		;;; occurrence of it.
		rparse(tl(precedence)) ->ss;
		if	thisop == orwds or thisop == exorwds or thisop == andwds
		then
			;;; allow them to associate to the right:
			while	try(thisop) then [%thisop(2),ss,rparse(nextops)%] ->ss close;
		elseif try(thisop) then	[%thisop(2),ss,rparse(nextops)%] ->ss
		close
	close
end;

;;; Now some printing routines.
vars rprint;

function truthpr(x);
	;;; print truth values as "f" or "t", everything else as normal.
	;;; if given a list of truth-values print it out without brackets.
	;;; if given a list of lists, separate them with newlines.
  if atom(x)
  then	ppr(if istrue(x) then "t" elseif isfalse(x) then "f" else x close)
  else	applist(x,truthpr); pr(newline)
  close
end;

function oppr(wd);
	if	wd == "-"
	then	pr(wd)
	else	ppr(wd)
	close
end;

function bracketpr(form);
	if atom(form)
	then	truthpr(form)
	elseif	hd(form)=="-"
	then	rprint(form)
	else	pr("("); rprint(form); pr(")")
	close
end;

function rprint(form);
	if	atom(form)
	then	truthpr(form)
	elseif hd(form)=="-"
	then	oppr("-");
	bracketpr(hd(tl(form)))
	else	bracketpr(hd(tl(form))),
	oppr(hd(form)), pr(space), bracketpr(hd(tl(tl(form))))
	close
end;

function rsay(form);
	function oppr(wd);
		if	wd=="-"
		then	pr(englishof(wd));
			pr(space)
		else	ppr(englishof(wd))
		close
	end;
	rprint(form);
end;

;;; The remaining functions are for generating a formula "at random".
;;; The function newgen creates the formula and assigns it to currformula.
;;; The complexity of the formula is controlled by the variables which follow.
;;; Some are altered as the program is used.
vars vfreq negfreq complexmax;
	2 ->vfreq;
	5 ->negfreq;
1 -> complexmax;
vars variables binaries;
[[p q r s] [ a b c d e]] ->variables;
[v v v & & & -> -> <->] ->binaries;
function randorder arglist;
	if	oneof([0 1])
	then	arglist
	else	rev(arglist)
	close
end;

vars complexity varlimit neglimit justnegated;		;;; local to newgen
function genform;
	vars vfreq negfreq justnegated;
	vfreq + random(2)*vfreq ->vfreq;
	negfreq + random(2)*negfreq ->negfreq;
   if	complexity >= 1 
   then	complexity - 1 ->complexity;
	if	complexity > 1
	and	random(varlimit) < vfreq
	then	oneof(variables);
		varlimit + 3*complexmax ->varlimit;
	elseif	not(justnegated)
	and	random(neglimit) < negfreq
	then	neglimit + 5*complexmax ->neglimit;
		round(negfreq/2) ->negfreq;
		true -> justnegated;
		[%"-", genform() %]
	else
		false -> justnegated;
		oneof(binaries)::randorder([%genform(), genform()%])
	close
   else	oneof(variables)
   close
end;

function newgen;
	;;; generates formulas a random, with gradually increasing complexity,
	;;; controlled by complexmax. Can be reset to 1 to reduce complexity.
	;;; See the command RS in operation DOLOGIC, below
	vars variables complexity varlimit neglimit;
	complexmax ->complexity;
	oneof(variables) -> variables;
	10 + (2 + random(4))*complexmax ->varlimit;
	50 + (3 + random(8))*complexmax ->neglimit;
	false -> justnegated;
	genform() ->currformula;
	if	complexmax == 1
	then	2
	else	oneof([ 0 1 0]) + complexmax
	close	->complexmax;
end;

macro nf; dl([newgen(); ppr('new formula ready');]) end;

macro rs; dl([1 -> complexmax;]) end;

popmess([%Getpid%]) ->ranseed;
false ->rnoisy;
vars operation 2 dologic;
macro logic; dl([dologic;]) end;
Uses fncomp; Uses exitto; Uses exitfrom;
Uses truthtable;
ppr('\nstill ');

;;; The next two functions are concerned with evaluating a formula.
function evalvar(vallist,varlist,var);
	;;; evaluate a variable using a binding environment defined by
	;;; a list of variables and a list of values
	until hd(varlist)==var then
		tl(varlist) -> varlist;
		tl(vallist) ->vallist;
		if varlist ==[] then error(var, 1, 'variable not in varlist') close
	close;
	istrue(hd(vallist))
end;

function eval(vallist,varlist,form);
	;;; evaluate a structure in an environment defined by varlist and vallist.
	;;; unless it is already a truth-value,
	;;; form is either a variable (an atom) or a list whose first element
	;;; is one of the operators "-", "<->", "->", "&" or "v", "x", and
	;;; whose remaining elements are structures (variables or lists)
	vars op arg1 arg2;
	if	istruthvalue(form)	then form
	elseif	atom(form)	then evalvar(vallist,varlist,form)
	else	dest(form) ->form ->op;
		if	op == "-"
		then	not(eval(vallist,varlist,form(1)))
		elseif	(dl(form) ->arg2 ->arg1;
			     eval(vallist,varlist,arg1) ->arg1;
			     eval(vallist,varlist,arg2) ->arg2;
			op == "<->")
		then	arg2 == arg1
		elseif	op == "->"
		then	not(arg1) or arg2
		elseif	op == "v"
		then	arg1 or arg2
		elseif	op == "x"
		then	not(arg1 == arg2)	;;; exclusive or
		elseif	op == "&"
		then	arg1 and arg2
		elseif	op == "nor"
		then	not(arg1 or arg2)
		elseif	op == "xor"	;;; exclusive "or"
		then	not(arg1 == arg2)
		else	error(0,'unknown operator')
		close
	close
end;

;;; Now some functions for partially evaluating a formula. I.e. only
;;; evaluate one subformula, replacing it by a truth-value, and return
;;; the modified formula.
function replacevars(vallist,varlist,form);
	;;; relace all atoms in the formula with the appropriate truth-values.
	if	atom(form)
	then	evalvar(vallist,varlist,form)
	else	hd(form) ::
			maplist(tl(form),
				lambda(form); replacevars(vallist,varlist, form) end)
	close
end;

ppr('compiling ...');

vars done1 evalslowly;
true -> evalslowly;		;;; Causes printeval to print out more intermediate stages.

function eval1(form)=>form;
	;;; the formula has had all its atoms replaced by truth-values.
	;;; evaluate one sub-expression, and replace it with its truth-value.
	vars list;
	unless	done1 or atom(form)
	then
		if	not(allof(tl(form),istruthvalue))
		then
			for	tl(form) ->list
			step	tl(list) -> list
			till	done1	or list == []
			then	eval1(hd(list)) ->hd(list)
			close
		else	eval([%true,false%],[%true,false%],form) ->form;
			if evalslowly then true ->done1 close;
		close
	close
end;

function printeval(vallist,varlist,form);
	;;; repeatedly replace a part of form with a truth-value got by
	;;; evaluating it, and print out the result, until the whole
	;;; thing has been evaluated.
	vars done1 x;
	1 -> x;		;;; controls indentation
	pr(newline);
	if	allof(tl(form),atom)
	then
		rprint(replacevars(vallist,varlist,form));
		;;; there are no embedded formulas, so just print the value.
		ppr(' is  ');
		truthpr(eval(vallist,varlist,form))
	exit;
	rprint(form);
	replacevars(vallist,varlist,form) ->form;
	pr(newline); rprint(form);
	until	istruthvalue(form)
	then
		false -> done1;
		eval1(form) ->form;
		pr(newline); sp(x);	x + 1 ->x;
		rprint(form)
	close
end;

function varsof(form) =>list;
	;;; form is a formula. Return a list of all its variables.
	function subvars(form);
		if	atom(form)
		then	unless	member(form,list)
			then	form::list ->list
			close
		elseif	hd(form) == "-" then subvars(form(2))
		else	subvars(form(2)), subvars(form(3))
		close
	end;
	[] -> list;
	subvars(form);
	sort(list) ->list;
end;

function finalcolumn(table,varlist,form);
	;;; form is a formula, varlist its variables, table the list of
	;;; possible truth-combinations.
	;;; return a list of truth-values for form, for each combination.
	maplist(table, eval(%varlist,form%))
end;

vars currformula, currvars, currtable, currcolumn;
	;;; currformula is read in using the parsing functions
function getcurrentvars();
	;;; prepare the environment for logical games, assuming a current formula.
	varsof(currformula) ->currvars;
	truthtable(length(currvars)) ->currtable;
	finalcolumn(currtable,currvars,currformula) ->currcolumn;
end;

function prtablefor(form);
	vars currformula currvars currtable currcolumn;
	unless form==currformula
	then 	form -> currformula;
		getcurrentvars()
	close;
	ppr(currvars); ppr('\t\t'); rprint(form);
	pr(newline);
	applist(currtable,
		lambda(l);
			applist(l,truthpr);
			ppr('\t\t\s\s');
			truthpr(dest(currcolumn) ->currcolumn);
			pr(newline)
		end)
end;

function gettruthvalues(mess)=>ll;
	;;; read in a row of truth-values, or possibly just one.
	;;; complain if something other than a truth-value is typed.
	vars list;
	getline(mess) -> ll;
	for	ll -> list
	step	tl(list) ->list
	till 	list == []
	then
		unless istruthvalue(hd(list))
		then	ppr([%hd(list)%' is not a truth-value, try again\n']);
			gettruthvalues(mess) ->ll
		exit;
	close;
	maplist(ll, istrue) -> ll
end;
ppr(' nearly');

function getvalue(truthrow)=>ll;
	;;; truthrow is a set of truth-values for the variables in the
	;;; current formula. This function asks you for the value of the
	;;; current formula.
	function pppr(row);
		if atom(row) then ppr(row)
		else	applist(row,truthpr);
			ppr('\t     ')
		close
	end;
	vars prompt;
	popmess(Prompt) ->prompt;
	'' ->popmess(Prompt);
	gettruthvalues(truthrow) ->ll;
	until length(ll) == 1
	then	gettruthvalues('Only one value please : ') -> ll;
	close;
	prompt ->popmess(Prompt);
	hd(ll) -> ll
end;

function morecomplex;
	;;; make tasks more complex.

	unless complexmax > 10 then oneof([0 1 0]) + complexmax ->complexmax close
end;

function lesscomplex;
	;;; make tasks less complex
	unless complexmax < 3 then complexmax - oneof([0 1]) -> complexmax close
end;

function helptable();
	vars currtable currcolumn mistakes;
	0 -> mistakes;
	for	ppr(['Type in the final column\n'
			^currvars '\t  ']),
		rprint(currformula), pr(newline)
	step	tl(currtable) -> currtable,
		tl(currcolumn) -> currcolumn
	till	currcolumn == []
	then
		unless getvalue(hd(currtable)) == hd(currcolumn)
		then	1 + mistakes -> mistakes;
			ppr('no, because:');
			printeval(hd(currtable),currvars,currformula);
			unless atom(tl(currcolumn)) then ppr('\n try next row\n') close
		close
	close;
	if	mistakes == 0
	then	ppr('** Good, no mistakes'); morecomplex()
	else	ppr('** Oh well, only '><mistakes><' mistakes');
		if	mistakes == 1	then ' not too bad' .ppr
		elseif	mistakes > 3 then lesscomplex() close
	close
end;

;;; Some commands for answering questions about the current formula.

operation 2 answer(x,reply);
	ppr(if x then 'it is ' else 'it isn\'t ' close);
	ppr(reply)
end;

function evalcurrform();
	vars row mistakes interrupt requestrow x ll prompt;
	lambda;	if	mistakes == 0
		then ppr('\ngood, no mistakes !'); morecomplex()
		else	ppr('\n** '><mistakes><' mistake'><
			if mistakes ==1 then '' else 's' close ><
			', altogether');
			if mistakes > 3 then lesscomplex() close;
		close;
	end fncomp interrupt -> interrupt;
	popmess(Prompt) ->prompt;
	'' -> popmess(Prompt);
	0 -> mistakes;
	length(currvars) ->x;
	newline::(currvars <>['\n\s']) -> requestrow;
	ppr('type STOP when fed up\n');
	ppr('the formula is :	'); rprint(currformula);
	ppr('\ngive values for the variables and for the formula');
	l:
		until	(gettruthvalues(requestrow) -> row;
			length(row) ->ll; ll == x or ll == x + 1)
		then	'** Wrong number of truthvalues, try again\n'.ppr
		close;
		if ll == x + 1
		then	dest(rev(row)) ->row ->ll;
			rev(row) -> row
		else	"noval" -> ll
		close;
		;;; if ll isn't "noval" its the value typed in for the whole formula.

		if	eval(row,currvars,currformula)==
				if ll == "noval" then getvalue('The final column? ')
				else ll
				close
		then	ppr('that\'s right')
		else	1 + mistakes ->mistakes;
			'No, because:\n'.ppr; printeval(row,currvars,currformula);
			'\ntry again'.ppr
		close;
	goto l;
end;

macro g;
	;;; ask for the guessing game
	;;; creates a new formula then lets you try to guess it.
	dl([guessform();]);
end;

vars guessform;	;;; defined below.
function getguess;
	vars form formvars;
	function rerror(l);
		ppr(l);
		exitto(getguess)
	end;
	getline('  guess?  ') -> form;
	ll:
	lambda;
		if isno(form) then exitto(guessform)
		elseif	isyes(form)
		then	getline('what\'s your guess? ') -> form
		close;
		form ->input;
		parse() -> form;
		varsof(form) -> formvars;
		if	form = currformula
		then	ppr('\ndead right - well done!\n');
			morecomplex();
			exitfrom(guessform)
		elseif	formvars /= currvars
		then	if	length(formvars) = length(currvars)
			and	finalcolumn(currtable,formvars,form) = currcolumn
			then	ppr('that is equivalent to the formula, namely:\n');
				rprint(currformula);
				ppr('\nassuming you meant: ');ppr(formvars);
				ppr('\nto stand for      : ');ppr(currvars);
				ppr('\nwell done!\n'); exitfrom(guessform);
			else	ppr('    your formula should contain the variables: ');
				ppr(currvars); ppr('  only\n');
				lesscomplex()
			close
		elseif	finalcolumn(currtable,currvars,form) = currcolumn
		then	ppr('good. that\'s equivalent to the formula, namely:\n');
			rprint(currformula);
			exitfrom(guessform)
		close
	end.apply;
	getline('\tanother guess? ') -> form;
  goto ll
end;

ppr('all compiled');

function guessform;
	vars table column;
	newgen();
	until	length(varsof(currformula)) > 1
	then	newgen()
	close;
	ppr('new formula ready. ');
	getcurrentvars();
	currtable ->table; currcolumn ->column;
	ppr('the variables are: '); ppr(currvars);
	ppr('\n here is the table:\n');
	ppr(currvars);
	pr(newline);
	until	column==[]
	then
		applist(hd(table),truthpr);
		ppr(' makes it :\t');
		truthpr(hd(column));
		getguess();
		tl(table) -> table;
		tl(column) -> column
	close;
	ppr('here\'s the formula:\n'); rprint(currformula);
end;

vars complexityforeq;
2 ->complexityforeq;

function testeq();
	;;; generate a formula and ask for an equivalent one to be typed in.
	vars variables complexmax form formvars interrupt mistakes tabledone;
	function rerror(l);
		ppr(l);
		repeat stacklength() times erase() close;
		[];		;;; a result for parse
		exitto(testeq)
	end;
	ppr(%'type tb if you want the formula\'s truth-table'%)
		fncomp interrupt ->interrupt;
	[[p q][p q]] -> variables;	;;; restrict formulas to two variables
	ppr('You\'ll be given a formula. Try typing in an equivalent one\n');
	start:
	complexityforeq ->complexmax;
	unless complexityforeq >=10 then oneof([1 0 0 0]) + complexityforeq -> complexityforeq close;
	complexityforeq ->complexmax;
	[] -> currformula;
	while	atom(currformula)
	then
		newgen()
	close;
	getcurrentvars();
	ppr(':  this is the formula:\n'); rprint(currformula);
	false ->tabledone;
	if	isyes(getline('\nwant its truth-table?') ->>input)
	then	prtablefor(currformula); true -> tabledone
	elseif	isno(input)
	then	[] ->input
	close;
	ll:
	if	length(input) < 2
	then	getline('equivalent formula?') ->input
	close;
	if	isno(input)
	then	interrupt()
	close;
	parse() ->form;
	if	form = []
	then	ppr('	bad luck\n')
	elseif	(varsof(form) ->>formvars) = currvars
	then	if	finalcolumn(currtable,formvars,form) = currcolumn
		then	ppr('good. that\'s equivalent');
			morecomplex();
			if	not(tabledone) and isyes(getline('want its truthtable? '))
			then	prtablefor(currformula)
			close;
			if	isyes(getline('want to try another? '))
			then	goto start
			else exit
		else	ppr('not quite. ');
			if	isyes(getline('want the truth-table of your formula?  '))
			then	prtablefor(form)
			close
		close
	else	ppr('variables not right '); lesscomplex()
	close;
	goto ll
end;

macro val;
	;;; provides exercises in checking validity
	dl([inferences();])
end;

vars inferencecomplexity; 1 ->inferencecomplexity;

function inferences();
	vars complexmax premisses antecedent conclusion numofpremisses printing ll variables;
	ll:
	unless	inferencecomplexity > 4
	then	oneof([0 1 0]) + inferencecomplexity -> inferencecomplexity
	close;
	inferencecomplexity -> complexmax;
	if complexmax < 3 then [[p q][p q]]
	else [[p q r][p q]]
	close -> variables;
	while (hd(getline('How many premisses?')) -> numofpremisses;
		not(isinteger(numofpremisses)))
	then	ppr('a number please\n')
	close;
	if	numofpremisses > 5
	then	ppr('that\'s too many. settle for 5');
		5 -> numofpremisses
	close;
	isyes(getline('do you want their individual truth-tables?')) ->printing;
	ppr('here are the premisses:\n');
	vars premissesok;
	false ->premissesok;
	until premissesok
	then
		inferencecomplexity ->complexmax;
 		[%repeat numofpremisses times
			newgen();
			currformula
		close%] -> premisses;
		;;; now check that the premisses are not inconsistent
		hd(premisses) -> antecedent;
		applist(tl(premisses), lambda x; [ & ^antecedent ^x] ->antecedent end);
		antecedent -> currformula;
		getcurrentvars();
		member(true,currcolumn) -> premissesok
	close;
	applist(premisses,
		lambda x;
			if	printing
			then	x -> currformula;
				getcurrentvars();
				prtablefor(x);
				ppr('\n-------------------\n')
			else	ppr('   '); rprint(x); pr(newline);
			close
		end);
	ppr('\n and the conclusion:\n');
	newgen() ;currformula ->conclusion;
	if	printing
	then	getcurrentvars(); prtablefor(currformula)
	else	ppr('   '); rprint(currformula); pr(newline)
	close;
	[%"->",antecedent,conclusion%] ->currformula;
	getcurrentvars();
	if	isno(getline('do you think the inference is valid?'))
	then	ppr('give values of variables making premisses true and conclusion false\n');
		ppr(currvars);
		popmess(Prompt); ' ' ->popmess(Prompt);
		gettruthvalues(newline) ->ll;
		while	length(ll) /== length(currvars)
		then	gettruthvalues('wrong number of truthvalues, try again\n') ->ll
		close;
		-> popmess(Prompt);
		if	eval(ll,currvars,currformula)
		then	ppr('that set doesn\'t invalidate the inference, because');
			printeval(ll, currvars, currformula);
			pr(newline)
		close
	close;
	if	member(false,currcolumn)
	then	ppr('it is invalid.  ')
	else	ppr('it is valid.  ')
	close;
	if	isyes(getline('want the complete truth-table?'))
	then	prtablefor(currformula)
	close;
	if	isyes(getline('want to try another?'))
	then	goto ll
	close
end;

vars prompt;	popmess(Prompt) ->prompt;
operation 2 dologic;
	vars errfun prompt oldform ll;
	 function interrupt;
		vars errfun interrupt;
		syserr ->errfun;
		setpop ->interrupt;
		prompt -> popmess(Prompt);
		repeat stacklength() times erase() close;
		ppr('\n*** restarting, ');
		exitto(nonop dologic);
	 end;
	getcurrentvars();
	rerror(%'something has gone wrong\n'%) ->errfun;

  ll:
     lambda;
	currformula ->oldform;
	getline('\ntype command or formula') ->ll;
	if	length(ll)==1
	and	isword(hd(ll)) and identprops(hd(ll)) == "macro"
	then	popval(ll);
	elseif	ll = [h]
	then	compile('/usr/lib/helplogic.p')
	elseif	ll = [rs]
	then	1 -> complexmax
	elseif	ll = [p]
	then	rprint(currformula); pr(newline)
	elseif	ll = [s]
	then	rsay(currformula); pr(newline)
	elseif	ll = [ht]
	then	helptable()
	elseif	ll = [eq]
	then	testeq()
	elseif	ll = [tb]
	then	prtablefor(currformula)
	elseif	ll=[ta]
	then	not(member(false,currcolumn)) answer 'a tautology'
	elseif ll = [in]
	then	not(member(true,currcolumn)) answer 'inconsistent'
	elseif	ll = [co]
	then	(member(true,currcolumn) and member(false,currcolumn)) answer 'contingent'
	elseif ll = [ev]
	then	evalcurrform()
	else	ll -> input;
		parse() -> currformula;
		ppr('** OK')
	close;
	unless	currformula == oldform
	then	getcurrentvars()
	close
     end.apply;
  goto ll
end;
ppr('\nall compiled at last\n');
