#include <param.h>
#include <systm.h>
#include <acct.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <inode.h>
#include <proc.h>
#include <seg.h>
#include <map.h>
#include <reg.h>
#include <buf.h>

extern struct map coremap[];
extern struct map swapmap[];
extern struct map umap[];
extern struct user u;

/*
 * Icode is the octal bootstrap
 * program executed in user mode
 * to bring up the system.
 */
int	icode[] =
{
	0104413,	/* sys exec; init; initp */
	0000014,
	0000010,
	0104401,	/* sys exit */
	0000021,	/* initp: init+5; 0 */
	0000000,
	0062457,	/* init: </etc/init\0> */
	0061564,
	0064457,
	0064556,
	0000164,
};
int	szicode = sizeof(icode);

/*
 * Machine-dependent startup code
 */
startup()
{
	register i, j;

	/*
	 * zero and free all of core
	 */

	i = ka6->r[0] + USIZE;
	UISD->r[0] = 077406;
	for(;;)
	{
		UISA->r[0] = i;
		if(fuibyte((caddr_t)0) < 0)
			break;
		clearseg(i);
		maxmem++;
		mfree(coremap, 1, i);
		i++;
	}
	/*
	 * Unfortunately, the 11/70 Stack Limit Register is very coarse
	 * (256 byte increments) which splits the PPDA into only 4 areas.
	 * Examination of core dumps reveals that the average stack depth
	 * is about 0500 bytes (0142000 to 0141300). Hence the size of the
	 * user structure must not exceed 0776 bytes.
	 */
	if (tkword(SLR))
		SLR->r[0] = &u.u_stack[0];
	/*
	 * Allocate memory for the buffer pool.
	 */
	if ((bufarea = malloc(coremap, NBUF * btoc(BSIZE))) == NULL)
		panic("No core for buffers");
	maxmem -= NBUF * btoc(BSIZE);			/* buffer pool */
	/*
	 * Now set up the unibus map to point to the buffer pool.
	 * This is purely for the benefit of any unibus DMA devices.
	 * Unfortunately it restricts the size of the buffer pool to
	 * 2^18-2^13 bytes i.e. NBUF must not exceed 16 * 31 = 496!
	 * Note also that the order of shorts in a unibus map register
	 * are opposite to the normal order in a long.
	 */
	{
		register unsigned *up;
		long l;

		l = (long)bufarea;
		l <<= 6;
		up = (unsigned *)UBMAP;
		j = (NBUF + 15) / 16;	/* 16 buffers / unibus map register */
		for (i = 0; i < j; i++)
		{
			*up++ = l & 0177777L;
			*up++ = (l >> 16) & 0177777L;
			l += 020000;
		}
	}
	/*
	 * Now free the remaining unibus map registers.
	 */
	mfree(umap, 31 - j, j + 1);	/* 31 unibus registers */

	printf("mem = %D\n", ctob((long)maxmem));
	if (MAXMEM < maxmem)
		maxmem = MAXMEM;
	mfree(swapmap, nswap, 1);
	swplo--;
}

/*
 * Determine which clock is attached, and start it.
 * panic: no clock found
 */
#define	CLOCK1	((physadr)0177546)	/* line clock */
#define	CLOCK2	((physadr)0172540)	/* programmable clock */
clkstart()
{
	lks = CLOCK1;
	if(tkword((caddr_t)lks) == 0)
	{
		lks = CLOCK2;
		if(tkword((caddr_t)lks) == 0)
			panic("no clock");
	}
	lks->r[0] = 0115;
}

/*
 * Let a process handle a signal by simulating an interrupt
 */
sendsig(p, signo)
caddr_t p;
{
	register unsigned n;

	n = u.u_ar0[R6] - 4;
	grow(n);
	suword((caddr_t)n+2, u.u_ar0[RPS]);
	suword((caddr_t)n, u.u_ar0[R7]);
	u.u_ar0[R6] = n;
	u.u_ar0[RPS] &= ~TBIT;
	u.u_ar0[R7] = (int)p;
}

/*
 * 11/70 routine to allocate the UNIBUS map and initialize for a unibus device.
 * A part of the UNIBUS map is set up to point to the physical buffer address.
 * The buffer header is then altered to use a unibus virtual address.
 */
int	maplock;

mapalloc(bp)
register struct buf *bp;
{
	register unsigned *up;
	register nreg;
	short a;
	long l;

	if (cputype != 70)
		return;
	nreg = (bp->b_bcount >> 13) & 07;
	spl6();
	while ((a = malloc(umap, nreg)) == NULL)
	{
		maplock = B_WANTED;
		sleep((caddr_t)&maplock, PSWP+1);
	}
	spl0();
	l = (long)bp->b_xmem << 16;
	l = l + (long)bp->b_addr;
	bp->b_addr = (--a) << 13;
	bp->b_xmem = (a >> 3) & 03;
	bp->b_flags |= B_MAP;
	up = ((unsigned *)UBMAP)->r[a << 1];
	do
	{
		*up++ = l & 0177777L;
		*up++ = (l >> 16) & 0177777L;
		l += 020000;
	} while (--nreg);
}

mapfree(bp)
struct buf *bp;
{
	register unsigned nreg, a;

	nreg = (bp->b_bcount >> 13) & 07;
	a = (((bp->b_addr >> 13) & 07) | (bp->b_xmem << 3)) + 1;
	mfree(umap, nreg, a);
	bp->b_flags &= ~B_MAP;
	if (maplock)
	{
		maplock = 0;
		wakeup((caddr_t)&maplock);
	}
}

/*
 * return special device number given inode address
 */
dev_t
getdevno(ip)
register struct inode *ip;
{
	return(makedev(ip->i_addr[2], ip->i_addr[1]));
}

/*
 * set special device number
 */
setdevno(ip, dev)
register struct inode *ip;
dev_t dev;
{
	ip->i_addr[2] = major(dev);
	ip->i_addr[1] = minor(dev);
}

/*
 * return block address given inode address and index
 */
daddr_t
getbno(ip, index)
register struct inode *ip;
short index;
{
	register char *cp0, *cp1;
	daddr_t temp;

	cp0 = &ip->i_addr[index * 3];
	cp1 = (char *)&temp;
	*cp1++ = *cp0++;
	*cp1++ = 0;
	*cp1++ = *cp0++;
	*cp1++ = *cp0++;
	return(temp);
}

/*
 * set block address
 */
setbno(ip, index, bno)
register struct inode *ip;
short index;
daddr_t bno;
{
	register char *cp0, *cp1;

	cp0 = &ip->i_addr[index * 3];
	cp1 = (char *)&bno;
	*cp0++ = *cp1++;
	cp1++;
	*cp0++ = *cp1++;
	*cp0++ = *cp1++;
}
