/*
**	Netd - network daemon
**
**	"netd [-O] hostname"
**
**	Bugs and comments to:	Piers Lauder
**				Dept of Comp Sci
**				Sydney University
**	June '80.
*/

#define	HZ	50
#include	<local-system>
#ifdef	Aus2
#include	<types.h>
#include	<stat.h>
#include	<dir.h>
#endif	Aus2
#ifdef	Aus1
#include	<stat16.h>
#include	<dir.h>
#endif	Aus1
#include	<signal.h>
#include	<setjmp.h>
#include	<stdio.h>
#include	<sgtty.h>
#include	<errno.h>
#ifdef	Aus2
#include	<fcntl.h>
#endif	Aus2
#ifdef	Aus1
#include	"lvl7.h"
#endif	Aus1
#include	"net.h"
#include	"netd.h"
#include	"neth.h"


char		net[]			= BIN_NET;	/* network spooler */
char		sh[]			= BIN_SH;	/* command interpreter */

#define		NETSTATE	3			/* state byte in name */
#define		NETACTIVE	'd'
#define		NETINACTIVE	'0'
#define		NETERROR	'E'
#define		NETSLEEPING	'S'
#define		NETOPENING	'o'

char		*name;					/* pointer to daemon name ( last char reflects status ) */
char		*remote;				/* remote host name */
char		spooldir[]		= SPOOLDIR;
#define		netdir			remote
FILE *		netdirfd;				/* file desc. of netdir */
char		lockfile[(DIRSIZ+1)+HOSTSIZE];
int		lockfd;					/* file desc. of lockfile */
struct lock	lokk;					/* contents of lock file */
char		netdev[2*(DIRSIZ+1)+HOSTSIZE]	= NETDEV;
int		net_fd;					/* file descriptor of net */
char		tempfile[(DIRSIZ+1)+HOSTSIZE];
int		tempfd;
char		logfile[(DIRSIZ+1)+HOSTSIZE];

enum f_states
{
	f_ok, f_stop, f_restart
}
		filestate;				/* cancel file flag */

enum up_types
{
	up_null, up_sent, up_rcvd, up_file, up_size
};

short		repf;					/* file repeated */
short		relinqf;				/* set when net not to be used */
struct sgttyb	netparams;				/* for stty calls */
long		files_sent;				/* files transmitted */
long		files_rcvd;				/* files received */
enum sync_proto	direction;				/* starts off sending */
jmp_buf		gg_jb, dto_jb;
char		exec_fail[]		= "can't exec %s";

extern char *	strcat();
extern char *	strncat();
extern int	strncmp();
extern char *	strcpy();
extern char *	strncpy();
extern char *	strrchr();
extern long	atol();
extern long	lseek();
extern char *	ctime();
extern long	time();
extern int	errno;


/*
**	Signals
*/

extern int	getgoing(), eflag(), restart(), repeat(), relinquish()
		,acquire(), shutdown(), togtrace();

struct csig
{
	int	s_signo;
	int	(*s_action)();
}
	csigs[] =
{
	{	SIGHUP,		SIG_IGN		}
	,{	SIGINT,		SIG_IGN		}
	,{	SIGQUIT,	SIG_IGN		}
	,{	NS_GETGOING,	SIG_IGN		}
	,{	NS_STOP,	eflag		}
	,{	NS_RESTART,	restart		}
	,{	NS_REPEAT,	repeat		}
	,{	NS_RELINQUISH,	relinquish	}
	,{	NS_AQUIRE,	acquire		}
	,{	SIGTERM,	shutdown	}
	,{	SIGSYS,		togtrace	}
};

#define	NCSIGS	(sizeof csigs/sizeof csigs[0])





main( argc, argv )
	register		argc;
	register char		*argv[];
{
	register struct csig	*sp;
	time_t			ltime;

	nice(-1);		/* some help */
	name = argv[0];		/* pointer to name */

	if ( argc >= 2 )
	{
		if ( argc == 3 )
		{
			if ( strcmp(argv[1], "-O") )
			{
				remote = "?";
				warn("unrecognised flag");
				return 1;
			}
			Oldnetd++;
		}
		remote = argv[argc-1];
	}
	else
	{
		remote = "?";
		warn("remote system name?");
		return 1;
	}

	fclose(stdin);
	fclose(stdout);

	strcat(strcpy(logfile, remote), LOGFILE);
	strcat(strcpy(tempfile, remote), TMPFILE);
	strcat(strcpy(lockfile, remote), LOCKFILE);
	strcat(netdev, remote);

	if( (chdir(spooldir) == SYSERROR) || ((netdirfd = fopen(netdir, "r")) == NULL) )
	{
		warn("%s%s inaccessible", spooldir, netdir);
		return 1;
	}

#	ifdef	Aus2
	umask(2);
#	endif	Aus2

	if( (lockfd = open(lockfile, O_RDWR)) != SYSERROR)
	{
		if( read(lockfd, (char *)&lokk, sizeof lokk) != sizeof lokk
			|| lokk.lk_pid == 0
			|| kill(lokk.lk_pid, 0) == SYSERROR
		  )
		{
			lseek(lockfd, (long)0, 0);
			lokk.lk_rcvd = 0;
			lokk.lk_sent = 0;
		}
		else
		{
			warn("%s: already active", remote);
			return 1;
		}
	}
	else
	{
		errno = 0;
#		ifdef	Aus2
		if ( mknod(lockfile, S_IFALK|0664, 0) == SYSERROR )
#		endif	Aus2
#		ifdef	Aus1
		if ( (lockfd = creat(lockfile, S_IFALK|0604)) == SYSERROR )
#		endif	Aus1
		{
			warn("cannot mknod %s", lockfile);
			return 1;
		}
#		ifdef	Aus1
		close(lockfd);
#		endif	Aus1

		chown(lockfile, DAEMONUID, DAEMONGID);
		lockfd = open(lockfile, O_RDWR);
	}

#	ifdef	Aus2
	setgid(DAEMONGID);
#	endif	Aus2
	setuid(DAEMONUID);

	switch ( lokk.lk_pid = fork() )
	{
	 default:
		lokk.lk_state = net_idle;
		if ( write(lockfd, (char *)&lokk, sizeof lokk) != sizeof lokk )
			warn("cannot write %s", lockfile);
		return 0;

	 case SYSERROR:
		warn("can't fork");
		return 1;

	 case 0:;	/* child will not inherit lock */
	}

	/* daemon now ready to talk */

	if ( lokk.lk_file[0] == '\0' || access(tempfile, 6) == SYSERROR )
	{
		lokk.lk_file[0] = '\0';
#		ifdef	Aus2
		if ( (tempfd = open(tempfile, O_CREAT|O_TRUNC|O_RDWR, 0600)) == SYSERROR )
#		endif	Aus2
#		ifdef	Aus1
		if ( (tempfd = creat(tempfile, 0600)) == SYSERROR )
#		endif	Aus1
		{
			warn("can't create %s", tempfile);
			return 1;
		}
		close(tempfd);
	}
	else
		if ( lokk.lk_state == net_recv )
			direction = recv_pro;	/* attempt restart */

	freopen(logfile, "a", stderr);
	errno = 0;
	chmod(logfile, 0664);

	ltime = time((long *)0);
	sync_time = ltime;

	warn("STARTED");

	for ( sp = csigs ; sp < &csigs[NCSIGS] ; sp++ )
		signal( sp->s_signo, sp->s_action );

#	ifdef	Aus2
	setpgrp();
#	endif	Aus2

	netchange('U');

	for ( ;; )
	{
		open_net();	/* Open link */

		while ( relinqf == 0 )
		{
			static short	wasdown;

			if ( direction == recv_pro )
			{
				if ( recv() && !hostdown )
					direction = send_pro;
			}
			else
			{
				if ( send() )
					direction = recv_pro;
			}

			if ( hostdown )
			{
				if ( !wasdown )
				{

					errno = 0;
					warn("host down");

					netchange('D');
					updtlock(net_down, up_null);

					wasdown = 1;
				}
				continue;
			}
			else
				wasdown = 0;

			if ( (sync_time - ltime) > ((long)24*60*60) )
			{
				ltime += (long)24*60*60;
				stats("check-point");
			}
		}

		close(net_fd);
		updtlock(net_closed, up_null);

		while( relinqf && (name[NETSTATE] = NETINACTIVE) )
			pause();
	}
}



/*
**	Netchange - exec shell on netchange command file
*/

netchange(type)
	char		type;
{
	char		temp[200];

	switch ( fork() )
	{
	 case 0:
		closeall();
		sprintf(temp, "%s -S%s -%c%s -C%s"
				,NETCHANGE
				,spooldir
				,type
				,remote
				,NETID
			);
		execl(sh, sh+5, "-c", temp, 0);
		warn(exec_fail, sh);
		exit(1);

	 case SYSERROR:
		warn("can't fork");
		break;

	 default:
		netwait();
	}
}



/*
**	Send - send jobs found described by files in netdir
*/

send()
{
	register FILE *	dfd;
	register char	*dfn;
	register	spoolfd;
	int		nogo;
	int		ncopies;
	int		ncsave;
	struct netr	netr;
	char		mesg[NETDATASIZE+1];
	uid_t		uid;
	extern char	*next();

	if ( (dfn = next(lokk.lk_file)) != NULL )
	{
		char *	rf = strrchr(dfn, '/')+1;

		if ( lokk.lk_file[0] != '\0' && strcmp(lokk.lk_file, rf) != 0 )
			lokk.lk_file[0] = '\0';

		uid = (uid_t)-1;
		nogo = 0;
		repf = 0;
		filestate = f_ok;
		ncsave = 1;
		mesg[0] = 0;

		while ( (dfd = fopen(dfn, "r")) == NULL )
		{
			if ( warn("cannot read %s", dfn) != ENFILE )
			{
				unlink(dfn);
				return 1;
			}
			sleep(60);
		}

		while ( fread((char *)&netr, sizeof netr, 1, dfd) == 1 )
		{
			ncopies = ncsave;

			switch( netr.nr_flag )
			{
			 case NC_MAIL:
			 case NC_FILE:
			 case NC_PRINT:
			 case NC_HSTUP:
			 case NC_HSTDN:
			 case NC_REMOVE:
			 case NC_STATE:
				if ( nogo )
break2:					break;

				while ((spoolfd = open( netr.nr_data, O_READ)) == SYSERROR)
				{
					if ( warn("can't open %s", netr.nr_data) != ENFILE )
						goto break2;
					sleep(60);
				}

				do
				{
					if ( !sync(send_pro, spoolfd, rf) )
					{
						close(spoolfd);
						fclose(dfd);
						return lokk.lk_file[0] == '\0';
					}

					if ( lokk.lk_file[0] != '\0' )
						warn("restart of %.14s succeeded", lokk.lk_file);

					updtlock(net_send, up_file, rf);

					if ( xproto(spoolfd) != r_ok )
					{
						updtlock(net_error, up_null);
						close(spoolfd);
						fclose(dfd);
						if ( !Oldnetd )
						{
							warn("transmission of %.14s aborted", lokk.lk_file);
							return 0;		/* xfer failed */
						}
						else
							return 1;		/* can't restart anyway */
					}

					files_sent++;
					updtlock(net_idle, up_sent, (char *)&proto_bytes);

					if( repf )
					{
						ncopies++;
						ncsave++;
						repf--;
					}

					if ( filestate == f_restart )
					{
						filestate = f_ok;
						repf = 0;
						ncopies = ncsave;
					}
				}
				while
					( --ncopies && filestate == f_ok );

				filestate = f_ok;

				close(spoolfd);

				break;

			 case NC_UNLINK:
				if ( unlink(netr.nr_data) == SYSERROR )
					warn("can't unlink %s", netr.nr_data);
				break;

			 case NC_SIZE:
				{
					long	l;

					l = atol(netr.nr_data);
					updtlock(net_idle, up_size, (char *)&l);
				}
				break;

			 case NC_MESSAGE:
				strncpy( mesg, netr.nr_data, NETDATASIZE );
				mesg[NETDATASIZE] = '\0';
				break;

			 case NC_UID:
				uid = (uid_t)atol(netr.nr_data);
				break;

			 case NC_TTYNAME:
				{
					char		ttyname[NETDATASIZE + 1];
					FILE		*fd = NULL;
					extern int	dtimeout();

					strncpy(ttyname, netr.nr_data, NETDATASIZE);
					ttyname[NETDATASIZE] = '\0';

					if ( setjmp(dto_jb) )
					{
						if ( fd != NULL )
							fclose(fd);
					}
					else
					{
#						ifdef	Aus2
						struct stat	statbuf;

						if ( stat(ttyname, &statbuf) == SYSERROR || statbuf.st_uid != uid )
							break;
#						endif	Aus2
#						ifdef	Aus1
						struct statbuf	statbuf;

						if ( newstat(ttyname, &statbuf) == SYSERROR || statbuf.sb_uid != uid )
							break;
#						endif	Aus2

						signal(SIGALRM, dtimeout);
						alarm(5);
						if ( (fd = fopen(ttyname, "w")) != NULL )
						{
							fprintf(fd , "\007\n%s: %s sending to host %s.\n"
								,mesg
								,(nogo||filestate==f_stop)?"stopped":"finished"
								,remote
							);
							fclose( fd );
						}
						alarm(0);
					}
				}
				break;

			 case NC_NOGO:
				nogo++;
				break;
			}
		}

		fclose(dfd);
		unlink(dfn);
	}

	return lokk.lk_file[0] == '\0';
}



dtimeout()
{
	longjmp(dto_jb, 1);
}



/*VARARGS2*/
updtlock(state, uptype, valp)
	enum netstates	state;
	enum up_types	uptype;
	char *		valp;
{
	writelock(lockfd);
	if ( state != net_inactive )
	{
		lseek(lockfd, (long)0, 0);
		read(lockfd, (char *)&lokk, sizeof lokk);
	}
	else
	{
		lokk.lk_pid = 0;
		if ( lokk.lk_state != net_idle )
			state = lokk.lk_state;
	}
	switch ( uptype )
	{
	 case up_null:	break;
	 case up_sent:
		lokk.lk_sent += *(long *)valp;
		*(long *)valp = 0;
		lokk.lk_file[0] = '\0';
		break;
	 case up_size:
		if ( (lokk.lk_size -= *(long *)valp) < 0 )
			lokk.lk_size = 0;
		break;
	 case up_rcvd:
		lokk.lk_rcvd += *(long *)valp;
		*(long *)valp = 0;
		lokk.lk_file[0] = '\0';
		break;
	 case up_file:
		strncpy(lokk.lk_file, valp, sizeof lokk.lk_file);
		break;
	}
	if ( proto_time )
		lokk.lk_rate = (lokk.lk_rcvd+lokk.lk_sent)/proto_time;
	lokk.lk_state = state;
	lseek(lockfd, (long)0, 0);
	write(lockfd, (char *)&lokk, sizeof lokk);
	unlock();
}


/*
**	next - find lexically smallest file name
**		returns NULL if nothing to do.
*/

char *
next(fn)
	char *			fn;	/* last active file name */
{
	register struct direct	*d1, *d2;
	register		f;
	struct direct		dirb[2];
	static char		dname[DIRSIZ + 1 + DIRSIZ + 1];

	sndflg = 0;

	rewind(netdirfd);
	d1 = &dirb[0];
	d2 = &dirb[1];

	while ( (f = fread((char *)d1, sizeof *d1, 1, netdirfd)) == 1 )
		if ( d1->d_ino && d1->d_name[0] != '.' )
			break;

	if ( f != 1 )
	{
		if ( lokk.lk_state == net_recv || lokk.lk_state == net_send || lokk.lk_size )
			updtlock(net_idle, up_size, (char *)&lokk.lk_size);
		return NULL;
	}

	sndflg = 1;

	if ( fn[0] == '\0' || strncmp(fn, d1->d_name, DIRSIZ) != 0 )
		while ( fread((char *)d2, sizeof *d2, 1, netdirfd) == 1 )
		{
			if ( d2->d_ino == 0 || d2->d_name[0] == '.' )
				continue;
	
			sndflg++;

			if ( fn[0] != '\0' && strncmp(fn, d2->d_name, DIRSIZ) == 0 )
			{
				d1 = d2;
				break;
			}

			if ( strncmp(d1->d_name, d2->d_name, DIRSIZ) > 0 )
			{
				f = (int)d1;
				d1 = d2;
				d2 = (struct direct *)f;
			}
		}

	return strncat(strcat(strcpy(dname, netdir), "/"), d1->d_name, DIRSIZ);
}


/*
**	Signal catching routines
*/

getgoing()
{
	signal(NS_GETGOING, SIG_IGN);
	longjmp(gg_jb, 1);
}

eflag()
{
	signal(NS_STOP,eflag);
	filestate = f_stop;
}

restart()
{
	signal(NS_RESTART, restart);
	filestate = f_restart;
}

repeat()
{
	signal(NS_REPEAT, repeat);
	repf++;
}

relinquish()
{
	signal(NS_RELINQUISH, relinquish);
	relinqf = 1;
}

acquire()
{
	signal(NS_AQUIRE, acquire );
	relinqf = 0;
}

shutdown()
{
	signal(SIGTERM, SIG_IGN);
	signal(NS_GETGOING, SIG_IGN);
	alarm(0);
	if ( lokk.lk_size == 0 && lokk.lk_file[0] == '\0' )
	{
		unlink(tempfile);
		unlink(lockfile);
	}
	else
		updtlock(net_inactive, up_null);
	stats("SHUTDOWN");
	exit(0);
}

stats(reason)
	char *	reason;
{
	struct
	{
		time_t	t_ut, t_st, t_cut, t_cst;
	}
		tbuf;

	errno = 0;
	warn(reason);
	fprintf(stderr, "%ld files received (%ld bytes)\n%ld files transmitted (%ld bytes)\n"
			,files_rcvd
			,lokk.lk_rcvd
			,files_sent
			,lokk.lk_sent
		);
	errors();
	times(&tbuf);
	fprintf(stderr, "Cpu secs: sys %ld, user %ld, child sys %ld, child user %ld. Total: %ld secs\n"
			,(tbuf.t_st)/HZ
			,(tbuf.t_ut)/HZ
			,(tbuf.t_cst)/HZ
			,(tbuf.t_cut)/HZ
			,(tbuf.t_st + tbuf.t_ut + tbuf.t_cst + tbuf.t_cut)/HZ
		);
	fflush(stderr);
}



/*
**	Open network link
*/

int
open_net()
{

	name[NETSTATE] = NETOPENING;
	updtlock(net_opening, up_null);

	while ( (net_fd = open(netdev, O_RDWR)) == SYSERROR && (name[NETSTATE] = NETERROR) )
	{
		updtlock(net_error, up_null);
		sleep(60);
	}

	updtlock(net_idle, up_null);

	/*
	**  Set up network characteristics
	*/

	gtty( net_fd, &netparams );
	netparams.sg_ispeed = netparams.sg_ospeed = B9600;
	netparams.sg_flags = RAW;
	stty( net_fd, &netparams );

	name[NETSTATE] = NETACTIVE;
}



/*
**	Receive job from network - pass to net
*/

recv()
{
	char	temp[sizeof lokk.lk_file];

	netwait();

again:
#	ifdef	Aus2
	tempfd = open(tempfile, (lokk.lk_file[0] == '\0')?O_RDWR|O_TRUNC:O_RDWR);
#	endif	Aus2
#	ifdef	Aus1
	if ( lokk.lk_file[0] == '\0' )
	{
		tempfd = creat(tempfile, 0600);
		close(tempfd);
	}
	tempfd = open(tempfile, O_RDWR);
#	endif	Aus1

	if ( tempfd == SYSERROR )
	{
		if ( warn("can't access %s", tempfile) != ENFILE )
			shutdown();
		sleep(60);
		goto again;
	}

	if ( !sndflg && lokk.lk_file[0] == '\0' )
		if ( setjmp(gg_jb) )
		{
			close(tempfd);
			return 1;
		}
		else
			signal(NS_GETGOING, getgoing);

	if ( sync(recv_pro, tempfd, temp) )
	{
		if ( !sndflg )
			signal(NS_GETGOING, SIG_IGN);

		if ( lokk.lk_file[0] != '\0' )
			warn("restart of %.14s succeeded", lokk.lk_file);

		updtlock(net_recv, up_file, temp);

		if ( rproto(tempfd) == r_ok )
		{
			switch ( fork() )
			{
			 case SYSERROR:	warn("can't fork!");
					break;
			 case 0:	close(0);
					dup(tempfd);
					close(1);
					lseek(0, (long)0, 0);
					execl(net, net+5, "-x", 0);
					warn("can't exec %s", net);
					exit(1);
			}

			files_rcvd++;
			updtlock(net_idle, up_rcvd, (char *)&proto_bytes);
			if ( !sndflg )
				netwait();
		}
		else
		{
			updtlock(net_error, up_null);
			close(tempfd);
			if ( !Oldnetd )
			{
				warn("receive of %.14s aborted", lokk.lk_file);
				return 0;	/* transfer aborted */
			}
			else
				return 1;	/* can't restart anyway */
		}
	}
	else
		if ( !sndflg )
			signal(NS_GETGOING, SIG_IGN);

	close(tempfd);
	return lokk.lk_file[0] == '\0';
}



/*
**	Netwait - for children
*/

netwait()
{
	int	state;

	errno = 0;

#	ifdef	Aus2
	while ( wait(&state) != SYSERROR )
#	endif	Aus2
#	ifdef	Aus1
	while ( waitx(&state) != SYSERROR )
#	endif	Aus1
	{
#		ifdef	Aus1
		fseek(stderr, (long)0, 2);
#		endif	Aus1
		if ( state & 0xff )
			warn("child died signal %d", state & 0xff);
		if ( (state>>8) & 0xff )
			warn("child error exit %d", (state>>8) & 0xff );
	}

	errno = 0;
}



closeall()
{
	register	i;

	close(0);	open("/dev/null", 0);
	close(1);	dup(2);

	for ( i = 3 ; ; i++ )
		if ( close(i) == SYSERROR )
			break;

	errno = 0;
}



/*
**	Warning message on stderr
*/

/*VARARGS1*/
int
warn(s, a, b)
	char *		s;
	char *		a;
	char *		b;
{
	register int	r;
	time_t		ltime = time((long *)0);

#	ifdef	GMT
	ltime += (long)GMT*60l*60l;
#	endif	GMT
	fprintf(stderr, "%s for %s @ %.15s: "
			,name
			,remote
			,ctime(ltime)+4
		);
	fprintf(stderr, s, a, b);
	if ( r = errno )
	{
		fflush(stderr);
		perror("\07");
	}
	else
	{
		putc('\n', stderr);
		fflush(stderr);
	}
	errno = 0;
	return r;
}



/*
**	Toggle trace output
*/

int	traceflag;

togtrace()
{
	signal(SIGSYS, togtrace);
	traceflag = !traceflag;
}
